/*
 * Loader tool
 *
 * Copyright (C) 2017  Appear TV AS - http://www.appeartv.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "btbl_parser.h"
#include "control.h"
#include "memory.h"
#include "utils.h"

#include <stdio.h>
#include <stdlib.h>


static void print_usage(char *arg0)
{
    printf("Usage: %s <boot table binary> -core<0-7>\n", arg0);
}


static dsp_core_id_t parse_for_core_id(char *arg)
{
    int temp;

    if (sscanf(arg, "-core%d", &temp) != 1) {
        return INVALID_CORE;
    }

    if ((temp < 0) || (temp > 7)) {
        return INVALID_CORE;
    }

    return CORE0 + temp;
}


int main(int argc, char **argv)
{
    dsp_core_id_t core_id;
    dsp_mem_type_t entry_mem_type;
    uint32_t i;
    int ret_val = 0;
    struct table_section section_temp;
    struct boot_table table;
    uint8_t *table_data_ptr;

    if (argc < 3) {
        print_usage(argv[0]);
        return -1;
    }

    core_id = parse_for_core_id(argv[2]);
    if (core_id == INVALID_CORE) {
        printf("Invalid core specified: %s\n", argv[2]);
        print_usage(argv[0]);
        return -1;
    }

    table_data_ptr = read_btbl_binary(&table, argv[1]);
    if (table_data_ptr == NULL) {
        print_usage(argv[0]);
        return -1;
    }

    printf("\nBoot table contains %u sections:\n", table.num_sections);
    for (i = 0; i < table.num_sections; i++) {
        section_temp = table.sections[i];
        printf("#%hhu 0x%08x (%d) : %10u bytes\n",
                i,
                section_temp.start_address,
                section_temp.memory_type,
                section_temp.length);
    }

    printf("\nEntry address is 0x%08x.\n", table.entry_address);

    if (assert_local_reset(core_id)) {
        printf("\nCould not assert local reset on core %d!\n", core_id);
        ret_val = -1;
        goto free_and_exit;
    }

    /* TODO: also check that we do not overwrite LL2 for any other cores */
    entry_mem_type = get_memory_type(table.entry_address);
    if ((entry_mem_type >= LL2_CORE0) && (entry_mem_type <= LL2_CORE7)) {
        if ((core_id - CORE0) != (entry_mem_type - LL2_CORE0)) {
            printf("\nCannot use entry address with specified core!\n");
            ret_val = -1;
            goto free_and_exit;
        }
    }

    if (write_table_sections(&table, core_id)) {
        ret_val = -1;
        goto free_and_exit;
    }

    if (set_core_boot_address(core_id, table.entry_address) ||
            deassert_local_reset(core_id)) {
        printf("\nCould not boot core %d!\n", core_id);
        ret_val = -1;
        goto free_and_exit;
    }

    printf("\nCore %d booted.\n", core_id);

free_and_exit:
    free(table_data_ptr);

    return ret_val;
}
