/*
 * Boot table parser
 *
 * Copyright (C) 2017  Appear TV AS - http://www.appeartv.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "btbl_parser.h"
#include "utils.h"

#include <stdio.h>
#include <stdlib.h>



/**
 * Read a big-endian uint32_t from specified input to output,
 * and return the number of bytes read.
 */
static inline int read_be_uint32(const uint8_t *in, uint32_t *out)
{
    uint8_t i;
    *out = *in++;

    for (i = 0; i < 3; i++) {
        *out <<= 8;
        *out |= *in++;
    }

    return 4;
}


/**
 * Read a big-endian uint32_t from specified input and write
 * it back to the same location in native endian.
 */
static void convert_be_u32(uint8_t *in, uint32_t length)
{
    uint8_t read_bytes;
    uint32_t *out;
    uint32_t temp;

    while (length) {
        /* TODO: figure out if this will give alignment issues.. */
        out = (uint32_t *)in;
        read_bytes = read_be_uint32(in, &temp);
        *out = temp;
        length -= read_bytes;
        in += read_bytes;
    }
}


/**
 * Parse a binary btbl that has been read into memory.
 */
int parse_btbl_binary(const uint8_t *btbl, size_t btbl_length, struct boot_table *table)
{
    const uint8_t *btbl_end = btbl + btbl_length;
    struct table_section temp_section;
    uint8_t alignment_miss;

    btbl += read_be_uint32(btbl, &table->entry_address);
    table->num_sections = 0;

    btbl += read_be_uint32(btbl, &temp_section.length);

    while (temp_section.length) {
        if (table->num_sections >= MAX_NUM_SECTIONS) {
            printf("Read too many sections!\n");
            return -1;
        }

        /* boot sections are 32-bit, so correct the length if it is not a multiple */
        alignment_miss = temp_section.length % 4;
        if (alignment_miss) {
            printf("Adjusting length of section #%u to multiple of 32 bits.\n", table->num_sections);
            temp_section.length += 4 - alignment_miss;
        }

        btbl += read_be_uint32(btbl, &temp_section.start_address);
        temp_section.data = (uint8_t *)btbl;
        temp_section.memory_type = get_memory_type(temp_section.start_address);

        /* convert endianness of the data */
        convert_be_u32(temp_section.data, temp_section.length);

        /* add this section to the array */
        table->sections[table->num_sections++] = temp_section;

        /* skip data */
        btbl += temp_section.length;

        if (btbl >= btbl_end) {
            printf("Reached end of binary without any termination record!\n");
            return -1;
        }

        btbl += read_be_uint32(btbl, &temp_section.length);
    }

    /* uh oh! we are _past_ the end! how?! */
    if (btbl > btbl_end) {
        printf("Read past the end of binary!\n");
        return -1;
    }

    return 0;
}


/**
 * Read boot table binary file and parse into specified table
 *
 * This function returns a pointer to the memory region which contains the
 * section data. That pointer should eventually be freed by the calling code.
 *
 * If an error occurs, NULL is returned instead.
 */
uint8_t *read_btbl_binary(struct boot_table *table_ptr, char *filename_ptr)
{
    uint8_t *btbl_ptr;
    FILE *file_ptr;
    size_t file_size;

    file_ptr = fopen(filename_ptr, "rb");
    if (file_ptr == NULL) {
        printf("Could not open file %s!\n", filename_ptr);
        return NULL;
    }

    fseek(file_ptr, 0L, SEEK_END);
    file_size = ftell(file_ptr);
    fseek(file_ptr, 0L, SEEK_SET);

    btbl_ptr = (uint8_t *)malloc(file_size);
    if (btbl_ptr == NULL) {
        printf("Not enough memory to fit %lu bytes of file!\n", file_size);
        goto failed_after_fopen;
    }

    if (fread(btbl_ptr, 1, file_size, file_ptr) != file_size) {
        printf("Failed reading entire file!\n");
        goto failed_after_malloc;
    }

    if (parse_btbl_binary(btbl_ptr, file_size, table_ptr)) {
        printf("Error parsing file!\n");
        goto failed_after_malloc;
    }

    fclose(file_ptr);

    return btbl_ptr;

failed_after_malloc:
    free(btbl_ptr);
failed_after_fopen:
    fclose(file_ptr);

    return NULL;
}