/*
 * Control utils
 *
 * Copyright (C) 2017  Appear TV AS - http://www.appeartv.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "control.h"
#include "utils.h"
#include "../common/device.h"
#include "../common/device_registers.h"

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>


#define MAX_COUNT  10


typedef struct {
    dsp_core_id_t core_id;
    uint32_t pid;
    uint32_t mid;
} core_map_entry_t;


static core_map_entry_t core_map[] = {
    { .core_id = CORE0, .pid = PD8,  .mid = LPSC_C0_TIM0 },
    { .core_id = CORE1, .pid = PD9,  .mid = LPSC_C1_TIM1 },
    { .core_id = CORE2, .pid = PD10, .mid = LPSC_C2_TIM2 },
    { .core_id = CORE3, .pid = PD11, .mid = LPSC_C3_TIM3 },
    { .core_id = CORE4, .pid = PD12, .mid = LPSC_C4_TIM4 },
    { .core_id = CORE5, .pid = PD13, .mid = LPSC_C5_TIM5 },
    { .core_id = CORE6, .pid = PD14, .mid = LPSC_C6_TIM6 },
    { .core_id = CORE7, .pid = PD15, .mid = LPSC_C7_TIM7 },
};


int assert_local_reset(dsp_core_id_t core_id)
{
    int file_desc;
    uint8_t count;
    uint8_t i;
    core_map_entry_t *map_entry_ptr;
    uint32_t temp;
    int ret_val = 0;

    map_entry_ptr = NULL;
    for (i = 0; i < NUM_ENTRIES(core_map); i++) {
        if (core_map[i].core_id == core_id) {
            map_entry_ptr = &core_map[i];
            break;
        }
    }

    if (map_entry_ptr == NULL) {
        printf("\nCore %d not found in core map!\n", core_id);
        return -1;
    }

    file_desc = open_control_file(PSC_BASE_ADDRESS);
    if (file_desc < 0) {
        return -1;
    }

    /* TODO: use flock to ensure atomic operations */

    /* clear LRST in MDCTL to assert local reset */
    lseek(file_desc, MDCTL(map_entry_ptr->mid), SEEK_SET);
    read(file_desc, &temp, sizeof(temp));
    temp &= ~MDCTL_LRST;
    lseek(file_desc, -sizeof(temp), SEEK_CUR);
    write(file_desc, &temp, sizeof(temp));

    usleep(1000);

    /* wait for LRSTDONE to get set */
    lseek(file_desc, MDSTAT(map_entry_ptr->mid), SEEK_SET);
    for (count = 0; count < MAX_COUNT; count++) {
        read(file_desc, &temp, sizeof(temp));
        lseek(file_desc, -sizeof(temp), SEEK_CUR);
        if ((temp & MDSTAT_LRSTDONE) != 0) {
            break;
        }
        usleep(1000);
    }
    if (count == MAX_COUNT) {
        printf("\nTimed out waiting for LRSTDONE on core %d!\n", core_id);
        ret_val = -1;
        goto close_and_exit;
    }

close_and_exit:
    close(file_desc);

    return ret_val;
}


int set_core_boot_address(dsp_core_id_t core_id, uint32_t entry_address)
{
    int file_desc;
    int ret_val = 0;
    uint32_t temp;

    if ((entry_address & 0x3ff) != 0) {
        printf("Boot address is not aligned on 1 kB boundary!\n");
        ret_val = -1;
        goto close_and_exit;
    }

    file_desc = open_control_file(CHIP_LEVEL_BASE_ADDRESS);
    if (file_desc < 0) {
        return -1;
    }

    temp = entry_address;
    lseek(file_desc, DSP_BOOT_ADDR(core_id - CORE0), SEEK_SET);
    write(file_desc, &temp, sizeof(temp));

    usleep(1000);

close_and_exit:
    close(file_desc);

    return ret_val;
}


int deassert_local_reset(dsp_core_id_t core_id)
{
    int file_desc;
    uint8_t i;
    core_map_entry_t *map_entry_ptr;
    uint32_t temp;
    int ret_val = 0;

    map_entry_ptr = NULL;
    for (i = 0; i < NUM_ENTRIES(core_map); i++) {
        if (core_map[i].core_id == core_id) {
            map_entry_ptr = &core_map[i];
            break;
        }
    }

    if (map_entry_ptr == NULL) {
        printf("\nCore %d not found in core map!\n", core_id);
        return -1;
    }

    file_desc = open_control_file(PSC_BASE_ADDRESS);
    if (file_desc < 0) {
        return -1;
    }

    /* TODO: use flock to ensure atomic operations */

    /* set LRST in MDCTL to deassert local reset */
    lseek(file_desc, MDCTL(map_entry_ptr->mid), SEEK_SET);
    read(file_desc, &temp, sizeof(temp));
    temp |= MDCTL_LRST;
    lseek(file_desc, -sizeof(temp), SEEK_CUR);
    write(file_desc, &temp, sizeof(temp));

    usleep(1000);

    close(file_desc);

    return ret_val;
}
