/*
 * Memory utils
 *
 * Copyright (C) 2017  Appear TV AS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "memory.h"
#include "utils.h"

#include <fcntl.h>
#include <unistd.h>


int write_section(struct table_section *section_ptr)
{
    int file_desc;
    dsp_mem_type_t mem_type;
    long int offset;
    int ret_val = 0;

    mem_type = section_ptr->memory_type;

    offset = section_ptr->start_address - get_memory_start(mem_type);
    if (offset == section_ptr->start_address) {
        printf("\nStart address not known for memory type %d!\n", mem_type);
        return -1;
    }

    file_desc = open_memory_file(mem_type);
    if (file_desc < 0) {
        printf("\nError opening device file for memory type %d!\n", mem_type);
        return -1;
    }

    if (lseek(file_desc, offset, SEEK_SET) != offset) {
        printf("\nError seeking %lx in device file for memory type %d!\n", offset, mem_type);
        ret_val = -1;
        goto close_and_exit;
    }

    if (write(file_desc, section_ptr->data, section_ptr->length) != section_ptr->length) {
        printf("\nFailed writing to device file for memory type %d!\n", mem_type);
        ret_val = -1;
    }

close_and_exit:
    close(file_desc);

    return ret_val;
}


int write_table_sections(struct boot_table *table_ptr, dsp_core_id_t core_id)
{
    uint32_t i;
    struct table_section section_temp;

    for (i = 0; i < table_ptr->num_sections; i++) {
        section_temp = table_ptr->sections[i];

        if (get_memory_type(section_temp.start_address) == LL2_LOCAL) {
            section_temp.memory_type = LL2_CORE0 + (core_id - CORE0);
            section_temp.start_address += LL2_START_CORE(core_id - CORE0) - LL2_START;
        }

        if (write_section(&section_temp)) {
            return -1;
        }
    }

    return 0;
}