/*
 * Common utils
 *
 * Copyright (C) 2017  Appear TV AS
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation, version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include "utils.h"
#include "../common/device.h"
#include "../atvdsp_module/atvdsp.h"

#include <fcntl.h>
#include <sys/ioctl.h>
#include <unistd.h>


typedef struct type_map_entry_s {
    dsp_mem_type_t mem_type;
    uint32_t start;
    uint32_t end;
    char *file_name;
} type_map_entry_t;


static type_map_entry_t type_map[] = {
    { .mem_type = LL2_LOCAL, .start = LL2_START,         .end = LL2_END,         .file_name = NULL },
    { .mem_type = LL2_CORE0, .start = LL2_START_CORE(0), .end = LL2_END_CORE(0), .file_name = ATVDSP_LL2_CORE0_PATH },
    { .mem_type = LL2_CORE1, .start = LL2_START_CORE(1), .end = LL2_END_CORE(1), .file_name = ATVDSP_LL2_CORE1_PATH },
    { .mem_type = LL2_CORE2, .start = LL2_START_CORE(2), .end = LL2_END_CORE(2), .file_name = ATVDSP_LL2_CORE2_PATH },
    { .mem_type = LL2_CORE3, .start = LL2_START_CORE(3), .end = LL2_END_CORE(3), .file_name = ATVDSP_LL2_CORE3_PATH },
    { .mem_type = LL2_CORE4, .start = LL2_START_CORE(4), .end = LL2_END_CORE(4), .file_name = ATVDSP_LL2_CORE4_PATH },
    { .mem_type = LL2_CORE5, .start = LL2_START_CORE(5), .end = LL2_END_CORE(5), .file_name = ATVDSP_LL2_CORE5_PATH },
    { .mem_type = LL2_CORE6, .start = LL2_START_CORE(6), .end = LL2_END_CORE(6), .file_name = ATVDSP_LL2_CORE6_PATH },
    { .mem_type = LL2_CORE7, .start = LL2_START_CORE(7), .end = LL2_END_CORE(7), .file_name = ATVDSP_LL2_CORE7_PATH },
    { .mem_type = MSM,       .start = MSM_START,         .end = MSM_END,         .file_name = ATVDSP_MSM_PATH },
    { .mem_type = DDR,       .start = DDR_START,         .end = DDR_END,         .file_name = ATVDSP_DDR_PATH },
};


dsp_mem_type_t get_memory_type(uint32_t address)
{
    uint8_t i;
    type_map_entry_t map_entry;

    for (i = 0; i < NUM_ENTRIES(type_map); i++) {
        map_entry = type_map[i];

        if ((address >= map_entry.start) && (address <= map_entry.end)) {
            return map_entry.mem_type;
        }
    }

    return INVALID_MEM_TYPE;
}


uint32_t get_memory_start(dsp_mem_type_t mem_type)
{
    uint8_t i;
    type_map_entry_t map_entry;

    for (i = 0; i < NUM_ENTRIES(type_map); i++) {
        map_entry = type_map[i];

        if (map_entry.mem_type == mem_type) {
            return map_entry.start;
        }
    }

    return 0;
}


int open_memory_file(dsp_mem_type_t mem_type)
{
    char *file_name_ptr;
    uint8_t i;
    type_map_entry_t map_entry;

    file_name_ptr = NULL;
    for (i = 0; i < NUM_ENTRIES(type_map); i++) {
        map_entry = type_map[i];

        if (map_entry.mem_type == mem_type) {
            file_name_ptr = map_entry.file_name;
            break;
        }
    }

    if (file_name_ptr == NULL) {
        printf("\nDevice file not known for memory type %d!\n", mem_type);
        return -1;
    }

    return open(file_name_ptr, O_RDWR | O_SYNC);
}


int open_control_file(uint32_t base_address)
{
    int file_desc;

    file_desc = open(ATVDSP_CONTROL_PATH, O_RDWR | O_SYNC);
    if (file_desc < 0) {
        return -1;
    }

    if (ioctl(file_desc, ATVDSP_IOC_WRITE_IB, base_address) != 0) {
        printf("\nControl file could not be rebased to address 0x%08x!\n", base_address);
        close(file_desc);
        return -1;
    }

    return file_desc;
}
