#include "atvdsp.h"
#include "pcie.h"
#include "../common/device.h"

#include <linux/cdev.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <asm/uaccess.h>
#include <asm/dma-mapping.h>
#include <linux/time.h>


MODULE_AUTHOR("Appear TV AS");
MODULE_LICENSE("GPL v2");


/* 8 * ll2 + msm + ddr */
#define NUM_ATVDSP_MEM_DEVS  10
#define NUM_ATVDSP_DEVS  (1 + NUM_ATVDSP_MEM_DEVS)
#define ATVDSP_MINOR  0


static int atvdsp_ctrl_open(struct inode *, struct file *);
static loff_t atvdsp_ctrl_llseek(struct file *, loff_t, int);
static ssize_t atvdsp_ctrl_read(struct file *, char *, size_t, loff_t *);
static ssize_t atvdsp_ctrl_write(struct file *, const char *, size_t, loff_t *);
static long int atvdsp_ctrl_ioctl(struct file *, unsigned int, unsigned long);

static loff_t atvdsp_mem_llseek(struct file *, loff_t, int);
static ssize_t atvdsp_mem_read(struct file *, char *, size_t, loff_t *);
static ssize_t atvdsp_mem_write(struct file *, const char *, size_t, loff_t *);
static int atvdsp_mem_open(struct inode *, struct file *);


static struct file_operations atvdsp_ctrl_fops = {
    .owner = THIS_MODULE,
    .open = atvdsp_ctrl_open,
    .llseek = atvdsp_ctrl_llseek,
    .read = atvdsp_ctrl_read,
    .write = atvdsp_ctrl_write,
    .unlocked_ioctl = atvdsp_ctrl_ioctl,
};

static struct file_operations atvdsp_mem_fops = {
    .owner = THIS_MODULE,
    .open = atvdsp_mem_open,
    .llseek = atvdsp_mem_llseek,
    .read = atvdsp_mem_read,
    .write = atvdsp_mem_write,
};


struct atvdsp_ctrl_dev_s {
    struct cdev cdev;
    size_t size;
};

struct atvdsp_mem_dev_s {
    struct cdev cdev;
    dsp_mem_type_t mem_type;
    size_t size;
};

static struct class *atvdsp_class_ptr;
static int atvdsp_major = 0;
static struct atvdsp_ctrl_dev_s atvdsp_ctrl_dev;
static struct atvdsp_mem_dev_s atvdsp_mem_devs[NUM_ATVDSP_MEM_DEVS];


long int atvdsp_ctrl_ioctl(struct file *file_ptr, unsigned int cmd, unsigned long arg)
{
	if ((_IOC_TYPE(cmd) != ATVDSP_IOC_MAGIC) || (_IOC_NR(cmd) > ATVDSP_IOC_MAXNR)) {
        return -ENOTTY;
    }

    switch (cmd) {
    case ATVDSP_IOC_READ_IB:
        return (uintptr_t)file_ptr->private_data;

    case ATVDSP_IOC_WRITE_IB:
        if (arg % 4) {
            return -EINVAL;
        }
        file_ptr->private_data = (void *)arg;
        return 0;

    default:
        return -ENOTTY;
    }
}


int atvdsp_ctrl_open(struct inode *inode_ptr, struct file *file_ptr)
{
    file_ptr->private_data = (void *)CHIP_LEVEL_BASE_ADDRESS;

    return 0;
}


int atvdsp_mem_open(struct inode *inode_ptr, struct file *file_ptr)
{
    struct atvdsp_mem_dev_s *dev_ptr;

    dev_ptr = container_of(inode_ptr->i_cdev, struct atvdsp_mem_dev_s, cdev);
    file_ptr->private_data = dev_ptr;

    return 0;
}


static loff_t _llseek_helper(size_t size, loff_t position, loff_t offset, int whence)
{
    loff_t new_position;

    switch(whence) {
    case SEEK_SET:
        new_position = offset;
        break;

    case SEEK_CUR:
        new_position = position + offset;
        break;

    case SEEK_END:
        new_position = size + offset;
        break;

    default:
        return -EINVAL;
    }

    if ((new_position < 0) || (new_position % 4)) {
        return -EINVAL;
    }

    return new_position;
}


loff_t atvdsp_ctrl_llseek(struct file *file_ptr, loff_t offset, int whence)
{
    loff_t new_position;

    new_position = _llseek_helper(atvdsp_ctrl_dev.size, file_ptr->f_pos, offset, whence);

    if (new_position >= 0) {
        file_ptr->f_pos = new_position;
    }

    return new_position;
}


loff_t atvdsp_mem_llseek(struct file *file_ptr, loff_t offset, int whence)
{
    struct atvdsp_mem_dev_s *dev_ptr = file_ptr->private_data;
    loff_t new_position;

    new_position = _llseek_helper(dev_ptr->size, file_ptr->f_pos, offset, whence);

    if (new_position >= 0) {
        file_ptr->f_pos = new_position;
    }

    return new_position;
}


static ssize_t atvdsp_ctrl_read(struct file *file_ptr, char *user_ptr, size_t size,
        loff_t *offset_ptr)
{
    int64_t ret_val;

    ret_val = atvdsp_read_registers((uintptr_t)file_ptr->private_data, *offset_ptr, user_ptr, size);

    if (ret_val < 0) {
        return ret_val;
    }

    *offset_ptr += size;

    return size;
}


static ssize_t atvdsp_ctrl_write(struct file *file_ptr, const char *user_ptr, size_t size,
        loff_t *offset_ptr)
{
    int64_t ret_val;

    ret_val = atvdsp_write_registers((uintptr_t)file_ptr->private_data, *offset_ptr, user_ptr, size);

    if (ret_val < 0) {
        return ret_val;
    }

    *offset_ptr += size;

    return size;
}


ssize_t atvdsp_mem_read(struct file *file_ptr, char *user_ptr, size_t size, loff_t *offset_ptr)
{
    struct atvdsp_mem_dev_s *dev_ptr;
    int64_t ret_val;

    dev_ptr = file_ptr->private_data;
    ret_val = atvdsp_read_memory(dev_ptr->mem_type, *offset_ptr, user_ptr, size);

    if (ret_val < 0) {
        return ret_val;
    }

    *offset_ptr += size;

    return size;
}


ssize_t atvdsp_mem_write(struct file *file_ptr, const char *user_ptr, size_t size, loff_t *offset_ptr)
{
    struct atvdsp_mem_dev_s *dev_ptr;
    int64_t ret_val;

    dev_ptr = file_ptr->private_data;
    ret_val = atvdsp_write_memory(dev_ptr->mem_type, *offset_ptr, user_ptr, size);

    if (ret_val < 0) {
        return ret_val;
    }

    *offset_ptr += size;

    return size;
}


static int atvdsp_setup_ctrl_dev(dev_t dev_no, const char *dev_name_ptr)
{
    struct device *device_ptr;
    int err;

    cdev_init(&atvdsp_ctrl_dev.cdev, &atvdsp_ctrl_fops);
    atvdsp_ctrl_dev.cdev.owner = THIS_MODULE;
    /* TODO: use a common size define with spiboot_pcieinit */
    atvdsp_ctrl_dev.size = 32 << 10;

    err = cdev_add(&atvdsp_ctrl_dev.cdev, dev_no, 1);
    if (err) {
        printk(KERN_NOTICE "atvdsp: failed with error %d when adding cdev %s.", err, dev_name_ptr);
    }

    device_ptr = device_create(atvdsp_class_ptr, NULL, dev_no, &atvdsp_ctrl_dev, dev_name_ptr);
    if (IS_ERR(device_ptr)) {
        cdev_del(&atvdsp_ctrl_dev.cdev);
        printk(KERN_NOTICE "atvdsp: failed with error %ld when creating device %s.",
                PTR_ERR(device_ptr), dev_name_ptr);
        return PTR_ERR(device_ptr);
    }

    printk(KERN_INFO "atvdsp: created %s device of size 0x%lx", dev_name_ptr, atvdsp_ctrl_dev.size);

    return 0;
}


static int atvdsp_setup_mem_dev(struct atvdsp_mem_dev_s *dev_ptr, dev_t dev_no,
        dsp_mem_type_t mem_type, size_t size, const char *dev_name_ptr)
{
    struct device *device_ptr;
    int err;

    dev_ptr->mem_type = mem_type;
    dev_ptr->size = size;

    cdev_init(&dev_ptr->cdev, &atvdsp_mem_fops);
    dev_ptr->cdev.owner = THIS_MODULE;

    err = cdev_add(&dev_ptr->cdev, dev_no, 1);
    if (err) {
        printk(KERN_NOTICE "atvdsp: failed with error %d when adding cdev %s.", err, dev_name_ptr);
    }

    device_ptr = device_create(atvdsp_class_ptr, NULL, dev_no, dev_ptr, dev_name_ptr);
    if (IS_ERR(device_ptr)) {
        cdev_del(&dev_ptr->cdev);
        printk(KERN_NOTICE "atvdsp: failed with error %ld when creating device %s.",
                PTR_ERR(device_ptr), dev_name_ptr);
        return PTR_ERR(device_ptr);
    }

    printk(KERN_INFO "atvdsp: created %s device of size 0x%lx", dev_name_ptr, dev_ptr->size);

    return 0;
}


int atvdsp_init_module(void)
{
    int result;
    dev_t dev_no = 0;

    result = atvdsp_pcie_init();
    if (result != 0) {
        return result;
    }

    result = alloc_chrdev_region(&dev_no, ATVDSP_MINOR, NUM_ATVDSP_DEVS, "atvdsp");
    atvdsp_major = MAJOR(dev_no);

    if (result < 0) {
        printk(KERN_ERR "atvdsp: can't get major %d.\n", atvdsp_major);
        return result;
    }

    atvdsp_class_ptr = class_create(THIS_MODULE, "atvdsp");
    if (IS_ERR(atvdsp_class_ptr)) {
        printk(KERN_ERR "atvdsp: can't create device class.\n");
        unregister_chrdev_region(dev_no, NUM_ATVDSP_DEVS);
        return PTR_ERR(atvdsp_class_ptr);
    }

    if (atvdsp_setup_ctrl_dev(dev_no++, "control") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[0], dev_no++, LL2_CORE0, LL2_SIZE, "ll2_core0") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[1], dev_no++, LL2_CORE1, LL2_SIZE, "ll2_core1") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[2], dev_no++, LL2_CORE2, LL2_SIZE, "ll2_core2") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[3], dev_no++, LL2_CORE3, LL2_SIZE, "ll2_core3") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[4], dev_no++, LL2_CORE4, LL2_SIZE, "ll2_core4") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[5], dev_no++, LL2_CORE5, LL2_SIZE, "ll2_core5") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[6], dev_no++, LL2_CORE6, LL2_SIZE, "ll2_core6") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[7], dev_no++, LL2_CORE7, LL2_SIZE, "ll2_core7") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[8], dev_no++, MSM, MSM_SIZE, "msm") ||
            atvdsp_setup_mem_dev(&atvdsp_mem_devs[9], dev_no++, DDR, DDR_SIZE, "ddr")) {
        return -EFAULT;
    }

    printk(KERN_NOTICE "atvdsp: all devices succesfully created!");

    return 0;
}


void atvdsp_cleanup_module(void)
{
    dev_t dev_no = MKDEV(atvdsp_major, ATVDSP_MINOR);
    uint8_t i;

    atvdsp_pcie_deinit();

    device_destroy(atvdsp_class_ptr, dev_no);
    cdev_del(&atvdsp_ctrl_dev.cdev);

    for (i = 0; i < NUM_ATVDSP_MEM_DEVS; i++) {
        device_destroy(atvdsp_class_ptr, dev_no + 1 + i);
        cdev_del(&atvdsp_mem_devs[i].cdev);
    }

    class_destroy(atvdsp_class_ptr);
    unregister_chrdev_region(dev_no, NUM_ATVDSP_DEVS);
}


module_init(atvdsp_init_module);
module_exit(atvdsp_cleanup_module);
