/**
 *  \file   sbl_rprc.c
 *
 *  \brief  This file contains functions to parse the multi-core
 *          image file & loads into CPU internal memory & external memory.
 *
 */

/*
 * Copyright (C) 2015-2016 Texas Instruments Incorporated - http://www.ti.com/
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the
 * distribution.
 *
 * Neither the name of Texas Instruments Incorporated nor the names of
 * its contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "sbl_rprc_parse.h"

/* TI-RTOS Header files */
#include <ti/drv/uart/UART_stdio.h>
#include <ti/csl/cslr_device.h>

#include <stdio.h>

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

typedef struct rprcFileHeader {
    uint32_t magic;
    uint32_t entry;
    uint32_t rsvd_addr;
    uint32_t SectionCount;
    uint32_t version;
} rprcFileHeader_t;

typedef struct rprcSectionHeader {
    uint32_t addr;
    uint32_t rsvd_addr;
    uint32_t size;
    uint32_t rsvdCrc;
    uint32_t rsvd;
} rprcSectionHeader_t;

typedef struct meta_header_start
{
    uint32_t magic_str;
    uint32_t num_files;
    uint32_t dev_id;
    uint32_t rsvd;
}meta_header_start_t;

typedef struct meta_header_core
{
    uint32_t core_id;
    uint32_t image_offset;
}meta_header_core_t;

typedef struct meta_header_end
{
    uint32_t rsvd;
    uint32_t magic_string_end;
}meta_header_end_t;

/* Magic numbers for gforge and sourceforge */
#define MAGIC_NUM_GF          (0xA1ACED00)
#define MAGIC_NUM_SF          (0x55424CBB)

/* Magic number and tokens for RPRC format */
#define RPRC_MAGIC_NUMBER   0x43525052
#define RPRC_RESOURCE       0
#define RPRC_BOOTADDR       5

#define MAX_INPUT_FILES 10
#define META_HDR_MAGIC_STR 0x5254534D /* MSTR in ascii */
#define META_HDR_MAGIC_END 0x444E454D /* MEND in ascii */

#if !defined(OMAPL137_BUILD)
#define SOC_OCMC_RAM1_SIZE          ((uint32_t) 0x80000)     /*OCMC1 512KB*/
#define SOC_OCMC_RAM2_SIZE          ((uint32_t) 0x100000)    /*OCMC2 1MB   */
#define SOC_OCMC_RAM3_SIZE          ((uint32_t) 0x100000)    /*OCMC3  1MB   */

#define MPU_IPU1_ROM                    (CSL_IPU_IPU1_TARGET_REGS)

#define MPU_IPU1_RAM                    (CSL_IPU_IPU1_TARGET_REGS + \
                                            (uint32_t) 0x20000)

#define MPU_IPU2_ROM                    (CSL_IPU_IPU1_ROM_REGS)

#define MPU_IPU2_RAM                    (CSL_IPU_IPU1_ROM_REGS + \
                                            (uint32_t) 0x20000)

#define MPU_DSP1_L2_RAM                 (0x40800000)
#define MPU_DSP1_L1P_CACHE              (0x40E00000)
#define MPU_DSP1_L1D_CACHE              (0x40F00000)
#define MPU_DSP2_L2_RAM                 (0x41000000)
#define MPU_DSP2_L1P_CACHE              (0x41600000)
#define MPU_DSP2_L1D_CACHE              (0x41700000)

#define SOC_DSP_L2_BASE                 (0x800000)
#define SOC_DSP_L1P_BASE                (0xe00000)
#define SOC_DSP_L1D_BASE                (0xf00000)
#endif

/* Function Pointer used while reading data from the storage. */
int32_t (*fp_readData)(void *dstAddr, void *srcAddr,
                        uint32_t length);

/* Function Pointer used while reading data from the storage. */
void  (*fp_seek)(void *srcAddr, uint32_t location);

/**
 * \brief           SBL_RprcImageParse function parse the RPRC executable image.
 *                  Copies individual section into destination location
 *
 * \param[in]    srcAddr - Pointer RPRC image
 * \param[out] entryPoint - CPU entry point address
 * \param[in]    CoreId - CPU ID to identify the CPU core
 *
 *
 * \return uint32_t: Status (success or failure)
 */
static int32_t SBL_RprcImageParse(void *srcAddr, uint32_t *entryPoint,
                              int32_t CoreId);

/**
 * \brief        SBL_BootCore function stores the CPU entry location into
 *               global pointer.
 *
 * \param[in]    entry - CPU Entry location
 * \param[in]    entryPoint - CPU ID
 *
 * \return   none
 */
void SBL_BootCore(uint32_t entry, uint32_t CoreID, sblEntryPoint_t *pAppEntry);

int32_t GetDeviceId(void);

/**
 * \brief           QSPIBootRprc function parses the multi-core app image
 *                  stored in the QSPI serial flash.
 *                  It Parses the AppImage & copies the section into CPU
 *                  internal memory & external memory.
 *                  CPUs entry loctions are stored into entry point
 *                  global pointers.
 *
 *
 * \param               none
 *
 *
 * \return          error status.If error has occured it returns a non zero
 *                  value.
 *                  If no error has occured then return status will be
 *                  zero.
 */

/******************************************************************************
**                       SBL Multicore RPRC parse functions
*******************************************************************************/

int32_t SBL_MulticoreImageParse(void *srcAddr,
                                uint32_t ImageOffset,
                                sblEntryPoint_t *pAppEntry)
{
    uint32_t            i;
    uint32_t            entryPoint = 0;
    meta_header_start_t mHdrStr;
    meta_header_core_t  mHdrCore[MAX_INPUT_FILES];
    meta_header_end_t   mHdrEnd;
    int32_t magic_str = META_HDR_MAGIC_STR;
    int32_t retVal    = E_PASS;

    if ((fp_readData == NULL) || (fp_seek == NULL))
    {
        retVal = E_FAIL;
    }
    else
    {
        /* Read Meta Header Start and get the Number of Input RPRC Files */
        fp_readData(&mHdrStr, srcAddr, sizeof (meta_header_start_t));
        if (mHdrStr.magic_str != (uint32_t) magic_str)
        {
            UART_printf("Invalid magic number in Single image header\r\n");
            retVal = E_FAIL;
        }
        else
        {
            if (mHdrStr.dev_id != GetDeviceId())
            {
                UART_printf("\nWARNING: Device Id Doesnot match\r\n");
            }

            /* Read all the Core offset addresses */
            for (i = 0; i < mHdrStr.num_files; i++)
            {
                fp_readData(&mHdrCore[i], srcAddr, sizeof (meta_header_core_t));
            }

            /* Add Base Offset address for All core Image start offset */
            for (i = 0; i < mHdrStr.num_files; i++)
            {
                mHdrCore[i].image_offset += ImageOffset;
            }

            /* Read Meta Header End */
            fp_readData(&mHdrEnd, srcAddr, sizeof (meta_header_end_t));

            /* Now Parse Individual RPRC files */

            for (i = 0; i < mHdrStr.num_files; i++)
            {
                if (mHdrCore[i].core_id != (0xFFFFFFFFU))
                {
                    fp_seek(srcAddr, mHdrCore[i].image_offset);
                    if (SBL_RprcImageParse(srcAddr, &entryPoint,
                                       mHdrCore[i].core_id) != E_PASS)
                    {
                        /* Error occurred parsing the RPRC file continue to
                         * parsing next
                         * image and skip booting the particular core */
                        UART_printf("RPRC parse error\n");
                        retVal = E_FAIL;
                    }
                    else
                    {
                        SBL_BootCore(entryPoint, mHdrCore[i].core_id, pAppEntry);
                    }
                }
            }
        }
    }
    return retVal;
}

void SBL_BootCore(uint32_t entry, uint32_t CoreID, sblEntryPoint_t *pAppEntry)
{
    switch (CoreID)
    {
        case MPU_CPU0_ID:
            /* CSL_MPU CPU0*/
            pAppEntry->entryPoint_MPU_CPU0 = entry;
            break;
#if defined(OMAPL137_BUILD) || defined(C6748_BUILD)  || defined(OMAPL138_BUILD)
        case DSP0_ID:
            /* DSP0*/
            pAppEntry->entryPoint_DSP0 = entry;
            break;
#endif    
#if defined(AM572x_BUILD) || defined(K2H_BUILD) || defined(K2E_BUILD) || defined(K2L_BUILD) || defined(K2K_BUILD) || defined(K2G_BUILD)
        case MPU_CPU1_ID:
            /* CSL_MPU CPU1 */
            pAppEntry->entryPoint_MPU_CPU1 = entry;
            break;
#endif

#if defined(K2H_BUILD) || defined(K2E_BUILD) || defined(K2K_BUILD) || defined(K2G_BUILD)
        case MPU_CPU2_ID:
            pAppEntry->entryPoint_MPU_CPU2 = entry;
            break;

        case MPU_CPU3_ID:
            pAppEntry->entryPoint_MPU_CPU3 = entry;
            break;

        case MPU_SMP_ID:
            pAppEntry->entryPoint_MPU_CPU0 = entry;
            pAppEntry->entryPoint_MPU_CPU1 = entry;
            pAppEntry->entryPoint_MPU_CPU2 = entry;
            pAppEntry->entryPoint_MPU_CPU3 = entry;
            break;
#endif

#if defined(AM571x_BUILD) || defined(AM572x_BUILD)
        case IPU1_CPU0_ID:
            /*IPU1 CPU0*/
            pAppEntry->entryPoint_IPU1_CPU0 = entry;
            break;

        case IPU1_CPU1_ID:
            /*IPU1 CPU1*/
            pAppEntry->entryPoint_IPU1_CPU1 = entry;
            break;

        case IPU1_CPU_SMP_ID:
            /*IPU1 CPU0 & CPU1*/
            pAppEntry->entryPoint_IPU1_CPU0 = entry;
            pAppEntry->entryPoint_IPU1_CPU1 = entry;
            break;
#endif

#if defined (AM572x_BUILD)
        case IPU2_CPU0_ID:
            /*IPU2 CPU0*/
            pAppEntry->entryPoint_IPU2_CPU0 = entry;
            break;

        case IPU2_CPU1_ID:
            /*IPU2 CPU1*/
            pAppEntry->entryPoint_IPU2_CPU1 = entry;
            break;

        case IPU2_CPU_SMP_ID:
            /*IPU2 CPU0 & CPU1*/
            pAppEntry->entryPoint_IPU2_CPU0 = entry;
            pAppEntry->entryPoint_IPU2_CPU1 = entry;
            break;
#endif

#if defined(K2H_BUILD) || defined(K2E_BUILD) || defined(K2L_BUILD) || defined(K2K_BUILD) || defined(K2G_BUILD)
        case DSP0_ID:
            pAppEntry->entryPoint_DSP0 = entry;
            break;
#endif

#if defined(AM572x_BUILD) || defined(AM571x_BUILD) || defined(K2H_BUILD) || defined(K2L_BUILD) || defined(K2K_BUILD)
        case DSP1_ID:
            /* DSP1*/
            pAppEntry->entryPoint_DSP1 = entry;
            break;
#endif

#if defined (AM572x_BUILD) || defined(K2H_BUILD) || defined(K2L_BUILD) || defined(K2K_BUILD)
        case DSP2_ID:
            /* DSP2*/
            pAppEntry->entryPoint_DSP2 = entry;
            break;
#endif

#if defined(K2H_BUILD) || defined(K2L_BUILD) || defined(K2K_BUILD)
        case DSP3_ID:
            pAppEntry->entryPoint_DSP3 = entry;
            break;
#endif

#if defined(K2H_BUILD)
        case DSP4_ID:
            pAppEntry->entryPoint_DSP4 = entry;
            break;

        case DSP5_ID:
            pAppEntry->entryPoint_DSP5 = entry;
            break;

        case DSP6_ID:
            pAppEntry->entryPoint_DSP6 = entry;
            break;

        case DSP7_ID:
            pAppEntry->entryPoint_DSP7 = entry;
            break;
#endif

            default:
            /* Wrong CPU ID */
            break;
    }
}

#if !defined(OMAPL137_BUILD) && !defined(OMAPL138_BUILD) && !defined(C6748_BUILD)
static int32_t SBL_RprcImageParse(void *srcAddr,
                                  uint32_t *entryPoint,
                                  int32_t CoreId)
{
    rprcFileHeader_t    header;
    rprcSectionHeader_t section;
    int32_t i;
    int32_t retVal = E_PASS;

    /*read application image header*/
    fp_readData(&header, srcAddr, sizeof (rprcFileHeader_t));

    /*check magic number*/
    if (header.magic != RPRC_MAGIC_NUMBER)
    {
        UART_printf("Invalid magic number in boot image. Expected: %x, received: %x\n", RPRC_MAGIC_NUMBER, header.magic);
        retVal = E_FAIL;
    }
    else
    {
        /* Set the Entry Point */
        *entryPoint = header.entry;

        /*read entrypoint and copy sections to memory*/
        for (i = 0; i < header.SectionCount; i++)
        {
            /*read new section header*/
            fp_readData(&section, srcAddr, sizeof (rprcSectionHeader_t));
#if defined(K2G_BUILD) || defined(K2E_BUILD) || defined(K2H_BUILD) || defined(K2K_BUILD) || defined(K2L_BUILD)
#if 0
            if ((section.addr >= CSL_MSMC_SRAM_REGS) &&
                (section.addr < CSL_MSMC_SRAM_REGS + CSL_MSMC_SRAM_REGS_SIZE))
            {

            }
            else if (section.addr >= CSL_DDR_0_DATA)
            {

            }
            else if ((section.addr >= CSL_C66X_COREPAC_LOCAL_L2_SRAM_REGS) &&
                (section.addr < CSL_C66X_COREPAC_LOCAL_L2_SRAM_REGS + CSL_C66X_COREPAC_LOCAL_L2_SRAM_REGS_SIZE))
            {
                section.addr |= 0x10000000;
            }
            else
            {
                
            }
#else
            if ((section.addr >= CSL_MSMC_SRAM_REGS) &&
                (section.addr < CSL_MSMC_SRAM_REGS + 0x01000000))
            {

            }
            else if (section.addr >= 0x80000000)
            {

            }
            else if ((section.addr >= CSL_C66X_COREPAC_LOCAL_L2_SRAM_REGS) &&
                (section.addr < CSL_C66X_COREPAC_LOCAL_L2_SRAM_REGS + 0x00100000))
            {
                section.addr |= 0x10000000;
            }
            else
            {
                
            }
#endif
#else
#if defined (AM572x_BUILD)
            if (((section.addr >= CSL_MPU_OCMC_RAM1_REGS) &&
                 (section.addr < (CSL_MPU_OCMC_RAM1_REGS + SOC_OCMC_RAM1_SIZE))) ||
                ((section.addr >= CSL_MPU_OCMC_RAM2_REGS) &&
                 (section.addr < (CSL_MPU_OCMC_RAM2_REGS + SOC_OCMC_RAM2_SIZE))) ||
                ((section.addr >= CSL_MPU_OCMC_RAM3_REGS) &&
                 (section.addr < (CSL_MPU_OCMC_RAM3_REGS + SOC_OCMC_RAM3_SIZE))))
#elif defined (AM571x_BUILD)
            if ((section.addr >= CSL_MPU_OCMC_RAM1_REGS) &&
                 (section.addr < (CSL_MPU_OCMC_RAM1_REGS + SOC_OCMC_RAM1_SIZE)))
#endif
            {
                /* Internal OCMC RAM Space for all the cores */
            }
            /*copy section to memory*/
            /*check for section mapped into CPU internal memories*/
            else if (section.addr < 0x80000000U)
            {
                switch (CoreId)
                {
                    case MPU_CPU0_ID:
                        /*No action*/
                        break;
#if defined (AM572x_BUILD)
                    case MPU_CPU1_ID:
                        /*No action*/
                        break;
#endif
                    case IPU1_CPU0_ID:
                    case IPU1_CPU1_ID:
                    case IPU1_CPU_SMP_ID:

                        if (((section.addr >= CSL_IPU_IPU1_BOOT_SPACE_REGS) &&
                             (section.addr <
                              (CSL_IPU_IPU1_BOOT_SPACE_REGS + 0x10000U))) ||
                            ((section.addr >= CSL_IPU_IPU1_RAM_REGS) &&
                             (section.addr < (CSL_IPU_IPU1_RAM_REGS + 0x10000))))
                        {
                            section.addr = section.addr & 0x000FFFFFU;
                            section.addr = MPU_IPU1_RAM + section.addr;
                        }
                        else
                        {
                            UART_puts("IPU1 - Invalid Memory section", -1);
                        }

                        break;
#if defined (AM572x_BUILD)
                    case IPU2_CPU0_ID:
                    case IPU2_CPU1_ID:
                    case IPU2_CPU_SMP_ID:
                        if (((section.addr >= CSL_IPU_IPU1_BOOT_SPACE_REGS) &&
                             (section.addr <
                              (CSL_IPU_IPU1_BOOT_SPACE_REGS + 0x10000U)))
                            ||
                            ((section.addr >= CSL_IPU_IPU1_RAM_REGS) &&
                             (section.addr < (CSL_IPU_IPU1_RAM_REGS + 0x10000))))
                        {
                            section.addr = section.addr & 0x000FFFFFU;
                            section.addr = MPU_IPU2_RAM + section.addr;
                        }
                        else
                        {
                            UART_puts("IPU2  - Invalid Memory section", -1);
                        }

                        break;
#endif
                    case DSP1_ID:

                        if ((section.addr >= SOC_DSP_L2_BASE) &&
                            (section.addr < (SOC_DSP_L2_BASE + 0x48000)))
                        {
                            /* L2 RAM Read the offset */
                            section.addr = section.addr - SOC_DSP_L2_BASE;
                            section.addr = MPU_DSP1_L2_RAM + section.addr;
                        }
                        else if ((section.addr >= SOC_DSP_L1P_BASE) &&
                                 (section.addr < (SOC_DSP_L1P_BASE + 0x8000)))
                        {
                            /* CSL_MPU_DSP1_L1P_CACHE Read the offset */
                            section.addr = section.addr - SOC_DSP_L1P_BASE;
                            section.addr = MPU_DSP1_L1P_CACHE + section.addr;
                        }
                        else if ((section.addr >= SOC_DSP_L2_BASE) &&
                                 (section.addr < (SOC_DSP_L2_BASE + 0x8000)))
                        {
                            /* CSL_MPU_DSP1_L1D_CACHE */
                            section.addr = section.addr - SOC_DSP_L2_BASE;
                            section.addr = MPU_DSP1_L1D_CACHE + section.addr;
                        }
                        else
                        {
                            UART_puts("DSP1 - Invalid Memory section", -1);
                        }

                        break;
#if defined (AM572x_BUILD)
                    case DSP2_ID:

                        if ((section.addr >= SOC_DSP_L2_BASE) &&
                            (section.addr < (SOC_DSP_L2_BASE + 0x48000)))
                        {
                            /* DSP2 L2 RAM Read the offset */
                            section.addr = section.addr - SOC_DSP_L2_BASE;
                            section.addr = MPU_DSP2_L2_RAM + section.addr;
                        }
                        else if ((section.addr >= SOC_DSP_L1P_BASE) &&
                                 (section.addr < (SOC_DSP_L1P_BASE + 0x8000)))
                        {
                            /* CSL_MPU_DSP2_L1P_CACHE Read the offset */
                            section.addr = section.addr - SOC_DSP_L1P_BASE;
                            section.addr = MPU_DSP2_L1P_CACHE + section.addr;
                        }
                        else if ((section.addr >= SOC_DSP_L1D_BASE) &&
                                 (section.addr < (SOC_DSP_L1D_BASE + 0x8000)))
                        {
                            /* CSL_MPU_DSP2_L1D_CACHE */
                            section.addr = section.addr - SOC_DSP_L1D_BASE;
                            section.addr = MPU_DSP2_L1D_CACHE + section.addr;
                        }
                        else
                        {
                            UART_puts("DSP2 - Invalid Memory section", -1);
                        }

                        break;
#endif
                    default:
                        break;
                }
            }
            else
            {
                /* To remove MISRA C error */
            }
#endif
            fp_readData((void *) section.addr, srcAddr, section.size);
        }
    }
    return retVal;
}
#else
/* */
static int32_t SBL_RprcImageParse(void *srcAddr,
                                  uint32_t *entryPoint,
                                  int32_t CoreId)
{
    rprcFileHeader_t    header;
    rprcSectionHeader_t section;
    int32_t i;
    int32_t retVal = E_PASS;

    /*read application image header*/
    fp_readData(&header, srcAddr, sizeof (rprcFileHeader_t));

    /*check magic number*/
    if (header.magic != RPRC_MAGIC_NUMBER)
    {
        UART_printf("Invalid magic number in boot image. Expected: %x, received: %x\n", RPRC_MAGIC_NUMBER, header.magic);
        retVal = E_FAIL;
    }
    else
    {
        /* Set the Entry Point */
        *entryPoint = header.entry;

        /*read entrypoint and copy sections to memory*/
        for (i = 0; i < header.SectionCount; i++)
        {
            fp_readData(&section, srcAddr, sizeof (rprcSectionHeader_t));
            fp_readData((void *) section.addr, srcAddr, section.size);
        }
    }
    return retVal;
}
#endif

/**
 * \brief  Function to read the device ID
 *
 *
 * \param   None.
 *
 * \return   Return the device id
 *
 **/
int32_t GetDeviceId(void)
{
    int32_t deviceId = 55;
    return (deviceId);
}
