import json
import subprocess
import os
num_asymmetric_keys = 0
num_symmetric_keys = 0
def generate_key_pair(key_size, block_number):
    private_key_file = f"build/private_key_{block_number}.pem"
    public_key_file = f"build/public_key_{block_number}.pem"
    der_key_file = f"build/public_key_{block_number}.der"

    try:
        # Generate private key
        subprocess.run(["openssl", "genrsa", "-out", private_key_file, str(key_size)], check=True)

        # Get public key
        subprocess.run(["openssl", "rsa", "-in", private_key_file, "-pubout", "-out", public_key_file], check=True)

        # Convert to DER format
        subprocess.run(["openssl", "pkey", "-in", public_key_file, "-pubin", "-outform", "DER", "-out", der_key_file], check=True)

        return der_key_file

    except subprocess.CalledProcessError as e:
        print(f"Error generating key pair for block {block_number}: {e}")
        return None

def generate_json():
    global num_asymmetric_keys
    global num_symmetric_keys
    num_keys = int(input("How many key blocks do you want to create (1-6)? "))
    if not 1 <= num_keys <= 6:
        print("Invalid number of keys. Please enter a number between 1 and 6.")
        return

    key_blocks = []
    used_ids = set()  # To track used key_ids

    for i in range(num_keys):
        print(f"Enter details for key block {i+1}:")

        # key_type validation
        while True:
            key_type = input("  key_type (1 - symmetric, 0 - asymmetric): ")
            if key_type in ("0", "1"):
                if key_type == "0":
                    num_asymmetric_keys += 1
                else:
                    num_symmetric_keys += 1
                break
            else:
                print("Invalid key_type. Please enter 0 or 1.")

        # key_id validation
        while True:
            key_id = input("  key_id (1-6, unique): ")
            try:
                key_id = int(key_id)
                if 1 <= key_id <= 6 and key_id not in used_ids:
                    used_ids.add(key_id)
                    break
                elif 1 <= key_id <= 6 and key_id in used_ids:
                    print("key_id already used. Please enter a unique ID.")
                else:
                    print("Invalid key_id. Please enter a number between 1 and 6.")
            except ValueError:
                print("Invalid key_id. Please enter a number.")

        # key_imageauth validation
        while True:
            key_imageauth = input("  key_imageauth (0 or 1): (0 - Not used for firmware authentication, 1 - Used for firmware authentication): ")
            if key_imageauth in ("0", "1"):
                break
            else:
                print("Invalid key_imageauth. Please enter 0 or 1.")

        # key_debugauth validation
        while True:
            key_debugauth = input("  key_debugauth (0 or 1): (0 - Not used for debug certificate authentication, 1 - Used for debug certificate authentication): ")
            if key_debugauth in ("0", "1"):
                break
            else:
                print("Invalid key_debugauth. Please enter 0 or 1.")

        # reserved values
        while True:
            reserved_0 = input("  reserved[0] (0 - sha512, 1 - sha384, 2 - sha256): ")
            if reserved_0 in ("0", "1", "2"):
                break
            else:
                print("Invalid reserved[0]. Please enter 0, 1, or 2.")

        while True:
            reserved_1 = input("  reserved[1] (0 - RSA4096, 1 - RSA3072): ")
            if reserved_1 in ("0", "1"):
                break
            else:
                print("Invalid reserved[1]. Please enter 0 or 1.")

        key_path = input("  key_path (path to .der file, leave blank to generate): ")
        if not key_path:
            if reserved_1 == "0":
                key_size = 4096
            else:
                key_size = 3072

            generated_key_path = generate_key_pair(key_size, i + 1)
            if generated_key_path:
                key_path = generated_key_path
                print(f"Generated key pair and set key_path to: {key_path}")
            else:
                print("Key generation failed.  Please provide a valid key path.")
                continue  

        if not key_path.lower().endswith(".der"):
            print("Warning: Key path does not end with .der.  Ensure the file is in DER format.")

        key_block = {
            "key_type": f"{key_type}U",
            "key_id": f"{key_id}U",
            "key_imageauth": f"{key_imageauth}U",
            "key_debugauth": f"{key_debugauth}U",
            "key_reserved": [f"{reserved_0}U", f"{reserved_1}U", "0U", "0U"],  # Fixed reserved[3] and reserved[4]
            "key_path": key_path
        }
        key_blocks.append(key_block)

    with open("build/keyring.json", "w") as f:
        json.dump(key_blocks, f, indent=4)

    print("keyring.json created successfully.")
    print("num_asymmetric_keys:", num_asymmetric_keys)
    print("num_symmetric_keys:", num_symmetric_keys)

