import json
import subprocess
import os

def calculate_hash(key_path, reserved_0):
    if reserved_0 == "0":
        hash_algorithm = "sha512"
        hash_size = 512 
    elif reserved_0 == "1":
        hash_algorithm = "sha384"
        hash_size = 384 
    elif reserved_0 == "2":
        hash_algorithm = "sha256"
        hash_size = 256
    else:
        print("Invalid reserved[0] value. Cannot calculate hash.")
        return None

    try:
        command = ["openssl", "dgst", f"-{hash_algorithm}", "-binary", key_path]
        result = subprocess.run(command, capture_output=True, check=True)
        hash_bytes = result.stdout
        hex_hash = "".join(f"{byte:02x}" for byte in hash_bytes)
        return hex_hash[:int(hash_size/4)]  
    except subprocess.CalledProcessError as e:
        print(f"Error calculating hash: {e}")
        return None

def generate_c_file():
    json_file = "build/keyring.json"

    try:
        with open(json_file, "r") as f:
            data = json.load(f)
    except FileNotFoundError:
        print(f"Error: {json_file} not found.")
        return
    except json.JSONDecodeError:
        print(f"Error: Invalid JSON format in {json_file}.")
        return

    key_blocks = data  
    c_file_lines = f'''
// file keyring_data.c
#include <stdint.h>

typedef uint8_t u8;

struct aux_pkh {{
    u8	key_type;
    u8	keyid;
    u8	imageauth;
    u8	debugauth;
    u8	reserved[4];
}} __attribute__((packed));

struct keyring_asymm {{
    struct aux_pkh	aux;
    u8 smpkh[64];
}} __attribute__((packed));
    '''
    c_file_lines += f'''
struct keyring_asymm instances[{len(key_blocks)}] = {{
'''    
    for i, key_block in enumerate(key_blocks):
        key_type = int(key_block["key_type"][:-1])  
        key_id = int(key_block["key_id"][:-1])
        imageauth = int(key_block["key_imageauth"][:-1])
        debugauth = int(key_block["key_debugauth"][:-1])
        reserved = [int(r[:-1]) for r in key_block["key_reserved"]]
        key_path = key_block["key_path"]

        smpkh = calculate_hash(key_path, str(reserved[0]))
        if smpkh is None:
            print(f"Skipping key block {i+1} due to hash calculation failure.")
            continue
        
        smpkh_bytes = [f"0x{smpkh[j:j+2]}" for j in range(0,len(smpkh), 2)]
        c_file_lines += f'  {{\n \t.aux = {{ {key_type}U, {key_id}U, {imageauth}U, {debugauth}U, {{ {reserved[0]}U, {reserved[1]}U, {reserved[2]}U, {reserved[3]}U }} }}, \n   \t.smpkh = {{ {", ".join(smpkh_bytes)} }}\n  }},\n'

    c_file_lines +=("};")

    with open("build/keyring_data.c", "w") as f:
        f.write(c_file_lines)

    print("keyring_data.c created successfully.")
    print("Compiling...")
# Compile the C file
    try:
        subprocess.run(["gcc", "-c", "build/keyring_data.c", "-o", "build/keyring_data.o"], check=True)
        print("keyring_data.c compiled successfully.")
    except subprocess.CalledProcessError as e:
        print(f"Error compiling keyring_data.c: {e}")
# Convert .o to .bin using objcopy
    try:
        subprocess.run(["objcopy", "--dump-section", ".data=build/keyring_data.bin", "build/keyring_data.o"], check=True)
        print("keyring_data.o converted to keyring_data.bin successfully.")
    except subprocess.CalledProcessError as e:
        print(f"Error converting keyring_data.o to keyring_data.bin: {e}")



