
/*#############################################################################

  Project:  Mailbox Test
  Revision: 1.0
  Board:    EVMAM5728
  Device:   AM5728
  Core:     ARM Cortex-A15

###############################################################################*/

/*=============================================================================
                                   HEADERS
  =============================================================================*/

/* Miscellany */
#include <stdio.h>
#include <stdlib.h>

/* XDCtools */
#include <xdc/std.h>
#include <xdc/cfg/global.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Error.h>

/* SYSBIOS */
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/knl/Semaphore.h>

/* AM5728 Device Drivers */
#include <ti/csl/soc.h>
#include <ti/board/board.h>
#include <ti/csl/csl_timer.h>
#include <ti/csl/csl_gpio.h>
#include <ti/csl/csl_mailbox.h>
#include <ti/drv/uart/UART_stdio.h>

/* AM5728 MMU and Cache */
#include <ti/sysbios/family/arm/a15/Cache.h>
#include <ti/sysbios/family/arm/a15/Mmu.h>

/*=============================================================================
                                    MACROS
  =============================================================================*/

/* Hardware Timer  */
#define TIMER_PERIOD         0.1
#define TIMER_FREQ           19200000
#define TIMER_PRESC          256
#define TIMER_INITIAL_COUNT  (0xFFFFFFFF - ((TIMER_PERIOD * TIMER_FREQ) / TIMER_PRESC))
#define TIMER_RLD_COUNT      TIMER_INITIAL_COUNT
#define TIMER_BASE_ADDR      CSL_MPU_TIMER3_REGS
#define TIMER_INTR_NUM       37

/* User LEDs */
#define LEDS_BASEADDR        CSL_MPU_GPIO7_REGS
#define LED0_ID              8
#define LED1_ID              9
#define LED2_ID              14
#define LED3_ID              15

/*=============================================================================
                                GLOBAL VARIABLES
  =============================================================================*/
int MyIntFlag_BiosTimer = 0;
int MyIntFlag_HwTimer   = 0;
int MyIntFlag_Mailbox   = 0;

int MailboxMessage[4] = {0,0,0,0};
int QueueCounter      = 0;
int QueueNumber       = 0;
int InterruptNumber   = 0;

/*=============================================================================
                             MISCELLANEOUS FUNCTIONS
  =============================================================================*/
void SetLed (int Led, int Status)
{
    switch (Led)
    {
        case 0: GPIOPinWrite(LEDS_BASEADDR, LED0_ID, Status); break;
        case 1: GPIOPinWrite(LEDS_BASEADDR, LED1_ID, Status); break;
        case 2: GPIOPinWrite(LEDS_BASEADDR, LED2_ID, Status); break;
        case 3: GPIOPinWrite(LEDS_BASEADDR, LED3_ID, Status); break;
    }
}

//-------------------------------------------------------------------
int GetLed (int Led)
{
    int i = -1;

    switch (Led)
    {
        case 0: i = GPIOPinRead(LEDS_BASEADDR, LED0_ID); break;
        case 1: i = GPIOPinRead(LEDS_BASEADDR, LED1_ID); break;
        case 2: i = GPIOPinRead(LEDS_BASEADDR, LED2_ID); break;
        case 3: i = GPIOPinRead(LEDS_BASEADDR, LED3_ID); break;
    }

    return (i);
}

//-------------------------------------------------------------------
void ToggleLed (int Led)
{
    SetLed(Led, (GetLed(Led)==0)? 1 : 0);
}

/*=============================================================================
                           INTERRUPT SERVICE ROUTINES
  =============================================================================*/
Void ISR_HwTimer(void)
{
    //-------------------------------------------------------------------------
    //                Timer 3 Interrupt Service Routine
    //                  Mapped to MPU_IRQ_39, HWI #71
    //-------------------------------------------------------------------------
    GPIOTriggerPinInt(LEDS_BASEADDR, GPIO_INT_LINE_1, LED0_ID);
    MyIntFlag_HwTimer = 1;
    TIMERIntStatusClear(TIMER_BASE_ADDR, TIMER_INT_OVF_EN_FLAG);
}

//-----------------------------------------------------------------------------
Void ISR_BiosTimer(void)
{
    //-------------------------------------------------------------------------
    //               BIOS Timer Interrupt Service Routine
    //-------------------------------------------------------------------------
    GPIOTriggerPinInt(LEDS_BASEADDR, GPIO_INT_LINE_1, LED1_ID);

    MailboxSendMessage(SOC_MAILBOX1_BASE, QueueCounter, QueueCounter);

    if (++QueueCounter > 2)
        QueueCounter = 0;

    MyIntFlag_BiosTimer = 1;
}

//-----------------------------------------------------------------------------
Void ISR_Mailbox1_User0(void)
{
    //-------------------------------------------------------------------------
    //               Mailbox1_User0 Interrupt Service Routine
    //                   Mapped to MPU_IRQ_49, HWI #81
    //-------------------------------------------------------------------------

    int i;

    for (i=0; i<=3; i++)
        MailboxMessage[i] = 0;

    i=0;
    while ((MailboxGetMessageCount(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_0) > 0) && (i<4))
        MailboxGetMessage(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_0, (uint32_t*) (&MailboxMessage[i++]) ) ;

    MailboxClrNewMsgStatus(SOC_MAILBOX1_BASE, 0, MAILBOX_QUEUE_0);

    QueueNumber = 10;
    InterruptNumber = 49;
    MyIntFlag_Mailbox = 1;
}

//-----------------------------------------------------------------------------
Void ISR_Mailbox1_User1(void)
{
    //-------------------------------------------------------------------------
    //               Mailbox1_User1 Interrupt Service Routine
    //                   Mapped to MPU_IRQ_50, HWI #82
    //-------------------------------------------------------------------------

    int i;

    for (i=0; i<=3; i++)
        MailboxMessage[i] = 0;

    i=0;
    while ((MailboxGetMessageCount(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_1) > 0) && (i<4))
        MailboxGetMessage(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_1, (uint32_t*) (&MailboxMessage[i++]) ) ;

    MailboxClrNewMsgStatus(SOC_MAILBOX1_BASE, 1, MAILBOX_QUEUE_1);

    QueueNumber = 11;
    InterruptNumber = 50;
    MyIntFlag_Mailbox = 1;
}

//-----------------------------------------------------------------------------
Void ISR_Mailbox1_User2(void)
{
    //-------------------------------------------------------------------------
    //               Mailbox1_User2 Interrupt Service Routine
    //                   Mapped to MPU_IRQ_51, HWI #83
    //-------------------------------------------------------------------------

    int i;

    for (i=0; i<=3; i++)
        MailboxMessage[i] = 0;

    i=0;
    while ((MailboxGetMessageCount(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_2) > 0) && (i<4))
        MailboxGetMessage(SOC_MAILBOX1_BASE, MAILBOX_QUEUE_2, (uint32_t*) (&MailboxMessage[i++]) ) ;

    MailboxClrNewMsgStatus(SOC_MAILBOX1_BASE, 2, MAILBOX_QUEUE_2);

    QueueNumber = 12;
    InterruptNumber = 51;
    MyIntFlag_Mailbox = 1;
}

//-----------------------------------------------------------------------------
Void ISR_Mailbox2_User0(void)
{
    //-------------------------------------------------------------------------
    //               Mailbox2_User0 Interrupt Service Routine
    //                   Mapped to MPU_IRQ_52, HWI #84
    //-------------------------------------------------------------------------

    int i;

    for (i=0; i<=3; i++)
        MailboxMessage[i] = 0;

    i=0;
    while ((MailboxGetMessageCount(SOC_MAILBOX2_BASE, MAILBOX_QUEUE_0) > 0) && (i<4))
        MailboxGetMessage(SOC_MAILBOX2_BASE, MAILBOX_QUEUE_0, (uint32_t*) (&MailboxMessage[i++]) ) ;

    MailboxClrNewMsgStatus(SOC_MAILBOX2_BASE, 0, MAILBOX_QUEUE_0);

    QueueNumber = 20;
    InterruptNumber = 52;
    MyIntFlag_Mailbox = 1;
}

/*=============================================================================
                               IDLE FUNCTION
  =============================================================================*/
void IdleFunction01(void)
{
    //-------------------------------------------------------------------------
    // Hw Timer Interrupt was activated...
    //-------------------------------------------------------------------------
    if (MyIntFlag_HwTimer == 1)
    {
        MyIntFlag_HwTimer = 0;

        ToggleLed(2);
        UART_printf(".");
    }

    //-------------------------------------------------------------------------
    // BIOS Timer Interrupt was activated...
    //-------------------------------------------------------------------------
    if (MyIntFlag_BiosTimer == 1)
    {
        MyIntFlag_BiosTimer = 0;
        ToggleLed(3);
    }

    //-------------------------------------------------------------------------
    // Mailbox Interrupt was activated...
    //-------------------------------------------------------------------------
    if (MyIntFlag_Mailbox == 1)
    {
        MyIntFlag_Mailbox = 0;

        UART_printf("\n\n > Message = {%d %d %d %d} @ Queue #%d, Hwi #%d \n\n",
                    MailboxMessage[0],
                    MailboxMessage[1],
                    MailboxMessage[2],
                    MailboxMessage[3],
                    QueueNumber,
                    InterruptNumber + 32);
    }

}

/*=============================================================================
                                     MAIN
  =============================================================================*/
int main(void)
{
    int i;

    //-------------------------------------------------------------------------
    //                           BOARD SETUP
    //-------------------------------------------------------------------------
    Board_initCfg boardCfg;

    boardCfg = BOARD_INIT_PINMUX_CONFIG |
        BOARD_INIT_MODULE_CLOCK |
        BOARD_INIT_UART_STDIO;

    Board_init(boardCfg);

    //-------------------------------------------------------------------------
    //                           GPIO7 SETUP
    //-------------------------------------------------------------------------
    GPIODirModeSet(LEDS_BASEADDR, LED0_ID, GPIO_DIR_OUTPUT);
    GPIODirModeSet(LEDS_BASEADDR, LED1_ID, GPIO_DIR_OUTPUT);
    GPIODirModeSet(LEDS_BASEADDR, LED2_ID, GPIO_DIR_OUTPUT);
    GPIODirModeSet(LEDS_BASEADDR, LED3_ID, GPIO_DIR_OUTPUT);

    GPIOIntTypeSet(LEDS_BASEADDR, LED0_ID, GPIO_INT_TYPE_RISE_EDGE);
    GPIOPinIntEnable(LEDS_BASEADDR, GPIO_INT_LINE_1, LED0_ID);

    //-------------------------------------------------------------------------
    //                           LEDs SETUP
    //-------------------------------------------------------------------------
    for (i=0; i<=3; i++)
        SetLed(i, 1);

    //-------------------------------------------------------------------------
    //                          TIMER3 SETUP
    //-------------------------------------------------------------------------
    TIMERDisable(TIMER_BASE_ADDR);
    TIMERReset(TIMER_BASE_ADDR);
    TIMERModeConfigure(TIMER_BASE_ADDR, TIMER_AUTORLD_NOCMP_ENABLE);
    TIMERPreScalerClkEnable(TIMER_BASE_ADDR, TIMER_PRESCALER_CLK_DIV_BY_256);
    TIMERCounterSet(TIMER_BASE_ADDR, TIMER_INITIAL_COUNT);
    TIMERReloadSet(TIMER_BASE_ADDR, TIMER_RLD_COUNT);
    TIMERIntStatusClear(TIMER_BASE_ADDR, TIMER_INT_OVF_IT_FLAG);
    TIMERIntEnable(TIMER_BASE_ADDR, TIMER_INT_OVF_EN_FLAG);
    TIMEREnable(TIMER_BASE_ADDR);

    //-------------------------------------------------------------------------
    //                         MAILBOX1 SETUP
    //-------------------------------------------------------------------------
    CSL_xbarMpuIrqConfigure(CSL_XBAR_INST_MPU_IRQ_49, CSL_XBAR_MAILBOX1_IRQ_USER0);
    CSL_xbarMpuIrqConfigure(CSL_XBAR_INST_MPU_IRQ_50, CSL_XBAR_MAILBOX1_IRQ_USER1);
    CSL_xbarMpuIrqConfigure(CSL_XBAR_INST_MPU_IRQ_51, CSL_XBAR_MAILBOX1_IRQ_USER2);
//    MailboxReset(SOC_MAILBOX1_BASE);
    MailboxEnableNewMsgInt(SOC_MAILBOX1_BASE, 0, MAILBOX_QUEUE_0);
    MailboxEnableNewMsgInt(SOC_MAILBOX1_BASE, 1, MAILBOX_QUEUE_1);
    MailboxEnableNewMsgInt(SOC_MAILBOX1_BASE, 2, MAILBOX_QUEUE_2);

    //-------------------------------------------------------------------------
    //                         MAILBOX2 SETUP
    //
    //  WARNING: EXCEPTION GENERATED WHEN ACCESSING MAILBOX 2..13 REGISTERS (!)
    //
    //-------------------------------------------------------------------------
    CSL_xbarMpuIrqConfigure(CSL_XBAR_INST_MPU_IRQ_52, CSL_XBAR_MAILBOX2_IRQ_USER0);
    MailboxEnableNewMsgInt(SOC_MAILBOX2_BASE, 0, MAILBOX_QUEUE_0);

    //-------------------------------------------------------------------------
    //                         SYSBIOS LAUNCH
    //-------------------------------------------------------------------------
    UART_printf("\n Mailbox Test running... \n");
    BIOS_start();
    return (0);
}
/*=============================================================================
                                    END
  =============================================================================*/
