
#include "my_timer.h"

#include <xdc/runtime/Types.h>
#include <xdc/runtime/System.h>
#include <ti/sysbios/family/c64p/EventCombiner.h>
#include <ti/sysbios/family/c64p/Hwi.h>
#include <ti/sysbios/family/shared/vayu/IntXbar.h>
#include <ti/csl/soc.h>
#include <ti/csl/csl_gpio.h>
#include <ti/csl/csl_timer.h>
#include <ti/csl/csl_types.h>
#include <xdc/runtime/Types.h>
#include <xdc/runtime/Timestamp.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Semaphore.h>

#define TIMER_CLOCK_FREQ 20000000
#define MY_TIMER_FREQ  20

// Timer reload value to get the required MY_TIMER_FREQ.
// Divide by 2 to get 50% duty cycle.
#define MY_TIMER_RELOAD_VALUE (0xffffffff - ((TIMER_CLOCK_FREQ / MY_TIMER_FREQ) / 2))

static uintptr_t timer13Base;
static Semaphore_Handle testSemaphore;

static volatile int bufferIndex;
static int curTime;
static volatile uint32_t times[2][20];
static uint32_t temp[19];

static void MyTimer_Handler(uintptr_t arg)
{
  times[bufferIndex][curTime] = Timestamp_get32();

  /* Disable the Timer interrupts */
  TIMERIntDisable(timer13Base, TIMER_INT_OVF_EN_FLAG);
  /* Clear the status of the interrupt flags */
  TIMERIntStatusClear(timer13Base, TIMER_INT_OVF_IT_FLAG);

  curTime++;
  if (curTime >= 20)
  {
    curTime = 0;
    bufferIndex ^= 1;
  }

  Semaphore_post(testSemaphore);

  TIMERIntEnable(timer13Base, TIMER_INT_OVF_EN_FLAG);
}

void MyTimer_Init(void)
{
  Types_FreqHz freq;
  Timestamp_getFreq(&freq);
  freq.lo /= 1000000; // Divide by a million to get times in microseconds.
  System_printf("MyTimer_Init enable timer 13\n");
  const uintptr_t l4perBaseVirt = SOC_L4PER_CM_CORE_BASE;
  // Enable timer 13.
  HW_WR_REG32(l4perBaseVirt + CM_L4PER3_TIMER13_CLKCTRL, 0x2);
  while ((HW_RD_REG32(l4perBaseVirt + CM_L4PER3_TIMER13_CLKCTRL) & (0x00030000)) != 0x0) ;

  timer13Base = SOC_TIMER13_BASE;

  System_printf("MyTimer_Init config timer 13 irq\n");
  IntXbar_connectIRQ(53, CSL_XBAR_TIMER13_IRQ);
  EventCombiner_dispatchPlug(53, (EventCombiner_FuncPtr)(&MyTimer_Handler), 53, (Bool)1);
  EventCombiner_enableEvent(53);

  Semaphore_Params semParams;
  Semaphore_Params_init(&semParams);
  semParams.mode = Semaphore_Mode_COUNTING;
  testSemaphore = Semaphore_create(1, &semParams, NULL);

  System_printf("MyTimer_Init config timer\n");
  TIMERModeConfigure(timer13Base, TIMER_AUTORLD_NOCMP_ENABLE);
  TIMERPreScalerClkDisable(timer13Base);
  TIMERCounterSet(timer13Base, MY_TIMER_RELOAD_VALUE);
  TIMERReloadSet(timer13Base, MY_TIMER_RELOAD_VALUE);
  TIMERPostedModeConfig(timer13Base, TIMER_NONPOSTED);
  TIMERReadModeConfig(timer13Base, TIMER_READ_MODE_NONPOSTED);
  TIMERIntStatusClear(timer13Base, TIMER_INT_TCAR_IT_FLAG | TIMER_INT_OVF_IT_FLAG | TIMER_INT_MAT_IT_FLAG);
  TIMERIntEnable(timer13Base, TIMER_INT_OVF_EN_FLAG);
  TIMEREnable(timer13Base);

  while (1)
  {
    int myIndex, i;
    uint32_t min, max;
    Semaphore_pend(testSemaphore, BIOS_WAIT_FOREVER);
    myIndex = bufferIndex ^ 1;
    for (i = 0; i < 19; i++)
    {
      temp[i] = (times[myIndex][i+1] - times[myIndex][i]) / freq.lo;
    }

    min = temp[0];
    max = temp[0];
    for (i = 1; i < 19; i++)
    {
      if (temp[i] < min)
        min = temp[i];
      if (temp[i] > max)
        max = temp[i];
    }

    System_printf("Collected L:%u U:%u %u %u %u %u %u %u\n", min, max, temp[0], temp[1], temp[2], temp[3], temp[4], temp[5]);
  }
}
