/*
 *
 * Copyright (C) 2011 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/
/***********************************************************************************************
 * FILE PURPOSE: Perform a 16 bit swap on a ccs data file
 ***********************************************************************************************
 * FILE NAME: swap16.c
 *
 * DESCRIPTION: Reads in a ccs data file, swaps the data on 16 bit boundaries.
 *				This is used when ccs data is loaded directly into the boot loader
 *
 *		usage:  swap16 [-in infile] [-out outfile]
 *
 ************************************************************************************************/

#include <stdio.h>
#include <string.h>
#include "ccsutil.h"


FILE *fin  = NULL;
FILE *fout = NULL;


/*************************************************************************************************
 * FUNCTION PURPOSE: Describe the invokation 
 *************************************************************************************************
 * DESCRIPTION: The program usage is displayed
 *************************************************************************************************/
int usage (char *inv)
{
  fprintf (stderr, "usage: %s [-in infile] [-out outfile]\n", inv);
  return (-1);

}

/*************************************************************************************************
 * FUNCTION PURPOSE: Parse the input args
 *************************************************************************************************
 * DESCRIPTION: The input args are checked
 *************************************************************************************************/
int parseit (int argc, char *argv[])
{
  int i;

  for (i = 1; i < argc;  ) {

     if (!strcmp (argv[i], "-in"))  {

         if (fin != NULL)
           return (usage (argv[0]));

         fin = fopen (argv[i+1], "r");
         if (fin == NULL)  {
            fprintf (stderr, "%s: Failed to open input file %s\n", argv[0], argv[i+1]);
            return (-1);
         }

         i = i + 2;

     }  else if (!strcmp (argv[i], "-out"))  {

        if (fout != NULL)
           return (usage (argv[0]));

        fout = fopen (argv[i+1], "w");
        if (fout == NULL)  {
            fprintf (stderr, "%s: Failed to open output file %s\n", argv[0], argv[i+1]);
            return (-1);
        }

        i = i + 2;

    }  else  {

        return (usage (argv[0]));

    }

  }

 if (fin == NULL)
    fin = stdin;

 if (fout == NULL)
    fout = stdout;

 return (0);

}
  



/*************************************************************************************************
 * FUNCTION PURPOSE: swap the data
 *************************************************************************************************
 * DESCRIPTION: The input file is read, 16 bit swapped, and written back out
 *************************************************************************************************/
int main (int argc, char *argv[])
{

  unsigned int *data, addr;
  unsigned int v, v1, v2;
  int i, n;

  if (parseit (argc, argv))
    return (-1);


  data = readCcsFile (fin, &n, &addr);
  if (data == NULL)  {
     fprintf (stderr, "%s: readCcsFile failed t read the input file\n", argv[0]);
     return (-1);
  }

  if (fin != stdin)
    fclose (fin);

  for (i = 0; i < n; i++)  {

    v = data[i];
    v1 = (v >> 16) & 0xffff;
    v2 = (v & 0xffff) << 16;
    v = v1 | v2;
    data[i] = v;

  }

  writeCcsFile (fout, data, n, addr); 

  if (fout != stdout)
    fclose (fout);
   
  return (0);

}







