/******************************************************************************
* FILE: omp_hello.c
* DESCRIPTION:
*   OpenMP Example - Hello World - C/C++ Version
*   In this simple example, the master thread forks a parallel region.
*   All threads in the team obtain their unique thread number and print it.
*   The master thread only prints the total number of threads.  Two OpenMP
*   library routines are used to obtain the number of threads and each
*   thread's number.
* AUTHOR: Blaise Barney  5/99
* LAST REVISED: 04/06/05
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#if defined(COMPILER_GNU)
#include <stdlib.h>

#if defined(_OPENMP)
#include <omp.h>
#endif

#include <time.h>
#define clock() ((double)clock())

#elif defined(COMPILER_C6X)

#if defined(_OPENMP)
#include <ti/runtime/openmp/omp.h>
#endif

#include <c6x.h>
#include <xdc/std.h>
//#include <xdc/runtime/IHeap.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/Memory.h>
#include <xdc/runtime/Error.h>
#include <xdc/runtime/IHeap.h>
//#include <ti/sysbios/heaps/IHeapBuf.h>
#include <xdc/cfg/global.h>
#define CLOCKS_PER_SEC 1e9
#define clock() ((double)_itoll(TSCH, TSCL))
#endif


#define ASSERT(cond, msg, param) if(cond){ } else { printf("\nError, assertion failed in file:'%s' at line: %d: \n %s requested size: %d\n", __FILE__, __LINE__, msg, param); printf("\n param = %d\n", param);exit(1); }
#define BUFF_SIZE (1024*10)
#define NUM_ITER (1)

typedef double mydt;


void process_some_data(mydt* buff_1, mydt* buff_2, mydt* buff_3){
    int i, j;
    for(j = 0; j < NUM_ITER; j++)
        for(i = 0; i < BUFF_SIZE; i++){
            buff_3[i] = buff_1[i]*3.2+(buff_2[i])-93;
        }
}


void omp_process_some_data(mydt* buff_1, mydt* buff_2, mydt* buff_3){
//    int j;
//    for(j = 0; j < NUM_ITER; j++)
//    {
#pragma omp parallel num_threads(8)
        {
        int i;
        #pragma omp for
            for(i = 0; i < BUFF_SIZE; i++){
                buff_3[i] = buff_1[i]*3.2+(buff_2[i])-93;
            }
        }
//    }
}


int main () {

#if defined(_OPENMP)
    omp_set_num_threads(8);
#endif

    mydt *ddr_buff_1, *ddr_buff_2, *ddr_buff_3;
    int msize = BUFF_SIZE*sizeof(mydt);
#if defined(COMPILER_GNU)
    ddr_buff_1 = malloc(msize);
    ddr_buff_2 = malloc(msize);
    ddr_buff_3 = malloc(msize);
#elif defined(COMPILER_C6X)
    Error_Block eb;
    Error_init(&eb);
    Error_init(&eb); ddr_buff_1 = Memory_alloc(NULL, msize, 0, &eb);
    Error_init(&eb); ddr_buff_2 = Memory_alloc(NULL, msize, 0, &eb);
    Error_init(&eb); ddr_buff_3 = Memory_alloc(NULL, msize, 0, &eb);
#endif

    ASSERT((int)ddr_buff_1 != 0, "Cannot allocate memory! msize: ", msize );
    ASSERT((int)ddr_buff_2 != 0, "Cannot allocate memory! msize: ", msize );
    ASSERT((int)ddr_buff_3 != 0, "Cannot allocate memory! msize: ", msize );

#if defined(COMPILER_C6X)
    TSCH = 0; TSCL = 0;
    long long start, end;
#elif defined(COMPILER_GCC)
    clock_t start, end;
#endif

    start = clock();
    process_some_data(ddr_buff_1, ddr_buff_2, ddr_buff_3);
    end = clock();
    printf("DDR3 Process Time: %f\n", (double)(end-start)/CLOCKS_PER_SEC);

    start = clock();
    omp_process_some_data(ddr_buff_1, ddr_buff_2, ddr_buff_3);
    end = clock();
    printf("OMP DDR3 Process Time: %f\n", (double)(end-start)/CLOCKS_PER_SEC);

#if defined(COMPILER_C6X)
    mydt *msmc_buff_1, *msmc_buff_2, *msmc_buff_3;

    Error_init(&eb); msmc_buff_1 = Memory_alloc(msmcHeap, msize, 0, &eb);
    Error_init(&eb); msmc_buff_2 = Memory_alloc(msmcHeap, msize, 0, &eb);
    Error_init(&eb); msmc_buff_3 = Memory_alloc(msmcHeap, msize, 0, &eb);

    ASSERT((int)msmc_buff_1 != 0, "Cannot allocate memory! msize: ", msize );
    ASSERT((int)msmc_buff_2 != 0, "Cannot allocate memory! msize: ", msize );
    ASSERT((int)msmc_buff_3 != 0, "Cannot allocate memory! msize: ", msize );
    start = clock();
    process_some_data(msmc_buff_1, msmc_buff_2, msmc_buff_3);
    end = clock();
    printf("MSMC Process Time: %f\n", (double)(end-start)/CLOCKS_PER_SEC);

    start = clock();
    omp_process_some_data(msmc_buff_1, msmc_buff_2, msmc_buff_3);
    end = clock();
    printf("OMP MSMC Process Time: %f\n", (double)(end-start)/CLOCKS_PER_SEC);
#endif

  return 0;
}
