/*
 *  ======== main.c ========
 */

//#include <cerrno>
#include <string.h>
//#include <stdlib.h>
//#include <stdio.h>

#include <xdc/std.h>

#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>

#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/family/c66/tci66xx/CpIntc.h>
#include <ti/sysbios/hal/Hwi.h>

#include "ti/platform/platform.h"
#include <ti/platform/evmc6657l/platform_lib/include/evmc665x_uart.h>

#include <ti/csl/soc/c6657/src/cslr_device.h>
#include <ti/csl/csl_uart.h>
#include <ti/csl/src/ip/uart/V0/cslr_uart.h>
#include <ti/csl/src/intc/csl_intc.h>
#include <ti/csl/csl_cpIntcAux.h>

#define CIC_HOST_INTERRUPT4UART         42
#define INTC_INTERRUPT4UART             9

#define UART_FIFO_SIZE                  14
#define TX_BUFFER_SIZE                  0x1000
#define RX_BUFFER_SIZE                  0x1000
#define TX_CIR_BUF_NEXT(x)              ((x+1)&(TX_BUFFER_SIZE-1))
#define TX_CIR_BUF_INC(x)               (x=(TX_CIR_BUF_NEXT(x)))
#define RX_CIR_BUF_NEXT(x)              ((x+1)&(RX_BUFFER_SIZE-1))
#define RX_CIR_BUF_INC(x)               (x=(RX_CIR_BUF_NEXT(x)))

#pragma DATA_ALIGN(Tx_Cir_Buf, 32)
Uint8   Tx_Cir_Buf[2][TX_BUFFER_SIZE];
Uint32  Tx_Buf_Head = 0,  Tx_Buf_Tail = 0;

#pragma DATA_ALIGN(Rx_Cir_Buf, 32)
Uint8   Rx_Cir_Buf[2][RX_BUFFER_SIZE];
Uint32  Rx_Buf_Head = 0,  Rx_Buf_Tail = 0;

Uint32  uart_isr_flag = 0;
Uint32  UART_Error = 0;
Uint32  UART_Receive_Con=0;
Uint32  uart_isr_cnt = 0;
Uint32  uart_isr_cnt1 = 0;

CSL_CPINTC_Handle           handleCIC0;
CSL_CPINTC_Handle           handleCIC1;
CSL_IntcContext             intcContext;
CSL_IntcEventHandlerRecord  EventHandler[30];
CSL_IntcObj                 intcObj;
CSL_IntcHandle              hTest;
CSL_IntcGlobalEnableState   state;
CSL_IntcEventHandlerRecord  EventRecord;
CSL_IntcParam               vectId;

/* Clock rate */
#define PLATFORM_BASE_CLK_RATE_MHZ (100)

/* PREDIV */
#define PLATFORM_PLL_PREDIV_val (1)

/* POSTDIV */
#define PLATFORM_PLL_POSTDIV_val (2)

/* Default PLL PLLM value (100 * 20/(1*2)) = 1.0GHz) */
#define  PLATFORM_PLL1_PLLM_val (20)

/* Default PLL PLLD value for 1.0GHz) */
#define  PLATFORM_PLL1_PLLD_val (1)

/* Input crystal frequency 100 MHz */
#define PLATFORM_UART_INPUT_CLOCK_RATE ((PLATFORM_BASE_CLK_RATE_MHZ * PLATFORM_PLL1_PLLM_val * 1000000)/(PLATFORM_PLL_PREDIV_val * 12 * PLATFORM_PLL1_PLLD_val)) /* SYSCLK7 = CPU_CLK/7 in Hz */

void UartSetBaudRate_self(uint32_t baudrate)
{
    uint8_t uiDLLVal = 0;
    uint8_t uiDLHVal = 0;
    Uint16 uiBaudRate=0;
    uiBaudRate = ((Uint16) (PLATFORM_UART_INPUT_CLOCK_RATE/(baudrate * 16)));

    hUartRegs->LCR = 0x80;
    uiDLLVal = (uint8_t )(0x00FF & uiBaudRate);
    uiDLHVal = (uint8_t )(0x00FF & (uiBaudRate  >> 8));

    // Set the baudrate,for accessing LCR[7] should be enable
    hUartRegs->DLL  = uiDLLVal;
    hUartRegs->DLH  = uiDLHVal;
    hUartRegs->LCR = 0x03;
}

void UartInit_self(void)
{
    hUartRegs->PWREMU_MGMT = 0x0;

    UartSetBaudRate_self(115200);
    hUartRegs->IER=0x03;
    hUartRegs->MCR = 0;
    hUartRegs->FCR = 0x7;
    hUartRegs->PWREMU_MGMT = 0x6001;

    return;
}

/******************************************************************************/
// write data from Tx circular buffer to UART0 port. Tail chase head,
// called from idle task.
/******************************************************************************/

void Write_UART()
{
    Uint8   ch;
    Uint32  i;
    Uint32 int_id;
    if (CSL_UART_LSR_THRE_FULL == (CSL_FEXT(hUartRegs->LSR, UART_LSR_THRE)))
        return;

    for (i = 0; i < UART_FIFO_SIZE; i++)
    {
        if (Tx_Buf_Tail == Tx_Buf_Head)
            break;  // no more to write.
        ch = Tx_Cir_Buf[0][Tx_Buf_Tail];
        CSL_FINS(hUartRegs->THR, UART_THR_DATA, ch);
        TX_CIR_BUF_INC(Tx_Buf_Tail);
    }

    //test and verify if the ETBEI and ERBI is changed
    while(1){
        int_id= CSL_FEXT(hUartRegs->IIR, UART_IIR_INITD);
    if(int_id != 0)
    {
         break;
    }

    }
} // Write_UART

/******************************************************************************/
// read data from UART0 port to rx circular buffer.  Head chase tail, but
// cannot equal to tail.  Called from isr.
/******************************************************************************/

void Read_UART()
{
    Uint8   ch;

    while (CSL_UART_LSR_DR_READY == (CSL_FEXT(hUartRegs->LSR, UART_LSR_DR)))
    {
        if (RX_CIR_BUF_NEXT(Rx_Buf_Head) == Rx_Buf_Tail)
        {
            UART_Error++;
//            LOGSTATUS(UART_INPUT_OVERRUN);
            break;
        }
        ch = CSL_FEXT(hUartRegs->RBR, UART_RBR_DATA);
        Rx_Cir_Buf[0][Rx_Buf_Head] = ch;
        RX_CIR_BUF_INC(Rx_Buf_Head);
    }
} // Read_UART

void uart_isr (UArg arg)
{
    Uint32      int_id;

    uart_isr_cnt++;
    CpIntc_disableHostInt(0, CIC_HOST_INTERRUPT4UART);
    CpIntc_clearSysInt(0, CSL_INTC0_UARTINT);

    int_id = CSL_FEXT(hUartRegs->IIR, UART_IIR_INITD);
    switch (int_id)
    {
        case CSL_UART_IIR_INITD_MODSTAT:
            break;
        case CSL_UART_IIR_INITD_THRE:
            Write_UART();
            break;
        case CSL_UART_IIR_INITD_RDA:
        case CSL_UART_IIR_INITD_CTI:
            uart_isr_cnt1++;
            Read_UART();
            break;
        case CSL_UART_IIR_INITD_RLS:
            break;
        default:
            break;
    }
    Hwi_clearInterrupt(INTC_INTERRUPT4UART);
    CpIntc_enableHostInt(0, CIC_HOST_INTERRUPT4UART);
}

void uart1ISRHandler (UArg arg){}

void Setup_UART_Interrupt()
{
    Hwi_Params params;
    int eventId;

    // Map the System Interrupt
    CpIntc_dispatchPlug(CSL_INTC0_UARTINT, &uart1ISRHandler, CSL_INTC0_UARTINT, TRUE);

    // We map system interrupt 164 to Host Interrupt 42
    CpIntc_mapSysIntToHostInt(0, CSL_INTC0_UARTINT, CIC_HOST_INTERRUPT4UART);

    // Enable the Host Interrupt
    CpIntc_enableHostInt(0, CIC_HOST_INTERRUPT4UART);

    // Enable the System Interrupt
    CpIntc_enableSysInt(0, CSL_INTC0_UARTINT);

    // Get the event id associated with the host interrupt
    eventId = CpIntc_getEventId(CIC_HOST_INTERRUPT4UART);

    Hwi_Params_init(&params);

    params.arg = CIC_HOST_INTERRUPT4UART;  // Host interrupt value
    params.eventId = eventId;              // Event id for the host interrupt
    params.enableInt = TRUE;               // Enable the Hwi

    // This plugs the interrupt vector 9 and the ISR function
    // When using CpIntc, you must plug the Hwi fxn with CpIntc_dispatch
    // so it knows how to process the CpIntc interrupts
    Hwi_create(INTC_INTERRUPT4UART, (ti_sysbios_interfaces_IHwi_FuncPtr)uart_isr, &params, NULL);

    Hwi_enableInterrupt(INTC_INTERRUPT4UART);

}

/******************************************************************************/
// application writes output data to UART tx circular buffer.  Head chase tail,
// but cannot equal to tail.
// return: number of bytes wrote.
/******************************************************************************/
Uint32  UART_Output(Int8 buf[], Uint32 size)
{
    Uint32 i;

    for (i = 0; i < size; i++)
    {
        if (TX_CIR_BUF_NEXT(Tx_Buf_Head) == Tx_Buf_Tail)
        {
            break;  // buffer over flow
        }
        Tx_Cir_Buf[0][Tx_Buf_Head] = buf[i];
        TX_CIR_BUF_INC(Tx_Buf_Head);
    }
    return i;
}

/******************************************************************************/
// Application reads input data from UART rx circular buffer.  Tail chase head.
// return: number of bytes read.
/******************************************************************************/
Uint32  UART_Input(Int8 buf[], Uint32 size)
{
    Uint32 i;

    for (i = 0; i < size; i++)
    {
        if (Rx_Buf_Tail == Rx_Buf_Head)
            break;  // no more data
        buf[i] = Rx_Cir_Buf[0][Rx_Buf_Tail];

        RX_CIR_BUF_INC(Rx_Buf_Tail);
    }
    return i;
}   // UART_Input

int UartTest()
{
    int8_t  message[] = "\r\n123456789012345678901234567890123456789012345678901234567890\r\n";
    int     length = strlen((char *)message);
    UART_Output(message, length);
    return(0);
}

/*
 *  ======== taskFxn ========
 */
Void taskFxn(UArg a0, UArg a1)
{

    int8_t buf[200]={0};
    uint32_t len=0;
    uint32_t i=0;

    UartTest();
    Write_UART();

    while(1){

        len=UART_Input(buf,50);
        if(len>0){
            System_printf("Len:%d\n",len);
            UART_Receive_Con+=len;
            System_printf("uart_isr_cnt:%d\n",uart_isr_cnt);
            System_printf("uart_isr_cnt1:%d\n",uart_isr_cnt1);
            for(i=0;i<len;i++)
                {
                    System_printf("%c",buf[i]);
                    System_printf("\n");
                }
            UART_Output(buf, len);
            Write_UART();
        }

    }

//    System_printf("exit taskFxn()\n");
}

/*
 *  ======== main ========
 */
Void main()
{
    Task_Handle task;
    Error_Block eb;

    Setup_UART_Interrupt();

    UartInit_self();

    Error_init(&eb);
    task = Task_create(taskFxn, NULL, &eb);
    if (task == NULL) {
        System_printf("Task_create() failed!\n");
        BIOS_exit(0);
    }

    BIOS_start();     /* enable interrupts and start SYS/BIOS */
}





