#include "draw_texture.h"
#define STB_IMAGE_IMPLEMENTATION
#include "stb/stb_image.h"
#include "renderutils.h"
#include "texture.h"
#define SHADER_PATH "shaders"

static char vert_shader[] = SHADER_PATH"/texture.vsh";
static char frag_shader[] = SHADER_PATH"/texture.fsh";

static const char vertexShaderSource[] =
"   #version 100\n"
"   attribute vec4 aVertexPosition;\n"
"   attribute vec2 aTextureCoord1;\n"
"   varying vec2 oTextureCoord1;\n"
"   void main(void) { \n"
"       gl_Position = aVertexPosition;\n"
"       oTextureCoord1 = aTextureCoord1;\n"
"   } \n";

static const char fragmentShaderSource[] =
"   #version 100\n"
"   precision mediump float;\n"
"   varying vec2 oTextureCoord1;\n"
"   uniform sampler2D uSampler0;\n "
"   void main(void) {\n"
"       gl_FragColor = texture2D(uSampler0, vec2(oTextureCoord1.s, (1.0-oTextureCoord1.t)));\n"
//"       gl_FragColor = vec4(1.0, 0.0, 0.0, 1.0);\n"
"   } \n";

static GLuint uSampler0_loc;
static GLuint aVertexPosition_loc;
static GLuint aTextureCoord1_loc;
static GLuint uOutputSize_loc;
static GLuint uTextureSize_loc;

static GLuint vertexData_ID;
static GLuint indices_ID;

int nrChannels, texWidth, texHeight;
GLuint Tex;
GLuint shaderProgram;



// Vertex data containing the positions of each point of the triangle
static const GLfloat vertexData[] = {
	-1.0f,  1.0f,  0.0f,   0.0f, 1.0f, // top left
	-1.0f, -1.0f,  0.0f,   0.0f, 0.0f, // bottom left
	 1.0f, -1.0f,  0.0f,   1.0f, 0.0f, // bottom right
	 1.0f,  1.0f,  0.0f,   1.0f, 1.0f  // top right
};

static GLushort indices[] = {
	0, 1, 2, // first triangle
	0, 2, 3  // second triangle
};

static int stride = 3+2; //coords, tex1


bool testGLError(const char* functionLastCalled)
{
	//	glGetError returns the last error that occurred using OpenGL ES, not necessarily the status of the last called function. The user
	//	has to check after every single OpenGL ES call or at least once every frame. Usually this would be for debugging only, but for this
	//	example it is enabled always
	GLenum lastError = glGetError();
	if (lastError != GL_NO_ERROR)
	{
		printf("%s failed (%x).\n", functionLastCalled, lastError);
		return false;
	}
	return true;
}

void createTex()
{
    unsigned int texbytes, readbytes;
    texbytes = texWidth * texHeight * 4;
	char filename[]= "data/test_rgba_1024x1024.png";

    glActiveTexture(GL_TEXTURE0);
    testGLError("glActiveTexture");
    glGenTextures(1, &Tex);
    testGLError("glGenTextures");
    glBindTexture(GL_TEXTURE_2D, Tex);
    testGLError("glBindTexture");
    glPixelStorei(GL_UNPACK_ALIGNMENT,1);
    testGLError("glPixelStorei");


	glEnable(GL_BLEND);
	testGLError("glEnable");
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	testGLError("glBlendFunc");
	glPixelStorei(GL_UNPACK_ALIGNMENT,1);
	testGLError("glPixelStorei");
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, 1920, 1080, 0, GL_RGBA, GL_UNSIGNED_BYTE, texture_data);
	testGLError("glTexImage2D");
	glGenerateMipmap(GL_TEXTURE_2D);


    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    testGLError("glTexParameteri");

    glUniform1i(uSampler0_loc, 0);
    testGLError("glUniform1i");

};

/*!*********************************************************************************************************************
\return		Whether the function succeeds or not.
\brief	Initializes shaders, buffers and other state required to begin rendering with OpenGL ES
***********************************************************************************************************************/
bool initializeShaders()
{
    shaderProgram = renderutils_createProgram(vertexShaderSource, fragmentShaderSource);
    glUseProgram(shaderProgram);
	testGLError("glUseProgram");

	uSampler0_loc       = glGetUniformLocation(shaderProgram, "uSampler0");
	uOutputSize_loc     = glGetUniformLocation(shaderProgram, "OutputSize");
	uTextureSize_loc    = glGetUniformLocation(shaderProgram, "TextureSize");
	aVertexPosition_loc = glGetAttribLocation(shaderProgram, "aVertexPosition");
	aTextureCoord1_loc  = glGetAttribLocation(shaderProgram, "aTextureCoord1");
	
	createTex();

	return true;
};

/*!*********************************************************************************************************************
\return		Whether the function succeeds or not.
\brief	Initializes shaders, buffers and other state required to begin rendering with OpenGL ES
***********************************************************************************************************************/
bool initializeBuffer()
{
	// set up vertices
	glGenBuffers(1, &vertexData_ID);
	glBindBuffer(GL_ARRAY_BUFFER, vertexData_ID);
	glBufferData(GL_ARRAY_BUFFER, sizeof(vertexData), vertexData, GL_STATIC_DRAW);
	testGLError("glBufferData");
	glVertexAttribPointer(aVertexPosition_loc, 3, GL_FLOAT, GL_FALSE, (stride)*sizeof(float), 0);
	glVertexAttribPointer(aTextureCoord1_loc, 2, GL_FLOAT, GL_FALSE, (stride)*sizeof(float), (GLvoid*)(3*sizeof(float)));
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	// set up indices
	glGenBuffers(1, &indices_ID);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, indices_ID);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indices), indices, GL_STATIC_DRAW);
	testGLError("glBufferData");
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	return true;
};


/*!*********************************************************************************************************************
\return		Whether the function succeeds or not.
\brief	Sets clear color, enables shaders, vertex arrays and draws triangle
***********************************************************************************************************************/
bool renderScene()
{

	glUseProgram(shaderProgram);
    testGLError("glUseProgram");
	
	// clear screen
    glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
    testGLError("glClearColor");
    glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
	testGLError("glClear");

	// set up vertices to draw
	glEnableVertexAttribArray(aVertexPosition_loc);
    glEnableVertexAttribArray(aTextureCoord1_loc);
	testGLError("glEnableVertexAttribArray");
	glBindBuffer(GL_ARRAY_BUFFER, vertexData_ID);
	testGLError("glBindBuffer");
	// glVertexAttribPointer must be called every draw
	glVertexAttribPointer(aVertexPosition_loc, 3, GL_FLOAT, GL_FALSE, (stride)*sizeof(float), 0);
	glVertexAttribPointer(aTextureCoord1_loc, 2, GL_FLOAT, GL_FALSE, (stride)*sizeof(float), (GLvoid*)(3*sizeof(float)));
	testGLError("glVertexAttribPointer");
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, indices_ID);
	testGLError("glBindBuffer");

	glDeleteTextures(1, &Tex);
	createTex();

	// draw
    glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_SHORT, 0);
    testGLError("glDrawElements");
	return true;
};

/*!*********************************************************************************************************************
\brief	Releases the resources created by "InitializeGLState"
***********************************************************************************************************************/
void deInitializeGLState()
{
	glDeleteProgram(shaderProgram);
	return;
};