/**
 * @file   UART_v1.c
 *
 * @brief  This file implements the combined EDMA based and non-EDMA based
 *         UART driver interface APIs for IP of version V1
 */
/*
 * Copyright (c) 2014-2017, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdint.h>
#include <ti/drv/uart/UART.h>
#include <ti/drv/uart/src/v1/UART_v1.h>
#include <ti/csl/src/ip/uart/V1/uart.h>
#include <ti/csl/src/ip/uart/V1/hw_uart.h>
#include <ti/drv/uart/src/UART_osal.h>
#ifdef UART_DMA_ENABLE
#include <ti/sdo/edma3/drv/edma3_drv.h>
#endif
#include <ti/csl/src/ip/uart/V1/cslr_uart.h>
#include <ti/drv/uart/src/UART_drv_log.h>

#define UART_TPOLL_MSECS                    ((uint32_t)1U)
/**< Poll time in milliSeconds                                                */

#define UART_TXRX_ENABLE                    ((uint32_t)0xE003U)
/**< Mask to enable Rx and Tx                                                 */

#define UART_TXRX_DISABLE                   ((uint32_t)0x0000U)
/**< Mask to edisable Rx and Tx                                               */

#define UART_OPT_TCINTEN_SHIFT              ((uint32_t)0x00000014U)
/**< Interrupt enable bit in OPT register for edma                            */

#define UART_OPT_TCINTEN          ((uint32_t)1U)
/**< Interrupt enable bit value in OPT register for edma                      */

#define UART_MAX_PWRM_EVENTS                (PWRM_INVALIDEVENT)
/**< Max number of PWRM events for which the uart will register               */

#define UART_MAX_PWRM_CONSTRAINTS           (PWRM_DISALLOWEDSLEEPSTATE_MASK)
/**< Max number of constraints for which a moudule can register               */

#define UART_TRANSMITTER_EMPTY              ((uint32_t)0x20U)
/**< Bit position to find if the transmitter is empty                         */

#define UART_FIFO_DISABLED_CHUNK_SIZE           ((uint32_t)1U)
/**< Chunk Size When FIFO is Disabled                                         */

#define UART_ISR_STATUS             ((uint32_t)0x01U)
/**< ISR status initial value                                                 */

#define UART_OS_CHECKTIMEOUT_COUNT_VALUE    (0xFFFFFFFFFFFFFFFFU)
/**< Count Value for OS CHECK TIME OUT                                        */

#define UART_DELAY_COUNT_VALUE          ((uint32_t)0xFFFFFU)
/**< Delay Count Value                                                        */

#define UART_INVALID_INT_NUM            ((uint32_t)0xFFU)
/**< Invalid Interrupt Number                                                 */

#define UART_EDMA_BASE_ADDR                 ((uint32_t)0x40000000U)
/**< EDMA_BASE_ADDR for DM814x DSP                                            */

#define UART_EDMA3CC_OPT_TCC_MASK           ((uint32_t)0x0003F000U)
/**< EDMA OPT TCC Mask value                                                  */

#define UART_EDMA3CC_OPT_TCC_SHIFT          ((uint32_t)0x0000000CU)
/**< EDMA OPT TCC Shift value                                                 */

#define UART_ERROR_COUNT            ((uint32_t)0x00FFFFFFU)
/**< Count Value to check error in the recieved byte                          */

#define UART_EDMA3CC_OPT_FIFO_WIDTH         ((uint32_t)0xFFFFF8FFU)
/**< Set FIFO Width for edma transfer                                         */

#define UART_EDMA3CC_OPT_SAM_CONST_MODE         ((uint32_t)0x00000001U)
/**< Set SAM in Constant Addressing Mode                                      */

#define UART_EDMA3CC_OPT_DAM_CONST_MODE         ((uint32_t)0x00000002U)
/**< Set DAM in Constant Addressing Mode                                      */

#define UART_EDMA3CC_OPT_SAM_INCR_MODE          ((uint32_t)0xFFFFFFFEU)
/**< Set SAM in Increment Mode                                                */

#define UART_EDMA3CC_OPT_DAM_INCR_MODE          ((uint32_t)0xFFFFFFFDU)
/**< Set DAM in Increment Mode                                                */

#define UART_EDMA3CC_OPT_SYNC_AB            ((uint32_t)0x00000004U)
/**< It is AB-synchronized                                                    */

#define UART_EDMA3CC_OPT_SYNC_MASK_VALUE            ((uint32_t)0xFFFFFFFFU)
/**< Mask Value for Transfer Synchronization                                  */

#define UART_EDMA3CC_PARAM_LINK_ADDRESS         ((uint32_t)0xFFFFU)
/**< Set link Address                                                         */

#define UART_EDMA3CC_PARAM_LINK_ADDR_MASK_VALUE         ((uint32_t)0x0000FFFFU)
/**< link Address Mask Value                                                  */

#define UART_EDMA3CC_PARAM_ACNT         ((uint32_t)1U)
/**< aCnt Value                                                               */

#define UART_EDMA3CC_SRC_BINDEX         ((uint32_t)1U)
/**< Src BIndex Value                                                         */

#define UART_EDMA3CC_DST_BINDEX         ((uint32_t)1U)
/**< Dst BIndex Value                                                         */

#define UART_EDMA3CC_OPT_SYNC_A         ((uint32_t)0x01U)
/**< It is A-synchronized                                                     */

#define UART_EDMA3CC_OPT_SYNCDIM_SHIFT          ((uint32_t)3U)
/**< Transfer synchronization dimension Shift Value                           */

#define UART_EDMA3CC_COUNT_VALUE           ((uint32_t)0xFFFFU)
/**< Count Value                                                              */

#define UART_TX_FIFO_WAIT_PERIOD        ((uint32_t)1U)

/* UART_v1 functions */
static void         UART_close_v1(UART_Handle handle);
static void         UART_init_v1(UART_Handle handle);
static UART_Handle  UART_open_v1(UART_Handle handle, const UART_Params *params);
static int32_t          UART_read_v1(UART_Handle handle, void *buffer,
                                      size_t size);
static int32_t          UART_readPolling_v1(UART_Handle handle, void *buf,
                                             size_t size);
static void         UART_readCancel_v1(UART_Handle handle);
static int32_t          UART_write_v1(UART_Handle handle, const void *buffer,
                                       size_t size);
static int32_t          UART_writePolling_v1(UART_Handle handle,
                                              const void *buf, size_t size);
static void         UART_writeCancel_v1(UART_Handle handle);

static  int32_t         UART_control_v1(UART_Handle handle,
                                               uint32_t cmd,
                                               void *arg);
static int32_t      UART_read2_v1(UART_Handle handle, UART_Transaction *transaction);
static int32_t      UART_write2_v1(UART_Handle handle, UART_Transaction *transaction);

static void         UART_v1_hwiIntFxn(uintptr_t arg);
static void UARTFifoWait(uint32_t baseAdd);
static bool UART_writeCancelNoCB(UART_Handle handle);
static bool UART_readCancelNoCB(UART_Handle handle);
#ifdef UART_DMA_ENABLE
static void UART_v1_receiveDMA(UART_Handle handle, const void *buffer, size_t size);
static void UART_transmitDMA(UART_Handle handle, const void *buffer, size_t size );
static int32_t  UART_configDMA(UART_Handle handle );
static void UART_rxIsrHandler(uint32_t tcc, EDMA3_RM_TccStatus status, void* appData);
static void UART_txIsrHandler(uint32_t tcc, EDMA3_RM_TccStatus status, void* appData);
static void UART_doNothing(uint32_t tcc, EDMA3_RM_TccStatus edmaStatus, void* appData);
#endif

/* UART function table for UART_v1 implementation */
const UART_FxnTable UART_FxnTable_v1 = {
    &UART_close_v1,
    &UART_control_v1,
    &UART_init_v1,
    &UART_open_v1,
    &UART_read_v1,
    &UART_readPolling_v1,
    &UART_readCancel_v1,
    &UART_write_v1,
    &UART_writePolling_v1,
    &UART_writeCancel_v1,
    &UART_read2_v1,
    &UART_write2_v1,

};

#ifdef UART_DMA_ENABLE
static void UART_doNothing(uint32_t tcc, EDMA3_RM_TccStatus edmaStatus, void* appData)
{

}
#endif

/*
 *  ======== UART_writeData_v1 ========
 *  Write and process data to the UART.
 */
static inline int32_t UART_writeData_v1(UART_Handle handle, int32_t size); /* for misra warnings*/
static inline int32_t UART_writeData_v1(UART_Handle handle, int32_t size)
{
    UART_V1_Object     *object;
    UART_HwAttrs const *hwAttrs;

    object = (UART_V1_Object*)handle->object;
    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

    uint8_t FIFO_FULL_FLAG = 0U;

    /* Send characters until FIFO is full or done. */
    while ((size != 0) && (FIFO_FULL_FLAG == 0U))
    {
        /* If mode is TEXT process the characters */
        if (object->params.writeDataMode == UART_DATA_TEXT)
        {
            if (object->writeCR)
            {
                if (UARTTxFIFOFullStatusGet(hwAttrs->baseAddr) ==
                    UART_TX_FIFO_FULL)
                {
                    /* Character was not sent, FIFO full */
                    FIFO_FULL_FLAG = 1U;
                }
                /* FIFO not full, write '\r' to FIFO */
                else
                {
                    UARTFIFOCharPut(hwAttrs->baseAddr, ((uint8_t)('\r')));
                    size--;
                    object->writeCount++;
                    object->writeCR = 0;

                    UART_drv_log1("UART:(0x%x) Wrote character ", hwAttrs->baseAddr);
                }
            }
            else
            {
                /* Add a return if next character is a newline. */
                if (*(uint8_t *)object->writeBuf == ((uint8_t)('\n')))
                {
                    size++;
                    object->writeCR = true;
                }

                if (UARTTxFIFOFullStatusGet(hwAttrs->baseAddr) ==
                    UART_TX_FIFO_FULL)
                {
                    /* FIFO full */
                    if (object->writeCR)
                    {
                        size--;
                        object->writeCR = false;
                    }
                    FIFO_FULL_FLAG = 1U;
                }
                /* FIFO not full, write to FIFO */
                else
                {
                    UARTFIFOCharPut(hwAttrs->baseAddr, *(uint8_t *)object->writeBuf);
                    object->writeBuf = (uint8_t *)object->writeBuf + 1;

                    UART_drv_log2("UART:(0x%x) Wrote character 0x%x",
                                  hwAttrs->baseAddr, *(uint8_t *)object->writeBuf);

                    size--;
                    object->writeCount++;
                }
            }
        }
        else
        {
            if (UARTTxFIFOFullStatusGet(hwAttrs->baseAddr) ==
                UART_TX_FIFO_FULL)
            {
                /* FIFO full */
                FIFO_FULL_FLAG = 1U;
            }
            /* FIFO not full, write character */
            else
            {
                UARTFIFOCharPut(hwAttrs->baseAddr, *(uint8_t *)object->writeBuf);
                object->writeBuf = (uint8_t *)object->writeBuf + 1;

                UART_drv_log2("UART:(0x%x) Wrote character 0x%x",
                              hwAttrs->baseAddr, *(uint8_t *)object->writeBuf);

                size--;
                object->writeCount++;
            }
        }
    }
    return (size);
}

/*
 *  ======== UART_v1_readData ========
 *  Read and process data from the UART.
 */
static inline int32_t UART_v1_readData(UART_Handle handle, int32_t size);   /* for misra warnings*/
static inline int32_t UART_v1_readData(UART_Handle handle, int32_t size)
{
    int32_t             readIn;
    UART_V1_Object     *object;
    UART_HwAttrs const *hwAttrs;

    object = (UART_V1_Object*)handle->object;
    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    readIn = (int32_t)UARTCharGetNonBlocking(hwAttrs->baseAddr);

    /* Receive chars until empty or done. */
    while ((size != 0) && (readIn != (int32_t)(-1)))
    {
        /* If data mode is set to TEXT replace return with a newline. */
        if (object->params.readDataMode == UART_DATA_TEXT)
        {
            if ((uint8_t)readIn == ((uint8_t)('\r')))
            {
                /* Echo character if enabled. */
                if (object->params.readEcho)
                {
                    UARTCharPut(hwAttrs->baseAddr, ((uint8_t)('\r')));
                }
                readIn = (int32_t)'\n';
            }
        }

        UART_drv_log2("UART:(0x%x) Read character 0x%x",
                      hwAttrs->baseAddr, (uint8_t)readIn);

        *(uint8_t *)object->readBuf = (uint8_t)readIn;
        object->readBuf = (uint8_t *)object->readBuf + 1;
        object->readCount++;
        size--;

        /* Echo character if enabled. */
        if (object->params.readEcho)
        {
            UARTCharPut(hwAttrs->baseAddr, (uint8_t)readIn);
        }

        /* If read return mode is newline, finish if a newline was received. */
        if ((object->params.readReturnMode == UART_RETURN_NEWLINE) &&
            ((uint8_t)readIn == ((uint8_t)('\n'))))
        {
            UART_drv_log1("UART:(0x%x) Newline character received, ",
            		      hwAttrs->baseAddr);
            size = 0;
            break;
        }
        readIn = (int32_t)UARTCharGetNonBlocking(hwAttrs->baseAddr);
    }

    return (size);
}

static void UART_v1_callback(UART_Handle handle, bool readTrans);   /* for misra warnings*/
static void UART_v1_callback(UART_Handle handle, bool readTrans)
{
    UART_V1_Object *object = (UART_V1_Object*)handle->object;

    /* Call back to application */
    if (readTrans == true)
    {
        if (object->params.readCallback2)
        {
            object->params.readCallback2(handle, object->readTrans);
        }
        else if (object->params.readCallback)
        {
            object->params.readCallback(handle,
                                        object->readBuf,
                                        object->readCount);
        }
        else
        {
            UART_osalPostLock(object->readSem);
        }
    }
    else
    {
        if (object->params.writeCallback2)
        {
            object->params.writeCallback2(handle, object->writeTrans);
        }
        else if (object->params.writeCallback)
        {
            object->params.writeCallback(handle,
                                         (void *)object->writeBuf,
                                         object->writeCount);
        }
        else
        {
            UART_osalPostLock(object->writeSem);
        }
    }
}

inline void UART_procLineStatusErr(UART_Handle handle);   /* for misra warnings*/
inline void UART_procLineStatusErr(UART_Handle handle)
{
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    uint32_t            lineStatus;

    if (object->readSize)
    {
        lineStatus = UARTReadStatus(hwAttrs->baseAddr);

        if (((lineStatus & UART_FIFO_PE_FE_BI_DETECTED) == UART_FIFO_PE_FE_BI_DETECTED) ||
            ((lineStatus & UART_OVERRUN_ERROR) == UART_OVERRUN_ERROR))
        {
            /* empty the RX FIFO which contains data with errors */
#ifdef UART_DMA_ENABLE
            if (hwAttrs->dmaMode == TRUE)
            {
                EDMA3_DRV_disableLogicalChannel(
                        (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                        (uint32_t) hwAttrs->rxDmaEventNumber,
                        (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);
                if (object->readTrans != NULL)
                {
                    object->readTrans->count = 0;
                }
                else
                {
                    object->readCount = 0;
                }
            }
            else
#endif
            {
                if (object->readTrans != NULL)
                {
                    object->readTrans->count = object->readCount;
                }
            }
            /* dummy read, clear the RX line status interrupt */
            UART_v1_readData(handle, object->readSize);
            UARTIntDisable(hwAttrs->baseAddr, UART_INT_RHR_CTI | UART_INT_LINE_STAT);

            /* Reset the read buffer and read count so we can pass it back */
            object->readBuf = (uint8_t *)object->readBuf - object->readCount;

            if (object->readTrans != NULL)
            {
                if (lineStatus & UART_BREAK_DETECTED_ERROR)
                {
                    object->readTrans->status = UART_TRANSFER_STATUS_ERROR_BI;
                }
                else if (lineStatus & UART_FRAMING_ERROR)
                {
                    object->readTrans->status = UART_TRANSFER_STATUS_ERROR_FE;
                }
                else if (lineStatus & UART_PARITY_ERROR)
                {
                    object->readTrans->status = UART_TRANSFER_STATUS_ERROR_PE;
                }
                else
                {
                    object->readTrans->status = UART_TRANSFER_STATUS_ERROR_OE;
                }
            }

            /* Call back to application if in callback mode */
            UART_v1_callback(handle, true);
            object->readTrans = NULL;
        }
    }
}

/*
 *  ======== UART_v1_hwiIntFxn ========
 *  Hwi function that processes UART interrupts.
 *
 *  In non-DMA mode, four UART interrupts are enabled:
 *    1. transmit FIFO is below the TX FIFO trigger level (THR)
 *    2. receive FIFO is above RX FIFO trigger level (RHR)
 *    3. line status rx error
 *    4. TX FIFO empty
 *
 *  ISR checks the three interrupts in a while(1) loop, to ensure that all
 *  the pending interrupts are handled before exiting.
 *
 *  If line status rx error is detected, ISR clears the last read error, update
 *  the actual number of bytes transferred and transfer status in readTrans and
 *  calls back to application in the callback mode or post the read semaphone
 *  in the blocking mode. ISR
 *
 *  If RHR interrupt is received, ISR calls the in-lined function readData to
 *  read the data out from the RX FIFO. When all the data are received, ISR updates
 *  the actual number of bytes transferred and transfer status in readTrans and
 *  calls back to the application in the callback mode or post the read semaphone
 *  in the blocking mode. if RX timeout is detected, ISR will log the timeout
 *  count.
 *
 *  If THR interrupt is received, ISR calls the in-lined function writeData,
 *  to write the data to the TX FIFO. After all the data are sent, TX FIFO empty 
 *  interrupt is enabled, ISR will update the actual number of bytes transferred 
 *  and transfer status in writeTrans and calls back to application in the callback 
 *  mode or post the read semaphone in the blocking mode.
 *  
 *  TX FIFO empty interrupt is enabled after all the data are written to the TX
 *  FIFO, ISR will 
 *
 *  In DMA mode, three UART interrupt is enabled:
 *    1. line status rx error
 *    2. receive FIFO is above RX FIFO trigger level (RHR)
 *    3. TX FIFO empty
 *
 *  If line status rx error is detected, ISR clears the last read error, update
 *  the actual number of bytes transferred and transfer status in readTrans,
 *  disables the DMA RX channel and calls back to application in the callback mode
 *  or post the read semaphone in the blocking mode.
 *
 *  DMA RX transfer completion is handled in UART_rxIsrHandler. If the read size 
 *  is not multiple of RX threshold size, RHR interrupt is enabled to receive the 
 *  remaining bytes in ISR.  
 *  
 *  TX transfer completion is handled in UART_txIsrHandler. After all the data are 
 *  sent, TX FIFO empty interrupt is enabled, ISR will update the actual number of 
 *  bytes transferred and transfer status in writeTrans and calls back to application 
 *  in the callback mode or post the read semaphone in the blocking mode.
 *
 *  @param(arg)         The UART_Handle for this Hwi.
 */
static void UART_v1_hwiIntFxn(uintptr_t arg);   /* for misra warnings*/
static void UART_v1_hwiIntFxn(uintptr_t arg)
{
    UART_Handle         handle = (UART_Handle)arg;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    uint32_t            intType;

    while (TRUE)
    {
        intType = UARTIntIdentityGet(hwAttrs->baseAddr);
        
        if ((intType & UART_INTID_RX_THRES_REACH) == UART_INTID_RX_THRES_REACH)
        {
            if ((intType & UART_INTID_RX_LINE_STAT_ERROR) ==
                UART_INTID_RX_LINE_STAT_ERROR)
            {
                /* RX line status error */
                UART_procLineStatusErr(handle);
            }
            else
            {
                if ((intType & UART_INTID_CHAR_TIMEOUT) == UART_INTID_CHAR_TIMEOUT)
                {
                    /* rx timeout, log the rx timeout errors */
                    object->rxTimeoutCnt++;
                }

                /* RX FIFO threshold reached */
                if (object->readSize)
                {
                    object->readSize = UART_v1_readData(handle, object->readSize);
                    if ((object->readSize) == 0U)
                    {
                        UARTIntDisable(hwAttrs->baseAddr, UART_INT_RHR_CTI | UART_INT_LINE_STAT);
                        /* Reset the read buffer so we can pass it back */
                        object->readBuf = (uint8_t *)object->readBuf - object->readCount;
                        if (object->readTrans != NULL)
                        {
                            object->readTrans->count = object->readCount;
                            object->readTrans->status = UART_TRANSFER_STATUS_SUCCESS;
                        }

                        /* Call back to application if in callback mode */
                        UART_v1_callback(handle, true);
                        object->readTrans = NULL;
                    }
                }
            }
        }
        else if ((intType & UART_INTID_TX_THRES_REACH) == UART_INTID_TX_THRES_REACH)
        {
            /* TX FIFO threshold reached */
            if (object->writeSize)
            {
                object->writeSize = UART_writeData_v1(handle, object->writeSize);
                if ((object->writeSize) == 0U)
                {
                    UARTIntDisable(hwAttrs->baseAddr, UART_INT_THR);
                    /* Reset the write buffer so we can pass it back */
                    object->writeBuf = (uint8_t *)object->writeBuf - object->writeCount;

                    if (object->writeTrans != NULL)
                    {
                        object->writeTrans->count = object->writeCount;
                        object->writeTrans->status = UART_TRANSFER_STATUS_SUCCESS;
                    }

                    object->txDataSent = TRUE;
                    UARTInt2Enable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);
                }
            }
        }
        else
        {
            break;
        }
    }

    if (object->txDataSent == TRUE)
    {
        intType = UARTInt2StatusGet(hwAttrs->baseAddr);
        if (intType & UART_INT2_TX_EMPTY)
        {
            UARTInt2Disable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);

            /* Call back to application if in callback mode */
            UART_v1_callback(handle, false);
            object->writeTrans = NULL;
            object->txDataSent = FALSE;
        }
    }
}

/*
 *  ======== UART_init_v1 ========
 */
static void UART_init_v1(UART_Handle handle)
{
    /* Mark the object as available */
    ((UART_V1_Object *)(handle->object))->isOpen = false;
}

static uint32_t UART_getTxTrigLvl(UART_TxTrigLvl trigLevel);   /* for misra warnings*/
static uint32_t UART_getTxTrigLvl(UART_TxTrigLvl trigLevel)
{
    uint32_t value;

    switch (trigLevel)
	{
        case UART_TXTRIGLVL_8:
            value = UART_FCR_TX_TRIG_LVL_8;
            break;

        case UART_TXTRIGLVL_16:
            value = UART_FCR_TX_TRIG_LVL_16;
            break;

        case UART_TXTRIGLVL_32:
            value = UART_FCR_TX_TRIG_LVL_32;
            break;

        case UART_TXTRIGLVL_56:
            value = UART_FCR_TX_TRIG_LVL_56;
            break;

        default:
        	value = 0;
            break;
	}

    return (value);
}

static uint32_t UART_getRxTrigLvl(UART_RxTrigLvl trigLevel);   /* for misra warnings*/
static uint32_t UART_getRxTrigLvl(UART_RxTrigLvl trigLevel)
{
    uint32_t value;

    switch (trigLevel)
	{
        case UART_RXTRIGLVL_8:
            value = UART_FCR_RX_TRIG_LVL_8;
            break;

        case UART_RXTRIGLVL_16:
            value = UART_FCR_RX_TRIG_LVL_16;
            break;

        case UART_RXTRIGLVL_56:
            value = UART_FCR_RX_TRIG_LVL_56;
            break;

        case UART_RXTRIGLVL_60:
            value = UART_FCR_RX_TRIG_LVL_60;
            break;

        default:
        	value = 0;
            break;
	}

    return (value);
}

/*
 *  ======== UART_open_v1 ========
 */
static UART_Handle UART_open_v1(UART_Handle handle, const UART_Params *params)
{
    uint32_t            key;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    SemaphoreP_Params   semParams;
    uint32_t            divisorValue;
    uint32_t            txTrigLevel;
    uint32_t            rxTrigLevel;
    uint32_t            regVal = 0x00;
    OsalRegisterIntrParams_t interruptRegParams;


    /* If params are NULL use defaults. */
    if (params == NULL) {
        UART_Params_init(&object->params);
    }
    else {
        /* Save UART parameters. */
        object->params = *params;
    }

    /* Disable preemption while checking if the UART is open. */
    key = UART_osalHardwareIntDisable();

    /* Check if the UART is open already with the base addr. */
    if(object->isOpen == true) {
        UART_osalHardwareIntRestore(key);
       UART_drv_log1("UART:(0x%x) already in use.", hwAttrs->baseAddr);

       handle = NULL;
    }

    /* UART is open by setting base addr */
    else
    {
        object->isOpen = true;
        UART_osalHardwareIntRestore(key);

        /* Set UART variables to defaults. */
        object->writeBuf = NULL;
        object->readBuf = NULL;
        object->writeCount = 0;
        object->readCount = 0;
        object->writeSize = 0;
        object->readSize = 0;
        object->writeCR = false;
        object->readTrans = NULL;
        object->writeTrans = NULL;
            /* Disable UART interrupts. */
        UARTIntDisable(hwAttrs->baseAddr,
                       UART_INT_RHR_CTI | UART_INT_THR | UART_INT_LINE_STAT);
        UARTInt2Disable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY | UART_INT2_RX_EMPTY);

        if(hwAttrs->enableInterrupt)
        {
            /* Construct Hwi object for this UART peripheral. */
            /* Initialize with defaults */
            Osal_RegisterInterrupt_initParams(&interruptRegParams);

             /* Populate the interrupt parameters */
            interruptRegParams.corepacConfig.arg=(uintptr_t)handle;
            interruptRegParams.corepacConfig.name=NULL;
            interruptRegParams.corepacConfig.isrRoutine=UART_v1_hwiIntFxn;
            interruptRegParams.corepacConfig.priority=0x20U;
            interruptRegParams.corepacConfig.corepacEventNum=hwAttrs->eventId; /* Event going in to CPU */
            interruptRegParams.corepacConfig.intVecNum=hwAttrs->intNum; /* Host Interrupt vector */

             /* Register interrupts */
            UART_osalRegisterInterrupt(&interruptRegParams,&(object->hwi));

            if(object->hwi == NULL)
            {
                handle = NULL;
            }
            else
            {
                UART_osalSemParamsInit(&semParams);
                semParams.mode = SemaphoreP_Mode_BINARY;
                /* If write mode is blocking construct a lock and set callback to NULL. */
                if (object->params.writeMode == UART_MODE_BLOCKING) {
                    semParams.name = "write";
                    object->writeSem = UART_osalCreateBlockingLock(0, &semParams);
                    object->params.writeCallback = NULL;
                    object->params.writeCallback2 = NULL;
                }

                /* If read mode is blocking create a lock and set callback to NULL. */
                if (object->params.readMode == UART_MODE_BLOCKING) {
                    semParams.name = "read";
                    object->readSem = UART_osalCreateBlockingLock(0, &semParams);
                    object->params.readCallback = NULL;
                    object->params.readCallback2 = NULL;
                }
            }
        }

        if (handle != NULL)
        {
            /* Set up the TX and RX FIFO Trigger levels. */
            txTrigLevel = UART_getTxTrigLvl(hwAttrs->txTrigLvl);
            rxTrigLevel = UART_getRxTrigLvl(hwAttrs->rxTrigLvl);
#ifdef UART_DMA_ENABLE
            if (hwAttrs->dmaMode == TRUE)
            {
                regVal = UART_FIFO_CONFIG(UART_TRIG_LVL_GRANULARITY_4,
                                          UART_TRIG_LVL_GRANULARITY_4,
                                          txTrigLevel,
                                          rxTrigLevel,
                                          1U,
                                          1U,
                                          UART_DMA_EN_PATH_FCR,
                                          UART_DMA_MODE_1_ENABLE);
                /* Configuring the FIFO settings. */
                UARTFIFOConfig(hwAttrs->baseAddr, regVal);

                UART_configDMA(handle);
            }
            else
#endif
            {
                regVal = UART_FIFO_CONFIG(UART_TRIG_LVL_GRANULARITY_4,
                                          UART_TRIG_LVL_GRANULARITY_4,
                                          txTrigLevel,
                                          rxTrigLevel,
                                          1U,
                                          1U,
                                          UART_DMA_EN_PATH_FCR,
                                          UART_DMA_MODE_0_ENABLE);
                /* Configuring the FIFO settings. */
                UARTFIFOConfig(hwAttrs->baseAddr, regVal);
            }
            /* Computing the Divisor Value for params.baudRate */
            divisorValue = UARTDivisorValCompute(hwAttrs->frequency,
                                                 object->params.baudRate,
                                                 UART16x_OPER_MODE,
                                                 UART_MIR_OVERSAMPLING_RATE_42);
            /* Configuring the Baud Rate settings. */
            UARTDivisorLatchWrite(hwAttrs->baseAddr, divisorValue);

            /* Configuring UART line characteristics */
            UARTLineCharacConfig(hwAttrs->baseAddr,
                                 (object->params.dataLength | object->params.stopBits),
                                 (object->params.parityType << CSL_UART_LCR_PARITY_EN_SHIFT));

            /* Disable write access to Divisor Latches. */
            UARTDivisorLatchDisable(hwAttrs->baseAddr);

            /* Disable Break Control. */
            UARTBreakCtl(hwAttrs->baseAddr, UART_BREAK_COND_DISABLE);
            if (hwAttrs->loopback == TRUE)
            {
                UARTLoopbackModeControl(hwAttrs->baseAddr, UART_LOOPBACK_MODE_ENABLE);
            }
            else
            {
                UARTLoopbackModeControl(hwAttrs->baseAddr, UART_LOOPBACK_MODE_DISABLE);
            }

            /* Enable UART */
            UARTOperatingModeSelect(hwAttrs->baseAddr, UART16x_OPER_MODE);

            UART_drv_log3("UART:(0x%x) CPU freq: %d; UART baudrate to %d",
                hwAttrs->baseAddr, hwAttrs->frequency, object->params.baudRate);

            UART_drv_log1("UART:(0x%x) opened", hwAttrs->baseAddr);
        }
    }
    /* Return the handle of the UART_Object. */
    return (handle);
}


/*
 *  ======== UART_close_v1 ========
 */
static void UART_close_v1(UART_Handle handle)
{
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

    UARTFifoWait(hwAttrs->baseAddr);

    /* Disable UART and interrupts. */
    UARTIntDisable(hwAttrs->baseAddr,
                   UART_INT_RHR_CTI | UART_INT_THR | UART_INT_LINE_STAT);
    UARTInt2Disable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);
    UARTOperatingModeSelect(hwAttrs->baseAddr, UART_DISABLED_MODE);

#ifdef UART_DMA_ENABLE
    if (hwAttrs->dmaMode == TRUE)
    {
        EDMA3_DRV_unlinkChannel(
                hwAttrs->edmaHandle,
                object->edmaLinkChId);

        EDMA3_DRV_freeChannel(
                    hwAttrs->edmaHandle,
                    object->edmaLinkChId);

        /* Free the DMA TX/RX channels */
        EDMA3_DRV_freeChannel(hwAttrs->edmaHandle,
                              hwAttrs->rxDmaEventNumber);

        EDMA3_DRV_freeChannel(hwAttrs->edmaHandle,
                              hwAttrs->txDmaEventNumber);
    }
#endif

    if(hwAttrs->enableInterrupt) {
      /* Delete the the SYS/BIOS objects. */
      UART_osalHardwareIntDestruct(object->hwi,hwAttrs->eventId);

       if (object->params.writeMode == UART_MODE_BLOCKING)
       {
          UART_osalDeleteBlockingLock(object->writeSem);
       }
       if (object->params.readMode == UART_MODE_BLOCKING)
       {
         UART_osalDeleteBlockingLock(object->readSem);
       }
    }

    object->isOpen = false;

    UART_drv_log1("UART:(0x%x) closed", hwAttrs->baseAddr);

}


static int32_t UART_write_v1(UART_Handle handle, const void *buffer, size_t size)
{
    uint32_t           key;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    int32_t            ret_val = 0;
    UART_Transaction   transaction;
    UART_HwAttrs *hwAttrs;

    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;


    if (hwAttrs->enableInterrupt) {
    /* Disable preemption while checking if the uart is in use. */
    key = UART_osalHardwareIntDisable();

  	if (object->writeSize)
      {
        UART_osalHardwareIntRestore(key);
        ret_val = UART_ERROR;
      }
      /* Save the data to be written and restore interrupts. */
      else
      {
        UART_osalHardwareIntRestore(key);

        transaction.buf = (void *)buffer;
    	transaction.timeout = object->params.writeTimeout;
    	transaction.count = (uint32_t)size;
    	UART_write2_v1(handle, &transaction);

    	ret_val = transaction.count;
      }
    }
    else
    {
        /* Use polling if interrupt is disabled */
       ret_val = UART_writePolling_v1(handle,buffer,size);
    }
    return(ret_val);
}

static int32_t UART_write2_v1(UART_Handle handle, UART_Transaction * transaction)
{
    uint32_t            key;
    UART_V1_Object     *object;
    UART_HwAttrs const *hwAttrs;
    int32_t             ret_val = UART_SUCCESS;
    SemaphoreP_Status   semStatus;

    /* Input parameter validation */
    OSAL_Assert(!((handle != NULL) && (transaction != NULL)));

    if (transaction->count == 0)
    {
        ret_val = UART_ERROR;
    }
    else
    {
        /* Get the pointer to the hwAttrs */
        hwAttrs = (UART_HwAttrs *)handle->hwAttrs;

        /* Get the pointer to the object */
        object = (UART_V1_Object*)handle->object;

        if(hwAttrs->enableInterrupt)
        {
            /* Disable preemption while checking if the uart is in use. */

             key = UART_osalHardwareIntDisable();

             if (object->writeSize)
             {
                 UART_osalHardwareIntRestore(key);

                 ret_val = UART_ERROR;
             }

             /* Save the data to be written and restore interrupts. */
             else
             {

                   transaction->status = UART_TRANSFER_STATUS_SUCCESS;

                 /*
                  * When legacy UART_write API is called in callback mode,
                  * object->params.writeCallback will save the callback function
                  * pointer passed from the application, otherwise
                  * object->params.writeCallback will be set to NULL.
                  *
                  * If legacy UART_write API is called in callback mode,
                  * transaction parameter passed to UART_write2_v1 is
                  * declared in the stack, and cannot be used in the ISR,
                  * otherwise will cause data corruption in the stack.
                  * The driver should only use object->writeTrans when it
                  * is NOT called from egacy UART_write API in callback mode
                  */
                 if (object->params.writeCallback == NULL)
                 {
                     object->writeTrans = transaction;
                 }
                 else
                 {
                     object->writeTrans = NULL;
                 }
                 object->writeBuf = transaction->buf;
                 object->writeCount = 0;
                 object->txDataSent = FALSE;

#ifdef UART_DMA_ENABLE
                 if (hwAttrs->dmaMode == TRUE)
                 {
                     /* DMA mode */
                     object->writeSize = transaction->count;
                     UART_osalHardwareIntRestore(key);

                     UART_transmitDMA(handle,(void*)object->writeBuf, object->writeSize);

                     /* If writeMode is blocking, block and get the status. */
                     if (object->params.writeMode == UART_MODE_BLOCKING)
                     {
                         /* Pend on semaphore and wait for Hwi to finish. */
                         semStatus = UART_osalPendLock(object->writeSem,
                                                       transaction->timeout);
                         if (semStatus == SemaphoreP_OK)
                         {
                             if (transaction->status == UART_TRANSFER_STATUS_SUCCESS)
                             {
                                 ret_val = UART_SUCCESS;
                             }
                             else
                             {
                                 /* RX errors */
                                 ret_val = UART_ERROR;
                             }
                         }
                         else
                         {
                             if (semStatus == SemaphoreP_TIMEOUT)
                             {
                                 transaction->status = UART_TRANSFER_STATUS_TIMEOUT;
                             }
                             else
                             {
                                 transaction->status = UART_TRANSFER_STATUS_ERROR_OTH;
                             }
                             /* Cancel the DMA without posting the semaphore */
                             UART_writeCancelNoCB(handle);
                             ret_val = UART_ERROR;
                         }
                     }
                     else
                     {
                         /*
                          * for callback mode, immediately return SUCCESS,
                          * once the transaction is done, callback function
                          * will return the transaction status and actual
                          * write count
                          */
                         transaction->count = 0;
                         ret_val = UART_SUCCESS;
                     }
                 }
                 else
#endif
                 {
                     /* non-DMA mode */
                     UART_osalHardwareIntRestore(key);

                     /* Disable TX interrupts */
                     UARTIntDisable(hwAttrs->baseAddr, UART_INT_THR);
                     UARTInt2Disable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);

                     object->writeSize = UART_writeData_v1(handle, (int32_t)(transaction->count));
                     if ((object->writeSize) == 0)
                     {
                         /* Write is finished. */
                         UART_drv_log2("UART(0x%x): Write finished, %d bytes written",
                                       hwAttrs->baseAddr, object->writeCount);

                         /* Reset the write buffer so we can pass it back */
                         object->writeBuf = (uint8_t *)object->writeBuf - object->writeCount;
                         if (object->writeTrans != NULL)
                         {
                             object->writeTrans->count = object->writeCount;
                             object->writeTrans->status = UART_TRANSFER_STATUS_SUCCESS;
                         }
                         object->txDataSent = TRUE;
                         UARTInt2Enable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);
                     }
                     /* If writeMode is blocking, block and get the status. */
                     else
                     {
                         UARTIntEnable(hwAttrs->baseAddr, UART_INT_THR);
                     }

                     if (object->params.writeMode == UART_MODE_BLOCKING)
                     {
                         /* Pend on lock and wait for Hwi to finish. */
                         semStatus = UART_osalPendLock(object->writeSem,
                                                       transaction->timeout);
                         if (semStatus == SemaphoreP_OK)
                         {
                             if (transaction->status == UART_TRANSFER_STATUS_SUCCESS)
                             {
                                 ret_val = UART_SUCCESS;
                             }
                             else
                             {
                                 ret_val = UART_ERROR;
                             }
                         }
                         else
                         {
                             if (semStatus == SemaphoreP_TIMEOUT)
                             {
                                 transaction->status = UART_TRANSFER_STATUS_TIMEOUT;
                             }
                             else
                             {
                                 transaction->status = UART_TRANSFER_STATUS_ERROR_OTH;
                             }
                             /* Cancel the DMA without posting the semaphore */
                             UART_writeCancelNoCB(handle);
                             ret_val = UART_ERROR;
                         }
                     }
                     else
                     {
                         /*
                          * for callback mode, immediately return SUCCESS,
                          * once the transaction is done, callback function
                          * will return the transaction status and actual
                          * write count
                          */
                         transaction->count = 0;
                         ret_val = UART_SUCCESS;
                     }
                 }
             }
        }
        else
        {
            /* Use polling if interrupt is disabled */
            transaction->count = UART_writePolling_v1(handle,transaction->buf,transaction->count);
            transaction->status = UART_TRANSFER_STATUS_SUCCESS;
        }
    }
    return(ret_val);
}

/*
 *  ======== UART_charPut_v1 ========
 *
 *  This function continuously does a non-blocking write, until it writes
 *  successfully or times out.
 */
static uint32_t UART_charPut_v1(UART_HwAttrs const *hwAttrs, uint8_t data, uint32_t *timeout);  /*for misra warnings*/
static uint32_t UART_charPut_v1(UART_HwAttrs const *hwAttrs, uint8_t data, uint32_t *timeout)
{
    uint32_t status;
    uint32_t timeoutCnt = *timeout;
    uint32_t retVal = TRUE;

    while (retVal == TRUE)
    {
        status = UARTCharPutNonBlocking(hwAttrs->baseAddr, data);
        if (status == FALSE)
        {
            if (timeoutCnt == 0)
            {
                retVal = FALSE;
            }
            else
            {
                UART_osalDelay(1U);
                timeoutCnt--;
            }
        }
        else
        {
            break;
        }
    }

    *timeout =  timeoutCnt;

    return (retVal);
}

/*
 *  ======== UART_writePolling_v1 ========
 */
static int32_t UART_writePolling_v1(UART_Handle handle, const void *buf,
                                     size_t size)
{
    int32_t             count = 0;
    uint8_t            *buffer = (uint8_t *)buf;
    uint32_t            timeout;
    uint32_t            timeoutErr = FALSE;

    /* Get the pointer to the object and hwAttrs */
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

    timeout = object->params.writeTimeout;

    /* Write characters. */
    while ((size != 0) && (timeoutErr == FALSE))
    {
        if ((object->params.writeDataMode == UART_DATA_TEXT) && (*buffer == ((uint8_t)('\n'))))
        {
            if (UART_charPut_v1(hwAttrs, ((uint8_t)('\r')), &timeout) == FALSE)
            {
                timeoutErr = TRUE;
            }
            else
            {
                count++;
            }
        }

        if (timeoutErr == FALSE)
        {
            if (UART_charPut_v1(hwAttrs, *buffer, &timeout) == FALSE)
            {
                timeoutErr = TRUE;
            }
            else
            {
                UART_drv_log2("UART:(0x%x) Wrote character 0x%x",
                    hwAttrs->baseAddr, *buffer);

                buffer++;
                count++;
                size--;
            }
        }
    }

    if (timeoutErr == TRUE)
    {
        UART_drv_log2("UART:(0x%x) Write polling timed out, %d bytes written",
                      hwAttrs->baseAddr, (int32_t)count);
    }
    else
    {
        UART_drv_log2("UART:(0x%x) Write polling finished, %d bytes written",
                      hwAttrs->baseAddr, (int32_t)count);
    }

    return (count);
}

/*
 *  ======== UART_writeCancel_v1 ========
 */
static void UART_writeCancel_v1(UART_Handle handle)
{
    UART_V1_Object     *object;
    UART_HwAttrs const *hwAttrs;

    if (UART_writeCancelNoCB(handle) == true)
    {
        object = (UART_V1_Object*)handle->object;
        hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

        /* enable the TX FIFO interrupt and callback to application in ISR */
        object->txDataSent = TRUE;
        UARTInt2Enable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);

        UART_drv_log2("UART:(%p) Write cancelled, %d bytes written",
                      ((UART_HwAttrs const *)(handle->hwAttrs))->baseAddr,
                      object->writeCount);
    }
}

#ifdef UART_DMA_ENABLE
/*
 *  ======== UART_v1_receiveDMA ========
 */

 static void UART_v1_receiveDMA(UART_Handle handle, const void *buffer, size_t size )
 {
     UART_HwAttrs const *hwAttrs;
     uint32_t            actualThreshold;
     int32_t             result = EDMA3_DRV_SOK;
     EDMA3_DRV_PaRAMRegs paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

     hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
     actualThreshold = hwAttrs->rxTrigLvl;

     if (result == EDMA3_DRV_SOK)
     {
         /* Get the PaRAM set for default parameters                               */
         EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                    (uint32_t) hwAttrs->rxDmaEventNumber, &paramSet);

         /* Fill the PaRAM Set with transfer specific information */
         paramSet.destAddr  = (uint32_t)(buffer);
         paramSet.srcAddr = (uint32_t) ((hwAttrs->baseAddr) + UART_RHR);
         /**
          * Be Careful !!!
          * Valid values for SRCBIDX/DSTBIDX are between -32768 and 32767
          * Valid values for SRCCIDX/DSTCIDX are between -32768 and 32767
          */
         paramSet.aCnt       = 1;
         paramSet.bCnt       = (uint16_t) actualThreshold;
         paramSet.srcBIdx    = 0;
         paramSet.srcCIdx    = 0;

         paramSet.destBIdx   = UART_EDMA3CC_DST_BINDEX;
         paramSet.destCIdx   =actualThreshold;

         /**
          * Be Careful !!!
          * Valid values for ACNT/BCNT/CCNT are between 0 and 65535.
          * ACNT/BCNT/CCNT must be greater than or equal to 1.
          * Maximum number of bytes in an array (ACNT) is 65535 bytes
          * Maximum number of arrays in a frame (BCNT) is 65535
          * Maximum number of frames in a block (CCNT) is 65535
          */
         if ((size <= UART_EDMA3CC_COUNT_VALUE))
         {
             paramSet.cCnt = (uint16_t)(UART_EDMA3CC_COUNT_VALUE &
                                        (size / actualThreshold));
         }
         else
         {
             result = -1;
         }

         /* For AB-synchronized transfers, BCNTRLD is not used. */
         paramSet.linkAddr   = 0xFFFFu;
         paramSet.bCntReload = (uint16_t) 0;

#if !defined(SOC_AM437x)
         /* Src is in INCR mode & Dest is in FIFO modes                            */
         paramSet.opt &= UART_EDMA3CC_OPT_DAM_INCR_MODE;
         paramSet.opt |= UART_EDMA3CC_OPT_SAM_CONST_MODE;
#endif
         /* FIFO width is 8 bit                                                    */
         paramSet.opt &= UART_EDMA3CC_OPT_FIFO_WIDTH;

         /* EDMA3_DRV_SYNC_AB                                                      */
         paramSet.opt &= UART_EDMA3CC_OPT_SYNC_MASK_VALUE;
         paramSet.opt |= UART_EDMA3CC_OPT_SYNC_AB;

         /* EDMA3_DRV_OPT_FIELD_TCINTEN                                            */
         paramSet.opt |= ((uint32_t) 1U << UART_OPT_TCINTEN_SHIFT);

         /* ADDED FOR ARM PLATFORM                                                 */
         /* update the transfer completion code                                    */
         paramSet.opt &= (~UART_EDMA3CC_OPT_TCC_MASK);
         paramSet.opt |= (( hwAttrs->edmaRxTCC) << UART_EDMA3CC_OPT_TCC_SHIFT);

         result = EDMA3_DRV_setPaRAM(hwAttrs->edmaHandle,
        		                     hwAttrs->rxDmaEventNumber,
        		                     &paramSet);
         if (result == 0)
         {
             result = EDMA3_DRV_enableTransfer(
                                 (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                 (uint32_t) hwAttrs->rxDmaEventNumber,
                                 (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);
         }
     }
}


static void UART_transmitDMA(UART_Handle handle, const void *buffer, size_t size )
{
    UART_HwAttrs const *hwAttrs;
    uint32_t            actualThreshold;
    UART_V1_Object     *object ;
    uint32_t            lineStatus, isTxFifoEmpty = FALSE, fifoLevel;
    uint32_t            edmaStatus = 0U, isEdmaEventPending = FALSE;
    int32_t             result = EDMA3_DRV_SOK;
    EDMA3_DRV_PaRAMRegs paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    actualThreshold = hwAttrs->txTrigLvl;
    object = (UART_V1_Object*)handle->object;
    if (result == EDMA3_DRV_SOK)
    {
        /* Get the PaRAM set for default parameters */
        EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                           (uint32_t) hwAttrs->txDmaEventNumber, &paramSet);

        /* Fill the PaRAM Set with transfer specific information */
        paramSet.srcAddr  = (uint32_t)(buffer);
        paramSet.destAddr = (uint32_t) ((hwAttrs->baseAddr) + UART_THR);

        /**
         * Be Careful !!!
         * Valid values for SRCBIDX/DSTBIDX are between ?32768 and 32767
         * Valid values for SRCCIDX/DSTCIDX are between ?32768 and 32767
         */
        paramSet.aCnt     = 1;
        if ((uint32_t)size < actualThreshold)
        {
            paramSet.bCnt   = (uint16_t) size;
        }
        else
        {
            paramSet.bCnt   = (uint16_t) actualThreshold;
        }
        paramSet.srcBIdx  = UART_EDMA3CC_SRC_BINDEX;
        paramSet.srcCIdx  = actualThreshold;
        paramSet.destBIdx = 0;
        paramSet.destCIdx = 0;

        /**
         * Be Careful !!!
         * Valid values for ACNT/BCNT/CCNT are between 0 and 65535.
         * ACNT/BCNT/CCNT must be greater than or equal to 1.
         * Maximum number of bytes in an array (ACNT) is 65535 bytes
         * Maximum number of arrays in a frame (BCNT) is 65535
         * Maximum number of frames in a block (CCNT) is 65535
         */
        if ((size <= UART_EDMA3CC_COUNT_VALUE) &&
            (size > actualThreshold))
        {
            paramSet.cCnt =
                (uint16_t) (UART_EDMA3CC_COUNT_VALUE &
                          (size / actualThreshold));
        }
        else if ((size <= actualThreshold) &&
                 (size > 0))
        {
            paramSet.cCnt = 1;
        }
        else
        {
            result = -1;
        }

        /* For AB-synchronized transfers, BCNTRLD is not used. */
        paramSet.linkAddr = (uint16_t) (object->edmaLinkChPhyAddr
                          & UART_EDMA3CC_PARAM_LINK_ADDR_MASK_VALUE);
        paramSet.bCntReload = (uint16_t) 0;

#if !defined(SOC_AM437x)
        /* Src is in INCR mode & Dest is in FIFO modes                            */
        paramSet.opt &= UART_EDMA3CC_OPT_SAM_INCR_MODE;
        paramSet.opt |= UART_EDMA3CC_OPT_DAM_CONST_MODE;
#endif
        /* FIFO width is 8 bit                                                    */
        paramSet.opt &= UART_EDMA3CC_OPT_FIFO_WIDTH;

        /* EDMA3_DRV_SYNC_AB                                                      */
        paramSet.opt &= UART_EDMA3CC_OPT_SYNC_MASK_VALUE;
        paramSet.opt |= UART_EDMA3CC_OPT_SYNC_AB;

        /* EDMA3_DRV_OPT_FIELD_TCINTEN                                            */
        paramSet.opt |= (UART_OPT_TCINTEN << UART_OPT_TCINTEN_SHIFT);

        /* update the transfer completion code                                    */
        paramSet.opt &= (~UART_EDMA3CC_OPT_TCC_MASK);
        paramSet.opt |= (( object->txTcc) << UART_EDMA3CC_OPT_TCC_SHIFT);

        result = EDMA3_DRV_setPaRAM(hwAttrs->edmaHandle, hwAttrs->txDmaEventNumber, &paramSet);
        if (result == 0)
        {
            result = EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                        (uint32_t) object->edmaLinkChId,
                        &paramSet);

            paramSet.aCnt     = 1;
            paramSet.linkAddr = UART_EDMA3CC_PARAM_LINK_ADDRESS;
            paramSet.opt     &=
                ~(UART_EDMA3CC_OPT_SYNC_A << UART_EDMA3CC_OPT_SYNCDIM_SHIFT);

            if (0 == result)
            {
                /* Now, write the PaRAM Set.                                          */
                result = EDMA3_DRV_setPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                            (uint32_t) object->edmaLinkChId,
                                            &paramSet);
            }
            if (0 == result)
            {
                result = EDMA3_DRV_linkChannel((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                               (uint32_t) hwAttrs->txDmaEventNumber,
                                               (uint32_t) object->edmaLinkChId);
            }
            if (0 == result)
            {
                /* Get TX FIFO status - This should be before EDMA status */
                lineStatus = (uint32_t)
                    UARTTxFIFOFullStatusGet((uint32_t) hwAttrs->baseAddr);
                if ((uint32_t) UART_LSR_TX_FIFO_E_MASK ==
                    (lineStatus & (uint32_t) UART_LSR_TX_FIFO_E_MASK))
                {
                    isTxFifoEmpty = (uint32_t) TRUE;
                }

                /* In FIFO mode, DMA events are generated as soon as there are
                 * TX threshold level bytes become free in the FIFO. Hence it is
                 * required to check the FIFO level as well in order to check if
                 * DMA events are consumed */
                fifoLevel = UARTTxFIFOLevelGet((uint32_t) hwAttrs->baseAddr);
                if (fifoLevel <= ((uint32_t)hwAttrs->txTrigLvl))
                {
                    isTxFifoEmpty = (uint32_t) TRUE;
                }

                /* Get EDMA event pending status */
                result = EDMA3_DRV_getChannelStatus(
                                    (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                    (uint32_t) hwAttrs->txDmaEventNumber,
                                    &edmaStatus);
                if (result != ((Int32) 0))
                {
                    result = (int32_t)(-1);
                }
                if ((uint32_t) EDMA3_DRV_CHANNEL_EVENT_PENDING ==
                        (edmaStatus & ((uint32_t) EDMA3_DRV_CHANNEL_EVENT_PENDING)))
                {
                    isEdmaEventPending = (uint32_t) TRUE;
                }

                /* Note: Getting the FIFO status and EDMA event pend should be
                 * done prior to enabling the EDMA channel */
                result = EDMA3_DRV_enableTransfer(
                                (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                (uint32_t) hwAttrs->txDmaEventNumber,
                                (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);
                if (result != ((Int32) EDMA3_DRV_SOK))
                {
                    result = (int32_t)(-1);
                }

                /* Workaround: In case of DMA mode, the UART IP generates the
                 * next EDMA event as soon as the FIFO gets emptied or when there are
                 * TX threshold level bytes free in FIFO. In the EDMA ISR,
                 * the DMA mode is disabled and EDMA channel is disabled.
                 * There is a race condition between software (CPU) disabling the DMA
                 * mode and the UART IP generating the DMA event to EDMA.
                 *
                 * In cases when the software is slower (multi-tasking systems or
                 * in heavily CPU loaded systems or for lower FIFO trigger levels),
                 * the UART IP DMA event gets generated before the CPU could disable
                 * the DMA event generation.
                 *
                 * When this happens, since the EDMA channel is enabled, the EDMA will
                 * acknowledge the EDMA event but it doesn't have proper param set to
                 * do the actual transfer to the UART FIFO. Hence the event gets lost
                 * and the UART IP doesn't generate the next EDMA event unless the FIFO
                 * is written. This becomes a deadlock!!
                 *
                 * To break the dead lock, first we need to identify whether this had
                 * happened by checking the UART FIFO level and the EDMA channel status.
                 * As per UART IP, in DMA mode, when ever the FIFO is empty or has TX
                 * threshold amount of free space it should have raised an DMA event.
                 * Hence we check if the UART FIFO has room and see if the event is
                 * latched in the EDMA register.
                 * So if FIFO is empty and event is not latched, then we are in a
                 * deadlock.
                 *
                 * To recover from the deadlock, we do a manual trigger for the first
                 * time and the rest of the transfer is taken care automatically
                 * by further UART events.
                 */
                if (((uint32_t) TRUE == isTxFifoEmpty) &&
                    ((uint32_t) FALSE == isEdmaEventPending))
                {
                    result = EDMA3_DRV_enableTransfer(
                                    (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                    (uint32_t) hwAttrs->txDmaEventNumber,
                                    (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_MANUAL);
                    if (result != ((Int32) EDMA3_DRV_SOK))
                    {
                        result = (int32_t)(-1);
                    }
                }
            }
        }
    }
}

static int32_t UART_configDMA(UART_Handle handle )
{
    UART_HwAttrs const           *hwAttrs;
    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    uint32_t      rxDmaEventNumber;
    uint32_t      txDmaEventNumber;
    uint32_t tcc = 0;
    UART_V1_Object        *object ;

    Int32 status = EDMA3_DRV_SOK;

    object = (UART_V1_Object*)handle->object;
    rxDmaEventNumber = hwAttrs->rxDmaEventNumber;


    txDmaEventNumber = hwAttrs->txDmaEventNumber;
    tcc = EDMA3_DRV_TCC_ANY;

    status = (Int32) EDMA3_DRV_requestChannel(
                        (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                        &rxDmaEventNumber,
                        &tcc,
                        (EDMA3_RM_EventQueue) hwAttrs->edmaRxTC,
                        &UART_rxIsrHandler,
                        (void *) handle);
    if(status == EDMA3_DRV_SOK ) {

        object->txTcc = EDMA3_DRV_TCC_ANY;
        status = (Int32) EDMA3_DRV_requestChannel(
                                        (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                        &txDmaEventNumber,
                                        &object->txTcc,
                                        (EDMA3_RM_EventQueue) 0,
                                        &UART_txIsrHandler,
                                        (void *) handle);
        if(status == EDMA3_DRV_SOK) {

             tcc = EDMA3_DRV_TCC_ANY;
             object->edmaLinkChId = EDMA3_DRV_LINK_CHANNEL;
             status = EDMA3_DRV_requestChannel(
                                hwAttrs->edmaHandle,
                                &object->edmaLinkChId,
                                &tcc,
                                0,
                                &UART_doNothing,
                                NULL);

             if (EDMA3_DRV_SOK == status)
                            {
                 status = EDMA3_DRV_getPaRAMPhyAddr(
                                        hwAttrs->edmaHandle,
                                        object->edmaLinkChId,
                                    &object->edmaLinkChPhyAddr);
             }
        }
    }
    return(status);
}
static void UART_rxIsrHandler(uint32_t tcc, EDMA3_RM_TccStatus status, void* appData)
{
    UART_Handle         handle;
    UART_HwAttrs const *hwAttrs;
    UART_V1_Object     *object ;
    EDMA3_DRV_PaRAMRegs paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    uint32_t            bytesRemain, bytesReceived;

    if(appData != NULL) {
        handle = (UART_Handle)appData;
        hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
        object = (UART_V1_Object*)handle->object;

        bytesRemain = object->readSize % hwAttrs->rxTrigLvl;
        if (object->readSize >= hwAttrs->rxTrigLvl)
        {
            bytesReceived = object->readSize - bytesRemain;
        }
        else
        {
            bytesReceived = bytesRemain;
        }

        EDMA3_DRV_disableLogicalChannel(
                (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                (uint32_t) hwAttrs->rxDmaEventNumber,
                (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);


        object->readCount = object->readSize;
        if (EDMA3_RM_XFER_COMPLETE != status) {

             /* Ensure to clear the error bits of EDMA channel                 */
            EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                       hwAttrs->rxDmaEventNumber);

            /* Get the PaRAM set for default parameters                       */
            EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                   hwAttrs->rxDmaEventNumber,
                                              &paramSet);
            /* calculate the amount of bytes remaining                        */
            object->readSize = 0;
            if (object->readTrans != NULL)
            {
                object->readTrans->count =
                    bytesReceived - (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);
                object->readTrans->status = UART_TRANSFER_STATUS_ERROR_OTH;
            }
            else
            {
                object->readCount =
                    bytesReceived - (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);
            }
            UARTIntDisable(hwAttrs->baseAddr, UART_INT_LINE_STAT);
            UART_v1_callback(handle, true);
        }
        else
        {
            object->readCount = bytesReceived;
            if ((object->readSize >= hwAttrs->rxTrigLvl) && (bytesRemain != 0))
            {
                /*
                 * read size is not multiple of RX threshold
                 * enable RHR interrupt to receive the
                 * remaining bytes in ISR
                 */
                object->readSize = bytesRemain;
                object->readBuf = (uint8_t *)object->readBuf + object->readCount;
                UARTIntEnable(hwAttrs->baseAddr, UART_INT_RHR_CTI);
            }
            else
            {
                /*
                 * all the data received
                 */
                if (object->readTrans != NULL)
                {
                    object->readTrans->count = object->readCount;
                    object->readTrans->status = UART_TRANSFER_STATUS_SUCCESS;
                }
                object->readSize = 0;
                UARTIntDisable(hwAttrs->baseAddr, UART_INT_LINE_STAT);
                UART_v1_callback(handle, true);
            }
        }
    }

}

static void UART_txIsrHandler(uint32_t tcc, EDMA3_RM_TccStatus status, void* appData)
{
    UART_Handle         handle;
    UART_HwAttrs const *hwAttrs;
    UART_V1_Object     *object ;
    EDMA3_DRV_PaRAMRegs paramSet = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    uint32_t            bytesRemain, bytesSent;

    if(appData != NULL) {

        handle = (UART_Handle)appData;
        hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
        object = (UART_V1_Object*)handle->object;

        bytesRemain = object->writeSize % hwAttrs->txTrigLvl;
        if (object->writeSize >= hwAttrs->txTrigLvl)
        {
            bytesSent = object->writeSize - bytesRemain;
        }
        else
        {
            bytesSent = bytesRemain;
        }

        EDMA3_DRV_disableLogicalChannel(
                            (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                            (uint32_t) hwAttrs->txDmaEventNumber,
                            (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);
        if (EDMA3_RM_XFER_COMPLETE != status) {

             /* Ensure to clear the error bits of EDMA channel                 */
            EDMA3_DRV_clearErrorBits((EDMA3_DRV_Handle)hwAttrs->edmaHandle,
                       hwAttrs->txDmaEventNumber);


            /* Get the PaRAM set for default parameters                       */
            EDMA3_DRV_getPaRAM((EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                                               hwAttrs->txDmaEventNumber,
                                               &paramSet);
            if (object->writeTrans != NULL)
            {
                object->writeTrans->count =
                    bytesSent - (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);
                object->writeTrans->status = UART_TRANSFER_STATUS_ERROR_OTH;
            }
            else
            {
                object->writeCount =
                    bytesSent - (paramSet.aCnt * paramSet.bCnt * paramSet.cCnt);
            }
            object->writeSize = 0;
            UART_v1_callback(handle, false);
        }
        else
        {
            if ((object->writeSize >= hwAttrs->txTrigLvl) && (bytesRemain != 0))
            {
                /*
                 * write size is not multiple of TX threshold
                 * kick off another DMA transfer to send the
                 * remaining bytes
                 */
                object->writeCount = bytesSent;
                object->writeSize = bytesRemain;
                UART_transmitDMA(handle, (const void *)((uint8_t *)object->writeBuf + bytesSent), bytesRemain);
            }
            else
            {
                /*
                 * all the data sent
                 */
                object->writeCount += bytesSent;
                if (object->writeTrans != NULL)
                {
                    object->writeTrans->count = object->writeCount;
                    object->writeTrans->status = UART_TRANSFER_STATUS_SUCCESS;
                }
                object->writeSize = 0;
                object->txDataSent = TRUE;
                UARTInt2Enable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);
            }
        }
    }

}
#endif

/*
 *  ======== UART_read_v1 ========
 */
static int32_t UART_read_v1(UART_Handle handle, void *buffer, size_t size)
{
    uint32_t               key;
    UART_V1_Object     *object;
    int32_t                ret_val = 0;
    UART_Transaction       transaction;
    UART_HwAttrs const *hwAttrs;

    /* Get the pointer to the object */
    object = (UART_V1_Object*)handle->object;
    /* Get the pointer to the hwAttrs */
    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

   if (hwAttrs->enableInterrupt) {
      /* Disable preemption while checking if the uart is in use. */
      key = UART_osalHardwareIntDisable();
      if (object->readSize)
      {
         UART_osalHardwareIntRestore(key);
         ret_val = UART_ERROR;
      }
      else
      {
         UART_osalHardwareIntRestore(key);

         transaction.buf = (void *)buffer;
    	 transaction.timeout = object->params.readTimeout;
    	 transaction.count = (uint32_t)size;
    	 UART_read2_v1(handle, &transaction);
    	 ret_val = transaction.count;
      }
   }
   else
   {
      ret_val = UART_readPolling_v1(handle,buffer,size);
   }

   return(ret_val);
}

/*
 *  ======== UART_read2_v1 ========
 */
static int32_t UART_read2_v1(UART_Handle handle, UART_Transaction *transaction)
{
    uint32_t               key;
    UART_V1_Object     *object;
    UART_HwAttrs const *hwAttrs;
    int32_t                ret_val = UART_SUCCESS;
    SemaphoreP_Status      semStatus;

    /* Input parameter validation */
    OSAL_Assert(!((handle != NULL) && (transaction != NULL)));

    if (transaction->count == 0)
    {
        return(UART_ERROR);
    }

    /* Get the pointer to the hwAttrs */
    hwAttrs = (UART_HwAttrs*)handle->hwAttrs;

    /* Get the pointer to the object */
    object = (UART_V1_Object*)handle->object;


    if(hwAttrs->enableInterrupt) {
    /* Disable preemption while checking if the uart is in use. */
    key = UART_osalHardwareIntDisable();
    if (object->readSize) {
        UART_osalHardwareIntRestore(key);

        ret_val = UART_ERROR;
    }
    else
    {
        /* Save the data to be read and restore interrupts. */
        transaction->status = UART_TRANSFER_STATUS_SUCCESS;

        /*
         * When legacy UART_read API is called in callback mode,
         * object->params.readCallback will save the callback function
         * pointer passed from the application, otherwise
         * object->params.readCallback will be set to NULL.
         *
         * If legacy UART_read API is called in callback mode,
         * transaction parameter passed to UART_read2_v1 is
         * declared in the stack, and cannot be used in the ISR,
         * otherwise will cause data corruption in the stack.
         * The driver should only use object->readTrans when it
         * is NOT called from egacy UART_read API in callback mode
         */
        if (object->params.readCallback == NULL)
        {
            object->readTrans = transaction;
        }
        else
        {
            object->readTrans = NULL;
        }
        object->readBuf = transaction->buf;
        object->readCount = 0;
        object->rxTimeoutCnt = 0;

#ifdef UART_DMA_ENABLE
        if (hwAttrs->dmaMode == TRUE)
        {
            /* DMA mode */
            object->readSize = transaction->count;
            UART_osalHardwareIntRestore(key);

            UARTIntEnable(hwAttrs->baseAddr, UART_INT_LINE_STAT);
            if (object->readSize < hwAttrs->rxTrigLvl)
            {
                /*
                 * read size less than RX threshold, no DMA trigger event
                 * enable RHR interrupt to receive data in ISR
                 */
                UARTIntEnable(hwAttrs->baseAddr, UART_INT_RHR_CTI);
            }
            else
            {
                UART_v1_receiveDMA(handle,(void*)object->readBuf, object->readSize);
            }

            /* If readMode is blocking, block and get the status. */
            if (object->params.readMode == UART_MODE_BLOCKING)
            {
                /* Pend on semaphore and wait for Hwi to finish. */
            	semStatus = UART_osalPendLock(object->readSem,
            			                      transaction->timeout);
                if (semStatus == SemaphoreP_OK)
                {
                    if (transaction->status == UART_TRANSFER_STATUS_SUCCESS)
                    {
                        ret_val = UART_SUCCESS;
                    }
                    else
                    {
                    	/* RX errors */
                        ret_val = UART_ERROR;
                    }
                }
                else
                {
                    if (semStatus == SemaphoreP_TIMEOUT)
                    {
                        transaction->status = UART_TRANSFER_STATUS_TIMEOUT;
                    }
                    else
                    {
                        transaction->status = UART_TRANSFER_STATUS_ERROR_OTH;
                    }
                	/* Cancel the read without posting the semaphore */
                    UART_readCancelNoCB(handle);
                    ret_val = UART_ERROR;
                }
            }
            else
            {
            	/*
            	 * for callback mode, immediately return SUCCESS,
            	 * once the transaction is done, callback function
            	 * will return the transaction status and actual
            	 * write count
            	 */
            	transaction->count = 0;
                ret_val = UART_SUCCESS;
            }
        }
        else
#endif
        {
            /* non-DMA mode */
            UART_osalHardwareIntRestore(key);

            /* Disable RX threshold and line status error interrupt */
            UARTIntDisable(hwAttrs->baseAddr, UART_INT_RHR_CTI | UART_INT_LINE_STAT);

            object->readSize = UART_v1_readData(handle, (int32_t)(transaction->count));
            if ((object->readSize) == 0)
            {

                UART_drv_log2("UART:(0x%x) Read finished, %d bytes read",
                              hwAttrs->baseAddr, object->readCount);

                /* Update the actual read count */
                transaction->count = object->readCount;

            	if (object->params.readMode == UART_MODE_CALLBACK)
                {
                    UART_v1_callback(handle, true);
                }
                ret_val = UART_SUCCESS;
            }
            else
            {
                UARTIntEnable(hwAttrs->baseAddr, UART_INT_RHR_CTI | UART_INT_LINE_STAT);
                if (object->params.readMode == UART_MODE_BLOCKING)
                {
                    /* Pend on lock and wait for Hwi to finish. */
                    semStatus = UART_osalPendLock(object->readSem,
                                                 transaction->timeout);
                    if (semStatus == SemaphoreP_OK)
                    {
                    	if (transaction->status == UART_TRANSFER_STATUS_SUCCESS)
                        {
                            ret_val = UART_SUCCESS;
                        }
                        else
                        {
                            ret_val = UART_ERROR;
                        }
                    }
                    else
                    {
                        if (semStatus == SemaphoreP_TIMEOUT)
                        {
                            transaction->status = UART_TRANSFER_STATUS_TIMEOUT;
                        }
                        else
                        {
                            transaction->status = UART_TRANSFER_STATUS_ERROR_OTH;
                        }
                    	/* Cancel the DMA without posting the semaphore */
                        UART_readCancelNoCB(handle);
                        ret_val = UART_ERROR;
                    }
                    transaction->count = object->readCount;
                }
                else
                {
                	/*
                	 * for callback mode, immediately return SUCCESS,
                	 * once the transaction is done, callback function
                	 * will return the transaction status and actual
                	 * read count
                	 */
                	transaction->count = 0;
                    ret_val = UART_SUCCESS;
                }
            }
        }
      }
    }
    else
    {
       transaction->count = UART_readPolling_v1(handle,transaction->buf,transaction->count);
       transaction->status = UART_TRANSFER_STATUS_SUCCESS;
    }
    return(ret_val);
}

/*
 *  ======== UART_charGet_v1 ========
 *
 *  This function continuously does a non-blocking read, until it reads
 *  successfully or times out.
 */
static uint32_t UART_charGet_v1(UART_HwAttrs const *hwAttrs, uint8_t *data, uint32_t *timeout);  /*for misra warnings*/
static uint32_t UART_charGet_v1(UART_HwAttrs const *hwAttrs, uint8_t *data, uint32_t *timeout)
{
    int8_t   rdData;
    uint32_t timeoutCnt = *timeout;
    uint32_t retVal = TRUE;

    while (retVal == TRUE)
    {
        rdData = UARTCharGetNonBlocking(hwAttrs->baseAddr);
        if (rdData == (int8_t)(-1))
        {
            if (timeoutCnt == 0)
            {
                retVal = FALSE;
            }
            else
            {
                UART_osalDelay(1U);
                timeoutCnt--;
            }
        }
        else
        {
            *data = (uint8_t)rdData;
            break;
        }
    }

    *timeout = timeoutCnt;

    return (retVal);
}

/*
 *  ======== UART_readPolling_v1 ========
 */
static int32_t UART_readPolling_v1(UART_Handle handle, void *buf, size_t size)
{
    int32_t                count = 0;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    uint8_t               *buffer = (uint8_t *)buf;
    uint32_t               timeout;
    uint8_t                ret_flag = 0U;
    uint32_t               timeoutErr = FALSE;

    timeout = object->params.readTimeout;

    /* Read characters. */
    while ((size != 0) && (timeoutErr == FALSE))
    {
        if (UART_charGet_v1(hwAttrs, buffer, &timeout) == FALSE)
        {
            timeoutErr = TRUE;
        }
        else
        {
            UART_drv_log2("UART:(0x%x) Read character 0x%x",
                hwAttrs->baseAddr, *buffer);

            count++;
            size--;

            if ((object->params.readDataMode == UART_DATA_TEXT) && (*buffer == ((uint8_t)('\r'))))
            {
                /* Echo character if enabled. */
                if (object->params.readEcho)
                {
                    if (UART_charPut_v1(hwAttrs, ((uint8_t)('\r')), &timeout) == FALSE)
                    {
                        timeoutErr = TRUE;
                    }
                }
                if (timeoutErr == FALSE)
                {
                    *buffer = ((uint8_t)('\n'));
                }
            }

            /* Echo character if enabled. */
            if ((timeoutErr == FALSE) && (object->params.readEcho))
            {
                if (UART_charPut_v1(hwAttrs, *buffer, &timeout) == FALSE)
                {
                    timeoutErr = TRUE;
                }
            }

            /* If read return mode is newline, finish if a newline was received. */
            if ((timeoutErr == FALSE) &&
                (object->params.readReturnMode == UART_RETURN_NEWLINE) &&
                (*buffer == ((uint8_t)('\n'))))
            {
                ret_flag = 1U;
                break;
            }
            else
            {
                buffer++;
            }
        }
    }

    if(timeoutErr == FALSE)
    {
        UART_drv_log2("UART:(0x%x) Read polling timed out, %d bytes read",
                      hwAttrs->baseAddr, (int32_t)count);
    }

    if(ret_flag == 0U)
    {
        UART_drv_log2("UART:(0x%x) Read polling finished, %d bytes read",
        hwAttrs->baseAddr, (int32_t)count);
    }

    return (count);
}

/*
 *  ======== UART_readCancel_v1 ========
 */
static void UART_readCancel_v1(UART_Handle handle)
{
    if (UART_readCancelNoCB(handle) == true)
    {
        UART_v1_callback(handle, true);

        UART_drv_log2("UART:(0x%x) Read cancelled, %d bytes read",
                      ((UART_HwAttrs const *)(handle->hwAttrs))->baseAddr,
                      object->readCount);
    }
}

/*!
 *  @brief      A function pointer to a driver specific implementation of
 *              UART_control().
 */
static int32_t UART_control_v1 (UART_Handle handle, uint32_t cmd, void *arg)
{
    return(0);
}

/**
 * \brief   This API waits indefinitely until the Transmitter FIFO
 *          (THR register in non-FIFO mode) and Transmitter Shift
 *          Register are empty.
 *
 * \param   baseAdd     Memory address of the UART instance being used
 *
 * \return  None
 */
static void UARTFifoWait(uint32_t baseAdd)
{
    while (UARTIsTransmitterEmpty(baseAdd) == FALSE);
}

static bool UART_writeCancelNoCB(UART_Handle handle)
{
    uint32_t            key;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    bool                retVal = true;

    UARTIntDisable(hwAttrs->baseAddr, UART_INT_THR);
    UARTInt2Disable(hwAttrs->baseAddr, UART_INT2_TX_EMPTY);

    /* Disable interrupts to avoid writing data while changing state. */
    key = UART_osalHardwareIntDisable();

    /* Return if there is no write. */
    if ((object->writeSize) == 0U)
    {
        UART_osalHardwareIntRestore(key);
        retVal = false;
    }
    else
    {
#ifdef UART_DMA_ENABLE
        if (hwAttrs->dmaMode == TRUE)
        {
            /* Set channel bit in the ENACLR register */
            EDMA3_DRV_disableLogicalChannel(
                        (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                        (uint32_t) hwAttrs->txDmaEventNumber,
                        (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);

            if (object->writeTrans != NULL)
            {
                object->writeTrans->count = 0;
            }
            else
            {
                object->writeCount = 0;
            }
        }
        else
#endif
        {
            /* Reset the write buffer so we can pass it back */
            object->writeBuf = (uint8_t *)object->writeBuf - object->writeCount;
            if (object->writeTrans != NULL)
            {
                object->writeTrans->count = object->writeCount;
            }
        }

        /* Set size = 0 to prevent writing and restore interrupts. */
        object->writeSize = 0;
        UART_osalHardwareIntRestore(key);
    }

    return (retVal);
}

static bool UART_readCancelNoCB(UART_Handle handle)
{
    uint32_t            key;
    UART_V1_Object     *object = (UART_V1_Object*)handle->object;
    UART_HwAttrs const *hwAttrs = (UART_HwAttrs*)handle->hwAttrs;
    bool                retVal = true;

    UARTIntDisable(hwAttrs->baseAddr,
                   UART_INT_RHR_CTI | UART_INT_LINE_STAT);

    /* Disable interrupts to avoid reading data while changing state. */
    key = UART_osalHardwareIntDisable();
    if ((object->readSize) == 0U)
    {
        UART_osalHardwareIntRestore(key);
        retVal = false;
    }
    else
    {
#ifdef UART_DMA_ENABLE
        if (hwAttrs->dmaMode == TRUE)
        {
            /* Set channel bit in the ENACLR register */
            EDMA3_DRV_disableLogicalChannel(
                    (EDMA3_DRV_Handle) hwAttrs->edmaHandle,
                    (uint32_t) hwAttrs->rxDmaEventNumber,
                    (EDMA3_DRV_TrigMode) EDMA3_DRV_TRIG_MODE_EVENT);

            if (object->readTrans != NULL)
            {
                object->readTrans->count = 0;
            }
            else
            {
                object->readCount = 0;
            }
        }
        else
#endif
        {
            /* Reset the read buffer so we can pass it back */
            object->readBuf = (uint8_t *)object->readBuf - object->readCount;
            if (object->readTrans != NULL)
            {
                object->readTrans->count = object->readCount;
            }
        }

        /* Set size = 0 to prevent reading and restore interrupts. */
        object->readSize = 0;
        UART_osalHardwareIntRestore(key);

        /* Flush the RX FIFO */
        while (UARTCharGetNonBlocking(hwAttrs->baseAddr) != (int8_t)(-1));
    }

    return (retVal);
}


