/*
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== ILoggerSnapshot.xdc ========
 */
import xdc.runtime.Types;
import xdc.runtime.Log;

/*!
 *  ======== ILoggerSnapshot ========
 *  Interface to service LogSnapshot events
 *
 *  A loggerSnapshot is responsible for recording, transmitting, or otherwise
 *  handling `{@link LogSnapshot#EventDesc LogSnapshot Events}` generated by clients of the
 *  `{@link LogSnapshot}` module. The `LogSnapshot` module uses modules that implement the
 *  `ILoggerSnapshot` interface to record the log events. Most application code will
 *  use the `LogSnapshot` module instead of directly calling the specific `ILoggerSnapshot`
 *  module implementation.
 *
 *  All loggerSnapshot implementations must inherit from this interface. The
 *  derived implementations must implement the functions defined in this
 *  interface but may also add additional configuration parameters and
 *  functions.
 *
 *  Since this interface inherits {@link xdc.runtime.ILogger}, loggerSnapshots
 *  can also be used to support events logged by the `{@link Log}` module and other
 *  modules that log events based on the ILogger interface.
 */
interface ILoggerSnapshot inherits IUIATransfer {

instance:
    /*!
     *  ======== writeMemoryRange ========
     *  Log an event along with values from a range of memory addresses
     *
     *  If the mask in the specified `LogSnapshot` event has any bit set which is
     *  also set in the current module's diagnostics mask, then this call to
     *  write will "raise" the given `LogSnapshot` event.
     *
     *  Note that this API supports null terminated strings, arrays of characters and
     *  memory mapped registgers as well as blocks of memory.  The LogSnapshot module provides macros that
     *  map the appropriate values to the writeMemoryRange API's arguments
     *
     *  @param(evt)         event to be logged
     *  @param(snapshotId)  0 = no other snapshot groups, Use value from
     *                      LogSnapshot.getSnapshotId() for all snapshots to be grouped.
     *  @param(fileName)    __FILE__ result
     *  @param(lineNum)     __LINE__ result
     *  @param(fmt)         a `printf` style format string
     *  @param(startAdrs)   value for first format conversion character
     *  @param(lengthInMAUs) value for second format conversion character
     *
     *  @see Log#Event
     *  @see Log#EventDesc
     *  @see Text#ropeText
     *  @see Types#getEventId
     */
    @DirectCall
    Void writeMemoryRange(Log.Event evt, Types.ModuleId mid, UInt32 snapshotId,
                         IArg fileName, IArg LineNum, IArg fmt, IArg startAdrs,
                         UInt32 lengthInMAUs);
}
