/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== BufferPacketizer.xdc ========
 */
import xdc.runtime.IHeap;
import xdc.runtime.Types;
import ti.uiactools.runtime.IUIAPacketTransfer;
import ti.uia.runtime.QueueDescriptor;
import xdc.runtime.Error;
import xdc.rov.ViewInfo;
import ti.uiactools.runtime.UIAPacket;

/*!
 *  ======== BufferPacketizer ========
 *  A logger which stores `Log` events in a non-blocking, streamable 
 *  circular buffer.
 */

@ModuleStartup      /* Initialize static instances */
@InstanceFinalize   /* this mod has cleanup fxn when instances are deleted */
@InstanceInitError  /* instance create can fail */
@CustomHeader       /* definitions for OST header etc. */
module BufferPacketizer inherits IUIAPacketTransfer
{

    /*!
     *  ======== BufferType ========
     *  Used to specify the type of ETB to drain
     *
     *  This enum is used by the instrumentation host to determine what
     *  the ETB type is. 
     */
    enum BufferType {
        BufferType_MEMORY = 0,
        BufferType_CPUTRACE_ETB = 1,
        BufferType_STM_ETB = 2
    };
    
    /*!
     *  ======== TransportType ========
     *  Used to specify the type of transport to use
     *
     *  This enum is used by the instrumentation host to determine what
     *  the transport is. It is not used by the target code.
     */
    enum TransportType {
        TransportType_UART = 0,
        TransportType_USB = 1, 
        TransportType_UDP = 2, 
        TransportType_TCP = 3, 
        TransportType_CUSTOM = 4
    };
    /*!
     *  @_nodoc
     *  ======== ModuleView ========
     */
    metaonly struct ModuleView {                
    }
    
    /*!
     *  @_nodoc
     *  ======== InstanceView ========
     */
    metaonly struct InstanceView {        
        String    label;
        Bool      enabled;
    	UInt32	  tracePktUpCount;
    	UInt32	  tracePktDownCount;
        Bits16    serial;
        UInt32    snapshotNumber;
        IUIAPacketTransfer.TransferType transferType;        
    }
    
    /*!
     *  @_nodoc
     *  ======== rovViewInfo ========
     */    
    @Facet
    metaonly config ViewInfo.Instance rovViewInfo =
        ViewInfo.create({
            viewMap: [
                ['Module',   
                    {
                        type: ViewInfo.MODULE,   
                        viewInitFxn: 'viewInitModule', 
                        structName: 'ModuleView'
                    }
                ],
                ['Instances',   
                    {
                        type: ViewInfo.INSTANCE,   
                        viewInitFxn: 'viewInitInstances', 
                        structName: 'InstanceView'
                    }
                ],
             ]
        });
	
    /*!
     *  ======== ExchangeFxnType ========
     *  Typedef for the exchange function pointer. 
     *  @param(Ptr)  first pointer is for the object instance handle
     *  @param(Ptr)  second pointer is the just-filled buffer containing the ETB packet to be sent
     */
    typedef Ptr (*ExchangeFxnType)(Ptr,Ptr);
    
    /*!
     *  ======== PrimeFxnType ========
     *  Typedef for the exchange function pointer. 
     *  @param(Ptr)  first pointer is for the object instance handle     
     */
    typedef Ptr (*PrimeFxnType)(Ptr);    

    /*!
     *  ======== transportType ========
     *  Transport used to send the records to an instrumentation host
     *
     *  This parameter is used to specify the transport that the 
     *  `{@link #exchangeFxn}` function will use to send the buffer to
     *  an instrumentation host (e.g. System Analyzer in CCS).
     *
     *  This parameter is placed into the generated UIA XML file. The 
     *  instrumentation host can use the XML file to help it auto-detect as
     *  much as possible and act accordingly.
     *
     *  If the desired transport is not in the `{@link #TransportType}` enum,
     *  select `{@link #TransportType_CUSTOM}` and set the 
     *  `{@link #customTransport}` string with the desired string.
     */
    config TransportType transportType = TransportType_UDP;  

    /*!
     *  ======== customTransportType ========
     *  Custom transport used to send the records to an instrumentation host
     *
     *  If the desired transport is not in the `{@link #TransportType}` enum,
     *  and `{@link #transportType}` is set to `{@link #TransportType_CUSTOM}`,
     *  this parameter must be filled in with the correct transport name.
     *
     *  If `{@link #transportType}` is NOT set to 
     *  `{@link #TransportType_CUSTOM}`, this parameter is ignored.
     */
    config String customTransportType = "None";
    
    /*!
     *  ======== initBuffer ========
     *  Initializes the UIA trace packet header. 
     *
     *  This API is used to initialize a buffer before it is given to
     *  BufferPacketizer (via priming or exchange). The function initializes
     *  the UIAPacket portion of the buffer. 
     *
     *  @param(buffer)    Pointer to the buffer that BufferPacketizer will
     *                    fill with Log events. The first four 32-bit words
     *                    will contain the UIAPacket_Hdr structure.
     *
     *  @param(src)       Used to initialize the UIA source address. For 
     *                    a single core device, this will generally be 0.
     *                    For multi-core devices, it generally corresponds
     *                    to the DNUM (on C6xxxx deviecs) or the Ipc
     *                    MultiProc id. It must be unique for all cores and
     *                    match the configuration in the System Analyzer 
     *                    endpoint configuration.
     *
     *  @param(processId) the process ID the packet is being sent by.
     *                    (0 if only a single process).
     */
    @Macro Void initBuffer(Ptr buffer, UInt16 src, UInt32 processId);
        
    /*! @_nodoc
     * ======== hasMetaData ========
     * Indicates that the LogSync module generates content for the uia.xml file.
     */
    override metaonly config Bool hasMetaData = true;    
   

      
instance:
    /*!
     *  ======== create ========
     *  Create a `BufferPacketizer` logger
     *
     *  @see BufferPacketizer#Params
     */
    @DirectCall
    create();

    /*!
     *  ======== packetSizeInMAUs ========
     *  Number of minimum addressable units (e.g. bytes) in the trace event packet buffer
     *
     *  NOTE: the packet size must contain an integer number of 32b words
     *  (e.g. if a MAU = 1 byte, then the packet size must be a multiple of 4).
     */
    config SizeT packetSizeInMAUs = 1400;     
	

	/*! 
	 * ======== etbCoreId ========
	 * the ID for the core that the ETB services
	 *
	 * For STM (default), use 0xFF 
	 */
	config UInt8 etbCoreId = 0xFF;

	/*!
	 * ======== bufferType ========
	 * Specifies the type of data that is being uploaded
	 * so that it can be properly decoded on the host.
	 */
	config BufferType bufferType = BufferType_STM_ETB;
	
    /*!
     * ======== endpointNumber ========
     * sender ID to insert in UIAPacket header
     * Set to 0xFF if this should be set to DNUM (C6X only)
     */
    config Bits16 endpointNumber = 0xFF;
    
    /*!
     *  ======== primeFxn ========
     *  Function pointer to the prime function. 
     */
    config PrimeFxnType primeFxn = null;
    
    /*!
     *  ======== exchangeFxn ========
     *  Function pointer to the exchange function. 
     *
     *  exchange function must return a pointer to a buffer that is word aligned, initialized with 
     *  a UIA header and the correct size.  This is called in the context of a log
     *  so generally the exchange function should be quick to execute.
     */
    config ExchangeFxnType exchangeFxn = null;

    /*!
     *  ======== sendPacket ========
     *  calls the exchangeFxn with the next packet payload of data
     */
    @DirectCall
    Void sendPacket();

         
    /*!
     *  ======== prime =========
     *  If PrimeFxn is not set the user must call prime with the first packet.
     */
    @DirectCall
    override Bool prime(Ptr packetBuffer);
    
    /*!
     *  ======== enable ========
     *  Enable a log
     *
     *  @a(returns)
     *  The function returns the state of the log (`TRUE` if enabled,
     *  `FALSE` if disabled) before the call. That allow clients to restore
     *  the previous state.
     */
    @DirectCall
    override Bool enable();

    /*!
     *  ======== disable ========
     *  Disable a log
     *
     *  Events written to a disabled log are silently discarded.
     *
     *  @a(returns)
     *  The function returns the state of the log (`TRUE` if enabled,
     *  `FALSE` if disabled) before the call. That allow clients to restore
     *  the previous state.
     */
    @DirectCall
    override Bool disable();

    /*!
     * ======== setBufferToUpload =========
     * Assigns a buffer to upload to System Analyzer (in Maus)
     *
     * Returns false if another buffer is already being uploaded.
     */
    @DirectCall
    Bool setBufferToUpload(UInt32* pBuffer, UInt32 bufferSizeInWords, Ptr pUploaderContext);
    
    /*!
     * ======== getUnsentSize =========
     * Determines how much unsent data is left to send (in Maus)
     */
    @DirectCall
    SizeT getUnsentSize();
    
	/*!
	 * ======== transferData ========
	 * Read and Transfer ETB data
	 * move the binary data to PC host via event packets for further
	 * decoding and analysis
	 * @param(maxNumPackets)  maximum number of packets to send in context of this API call
	 */
	@DirectCall
	override Bool transferData( UInt32 maxNumPackets);    

	/*! @nodoc
	 * ======== transferSTMConfig ========
	 * Status of the STM - this is required to decode the compressed data
	 */
	@DirectCall
	Void transferSTMConfig();
	
	/*! @nodoc
	 * ======== init ========
	 * Used internally to initialize data structures as part of prime function
	 */
	@DirectCall
	Bool init();
	
        
    @DirectCall
    Void enableDebugMessages(Bool enable);
    
    @DirectCall
    Bool isFirstPacketFlagSet();
    
    
internal:

	
    struct Instance_State {
        Bool        enabled;       /* Enabled state */
        UInt32      *pPacketBuffer;       /* Ptr to packet buffer */
        UInt32      *wrPacket ;              /* Ptr to write location in packet buffer*/
        UInt32      *rdUploadBuffer ;              /* Ptr to read location in buffer to be uploaded */
        UInt32      *pPacketEndPlusOne;
        Bool        isInitialized;
        Bits16 serial;
        UInt32 snapshotNumber;
        UInt32 transferNumber;
        Bits16 endpointNumber;
        Bool  isFirstPacket;
        Bits16 tracePktDownCount;
        Bits16 tracePktUpCount;
        UIAPacket.HdrType traceHdrType;
        IUIAPacketTransfer.TransferType transferType;
        UInt32 packetSizeInMAUs;
        UInt8 etbCoreId;
		BufferType bufType;
        UInt32* pUploadBuffer;

	    SizeT uploadBufferSizeInWords;
	    Ptr pUploaderContext;   
        Bool primeStatus;
        ExchangeFxnType exchangeFxn;
        PrimeFxnType primeFxn;
        Bool printDebugMessages;

    };
}


