/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== LoggerSTM.xdc ========
 */
package ti.uiactools.runtime;

import xdc.runtime.Diags;
import xdc.runtime.Log;
import ti.uia.runtime.LoggerTypes;
import xdc.runtime.Types;
import xdc.runtime.IHeap;
import xdc.runtime.ILogger;
import ti.uia.runtime.IUIATransfer;
import ti.uia.runtime.ILoggerSnapshot;


/*!
 *  ======== LoggerSTM ========
 *  A logger which writes `Log` events to the System Trace Module (STM).
 *
 *  This module provides a logger which writes `{@link Log}` events to the
 *  System Trace Module in realtime. The `Log` events stored in the buffer are
 *  unformatted; `Log` event formatting is deferred until some client reads
 *  the raw event data from the buffer.
 *
 *
 * @a(Examples)
 * Configuration Example 1: The following XDC configuration script statements
 * show how to use the ti.uiactools.sysbios.LoggingSetupSTM module to enable
 * instrumentation for SysBIOS modules, with the resulting UIA software events 
 * being logged to System Trace as STM software messages:
 *
 * @p(code)
 * var LoggerSTM = xdc.useModule('ti.uiactools.runtime.LoggerSTM');
 * LoggerSTM.TimestampProxy = xdc.useModule('xdc.runtime.Timestamp');
 * 
 * // The following stmBaseAdrs setting is valid for C66X devices only.
 * // For other devices, please check the device's Technical Reference / Data
 * // Manual for the STM base address.
 *
 * LoggerSTM.stmBaseAdrs = 0x20000000;
 * LoggerSTM.stmChannelResolution = 0x1000;
 * 
 * LoggerSTM.isStringOutputModeEnabled = false;
 * 
 *
 * // If you wish to provide a custom memory allocation
 * // procedure for use by LoggerSTM and stmLib, you should
 * // set the following to false.  You will then need to provide 
 * // your own implementation of the following helper functions:
 * // void * cTools_memAlloc(size_t sizeInBytes);
 * // void cTools_memFree(void * ptr);
 * // void * cTools_memMap(unsigned int phyAddr, unsigned int mapSizeInBytes);
 * // void cTools_memUnMap(void * vAddr, unsigned int mapSizeInBytes);
 * //
 * // See ti.uiactools.runtime.LoggerSTM.xdt for the default implementation
 * // of these functions.
 *
 * LoggerSTM.isGenerateSTMLibSupportFnsEnabled = true;
 * 
 * // Configure which STM channels the messages are to be logged to.
 * // The channel number is incremented (and wrapped around if necessary)
 * // for each message so that, if the code logging the message is preempted
 * // by another task that also needs to log a message, the messages can be
 * // properly decoded by the host.
 * // Typically, channel 200 is reserved for synchronization events.
 * // (See the cdoc documentation for ti.uiactools.runtime.SyncSTM
 * // for more information).
 *
 * var LoggerSTMParams = new LoggerSTM.Params;
 * LoggerSTMParams.stmChannelRangeMin = 128;
 * LoggerSTMParams.stmChannelRangeMax = 199;
 * var loggerSTM0 = LoggerSTM.create(LoggerSTMParams);
 * Program.global["loggerSTMInstStatic"] = loggerSTM0;
 * 
 * //Configure the LogSnapshot module to log snapshot events
 * //(e.g. dynamically allocated strings and memory ranges)
 * //to STM
 * var LogSnapshot = xdc.useModule('ti.uia.runtime.LogSnapshot');
 * LogSnapshot.common$.logger = loggerSTM0;
 * 
 * var LoggingSetup = xdc.useModule('ti.uiactools.sysbios.LoggingSetupSTM');
 * LoggingSetup.logger = loggerSTM0;
 * LoggingSetup.sysbiosTaskLogging = true;
 * LoggingSetup.loadLogging = false;
 * LoggingSetup.mainLogging = true;
 *
 * // The following lines enable correlation of STM events 
 * // with UIA software events:
 * var LogSync = xdc.useModule('ti.uia.runtime.LogSync');
 * LoggerSTMParams.stmChannelRangeMin = 200;
 * LoggerSTMParams.stmChannelRangeMax = 200;
 * var loggerSTMsync = LoggerSTM.create(LoggerSTMParams);
 * LogSync.syncLogger = loggerSTMsync;
 *
 * // If correlation of STM events with UIA software events is not
 * // required (e.g. to reduce the code footprint), the
 * // above lines should be commented out and the following line
 * // enabled:
 * // LoggingSetup.disableMulticoreEventCorrelation = false;
 * 
 * @p
 *  @a(Examples)
 *  Configuration Example 2: The following XDC configuration script statements
 *  create a logger instance, assign it as the default logger for all
 *  modules, and enable `USER1` logging in all modules of the package
 *  `my.pkg`. See the `{@link Diags#setMaskMeta Diags.setMaskMeta()}` function
 *  for details on specifying the module names.
 *
 *  @p(code)
 *  var Defaults = xdc.useModule('xdc.runtime.Defaults');
 *  var Main = xdc.useModule('xdc.runtime.Main');
 *  var Diags = xdc.useModule('xdc.runtime.Diags');
 *  var LoggerSTM = xdc.useModule('ti.uiactools.runtime.LoggerSTM');
 *
 *  // Configure LoggerSTM to log to channels 100-128.
 *  // (assumes that the device has the STM base address at the
 *  // default address of 0x20000000).
 *  var LoggerSTMParams = new LoggerSTM.Params();
 *  LoggerSTMParams.stmChannelRangeMin = 100;
 *  LoggerSTMParams.stmChannelRangeMax = 128;
 *  var loggerSTM0 = LoggerSTM.create(LoggerSTMParams);
 * 
 *  // Configure which modules are to log 
 *  // events using the LoggerSTM instance that was created above.
 *  Main.common$.logger = loggerSTM0;
 *  Defaults.common$.logger = loggerSTM0;
 *
 *  // Configure which event masks are enabled.
 *  Main.common$.Diags_USER1 = Diags.RUNTIME_ON;
 *  Main.common$.Diags_ANALYSIS = Diags.RUNTIME_ON;
 *  Diags.setMaskMeta("my.pkg.%", Diags.USER1, Diags.RUNTIME_ON);
 *  @p
 *  The implementation of this logger is fast with minimal stack usage
 *  making it appropriate for a realtime application.
 *  This logger uses different STM channels for writing different numbers of
 *  parameters in order to avoid the need to have a header indicating the length
 *  in bytes of each event.  As a  result, the execution time of all `Log` methods
 *  bound to this type of logger are deterministic (and quite short) because there are no
 *  additional memory allocation calls required.
 *
 *  If this logger is used in a preemptive environment, then an appropriate
 *  gate must be assigned to the module. By default, the xdc.runtime.Gate module's system gate
 *  is used.  The following configuration script shows how to configure a different gate:
 *
 *  @p(code)
 *  var LoggerSTM = xdc.useModule('ti.uiactools.runtime.LoggerSTM');
 *  LoggerSTM.common$.gate = ...some gate instance...
 *  @p
 */
@ModuleStartup
/* Initialize static instances */
@InstanceFinalize
/* this mod has cleanup fxn when instances are deleted */
@InstanceInitError
/* instance create can fail */
@CustomHeader
/* definitions for OST header etc. */
@Template("./LoggerSTM.xdt")
module LoggerSTM inherits ti.uia.runtime.ILoggerSnapshot {

    /*!
     *  ======== TimestampProxy ========
     *  User supplied time-stamp proxy
     *
     *  This proxy allows `LoggerSTM` to use a timestamp server different
     *  from the server used by `{@link xdc.runtime.Timestamp}`. However, if
     *  not supplied by a user, this proxy defaults to whichever timestamp
     *  server is used by `Timestamp`.
     */
    proxy TimestampProxy inherits xdc.runtime.ITimestampClient;

    /*!
     *  ======== endpointId ========
     *  Endpoint ID for the CPU that is logging the event
     *
     *  The endpointId corresponds to Ipc's MultiProc value.
     *
     *  For single processor systems this value is ignored.
     *  For C64P devices, leave this value as 0xFFFF to configure
     *  the logger to use the DNUM value as the endpointId. 
     */
    config UInt16 endpointId = 0xFFFF;

    /*!
     * ======== getOSTHeader ========
     * returns the OST Header to use as the first word of the STM event
     *
     * param(numBytes)  number of bytes in the event, not including the OST header itself (max 255)
     */
    @Macro Int32 getOSTHeader(Int numBytes);

    /*!
     *  ======== stmChannelResolution ========
     *  Base address for the STM channel that events are to be written to
     */
    config Int stmChannelResolution = 0x1000; /* 4K channel channel resolution */
    /*!
     *  ======== stmBaseAdrs ========
     *  Base address for the STM channel that events are to be written to
     */
    config Ptr stmBaseAdrs = 0x54000000; /* OMAP4430 CortexA9 MIPI_STM 4K channel base address */

    /*!
     *  ======== setSTMBaseAdrs ========
     *  Sets the base address (STM channel 0) that events are to be written to
     *
     * @param(adrs) the STM base address (STM channel 0) that events are to be written to
     */
    @DirectCall
    Void setSTMBaseAdrs(Ptr adrs);

    /*!
     *  ======== getSTMBaseAdrs ========
     *  Gets the STM base address (STM channel 0) that events are to be written to
     */
    @DirectCall
    Ptr getSTMBaseAdrs();

   
    /*!
     *  ======== level1Mask ========
     *  Mask of diags categories whose initial filtering level is Diags.LEVEL1
     *
     *  See '{@link #level4Mask}' for details.
     */
    config Diags.Mask level1Mask = 0;

    /*!
     *  ======== level2Mask ========
     *  Mask of diags categories whose initial filtering level is Diags.LEVEL2
     *
     *  See '{@link #level4Mask}' for details.
     */
    config Diags.Mask level2Mask = 0;

    /*!
     *  ======== level3Mask ========
     *  Mask of diags categories whose initial filtering level is Diags.LEVEL3
     *
     *  See '{@link #level4Mask}' for details.
     */
    config Diags.Mask level3Mask = 0;

    /*!
     *  ======== level4Mask ========
     *  Mask of diags categories whose initial filtering level is Diags.LEVEL4
     *
     *  If 'filterByLevel' is true, then all LoggerBuf instances will filter
     *  incoming events based on their event level.
     *
     *  The LoggerCircBuf module allows for specifying a different filter level for
     *  every Diags bit. These filtering levels are module wide; LoggerBuf does
     *  not support specifying the levels on a per-instance basis.
     *
     *  The setFilterLevel API can be used to change the filtering levels at
     *  runtime.
     *
     *  The default filtering levels are assigned using the 'level1Mask' -
     *  'level4Mask' config parameters. These are used to specify, for each of
     *  the four event levels, the set of bits which should filter at that
     *  level by default.
     *
     *  The default filtering configuration sets the filter level to
     *  Diags.LEVEL4 for all logging-related diags bits so that all events are
     *  logged by default.
     */
    config Diags.Mask level4Mask = Diags.ALL_LOGGING;    
    
    /*!
     *  ======== isGenerateSTMLibSupportFnsEnabled ========
     *  Configures whether helper functions required by STMLib are automatically generated (default) or not.
     *
     *  if LoggerSTM.isGenerateSTMLibSupportFnsEnabled is true AND 
     *  LoggerSTM.isGenerateSTMLibSupportFnsEnabled is false, 
     *  the following helper functions required by STMLib are
     *  automatically generated by the LoggerSTM.xdt template:
     *  (see ti/ctools/stmLib/StmLibrary.h for details)
     *  @p(blist)
     *  - cTools_memAlloc(size_t sizeInBytes); 
     *  - void * cTools_memMap(unsigned int phyAddr, unsigned int mapSizeInBytes); 
     *  - void cTools_memFree(void * ptr); 
     *  - void cTools_memUnmap(void * vAddr, unsigned int mapSizeInBytes);
     *  @p
     *  If you wish to use STMLib directly, set this to false
     *  and provide implementations of the above functions in your
     *  application in order to handle memory allocation and mapping.
     */
    config Bool isGenerateSTMLibSupportFnsEnabled = true;
        
    /*!
     *  ======== isStringOutputModeEnabled ========
     *  if true, events are logged to STM as strings instead
     *  of raw UIA data.  Only enable this mode if not using
     *  System Analyzer to analyze and display events.
     *
     *  Note that setting LoggerSTM.isStringOutputModeEnabled to true
     *  will require that your application implement the STMLib helper
     *  functions in order to properly provide memory management for STMLib.
     *  The LoggerSTM.isGenerateSTMLibSupportFnsEnabled will be forced to
     *  'false' if LoggerSTM.isStringOutputModeEnabled is set to 'true'.  
     *  @see #isGenerateSTMLibSupportFnsEnabled
     */
    config Bool isStringOutputModeEnabled = false;
        
    /*!
     *  ======== printToSTM ========
     *  Helper function that prints formatted strings to STM
     */
     @DirectCall
     Void printToSTM(int chanNum, Log.Event evt, IArg a1,IArg a2,IArg a3,IArg a4, IArg a5, IArg a6, IArg a7, IArg a8);
         
    /*!
     *  ======== setProcessId ========
     *  Sets the active process ID to log with the event (default is 0)
     */
    @DirectCall
    Void setProcessId(UInt32 processId);
	
instance:
    /*!
     *  ======== enabled ========
     *  controls whether logging is enabled or disabled
     */
    config Bool enabled = true;

    /*!
     *  ======== stmChannelRangeMin ========
     *  The lowest STM channel number that events are to be written to
     */
    config Bits8 stmChannelRangeMin = 128;
    /*!
     *  ======== setSTMChannelRangeMin ========
     *  Sets the lowest STM channel number that events are to be written to
     */
    @DirectCall
    Void setSTMChannelRangeMin(Bits8 chanNum);


    /*!
     *  ======== getSTMChannelRangeMin ========
     *  Gets the lowest STM channel number that events are to be written to
     */
    @DirectCall
    Bits8 getSTMChannelRangeMin();

    /*!
     *  ======== stmChannelRangeMax ========
     *  The highest STM channel number that events are to be written to
     */
    config Bits8 stmChannelRangeMax = 200;

    /*!
     *  ======== setSTMChannelRangeMax ========
     *  Sets the highest STM channel number that events are to be written to
     */
    @DirectCall
    Void setSTMChannelRangeMax(Bits8 chanNum);

    /*!
     *  ======== getSTMChannelRangeMax ========
     *  Gets the highest STM channel number that events are to be written to
     */
    @DirectCall
    Bits8 getSTMChannelRangeMax();
    
    /*!
     * ======== maxEventSize ========
     * The maximum event size (in Maus) that can be written with a single event.
     *
     * Note: must be smaller than 240 due to STM small block size limitation.
     * Must also be an integer multiple of sizeof(UInt32).
     */
    config SizeT maxEventSize = 200;
    
    /*!
     *  ======== create ========
     *  Create a `LoggerSTM` logger
     *
     *  @see LoggerSTM#Params
     */
    @DirectCall
    create();


    /*!
     *  ======== write0 ========
     *  Process a log event with 0 arguments
     */
    @DirectCall
    override Void write0(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid);

    /*!
     *  ======== write1 ========
     *  Process a log event with 1 argument
     */
    @DirectCall
    override Void write1(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid, IArg a1);

    /*!
     *  ======== write2 ========
     *  Process a log event with 2 arguments
     */
    @DirectCall
    override Void write2(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid, IArg a1, IArg a2);

    /*!
     *  ======== write4 ========
     *  Process a log event with 4 arguments
     */
    @DirectCall
    override Void write4(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid,  IArg a1, IArg a2, IArg a3, IArg a4);

    /*!
     *  ======== write8 ========
     *  Process a log event with 8 arguments
     */
    @DirectCall
    override Void write8(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid, IArg a1, IArg a2, IArg a3, IArg a4, IArg a5, IArg a6, IArg a7, IArg a8);

    /*!
     *  ======== memoryRange ========
     *  Process a log a snapshot of a range of memory values (e.g. array values, memory mapped registers, strings, etc.)
     */
    @DirectCall
    override Void writeMemoryRange(xdc.runtime.Log.Event evt, xdc.runtime.Types.ModuleId mid, UInt32 snapshotId, IArg fileName, IArg LineNum, IArg fmt, IArg startAdrs, UInt32 lengthInMAUs);

    /*!
     *  ======== setFilterLevel ========
     *  Sets the level of detail that instances will log.
     *
     *  Events with the specified level or higher will be logged, events
     *  below the specified level will be dropped.
     *
     *  Events are filtered first by diags category, then by level. If an
     *  event's diags category is disabled in the module's diags mask, then it
     *  will be filtered out regardless of level. The event will not even be
     *  passed to the logger.
     *
     *  This API allows for setting the filtering level for more than one
     *  diags category at a time. The mask parameter can be a single category
     *  or multiple categories combined, and the level will be set for all of
     *  those categories.
     *
     *  @param(mask) The diags categories to set the level for
     *  @param(filterLevel) The new filtering level for the specified
     *                      categories
     */
    @DirectCall
    override Void setFilterLevel(xdc.runtime.Diags.Mask mask, xdc.runtime.Diags.EventLevel filterLevel);

    /*!
     *  ======== getFilterLevel ========
     *  Returns the mask of diags categories currently set to the specified
     *  level.
     *
     *  See '{@link #setFilterLevel}' for an explanation of level filtering.
     */
    @DirectCall
    override xdc.runtime.Diags.Mask getFilterLevel(xdc.runtime.Diags.EventLevel level);

internal:

        @DirectCall
    Bool filterOutEvent(Diags.Mask mask);

    struct Module_State {
        Bits16 sequenceNum;
        Diags.Mask level1;
        Diags.Mask level2;
        Diags.Mask level3;
        Bool isInitialized;
        Ptr stmBaseAdrs;
        Int stmChannelResolution;
        UInt majorVersion;
        UInt minorVersion;
        UInt32 endpointId;
        UInt32 processId;
    };

    struct Instance_State {
        Bool enabled;
        SizeT maxEventSizeInBits32;
        Bits8 stmChannel;
        Bits8 stmChannelRangeMin;
        Bits8 stmChannelRangeMax;		
    };

}



