/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
  *  ======== StmEvents.h ========
 * The following macros are used to enable writing event data to STM with
 * very low overhead.
 *
 * To use them,:
 * - define a variable of type STMEventHandle
 * - create an STM Handle by calling STMXport_open(NULL); (To enable blocking IO pSTMBufObj must be NULL.)
 * - call STM_writeEventStart, passing in the address of the STMeventHandle variable
 * - for all but the last Int32 word of the event, call STM_writeEventWord, passing in the value of the STMeventHandle variable
 * - for the last Int32 word of the event, call STM_writeEventLastWord
 *
 */

#ifndef STMEVENTS_H_
#define STMEVENTS_H_

#include <stdio.h>


#include <ti/ctools/stmLib/StmLibrary.h>
#include <ti/ctools/stmLib/StmSupport.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef UInt32 STMEventHandle;

#ifdef _CIO

#define STMX_getStmEventHandle(pSTMHandle, chNum) \
  ( (STMEventHandle)STM_getEventStartAddress(pSTMHandle, chNum))

#define STMX_writeEventStart(pSTMHandle, pSTMEventHandle, chNum, numBytesInEvent) \
  (  fprintf(((STMHandle *)pSTMHandle)->pLogFileHdl,"STM_writeEvent (MasterID=%d, chNum=%d): ", ((STMHandle *)pSTMHandle)->BufIOInfo.pMasterId, chNum) )


#define STMX_writeEventWord(pSTMHandle,stmEventHandle, dataWord) \
  (  fprintf (((STMHandle *)pSTMHandle)->pLogFileHdl,"0x%8.8X ", dataWord) )



#define STMX_writeEventLastWord(pSTMHandle,pSTMEventHandle, dataWord) \
  { fprintf (((STMHandle *)pSTMHandle)->pLogFileHdl,"0x%8.8X\n", dataWord); \
      fflush(((STMHandle *)pSTMHandle)->pLogFileHdl);}

#else /* _CIO undefined */

/* internal helper function */
#define STM_getEventStartAddress(pSTMHandle,chNum) \
   ((STMEventHandle)((UInt32)((STMHandle *)pSTMHandle)->ulBaseAddress)+ (((UInt32)((STMHandle *)pSTMHandle)->chResolution) *(UInt32)chNum))

/* internal helper function */
#define STM_getEventEndAddress(pSTMHandle,ulMsgHdrAddress) \
   ((STMEventHandle)((UInt32)ulMsgHdrAddress + (UInt32)(((STMHandle *)pSTMHandle)->chResolution/2)))
/*
\par STMX_getStmEventHandle

    Internal Message: This method is meant to be used by
    the Unified Instrumentation Architecture to log UIA events.
    This function does not require any knowedge by the decoder as to which library was the source
    of the data.

    \param[in] pSTMHandle  A pointer to an STM Handle returned by STMXport_open().
    \param[in] chNum       An STM Channel number in the range of 0 to 254 (255 reserved).
                           The most common usage is to identify messages coming from
                           different OS tasks or processes. It's expected the application
                           will manage it's channel usage.
    \return     stmEventHandle  the STM Event Handle to use in calls to STMX_writeEventStart and STMX_writeEventWord

    This function is utilized in order to get a valid stmEventHandle
    prior to writing event data to STM.  It should be called prior to
    writing the first word of each event.
*/
#define STMX_getStmEventHandle(pSTMHandle, chNum) \
   ((STMEventHandle)((UInt32)((STMHandle *)pSTMHandle)->ulBaseAddress)+ (((UInt32)((STMHandle *)pSTMHandle)->chResolution) *(UInt32)chNum))

/*
\par STMX_getStmEventHandleForLastWord

    Internal Message: This method is meant to be used by
    the Unified Instrumentation Architecture to log UIA events.
    This function does not require any knowedge by the decoder as to which library was the source
    of the data.

    \param[in] pSTMHandle  A pointer to an STM Handle returned by STMXport_open().
    \param[in] stmEventHandle a variable of type STMEventHandle that was initialized by calling STMX_getStmEventHandle

    \return     stmEventHandle  the STM Event Handle to use to call STMX_writeEventWordForLastWord

    This function is utilized after calling STMX_getStmEventHandle
    in order to update the stmEventHandle in order to write the last
    word of event data to STM.
*/
#define STMX_getStmEventHandleForLastWord(pSTMHandle, stmEventHandle) \
   ((STMEventHandle)((UInt32)stmEventHandle)+ (UInt32)(((STMHandle *)pSTMHandle)->chResolution/2))


/*
\par STMX_writeEventStart

    Internal Message: This method is meant to be used by
    the Unified Instrumentation Architecture to log UIA events.
    This function does not require any knowedge by the decoder as to which library was the source
    of the data.

    \param[in] pSTMHandle  A pointer to an STM Handle returned by STMXport_open(NULL).
    \param[in] pSTMEventHandle A pointer to a variable of type STMEventHandle provided by the caller
    \param[in] chNum       A STM Channel number in the range of 0 to 254 (255 reserved).
                           The most common usage is to identify messages coming from
                           different OS tasks or processes. It's expected the application
                           will manage it's channel usage.
    \param[in] numBytesInEvent The number of bytes in the event (not including the OST header)

    This function is utilized to start writing event data to STM using blocking writes.
    All elements are transported using 32-bit STM transfers.
*/


#define STMX_writeEventStart(pSTMHandle, stmEventHandle, chNum, numBytesInEvent) \
    (   *(UInt32 *)((STMEventHandle)stmEventHandle) = ((UInt32)OST_VERSION | (UInt32) OST_ENTITY_ID | (UInt32)OST_PROTOCOL_MSGSTREAM | (numBytesInEvent) ))

 /*
\par STMX_writeEventWord

    Internal Message: This method is meant to be used by
    the Unified Instrumentation Architecture to log UIA events.
    This function does not require any knowedge by the decoder as to which library was the source
    of the data.

    \param[in] pSTMHandle  A pointer to an STM Handle returned by STMXport_open().
    \param[in] stmEventHandle the value of a variable of type STMEventHandle provided by the caller (see STMX_writeEventStart)
    \param[in] dataWord    The Int32 value to be written to STM

    This function is utilized after calling STMX_writeEventStart
    in order to write one word of event data to STM using a blocking
    32-bit STM transfer.
*/
#define STMX_writeEventWord(pSTMHandle, stmEventHandle, dataWord) \
    (   *(UInt32 *)((STMEventHandle)stmEventHandle) = (Int32)dataWord )

 /*
\par STMX_writeEventLastWordInEvent

    Internal Message: This method is meant to be used by
    the Unified Instrumentation Architecture to log UIA events.
    This function does not require any knowedge by the decoder as to which library was the source
    of the data.

    \param[in] pSTMHandle  A pointer to an STM Handle returned by STMXport_open().
    \param[in] stmEventHandle the value of a variable of type STMEventHandle provided by the caller (see STMX_getStmEventHandleForLastWord)
    \param[in] dataWord    The Int32 value to be written to STM

    This function is utilized after calling STMX_writeEventStart
    in order to write one word of event data to STM using a blocking
    32-bit STM transfer.
*/

#define STMX_writeEventLastWord(pSTMHandle,stmEventHandle, dataWord) \
    (   *(UInt32 *)((STMEventHandle)stmEventHandle) = (Int32)dataWord )


#endif /* _CIO */

#ifdef __cplusplus
}
#endif

#endif /*STMEVENTS_H_*/

/*
 */
/*
 *  @(#) ti.uia.stm; 1, 0, 0, 0,46; 9-16-2010 17:37:53; /db/vtree/library/trees/uia/uia.git/src/ null
 */

