/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== SyncSTM.c ========
 */

#include <xdc/std.h>
#include <xdc/runtime/Types.h>
#include <xdc/runtime/Timestamp.h>
#include <xdc/runtime/Gate.h>
#include <ti/uia/events/UIASync.h>
#include <ti/uia/runtime/IUIATraceSyncProvider.h>
#include <ti/uiactools/runtime/LoggerSTM.h>
#include <ti/uia/runtime/LogSync.h>

#ifdef xdc_target__isaCompatible_64P
#include <c6x.h>
#endif

#include "package/internal/SyncSTM.xdc.h"

/*! ======== getSyncWord ========
 * Returns a single 32b word of the sync point data to be written into the 
 *  C6X OVERLAY register if SyncSTM_isC6XTraceSyncEnabled is true
 *
 * Use this macro with ContextType values 0-3.
 */
#define getSyncWord(serialNum, ctxType) \
        (((ctxType & 0x0F)<<26) | ((serialNum << 2)& 0x03FFFFFC))

/*!
 * ======== getSyncWord1 ========
 * Returns the first word of a 2 word sync point to be written into the 
 *  C6X OVERLAY register if SyncSTM_isC6XTraceSyncEnabled is true
 *
 * @see getSyncWord1
 */
#define getSyncWord1(serialNum,ctxType,ctxData)\
        (0x80000000 | ((ctxType & 0x0F)<<26) | ((ctxData >> 10)& 0x03FFFFFC))

/*! ======== getSyncWord2 ========
 * Returns the second word of a 2 word sync point to be written into the 
 *    OVERLAY register if SyncSTM_isC6XTraceSyncEnabled is true
 *
 */
#define getSyncWord2(serialNum,ctxType,ctxData)\
        ((0xC0000000 | (0x3FFFFC & serialNum)) | ((ctxData & 0x0FF) << 2))

/*
 *  ======== SyncSTM_putSyncPoint ========
 *  Logs two successive sync point events to get STM local timestamp
 *  to scale to highest-resolution scale.
 */
Void SyncSTM_putSyncPoint(UInt32 serialNum){
    Types_Timestamp64 globalTStamp;
    Types_Timestamp64 cpuTStamp;
    Types_FreqHz globalFreq;
    Types_FreqHz cpuFreq;
    xdc_runtime_Log_Event evt = UIASync_syncPoint;

    LogSync_CpuTimestampProxy_getFreq(&cpuFreq);

    LogSync_GlobalTimestampProxy_get64(&globalTStamp); 
#ifndef xdc_target__isaCompatible_64P
    LogSync_CpuTimestampProxy_get64(&cpuTStamp); 
#else    
    cpuTStamp.lo = TSCL;     
    cpuTStamp.hi = TSCH;
#endif   
        /* Log the events directly without any filtering based on the 
         * diags mask. */
    xdc_runtime_ILogger_write8(ti_uiactools_runtime_SyncSTM_Module__loggerObj__C,
                evt,ti_uiactools_runtime_SyncSTM_Module__id__C, serialNum, 
                cpuTStamp.lo, cpuTStamp.hi, globalTStamp.lo,globalTStamp.hi,
                LogSync_cpuTimestampCyclesPerTick,cpuFreq.lo,cpuFreq.hi);
	/* If the LogSync module is logging its events to STM, then no need to output 
	 * the following events, since they will be logged by the LogSync module.  Otherwise,
	 * log the sync point info again to STM to reduce the timestamp scale to its minimum
	 * and log the global freq info to STM so that there is enough info to determine
	 * the frequency of operation of STM, the CPU and the global timestamp.
	 */
    if (ti_uia_runtime_LogSync_Module__loggerObj__C != ti_uiactools_runtime_SyncSTM_Module__loggerObj__C){
		LogSync_GlobalTimestampProxy_get64(&globalTStamp); 
#ifndef xdc_target__isaCompatible_64P
		LogSync_CpuTimestampProxy_get64(&cpuTStamp); 
#else    
	    cpuTStamp.lo = TSCL;     
	    cpuTStamp.hi = TSCH;
#endif   
	/* Log the events directly without any filtering based on the 
	 * diags mask. */
	    xdc_runtime_ILogger_write8(ti_uiactools_runtime_SyncSTM_Module__loggerObj__C,
					evt,ti_uiactools_runtime_SyncSTM_Module__id__C, serialNum, 
					cpuTStamp.lo, cpuTStamp.hi, globalTStamp.lo,globalTStamp.hi,
					LogSync_cpuTimestampCyclesPerTick,cpuFreq.lo,cpuFreq.hi);
	    LogSync_GlobalTimestampProxy_getFreq(&globalFreq);					
	    xdc_runtime_ILogger_write4(ti_uiactools_runtime_SyncSTM_Module__loggerObj__C,
					UIASync_globalTimerFreq,ti_uiactools_runtime_SyncSTM_Module__id__C,
					serialNum, LogSync_globalTimestampCpuCyclesPerTick,globalFreq.lo,
					globalFreq.hi);					
	}
}		

/*
 * ====== injectIntoTrace ======
 * Inject syncPoint info into GEM Trace
 * This method logs a sync point event and injects
 * correlation info into the trace stream (if available)
 * to enable correlation between software events and hardware trace.
 *
 */
Void SyncSTM_injectIntoTrace(UInt32 serialNumber, 
        IUIATraceSyncProvider_ContextType ctxType){
    volatile UInt32 syncWord;

#ifdef xdc_target__isaCompatible_64P
	if (SyncSTM_isC6XTraceSyncEnabled){
	    /* build word to inject into overlay register without using the OVERLAY 
         * register as scratchpad */
		syncWord= getSyncWord(serialNumber,ctxType);
		OVERLAY = syncWord;
	}
#endif
    SyncSTM_putSyncPoint(serialNumber);

}
/*
 * ====== injectCtxDataIntoTrace ======
 * Inject syncPoint info into GEM Trace
 * This method logs a sync point event and injects
 * correlation info into the trace stream (if available)
 * to enable correlation between software events and hardware trace.
 *
 */
Void SyncSTM_injectCtxDataIntoTrace(UInt32 serialNumber, 
        IUIATraceSyncProvider_ContextType ctxType, UInt32 ctxData){
    volatile UInt32 syncWord1;
    volatile UInt32 syncWord2;

#ifdef xdc_target__isaCompatible_64P
    switch(ctxType){
    /* Sync Point event sequence number */
    case IUIATraceSyncProvider_ContextType_SyncPoint: 
    /* Context Change event sequence number */
    case IUIATraceSyncProvider_ContextType_ContextChange:
    /* Snapshot event Snapshot ID  */
    case IUIATraceSyncProvider_ContextType_Snapshot: 

	    if (SyncSTM_isC6XTraceSyncEnabled){
			/* build word to inject into overlay register without using the 
			 * OVERLAY register as scratchpad */
			syncWord1= getSyncWord(ctxType,serialNumber);

			OVERLAY = syncWord1;
        }

        break;
    default:


        /* build the two words to inject into overlay register without using 
         * the OVERLAY register as scratchpad */
	    if (SyncSTM_isC6XTraceSyncEnabled){		 
			syncWord1= getSyncWord1(ctxType,ctxData,serialNumber);
			syncWord2= getSyncWord2(ctxType,ctxData,serialNumber);

			OVERLAY = syncWord1;
			OVERLAY = syncWord2;
		}


    }
#endif	
    SyncSTM_putSyncPoint(serialNumber);	
}

