/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== BufferUpload.xdc ========
 *  UIA RealTime Analysis module
 *
 *  Responsible for uploading the contents of a buffer to
 *  an instrumentation host. BufferUpload only services buffer packetizers that inherit from
 *  the {@link ti.uia.runtime.IUIATransfer} interface (e.g. 
 *   {@link ti.uia.runtime.EtbCircBuf}.
 *
 *  BufferUpload sends records to the host from one buffer packetizer instance at a
 *  time--it will finish sending the configured number of packets for one buffer before 
 *  moving on to the next one.
 *
 *  The BufferUpload service runs in the framework of the UIA ServiceMgr framework.
 *  Configuration of the how the logs are actually moved off the target
 *  is managed by the {@link ti.uia.runtime.ServiceMgr} modules.
 *
 *  Before sending the records to the host, the buffer packetizer will copy them into
 *  a packet buffer. The size of this packet buffer is configurable using
 *  the {@link ti.uia.runtime.ServiceMgr} module.
 *
 *  Because the application may generate logs faster than the buffer packetizer can send
 *  them to the host, there is a limit to how many packets the buffer packetizer will send
 *  from one buffer before moving on to the next one. 
 *
 *  The BufferUpload module also responds to commands from the host to configure logging
 *  dynamically on the target, such as changing modules' diagnostic masks and
 *  enabling or disabling buffer packetizers.
 *
 *  To enable BufferUpload, simply do an xdc.useModule in your configuration file.
 *  @p(code)
 *  var BufferUpload = xdc.useModule('ti.uia.services.BufferUpload');
 *  @p
 *  The ServiceMgr is brought in automatically be this statement. If the
 *  configuration of the ServiceMgr needs to be changed, the application must
 *  do an xdc.useModule on ti.uia.runtime.ServiceMgr and configured it as
 *  needed.
 */

import xdc.runtime.Diags;
import xdc.runtime.Log;
import ti.uia.runtime.ServiceMgr;
import ti.uia.runtime.UIAPacket;
import ti.uiactools.runtime.IUIAPacketTransfer;
import xdc.rov.ViewInfo;

module BufferUpload inherits ti.uia.events.IUIAMetaProvider{

    /* Structure of response packet sent back to host */
    struct Packet {
        UIAPacket.Hdr hdr;
        Bits32        arg0;
        Bits32        arg1;
    }

    /*!
     *  @_nodoc
     *  ======== ModuleView ========
     */
    metaonly struct ModuleView {
        Int         serviceId;
        Bool        enabled;
        Bool        snapshotMode;
        Int         period;
        UInt        numBufs;
        String      hBufferPacketizers[];
        Bits16      sequence;
        UInt        totalPacketsSent;
    }

    /*!
     *  @_nodoc
     *  ======== rovViewInfo ========
     */
    @Facet
    metaonly config ViewInfo.Instance rovViewInfo =
        ViewInfo.create({
            viewMap: [['Module',   {type: ViewInfo.MODULE,
                                    viewInitFxn: 'viewInitModule',
                                    structName: 'ModuleView'}
                      ]]
        });

    /*! Logged on every packet is sent call from the Rta */
    config Log.Event LD_recordsSent = {
        mask: Diags.USER2,
        msg: "BufferUpload LD_recordsSent:  Sent %d bytes from logger [%d] 0x%x"
    };

    /*! Logged when the Rta receives a command */
    config Log.Event LD_cmdRcvd = {
        mask: Diags.USER2,
        msg: "BufferUpload LD_cmdRcvd: Received command: %d, arg0: 0x%x, arg1: 0x%x"
    };
    
    /*!
     *  ======== periodInMs ========
     *  Period in miliseconds of the BufferUpload Transfer Agent Task
     *
     *  Configures how often the BufferUpload should collect events. The minimum
     *  value is 100ms.
     *
     *  This value does not guarantee that the collection will run
     *  at this rate. Even if the period has expired, the collection
     *  will not occur until the current running Task has yielded and there
     *  are no other higher priority Tasks ready.
     *
     *  Setting the period to 0, disables all collection of events. There
     *  must be a setPeriod message sent from an instrumentation host to
     *  BufferUpload to enable it.
     *
     *  Default is 100 milliseconds.
     */
    config Int periodInMs = 100;

    /*! @_nodoc
     *  ======== processCallback ========
     *  Function registered with the ServiceMgr module
     */
	@DirectCall	 
    Void processCallback(ServiceMgr.Reason reason, UIAPacket.Hdr *cmd);
    
    /*! @_nodoc
     *  ======== registerBuffers ========
     *  Register all buffer instances with the Buffer Packetizer so that it can service
     *  them on the target.
     */
    metaonly function registerBuffers(modObj);
    
    /*!
     *  ======== disableAllBuffers ========
     *  Function to disable all the buffer packetizers being processed by BufferUpload
     *
     *  Runtime function to disable all the logs that are being 
     *  processed/read by BufferUpload. When disabled, all new Etb records
     *  are discarded.
     *
     *  Please realize that external instrumentation host (e.g.
     *  System Analyzer) might be sending down similar requests.
     */
	@DirectCall	 
    Void disableAllBuffers();
    
    /*!
     *  ======== enableAllBuffers ========
     *  Function to enable all the buffer packetizers being processed by BufferUpload
     *
     *  Runtime function to enable disable all the logs that are being 
     *  processed/read by BufferUpload.
     *
     *  Please realize that external instrumentation host (e.g.
     *  System Analyzer) might be sending down similar requests
     */
	@DirectCall	 
    Void enableAllBuffers();
    
    /*!
     *  ======== snapshotAllBuffers ========
     *  Function to delay processing of the BufferUpload service
     *
     *  This function informs BufferUpload to delay for the specified waitPeriod (in ms).
     *  After the waitPeriod has expired, BufferUpload will process all the etbs
     *  that it manages. The state of BufferUpload (e.g. started or stopped) will
     *  be maintained after the waitPeriod is expired and all the logs 
     *  processed. 
     *
     *  The reset flag determines whether to reset all the etbs at the 
     *  start of the waitPeriod (true -> reset). The state of the etbs
     *  (e.g. enabled or disabled) is not changed by this flag.
     *
     *  @param(reset)   Flag to denote whether to reset the etbs or not.
     *                  TRUE means reset all the etbs processed by BufferUpload.
     *                  FALSE means do not reset any of the etbs processed
     *                  by BufferUpload.
     * 
     *  @param(waitPeriod) Duration in milliseconds to wait to run the BufferUpload
     *                     service.
     */
	@DirectCall	 
    Void snapshotAllBuffers(UArg reset, UArg waitPeriod);    
    
    /*!
     *  ======== resetAllBuffers ========
     *  Function to reset all the buffer packetizers being processed by BufferUpload
     *
     *  Runtime function to enable resets all the buffer packetizers that are being 
     *  processed/read by BufferUpload. 
     *  The state of the buffer packetizers (e.g. enabled or disabled) are not changed.
     *
     *  Please realize that external instrumentation host (e.g.
     *  System Analyzer) might be sending down similar requests     
     */
	@DirectCall
    Void resetAllBuffers();
    
    /*!
     *  ======== startDataTx ========
     *  Function to start the BufferUpload service
     *
     *  This function allows the BufferUpload service to be turned on.
     *
     *  Please realize that external instrumentation host (e.g.
     *  System Analyzer) might be sending down similar requests     
     */
	 @DirectCall
	 Void startDataTx();
    
    /*!
     *  ======== stopDataTx ========
     *  Function to stop the BufferUpload service
     *
     *  This function allows the BufferUpload service to be turned off.
     *
     *  Please realize that external instrumentation host (e.g.
     *  System Analyzer) might be sending down similar requests     
     */
	@DirectCall
    Void stopDataTx();
    
    /*!
     * ====== isDataTx ========
     * returns true if DataTx is active 
     */
	@DirectCall
    Bool isDataTx();
 
    /*!
     * ====== packetBufferExchange ========
     * Exchange function called by the buffer packetizer when it has
     * a packet of buffer data ready to send.	 
     */ 
	Ptr packetBufferExchange(Ptr hBufferPacketizer, Ptr full);
internal:

struct Module_State {
        IUIAPacketTransfer.Handle  hBufferPacketizers[];
        UInt                 numBufs;
        UInt                 totalPacketsSent;
        Int                  period;
        Bits16               seq;
        Bool                 txData; 
        Bool                 snapshot; 
    };
    /*
     *  ======== SERVICEID ========
     *  Method to obtain serviceId
     */
    readonly config ServiceMgr.ServiceId SERVICEID;


    /*
     *  ======== sendEvents ========
     *  Send out events
     */
    Void sendEvents();

    /*
     *  ======== processMsg ========
     *  Process an incoming message
     */
    Void processMsg(UIAPacket.Hdr *cmd);

    /*
     *  ======== flushBuffer ========
     *  Flushes a buffer.
     * @param(hBufferPacketizer) the handle of the hBufferPacketizer to flush
     * @param(bufNum) the hBufferPacketizer's InstanceId
     */
    Void flushBuffer(IUIAPacketTransfer.Handle hBufferPacketizer, UInt bufNum);

    /* Control APIs */
    Void acknowledgeCmd(Packet *resp);
    Void enableBuffer(UArg log);
    Void disableBuffer(UArg log);
    Void getCpuSpeed(Packet *resp);
    Void resetBuffer(UArg log);
    Void changePeriod(UArg period);
    
    /* Command ids */
    enum Command {
        Command_BUF_OFF = 2,
        Command_BUF_ON = 3,
        Command_GET_CPU_SPEED = 4,
        Command_RESET_BUF = 5,
        Command_CHANGE_PERIOD = 6,
        Command_START_TX = 7,
        Command_STOP_TX = 8,
        Command_BUF_OFF_ALL = 9,
        Command_BUF_ON_ALL = 10,
        Command_RESET_BUF_ALL = 11,
        Command_SNAPSHOT_ALL = 12        
    };

    enum ErrorCode {
        ErrorCode_NULLPOINTER = 0        
    };
    
    /*!
     * ======== maxNumConsecutivePackets ========
     * the maximum number of packets that are to be sent at one time
     * in response to the ServiceMgr's periodic call to the processCallback function.
     * A larger number drains the buffer more quickly,
     * a smaller number conserves the number of packets that need to be in flight
     * simultaneously
     */
	config UInt32 maxNumConsecutivePackets = 1;    

    /*! @_nodoc
     * ======== hasMetaData ========
     * Indicates that the SyncSTM module generates content for the uia.xml file.
     */
    override metaonly config Bool hasMetaData = true;    

}

