/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== BufferUpload.xs ========
 */

var BufferUpload;
var IUIATransfer;
var IEtb
var Etb;
var ServiceMgr;
var UIAPacket;
var Timestamp
var Error
var UIAMetaData;
var isEnableDebugPrintf = false;

/*
 *  ======== module$use ========
 */
function module$use()
{
    BufferUpload = this;

    /* Bring in all modules used directly by the BufferUpload. */
    IUIAPacketTransfer = xdc.useModule('ti.uiactools.runtime.IUIAPacketTransfer');
    ServiceMgr   = xdc.useModule('ti.uia.runtime.ServiceMgr');
    UIAPacket    = xdc.useModule('ti.uia.runtime.UIAPacket');
    Timestamp    = xdc.useModule('xdc.runtime.Timestamp');
    Error        = xdc.useModule('xdc.runtime.Error');
    UIAMetaData  = xdc.useModule('ti.uia.runtime.UIAMetaData');

//TODO:    LogSync.isUsedByBufferUpload();

}

/*
 *  ======== module$static$init ========
 */
function module$static$init(mod, params)
{
    if (xdc.om.$name != "cfg") {
        return;
    }    
    
    /* Register all of the etb instances with the module */
    registerEtbs(mod);    
    
    /*
     * Initialize totalPacketsSent, which simply tracks how many UIA packets
     * have been sent up for debugging purposes.
     */
    mod.totalPacketsSent = 0;

    /* Initialize the starting event sequence number */
    mod.seq = 0;

    /* 
     *  If there is support control, wait for a start command.
     *  If there is no support control, start sending...unless
     *  the period is zero. Give a remark to the user for this case.
     */
    if (ServiceMgr.supportControl == true) {
       mod.txData = false;
    } 
    else if (params.periodInMs != 0) {
        mod.txData = true;           
    }
    else { 
        mod.txData = false;
        print("\nremark: The BufferUpload.periodInMs is zero and the " +
              "ServiceMgr.supportControl is false. Therefore packets " +
              "cannot be uploaded to an instrumentation host.");    
    }
    
    mod.period = params.periodInMs;
    mod.snapshot = false;

    /* Plug in process function */
    ServiceMgr.register(BufferUpload.SERVICEID, '&ti_uiactools_services_BufferUpload_processCallback',
                        params.periodInMs);
 }

/*
 *  ======== registerEtbs ========
 *  Register all etb instances with the Agent so that it can service
 *  them on the target.
 */
function registerEtbs(modObj)
{
	var IUIAPacketTransfer = xdc.module('ti.uiactools.runtime.IUIAPacketTransfer');
    /* Add all instances of modules which implement IUIATransfer. */    
    for (var i = 0; i < xdc.om.$modules.length; i++) {
        var mod = xdc.om.$modules[i];
        
        /* Only handle IUIATransfer modules. */
        if (!(mod instanceof IUIAPacketTransfer.Module)) {
            continue;
        }
        
        /* Add each of the module's instances. */
        for (var j = 0; j < mod.$instances.length; j++) {            
            addBuffer(mod.$instances[j], modObj.hBufferPacketizers);
        }

        /* get embedded hBufferPacketizers too */
        for (var j = 0; j < mod.$objects.length; j++) {
            addBuffer(mod.$objects[j], modObj.hBufferPacketizers);
        }
    }

    /* Set numBufs so the target can iterate over them. */
    modObj.numBufs = modObj.hBufferPacketizers.length;
}

/*
 *  ======== addBuffer ========
 *  Add buffer to array if not already present.
 */
function addBuffer(bufferPacketizer, bufferPacketizers) 
{
    // TODO SDOCM00076661 get the etb's id

    for (var i = 0; i < bufferPacketizers.length; i++) {
        if (bufferPacketizer.$module.getBufferInstanceId(bufferPacketizer) == 
            bufferPacketizers[i].$module.getBufferInstanceId(bufferPacketizers[i])) {
            return;
        }
    }
    hBufferPacketizers.$add(etb);
}

/*
 *  ======== viewInitModule ========
 *  Initializes the Module view in ROV.
 */
function viewInitModule(view, mod)
{
    BufferUpload  = xdc.useModule("ti.uiactools.services.BufferUpload");
    
    var BufferUploadModConfig = Program.getModuleConfig(BufferUpload.$name);    
    
    /* Display whether ServiceId */
    view.serviceId = BufferUploadModConfig.SERVICEID.val; 
    
    /* Display whether BufferUpload is enabled to transmit */
    view.enabled = mod.txData; 
    
    /* Display whether BufferUpload is in snapshot waiting mode */
    view.snapshotMode = mod.snapshot; 
    
    /* Display the current period */
    view.period = mod.period; 
    
    /* Display the current sequence number */
    view.sequence = mod.seq; 
    
    /* Display the count of all records sent by the . */
    view.totalPacketsSent = mod.totalPacketsSent;    
    
    /* Display the number of hBufferPacketizers serviced by the Transfer Agent. */
    view.numBufs = mod.numBufs;    
    
    /* 
     * Display an array of the UIATransfer instances that the BufferUpload is servicing.
     * Also show the buffer id (the index in the array).
     */
    try {
        /* Get the array of bufferPacketizer handles. */
        var hBufferPacketizers = Program.fetchArray(mod.hBufferPacketizers$fetchDesc, 
                                               mod.hBufferPacketizers, mod.numBufs);
        
        /* For each handle, get its view, then get its label or address. */
        // TODO support all UIATransfer hBufferPacketizers SDOCM00077586
        for (var i = 0; i < hBufferPacketizers.length; i++) {
            etbView = Program.scanHandleView('ti.uiactools.runtime.EtbDrainer', 
                                                hBufferPacketizers[i], 'Instances');
            
            /* If a label is defined, display that. */
            if ((etbView.label != undefined) && (etbView.label != "")) {
                view.hBufferPacketizers.$add('[' + i + '] Label: ' + etbView.label);
            }
            /* Otherwise display the address. */
            else {
                view.hBufferPacketizers.$add('[' + i + '] Handle: ' + etbView.address);
            }
        }
    }
    catch (e) {
        view.$status["hBufferPacketizers"] = "Caught exception while processing " +
                                  "hBufferPacketizers: " + e.toString();
    }
}

/*
 *  ======== writeUIAMetaData ========
 * generates XML metadata describing what streams are being synchronized for correlation
 *
 *  NOTE: This should only be called by ti.uiactools.runtime.UIAMetaData.xs in order
 *  to ensure that the UIA xml file has been properly opened and is ready for
 *  writing.
 *  @param(inst) instance of the module to generate data for, null for module-level info 
 *  @param(instNum) the instance number used in .h files
 *  @param(indent) the number of spaces to preceed the xml tag with     
 */
function writeUIAMetaData(inst, instNum, indent)
{
	var mod = xdc.module('ti.uiactools.services.BufferUpload');
	var UIAMetaData = xdc.module('ti.uia.runtime.UIAMetaData');
	var ServiceMgr = xdc.module("ti.uia.runtime.ServiceMgr");
	if (isEnableDebugPrintf) {
		print("ti.uiactools.services.BufferUpload.writeUIAMetaData instNum = "+instNum);
		print("ti.uiactools.services.BufferUpload.SERVICEID = "+""+mod.SERVICEID);
	}

	UIAMetaData.genXmlEntryOpen(indent,"services");
	indent += 3;
	UIAMetaData.genXmlEntryOpen(indent,"service");
	indent += 3;
	UIAMetaData.genXmlEntry(indent,"name", mod.$name);
	for each (var prop in mod) {
		if (prop instanceof ServiceMgr.ServiceId) {
			 UIAMetaData.genXmlEntry(indent,"serviceId",prop.val.toString());
			 break;
		}
	}		
	indent -= 3;
	UIAMetaData.genXmlEntryClose(indent,"service");
	indent -= 3;
	UIAMetaData.genXmlEntryClose(indent,"services");
}    
