/* 
 * Copyright (c) 2012, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * */

/*
 *  ======== LoggingSetupSTM.xdc ========
 */

package ti.uiactools.sysbios;

/*!
 *  ======== LoggingSetupSTM ========
 *  Module to aid in configuring SYSBIOS logging when using ti.uiactools.loggers.stream.LoggerStreamer2
 * @p
 * The LoggingSetupSTM module automates the process of configuring an application
 * to use UIA events, and configures SYS/BIOS modules to capture user-specified
 * information such as CPU Load, Task Load and Task Execution so that it can
 * be displayed by System Analyzer.  It also automates the creation of 
 * infrastructure modules such as loggers.  If you are do not want to use LoggerStreamer2 
 * to log events, please use ti.uiactools.sysbios.LoggingSetup instead.
 * @p
 *  The following configuration script demonstrates the use of the LoggingSetupSTM
 *  module in the XDC configuration file for the application:
 *
 * @a(Example)
 * Example 1: Configuring an application to use the default settings provided
 * by LoggingSetupSTM.  The following default settings are automatically applied:
 * @p(blist)
 *    - Logging UIA events from user-provided C code.  User provided C code is
 *    treated as part of the xdc.runtime.Main module.  A LoggerStreamer2 logger with
 *    a buffer size of 1400 bytes is enabled by default to support this.
 *   - Event logging is enabled for the SYS/BIOS Load and Task modules in order to 
 *   allow System Analyzer to display CPU Load, Task Load, and Task Execution 
 *   information.  Logging of SWI and HWI events is disabled by default.
 *   In order to optimize event capture and minimize event loss, two loggers 
 *   are created: one to store events from the SYS/BIOS Load module
 *   and the other to store events from other SYS/BIOS modules.
 *   - Multicore event correlation is enabled by default.  This enables the
 *    LogSync module and creation of a dedicated logger for sync point events
 *    to be logged to the LoggerStreamer2 logger.
 *   - The Event Upload Mode is configured for STREAMER.  This
 *    allows events to be uploaded in real-time via application provided code
 *    in response to the exchange callback.
 *  @p 
 *  @p(code)
 *  // the Log module provides logging APIs for use by the user's software
 *  var Log = xdc.useModule('xdc.runtime.Log');
 *  // the LoggingSetupSTM module's default settings configure much of the UIA infrastructure.
 *  var LoggingSetupSTM = xdc.useModule('ti.uiactools.sysbios.LoggingSetupSTM');
 *  @p
 *  @p(html)
 *  <hr />
 *  @p
 * @a(Example)
 * Example 2: A number of 'template' applications are available that 
 * provide predefined XDC configuration scripts and C code for use in new
 * projects.  These templates provide good examples of how to configure all 
 * of the various modules that are involved in setting up the UIA infrastructure,
 * including the LoggingSetupSTM module.  The templates can be downloaded from
 * the System Analyzer Wiki site at 
 *  @p(html)
 *  <a href="http://processors.wiki.ti.com/index.php/System_Analyzer">
 * http://processors.wiki.ti.com/index.php/System_Analyzer</a>
 * @p
 * They can also be generated directly by CCS, using the CCS New Project 
 * Wizard.  The following steps show how to use CCSv5.0.x to generate a new project
 * that configure the NDK to provide an Ethernet transport for uploading events 
 * from the target to the host, please perform the following steps in CCS. :
 * @p(blist)
 *   - File / New / CCS Project : opens the 'New CCS Project' dialog
 *   - <enter a name for the project>
 *   - <select the type of project>
 *   - <accept defaults for Additional Project Settings>
 *   - <configure the project settings for compiler options, etc.>
 *   - For Project Templates, expand the UIA and System Analyzer Examples section
 *   - select evm6472: Stairstep for a single core with UIA to see how
 *   to configure the NDK to use Ethernet as a transport for UIA events and
 *   commands.  
 *  - select evm6472: MessageQ (single image for all cores) with UIA
 *   to see how to create a multicore application that uses the same application
 *   software for all cores, and uses IPC to move event data from CPU cores 1-5
 *   to CPU core 0, with CPU core 0 using the NDK for Ethernet communications
 *   with the host.
 *  @p
 * 
 *  @p(html)
 *  <hr />
 */ 
metaonly module LoggingSetupSTM 
{

    enum UploadMode {
        UploadMode_SIMULATOR = 1, /* note that simulators use probe points */
        UploadMode_PROBEPOINT = 2,
        UploadMode_JTAGSTOPMODE = 3,
        UploadMode_JTAGRUNMODE = 4,
        UploadMode_NONJTAGTRANSPORT = 5,
        UploadMode_NONJTAG_AND_JTAGSTOPMODE = 6,
        UploadMode_STREAMER = 7,
        UploadMode_IDLE = 8,
        UploadMode_STM = 9
    }
        
    /*!
     *  ======== logger ========
     *  LoggerStreamer used for logging events
     * @a(Example)
     * Example 1: Configuring an loggerStreamer with custom settings for use
     * by LoggingSetupSTM.  
     *  @p 
     *  The following configuration script shows how to create a LoggerStreamer2
     *  module and hook it in to the LoggingSetupSTM module so that it is used to
     *  log SysBios and application events:
     *  @p(code)
     *  // the Log module provides logging APIs for use by the user's software
     *  var Log = xdc.useModule('xdc.runtime.Log');
     *  var LoggerStreamer = xdc.useModule('ti.uiactools.loggers.stream.LoggerStreamer2');
     *  // A packet length of 1500 bytes or less is recommended for streaming over UDP.
     *  // The UDP packet length includes a 42 byte UDP packet header, which must be pre-pended
     *  // in front of the UIA event packet that is generated by LoggerStreamer2.
     *  // Also, the LoggerStreamer buffer must be word aligned and an integer multiple of 4
     *  // (i.e. contain an even number of UInt32 words).
     *  LoggerStreamer.bufSize = 1498-42; 
     *  LoggerStreamer.isTimestampEnabled = true;
     *  LoggerStreamer.primeFxn = '&prime';
     *  LoggerStreamer.exchangeFxn = '&exchange';
     *  var logger0 = LoggerStreamer.create();
     *
     *  var Defaults = xdc.useModule('xdc.runtime.Defaults');
     *  Defaults.common$.logger = logger0;
     *  // the LogSnapshot module supports logging dynamic strings and blocks of memory values.
     *  // Configure it to use the LoggerStreamer2 logger.
     *  var LogSnapshot = xdc.useModule('ti.uiactools.runtime.LogSnapshot');
     *  LogSnapshot.common$.logger = logger0;
     *
     *  // Configure LoggingSetup to use the same LoggerStreamer logger for Bios and application events.
     *  var LoggingSetup = xdc.useModule('ti.uiactools.sysbios.LoggingSetupSTM');
     *  LoggingSetup.eventUploadMode = LoggingSetup.UploadMode_STM;
     *  LoggingSetup.logger = logger0;
     * @p
     */
    config xdc.runtime.ILogger.Handle logger = null;
    
   
    /*!
     *  ======== loadLogging ========
     *  Enable the Load module event logging.
     *
     *  If this is false, the events will be disabled. Otherwise the events 
     *  will be enabled.
     *  Use the {@link #loadLoggingRuntimeControl} parameter 
     *  to determine whether the state can be modified during runtime.
     */
    metaonly config Bool loadLogging = true;
    
    /*!
     *  ======== loadLoggingRuntimeControl ========
     *  Specify whether load logging can be enabled / disabled at runtime.
     *
     *  This determines what diags settings are applied to the module's diags
     *  mask. If 'false', the diags bits will be configured as 
     *  ALWAYS_ON, meaning they can't be changed at runtime. If 'true', the 
     *  bits will be configured as 'RUNTIME_ON'.
     *
     *  Use the {@link #loadLogging} parameter 
     *  to determine whether the event is ON or OFF. For example, the
     *  following two lines set the Load modules events to 
     *  initially be 'ALWAYS_ON'.
     *
     *  @p(code)
     *  LoggingSetupSTM.loadLogging = true;   
     *  LoggingSetupSTM.loadLoggingRuntimeControl = false;
     *  @p
     */
    metaonly config Bool loadLoggingRuntimeControl = true;
    
   
   
    /*!
     *  ======== mainLogging ========
     *  Enable main and non-XDC modules event logging
     *
     *  If this is false, the events will be disabled. Otherwise the events 
     *  will be enabled.
     *  Use the {@link #mainLoggingRuntimeControl} parameter 
     *  to determine whether the state can be modified during runtime.
     */
    metaonly config Bool mainLogging = true;
    
    /*!
     *  ======== mainLoggingRuntimeControl ========
     *  Specify whether main logging can be enabled / disabled at runtime.
     *
     *  This determines what diags settings are applied to the module's diags
     *  mask. If 'false', the diags bits will be configured as 
     *  ALWAYS_ON, meaning they can't be changed at runtime. If 'true', the 
     *  bits will be configured as 'RUNTIME_ON'.
     *
     *  Use the {@link #mainLogging} parameter 
     *  to determine whether the event is ON or OFF. For example, the
     *  following two lines set the Load modules events to 
     *  initially be 'ALWAYS_ON'.
     *
     *  @p(code)
     *  LoggingSetupSTM.mainLogging = true;   
     *  LoggingSetupSTM.mainLoggingRuntimeControl = false;
     *  @p
     */
    metaonly config Bool mainLoggingRuntimeControl = true;
    
   
    /*!
     *  ======== sysbiosHwiLogging ========
     *  Enable SYSBIOS Hwi and Clock modules' event logging
     *
     *  If this is false, the events will be disabled. Otherwise the events 
     *  will be enabled.
     *  Use the {@link #sysbiosHwiLoggingRuntimeControl} parameter 
     *  to determine whether the state can be modified during runtime.
     */
    metaonly config Bool sysbiosHwiLogging = false;
    
    /*!
     *  ======== sysbiosHwiLoggingRuntimeControl ========
     *  Specify whether Hwi and Clock logging can be enabled / disabled at runtime
     *
     *  This determines what diags settings are applied to the module's diags
     *  mask. If 'false', the diags bits will be configured as 
     *  ALWAYS_ON, meaning they can't be changed at runtime. If 'true', the 
     *  bits will be configured as 'RUNTIME_ON'.
     *
     *  Use the {@link #sysbiosHwiLogging} parameter 
     *  to determine whether the event is ON or OFF. For example, the
     *  following two lines set the Load modules events to 
     *  initially be 'ALWAYS_ON'.
     *
     *  @p(code)
     *  LoggingSetupSTM.sysbiosHwiLogging = true;   
     *  LoggingSetupSTM.sysbiosHwiLoggingRuntimeControl = false;
     *  @p
     */
    metaonly config Bool sysbiosHwiLoggingRuntimeControl = false;
    
    /*!
     *  ======== sysbiosSwiLogging ========
     *  Enable SYSBIOS Swi module's event logging
     *
     *  If this is false, the events will be disabled. Otherwise the events 
     *  will be enabled.
     *  Use the {@link #sysbiosSwiLoggingRuntimeControl} parameter 
     *  to determine whether the state can be modified during runtime.
     */
    metaonly config Bool sysbiosSwiLogging = false;
    
    /*!
     *  ======== sysbiosSwiLoggingRuntimeControl ========
     *  Specify whether Swi logging can be enabled / disabled at runtime.
     *
     *  This determines what diags settings are applied to the module's diags
     *  mask. If 'false', the diags bits will be configured as 
     *  ALWAYS_ON, meaning they can't be changed at runtime. If 'true', the 
     *  bits will be configured as 'RUNTIME_ON'.
     *
     *  Use the {@link #sysbiosSwiLogging} parameter 
     *  to determine whether the event is ON or OFF. For example, the
     *  following two lines set the Load modules events to 
     *  initially be 'ALWAYS_ON'.
     *
     *  @p(code)
     *  LoggingSetupSTM.sysbiosSwiLogging = true;   
     *  LoggingSetupSTM.sysbiosSwiLoggingRuntimeControl = false;
     *  @p
     */
    metaonly config Bool sysbiosSwiLoggingRuntimeControl = false;
    
    /*!
     *  ======== sysbiosTaskLogging ========
     *  Enable SYSBIOS Task module's event logging
     *
     *  If this is false, the events will be disabled. Otherwise the events 
     *  will be enabled.
     *  Use the {@link #sysbiosTaskLoggingRuntimeControl} parameter 
     *  to determine whether the state can be modified during runtime.
     */
    metaonly config Bool sysbiosTaskLogging = true;
    
    /*!
     *  ======== sysbiosTaskLoggingRuntimeControl ========
     *  Specify whether Task logging can be enabled / disabled at runtime.
     *
     *  This determines what diags settings are applied to the module's diags
     *  mask. If 'false', the diags bits will be configured as 
     *  ALWAYS_ON, meaning they can't be changed at runtime. If 'true', the 
     *  bits will be configured as 'RUNTIME_ON'.
     *
     *  Use the {@link #sysbiosTaskLogging} parameter 
     *  to determine whether the event is ON or OFF. For example, the
     *  following two lines set the Load modules events to 
     *  initially be 'ALWAYS_ON'.
     *
     *  @p(code)
     *  LoggingSetupSTM.sysbiosTaskLogging = true;   
     *  LoggingSetupSTM.sysbiosTaskLoggingRuntimeControl = false;
     *  @p
     */
    metaonly config Bool sysbiosTaskLoggingRuntimeControl = true;

   
    /*!
     *  ========= eventUploadMode ========
     *  Event upload mode
     *  @p
     *  Upload_STREAMER: events are uploaded by the application
     *  when the LoggerStreamer2 logger calls the application-provided
     *  buffer exchange function.  If a different type of logger is 
     *  required, please use ti.uiactools.sysbios.LoggingSetup instead.  
     *  @p 
     *  
     *  @a(Example)
     *  The following is an example of the configuration script used
     *  to configure the system to use a LoggerStreamer logger in order to
     *  stream events from the target to the host while the target is running.  
     *   
     *  @p(code)
     *  var LoggingSetupSTM = xdc.useModule('ti.uiactools.sysbios.LoggingSetupSTM');
     *  LoggingSetupSTM.eventUploadMode = LoggingSetupSTM.UploadMode_STM;
     *   
     */
    metaonly config UploadMode eventUploadMode = UploadMode_STM; 

    /*!
     * ======== disableMulticoreEventCorrelation ========
     * Set to true for single core applications.
     * 
     * When true the LoggingSetupSTM module will not automatically
     * include the the LogSync module.  The LogSync module is
     * required in order to enable events from multiple CPU cores
     * to be correlated with each other.
     * @see ti.uia.runtime.LogSync  
     * 
     */
    metaonly config Bool disableMulticoreEventCorrelation = false;
   
    /*! @_nodoc
     * ======== createLogger =========
     * Internal helper function that creates the type of logger
     * appropriate for the LoggingSetupSTM.uploadMode that has been configured.
     * 
     * @param(loggerSize): the size of the logger in MAUs
     * @param(loggerInstanceName): the name to assign to the logger instance
     * @param (loggerPriority): the IUIATransfer.Priority to assign to the logger instance
     * @a(return) returns the logger instance that was created
     */
     metaonly function createLogger(loggerSize,loggerInstanceName,loggerPriority);
}

/*
 */

/*
 *  @(#) ti.uiactools.sysbios; 1, 0, 0, 0,1; 2-13-2012 15:15:39; /db/vtree/library/trees/uia/uia-c35/src/ xlibrary

 */

