/*****************************************************************************\
*           TEXAS INSTRUMENTS INCORPORATED PROPRIETARY INFORMATION           
*                                                                            
*  Property of Texas Instruments 
*  For  Unrestricted  Internal  Use  Only 
*  Unauthorized reproduction and/or distribution is strictly prohibited.  
*  This product is protected under copyright law and trade secret law 
*  as an unpublished work.  
*  (C) Copyright 2009 Texas Instruments.  All rights reserved.
*------------------------------------------------------------------------------
*  Filename       : vcp2_test.c
*  Description    : Performs VCP2 Decoding with EDMA3
*                   r = 1/4 K = 9 FL = 120 Hard Decision Tailed Mode
*                   r = 1/2 K = 5 FL = 1000 Soft Decision Tailed Mode
*                   r = 1/3 K = 7 FL = 6000 Hard Decision Convergent Mode
*                   r = 1/2 K = 9 FL = 6000 Hard Decision Mixed Mode
\*****************************************************************************/

#include <stdio.h>
#include <ti/csl/csl.h>
#include <ti/csl/csl_edma3.h>
#include <ti/csl/csl_vcp2.h>
#include <ti/csl/csl_vcp2Aux.h>
#include <csl_intc.h>
#include <ti/csl/csl_chip.h>
#include <ti/csl/csl_tsc.h>
#include <ti/csl/csl_psc.h>
#include <ti/csl/csl_pscAux.h>
#include <ti/csl/csl_cacheAux.h>


#include "val_util.h"
#include "vcp2_testInit.h"
#include "vcp2_parameters.h"
#include "vcp2_edma3.h"
#include "vcp2_test.h"
#include "string.h"

// Constant definitions
#define NUM_USERS                   4
#define VCP2_POWER_ON_IN_PROGRESS   0xFEFEFEFE
#define VCP2_POWER_ON_COMPLETE      0xABCDABCD

// Global variable declaration
VCP2Handle  hVcp2;
VCP2Obj     pVcp2Obj;
Uint32      coreNum;

volatile Uint32 VcpError = 0;

volatile Uint32 VCP_IPTX_TimeStamp_Start_Count = 0;
volatile Uint32 VCP_IPTX_TimeStamp_Stop_Count = 0;
volatile Uint32 VCP_OPRX_TimeStamp_Stop_Count = 0;

volatile Uint32 TxEdmaChannelBCount = 0;
volatile Uint32 ChannelTXCount = 0;
volatile Uint32 ChannelRXCount = 0;

Uint32 profileIPXmtStartLow[8], profileIPXmtStartHigh[8];
Uint32 profileIPXmtStopLow[8], profileIPXmtStopHigh[8];
Uint32 profileOPRcvStopLow[8], profileOPRcvStopHigh[8];
CSL_Uint64 profileIPXmtStartLow64[8];
CSL_Uint64 profileIPXmtStopLow64[8];
CSL_Uint64 profileOPRcvStopLow64[8];

/* Stores profiling data */
Uint32 frameDecodeTime[20];
volatile unsigned long long int Profile64;

#pragma DATA_ALIGN(configIcArray, 8)   /* Should be double-word aligned   */
VCP2_ConfigIc configIcArray[4] ;  /* VCP Input Configuration Reg */

Uint32 RefoutParams[4][2] = {{0x05aa07f1, 0x00000001},
                            {0x0a8c0af0, 0xf},
                            {0x16a917dd,0x0},
                            {0x027b034f, 0x6a}} ;

Uint32 NumofDecisions[4] = {4,32, 188, 188};

/* The coreNum dimension is required only for DDR3 endpoints as DDR3 is shared
 * by all cores unlike L2 which is core specific but its defined uncoditionally
 * to keep the listing simple.
 */
#ifdef DDR3_OUTPUT
#pragma DATA_SECTION(decisions, ".ddrData")
#else
#pragma DATA_SECTION(decisions, ".asmBM")
#endif
#pragma DATA_ALIGN(decisions, 8) /* Should be double-word aligned */
Uint32 decisions[4][4][1024]; //[coreNum][blocknum][]

/* The coreNum dimension is required only for DDR3 endpoints as DDR3 is shared
 * by all cores unlike L2 which is core specific but its defined uncoditionally
 * to keep the listing simple.
 */
#ifdef DDR3_OUTPUT
#pragma DATA_SECTION(outParams, ".ddrData")
#else
#pragma DATA_SECTION(outParams, ".asmBM")
#endif
#pragma DATA_ALIGN(outParams, 8)  /* Should be double-word aligned */
Uint32 outParams[4][4][2]; //[coreNum][][]

// External global variables
extern CSL_IntcHandle hIntcEdma;
extern CSL_IntcHandle hIntcVcp;
extern Uint32 *profilePtr;

/******************************************************************************
 * 
 * Function: doVCP2Test ()
 *
 *****************************************************************************/

Uint32 doVCP2Test()
{
    Uint32 i;
    int pStatus;
    volatile unsigned int *addr;
    uint32_t marIdx = 0;

    // Disable L1D cache by setting its size to 0
    CACHE_setL1DSize(CACHE_L1_0KCACHE);

    /* Any location In MSMC (It has to be in DDR3 or MSMC as its a shared
     * variable used across cores to signal power enable completion).
     */
    volatile Uint32* powerOnFlagPtr = (volatile Uint32*)0x0C000000;

    //Get current DSP core number
    coreNum = CSL_chipReadReg(CSL_CHIP_DNUM);

    //Enable VCP2 corresponding to the coreId (any VCP2 can be used on any core).
    if(coreNum == 0)
    {
        *(powerOnFlagPtr) = VCP2_POWER_ON_IN_PROGRESS;

        for(i =  CSL_VCP_0; i <=  CSL_VCP_3; i++)
        {
            printf("\n[Core%d][doVCP2Test]: Enabling VCP2-%d", coreNum, i);
            if(enableVcp2(i) != SUCCESS)
            {
                printf("\n[Core%d][doVCP2Test]: Error, Couldn't enable VCP2-%d", coreNum, i);
                return(FAILURE);
            }
        }

        *(powerOnFlagPtr) = VCP2_POWER_ON_COMPLETE;

    }
    else
    {
        while(*(powerOnFlagPtr) != VCP2_POWER_ON_COMPLETE)
            ;
        printf("\n[Core%d][doVCP2Test]: Power enable complete.", coreNum);
    }
    
    /* Clear out profiling data */
    memset((void*)frameDecodeTime, 0x00, sizeof(frameDecodeTime) * sizeof(Uint32));

    // Get handle to desired VCP2 instance
    hVcp2 = VCP2_init(&pVcp2Obj, coreNum, &pStatus);

    //Initialize EDMA3 (instance CSL_TPCC_2)
    edma3ModuleInit(coreNum);

    /* VCP_NUM_USERS */
    for (i = 0; i < NUM_USERS  ; i++)
    {
        /* Calculate the VCP IC values */      
        VcpCslCompat(&vcpParameters[i],&vcpParameters[i]);
        VCP2_genIc(&vcpParameters[i], &configIcArray[i]);
        VCP2_setBmEndian(hVcp2, 0);
        VCP2_setSdEndian(hVcp2, 0);

        /* --- Setup DMA Chanels for VCP Processing*/
        setupVcpTputEdma3( &configIcArray[i],
                vcpUserData[i],
                &vcpParameters[i],
                decisions[coreNum][i],
                outParams[coreNum][i],
                &i,
                coreNum);

    } /* end for i */


    /* Enable DMA Channels */
    edma3MultChnlEnable(coreNum);

    /* Enable DMA & VCP Interrupts */
    Setup_Interrupt(); 

    /*------------------------------------------------------------------*/
    /* Start the Timer                                                  */
    /*------------------------------------------------------------------*/
    CSL_tscEnable();

    /* Start VCP */
    profileIPXmtStartLow64[VCP_IPTX_TimeStamp_Start_Count++]=CSL_tscRead();
    VCP2_start(hVcp2);

    /* Check if the TX EDMA3 is comlpeted */
    //while((ChannelTXCount != NUM_USERS) || (ChannelRXCount!= NUM_USERS))
    while(1)
    {
        /*printf("\nCore:%d, ChannelTxCount:%d, ChannelRxCount:%d",
                DNUM, ChannelTXCount, ChannelRXCount); */
        if (VcpError != 0) 
        {
            VAL_TRACE(0xDEADEEE1);
            VAL_TRACE(VcpError);
            break;
        }
    }

#if 0
    /*------------------------------------------------------------------*/
    /* Measure the time taken from the start of Input parameter transfer*/
    /* to decode completion (output parameters RX)                      */
    /*------------------------------------------------------------------*/
    
    profilePtr = (Uint32 *)(&frameDecodeTime);
    for (i=0; i<NUM_USERS; i++)
    {
        profileIPXmtStartLow[i]  = profileIPXmtStartLow64[i];
        profileIPXmtStartHigh[i] = profileIPXmtStartLow64[i] >> 32; 
        profileOPRcvStopLow[i]   = profileOPRcvStopLow64[i];
        profileOPRcvStopHigh[i]  = profileOPRcvStopLow64[i] >> 32;
        CalcTimeStamp(profileIPXmtStartLow[i], profileIPXmtStartHigh[i], 
        profileOPRcvStopLow[i], profileOPRcvStopHigh[i]);
    }

    /* VCP_NUM_USERS */
    for (i = 0; i < NUM_USERS  ; i++)
    {
        /* Verify output decisions */
        VcpError |= vcpCheckResults(decisions[coreNum][i], referenceDec[i], NumofDecisions[i]);

        if (VcpError)
        {
            /* Close the Interrup and EDMA3 Handles */
            edma3CloseHandles();
            CSL_intcClose(hIntcEdma);
            CSL_intcClose(hIntcVcp);
            VAL_TRACE(0xDEADDEAD);
            return (FAILURE);
        }
     
        /* Verify output parameters */
        VcpError += vcpCheckResults(outParams[coreNum][i], RefoutParams[i], 2);
        if (VcpError)
        {
            /* Close the Interrup and EDMA3 Handles */
            edma3CloseHandles();
            CSL_intcClose(hIntcEdma);
            CSL_intcClose(hIntcVcp);
            VAL_TRACE(0xDEADDEAD);
            return (FAILURE);
        }


    } /* end for i */


    VAL_TRACE(0xABCDABCD);

    /* Close the Interrup and EDMA3 Handles */
    edma3CloseHandles();
    CSL_intcClose(hIntcEdma);
    CSL_intcClose(hIntcVcp);

    VAL_TRACE(0xABCDABCD);
    VAL_TRACE(0xABCDABCD);
#endif
    return (SUCCESS);

} /* End of DoVcp2Test() */

/******************************************************************************
 * 
 * Function: enableVcp2 (Uint32 vcp2Id)
 *
 *****************************************************************************/
Uint32 enableVcp2 (Int32 vcp2Id)
{

    Uint32 pdNum;
    Uint32 lpscNum;

    if((vcp2Id <  CSL_VCP_0) && (vcp2Id >  CSL_VCP_3))
    {
        printf("\n[Core%d][enableVcp2] : Wrong VCP2 Id specified.", coreNum);
        return FAILURE;
    }

    pdNum   = CSL_PSC_PD_VCP_0123;
    lpscNum = (vcp2Id == CSL_VCP_0) ? CSL_PSC_LPSC_VCP_0  : (CSL_PSC_LPSC_VCP_1 + (vcp2Id-1));

    /* Enable VCP2-B/C/D power domain */
    CSL_PSC_enablePowerDomain (pdNum);

    /* Enable the clocks */
    CSL_PSC_setModuleNextState (lpscNum, PSC_MODSTATE_ENABLE);

    /* Start the state transition */
    CSL_PSC_startStateTransition (pdNum);

    /* Wait until the state transition process is completed. */
    while (!CSL_PSC_isStateTransitionDone (pdNum));

    if ((CSL_PSC_getPowerDomainState (pdNum) == PSC_PDSTATE_ON) &&
            (CSL_PSC_getModuleState (lpscNum) == PSC_MODSTATE_ENABLE))
    {
        return SUCCESS;
    }
    else
    {
        return FAILURE;
    }

}


/*****************************************************************************\
 * End of vcp2_test.c
\*****************************************************************************/

