/*
 *  Copyright (c) Texas Instruments Incorporated 2012-2017
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  Redistributions of source code must retain the above copyright
 *  notice, this list of conditions and the following disclaimer.
 *
 *  Redistributions in binary form must reproduce the above copyright
 *  notice, this list of conditions and the following disclaimer in the
 *  documentation and/or other materials provided with the
 *  distribution.
 *
 *  Neither the name of Texas Instruments Incorporated nor the names of
 *  its contributors may be used to endorse or promote products derived
 *  from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/**
 *  \file Loopback_main.c
 *
 *  \brief Capture->Display loopback sample application main file.
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <Loopback_priv.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

static void LpbkApp_createTsk(void);
static void LpbkApp_tskReset(UArg arg0, UArg arg1);
static void LpbkApp_tskMain(UArg arg0, UArg arg1);

static void LpbkApp_init(LpbkApp_Obj *appObj);
static void LpbkApp_deInit(LpbkApp_Obj *appObj);


/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* Align stack memory to integer boundary. */
#if defined (__GNUC__) && !defined (__ti__)
DATA_ALIGN(32) DATA_SECTION(".bss:taskStackSection")
#else
#pragma DATA_ALIGN(gLpbkAppTskStackMain, 32)
/* Place the stack in stack section. */
#pragma DATA_SECTION(gLpbkAppTskStackMain, ".bss:taskStackSection")
#endif
/* Test application stack */
static UInt8 gLpbkAppTskStackMain[LPBK_APP_TSK_STACK_MAIN];

#if defined (__GNUC__) && !defined (__ti__)
DATA_ALIGN(32) DATA_SECTION(".bss:taskStackSection")
#else
/* Align stack memory to integer boundary. */
#pragma DATA_ALIGN(gLpbkAppTskStackReset, 32)
/* Place the stack in stack section. */
#pragma DATA_SECTION(gLpbkAppTskStackReset, ".bss:taskStackSection")
#endif
/* Test application stack */
static UInt8 gLpbkAppTskStackReset[LPBK_APP_TSK_STACK_MAIN];

/* Test application object */
static LpbkApp_Obj        gLpbkAppObj;

/* Task struct Object for static allocation */
static Task_Struct        gResetTskStruct;
static Task_Struct        gTskStruct;

static volatile UInt32    gExitApp;

static LpbkApp_TestParams gLpbkAppTestPrms[] =
{
    {
        "OV2659            -> VIP2_S1_PORTB (1280x720@30FPS)  -> YUYV422      -> DSS Video1 -> DPI1 LCD",
        1U,                                         /* numCaptHandles */
        1U,                                         /* numDispHandles */
        /* captInstId[] */
        {
            VPS_CAPT_VIP_MAKE_INST_ID(VPS_VIP2, VPS_VIP_S1, VPS_VIP_PORTB),
        },
        /* dispInstId[] */
        {
            VPS_DISP_INST_DSS_VID1
        },

        /* decDrvId[] */
        {
            FVID2_VID_SENSOR_OV2659_DRV,
        },
        /* encDrvId[] */
        {
            FVID2_LCD_CTRL_DRV,
        },
        /* vencId */
        {
            VPS_DCTRL_DSS_VENC_LCD1
        },
        FALSE,                                      /* isSdVenc */
        /* captStandard */
        FVID2_STD_720P_60,
        /* dispStandard */
        {
            FVID2_STD_CUSTOM,
        },

        FVID2_VIFM_SCH_DS_HSYNC_VSYNC,           /* captVideoIfMode */
        FVID2_VIFW_8BIT,                         /* captVideoIfWidth */
        FVID2_SF_PROGRESSIVE,                    /* captScanFormat */
        FVID2_DF_YUV422P,                        /* captInDataFmt */
        FVID2_DF_YUV422I_UYVY,                   /* captOutDataFmt */

        LPBK_APP_SC_DISABLE_ALL,                 /* captScEnable */
        1280U,                                   /* captInWidth */
        720U,                                    /* captInHeight */
        1280U,                                   /* captOutWidth */
        720U,                                    /* captOutHeight */

        1280U,                                   /* dispInWidth */
        720U,                                    /* dispInHeight */
        LPBK_APP_USE_LCD_WIDTH,                  /* Target width */
        LPBK_APP_USE_LCD_HEIGHT,                 /* Target Height */

        LPBK_APP_RUN_COUNT,                      /* runCount */
        BSP_BOARD_MODE_VIDEO_8BIT,               /* captBoardMode */
        BSP_BOARD_MODE_DEFAULT,                  /* dispBoardMode */
        /* boardId */
        (BSP_BOARD_UNKNOWN | BSP_BOARD_IDK_AM57XX)
    },

};

#define LPBK_APP_NUM_TESTS              (sizeof (gLpbkAppTestPrms) / \
                                         sizeof (gLpbkAppTestPrms[0U]))

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */
#if defined (SOC_AM574x) || defined (SOC_AM572x) || defined (SOC_AM571x)
#define PDK_RAW_BOOT
static Board_STATUS CapApp_boardInit(void)
{
    Board_STATUS ret;
    Board_initCfg boardCfg;

#ifdef PDK_RAW_BOOT
    boardCfg = BOARD_INIT_PINMUX_CONFIG | BOARD_INIT_MODULE_CLOCK;
#endif

    boardCfg |= BOARD_INIT_UART_STDIO;
    ret = Board_init(boardCfg);
    return (ret);
}
#endif
/*
 * Application main
 */
int main(void)
{
    static Char stdin_buf[BUFSIZ];
#if defined (SOC_AM574x) || defined (SOC_AM572x) || defined (SOC_AM571x)
    UInt32                  tempFuncPtr;
    BspOsal_InitParams_t    bspOsalInitPrms = {0};

    CapApp_boardInit();
    /* Initialize the UART prints */
    BspUtils_uartInit();
    tempFuncPtr = (UInt32) & BspUtils_uartPrint;
    bspOsalInitPrms.printFxn = (BspOsal_PrintFxn) (tempFuncPtr);
    BspOsal_Init(&bspOsalInitPrms);
#endif
    /* Provide buffer so that STDIO library will use this memory instead of
     * allocating from system heap, which might lead to false system leak
     * warning */
    setvbuf(stdin, stdin_buf, _IOLBF, BUFSIZ);

    /* Create test task */
    LpbkApp_createTsk();

    /* Start BIOS */
    BIOS_start();

    return (0);
}

/*
 * Create test task
 */
static void LpbkApp_createTsk(void)
{
    Task_Params tskPrms;

    /* Create task to wait for overflow to occur.
     * When it occurs, this task will just reset VIP */
    gExitApp = FALSE;
    Task_Params_init(&tskPrms);
    tskPrms.priority  = LPBK_APP_TSK_PRI_MAIN;
    tskPrms.stack     = gLpbkAppTskStackReset;
    tskPrms.stackSize = sizeof (gLpbkAppTskStackReset);
    tskPrms.arg0      = (UArg) & gLpbkAppObj;
    Task_construct(&gResetTskStruct, LpbkApp_tskReset, &tskPrms, NULL);
    gLpbkAppObj.resetTskHandle = Task_handle(&gResetTskStruct);
    GT_assert(BspAppTrace, (gLpbkAppObj.resetTskHandle != NULL));

    /* Create test task */
    Task_Params_init(&tskPrms);
    tskPrms.priority  = LPBK_APP_TSK_PRI_MAIN;
    tskPrms.stack     = gLpbkAppTskStackMain;
    tskPrms.stackSize = sizeof (gLpbkAppTskStackMain);
    tskPrms.arg0      = (UArg) & gLpbkAppObj;
    Task_construct(&gTskStruct, LpbkApp_tskMain, &tskPrms, NULL);
    gLpbkAppObj.tskHandle = Task_handle(&gTskStruct);
    GT_assert(BspAppTrace, (gLpbkAppObj.tskHandle != NULL));

    /* Register the task to the load module for calculating the load */
    BspUtils_prfLoadRegister(gLpbkAppObj.tskHandle, APP_NAME);

    return;
}

/*
 *  Reset task
 * When overflow occurs, this task will run.
 * It will stop the driver, reset it and again start it.
 */

static void LpbkApp_tskReset(UArg arg0, UArg arg1)
{
    LpbkApp_Obj     *appObj = (LpbkApp_Obj *) arg0;
    LpbkApp_InstObj *instObj;
    UInt32           captInstCnt = 0;
    Int32 retVal;
    overflowSem = BspOsal_semCreate(0, FALSE);

    GT_assert(BspAppTrace, (NULL != appObj));
    GT_assert(BspAppTrace, (NULL != overflowSem));

    while (1U)
    {
        BspOsal_semWait(overflowSem, BSP_OSAL_WAIT_FOREVER);
        if ((UInt32) TRUE == gExitApp)
        {
            BspOsal_semDelete(&overflowSem);
            break;
        }
        for (captInstCnt = 0; captInstCnt < appObj->testPrms.numCaptHandles;
             captInstCnt++)
        {
            Vps_CaptOverflowCheckParams overflowCheckParams;
            instObj = &appObj->instObj[captInstCnt];
            instObj->overflowCount = 0;
            retVal = Fvid2_control(
                instObj->captDrvHandle,
                IOCTL_VPS_CAPT_CHECK_OVERFLOW,
                &overflowCheckParams,
                NULL);
            if (TRUE == overflowCheckParams.isOverflowOccured)
            {
                /*Stop the driver*/
                retVal = Fvid2_stop(instObj->captDrvHandle, NULL);
                if (retVal != FVID2_SOK)
                {
                    GT_0trace(BspAppTrace, GT_ERR,
                              APP_NAME ": Capture Stop Failed!!!\r\n");
                }

                if (FVID2_SOK == retVal)
                {
                    /* Reset the VIP */
                    retVal = Fvid2_control(
                        instObj->captDrvHandle,
                        IOCTL_VPS_CAPT_RESET_VIP,
                        NULL,
                        NULL);

                    if (retVal != FVID2_SOK)
                    {
                        GT_0trace(BspAppTrace, GT_ERR,
                                  APP_NAME ": VIP Port Reset Failed!!!\r\n");
                    }
                }

                if (FVID2_SOK == retVal)
                {
                    /* Restart the driver */
                    retVal = Fvid2_start(instObj->captDrvHandle, NULL);
                    if (retVal != FVID2_SOK)
                    {
                        GT_0trace(BspAppTrace, GT_ERR,
                                  APP_NAME ": Capture Start Failed!!!\r\n");
                    }
                }
                if (FVID2_SOK != retVal)
                {
                    break;
                }
            }
        }
    }

    return;
}

/*
 * Test task main
 */
static void LpbkApp_tskMain(UArg arg0, UArg arg1)
{
    Int32       testId;
    Bsp_BoardId boardId;
    LpbkApp_Obj           *appObj = (LpbkApp_Obj *) arg0;
    LpbkApp_TestParams    *testPrms;
    BspUtils_MemHeapStatus startHeapStat, startHeapStat1;

    GT_0trace(BspAppTrace, GT_INFO,
              APP_NAME ": Sample Application - STARTS !!!\r\n");
    BspUtils_memGetHeapStat(&startHeapStat);
    LpbkApp_init(appObj);

    appObj->enableAutoRun = (UInt32) FALSE;
    boardId = Bsp_boardGetId();
    while (1U)
    {
        /* Get the TestId */
        testId = 0;
        if ((testId >= 0) && (testId < LPBK_APP_NUM_TESTS))
        {
            testPrms = &gLpbkAppTestPrms[testId];
            if (testPrms->boardId & boardId)
            {
                BspUtils_memGetHeapStat(&startHeapStat1);

                testPrms->testId = testId;
                LpbkApp_runTest(appObj, testPrms);

                BspUtils_memCheckHeapStat(&startHeapStat1);
            }
            else
            {
                GT_1trace(BspAppTrace, GT_INFO,
                          APP_NAME ": Skipping test case %d!!!\r\n", testId);
            }
        }
        else
        {
            /* Exit */
            break;
        }
    }

    LpbkApp_deInit(appObj);

    gExitApp = (UInt32) TRUE;
    BspOsal_semPost(overflowSem);
    BspOsal_sleep(500);         /* Wait for reset task to exit */

    BspUtils_memCheckHeapStat(&startHeapStat);
    GT_1trace(BspAppTrace, GT_INFO,
              APP_NAME ": Max stack used for test task: %d bytes\r\n",
              BspOsal_getTaskStackUsage(NULL));
    GT_1trace(BspAppTrace, GT_INFO,
              APP_NAME ": Max system stack used (ISR): %d bytes\r\n",
              BspOsal_getSystemStackUsage());
    BspUtils_appPrintMemStatus();

    GT_0trace(BspAppTrace, GT_INFO,
              APP_NAME ": Sample Application - DONE !!!\r\n");

    return;
}

static void LpbkApp_init(LpbkApp_Obj *appObj)
{
    Int32  retVal;
    UInt32 isI2cInitReq;
    UInt32 defPixelClk;

    /* System init */
    isI2cInitReq = TRUE;
    retVal       = BspUtils_appDefaultInit(isI2cInitReq);
    if (retVal != FVID2_SOK)
    {
        GT_0trace(BspAppTrace, GT_ERR,
                  APP_NAME ": System Init Failed!!!\r\n");
    }
    if (FVID2_SOK == retVal)
    {
        /* Create global capture handle, used for common driver configuration */
        appObj->fvidHandleAll = Fvid2_create(
            FVID2_VPS_CAPT_VID_DRV,
            VPS_CAPT_INST_ALL,
            NULL,                       /* NULL for VPS_LPBK_INST_ALL */
            NULL,                       /* NULL for VPS_LPBK_INST_ALL */
            NULL);                      /* NULL for VPS_LPBK_INST_ALL */
        if (NULL == appObj->fvidHandleAll)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Global Handle Create Failed!!!\r\n");
            retVal = FVID2_EBADARGS;
        }
    }
    if (FVID2_SOK == retVal)
    {
        /* Create DCTRL handle, used for common driver configuration */
        appObj->dctrlHandle = Fvid2_create(
            FVID2_VPS_DCTRL_DRV,
            VPS_DCTRL_INST_0,
            NULL,                       /* NULL for VPS_DCTRL_INST_0 */
            NULL,                       /* NULL for VPS_DCTRL_INST_0 */
            NULL);                      /* NULL for VPS_DCTRL_INST_0 */
        if (NULL == appObj->dctrlHandle)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": DCTRL Create Failed!!!\r\n");
            retVal = FVID2_EBADARGS;
        }
    }

    if ((BSP_PLATFORM_ID_EVM == Bsp_platformGetId()) && (FVID2_SOK == retVal))
    {
        /* Set default video PLL clock - This will be changed later based
         * on detected LCD */
//        defPixelClk = 29232U * 4U;
//
//        /* No Internal Dividers present in DSS in Tda3xx platform,
//         * so default it to 29232 itself. */
//        if (TRUE == Bsp_platformIsTda3xxFamilyBuild())
//        {
//            defPixelClk = 29232U;
//        }
//
//        retVal = LpbkApp_configureVideoPllAndClkSrc(appObj, defPixelClk);
//        if (retVal != FVID2_SOK)
//        {
//            GT_0trace(BspAppTrace, GT_ERR,
//                      APP_NAME ": Configuring PLL Failed!!!\r\n");
//        }
//
//        if (FVID2_SOK == retVal)
//        {
//            /* Needed onlt for TDA2xx platform. Other platform will do nothing
//             * and return OK */
//            retVal = Bsp_platformEnableHdmiPll(TRUE);
//            if (retVal != FVID2_SOK)
//            {
//                GT_0trace(BspAppTrace, GT_ERR,
//                          APP_NAME ": Enabling HDMI PLL Failed!!!\r\n");
//            }
//        }
//
//        if (FVID2_SOK == retVal)
//        {
//            if (BSP_BOARD_MULTIDES == Bsp_boardGetId())
//            {
//                retVal =
//                    BspUtils_appInitSerDeSer();
//                if (retVal != FVID2_SOK)
//                {
//                    GT_0trace(BspAppTrace, GT_ERR,
//                              APP_NAME ": MultiDes Board Init failed!!!\r\n");
//                }
//            }
//        }
    }

    if (FVID2_SOK == retVal)
    {
        GT_0trace(BspAppTrace, GT_INFO,
                  APP_NAME ": LpbkApp_init() - DONE !!!\r\n");
    }

    return;
}

static void LpbkApp_deInit(LpbkApp_Obj *appObj)
{
    Int32  retVal;
    UInt32 isI2cDeInitReq;

    /* Delete DCTRL handle */
    retVal = Fvid2_delete(appObj->dctrlHandle, NULL);
    if (retVal != FVID2_SOK)
    {
        GT_0trace(BspAppTrace, GT_ERR,
                  APP_NAME ": DCTRL handle delete failed!!!\r\n");
    }

    if (FVID2_SOK == retVal)
    {
        /* Delete global VIP capture handle */
        retVal = Fvid2_delete(appObj->fvidHandleAll, NULL);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Global handle delete failed!!!\r\n");
        }
    }

    if (FVID2_SOK == retVal)
    {
        if (BSP_BOARD_MULTIDES == Bsp_boardGetId())
        {
            retVal =
                BspUtils_appDeInitSerDeSer();
            if (retVal != FVID2_SOK)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": MultiDes Board DeInit failed!!!\r\n");
            }
        }
    }

    if (FVID2_SOK == retVal)
    {
        /* System de-init */
        isI2cDeInitReq = TRUE;
        retVal         = BspUtils_appDefaultDeInit(isI2cDeInitReq);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": System De-Init Failed!!!\r\n");
        }
    }

    if (FVID2_SOK == retVal)
    {
        /* Needed onlt for TDA2xx platform. Other platform will do nothing
         * and return OK */
        retVal = Bsp_platformEnableHdmiPll(FALSE);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Disabling HDMI PLL Failed!!!\r\n");
        }
    }

    if (FVID2_SOK == retVal)
    {
        GT_0trace(BspAppTrace, GT_INFO,
                  APP_NAME ": LpbkApp_deInit() - DONE !!!\r\n");
    }

    return;
}


