

/**
 *  \file Loopback_test.c
 *
 *  \brief Capture->Display loopback sample application test file.
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include <Loopback_priv.h>

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structure Declarations                             */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Function Declarations                             */
/* ========================================================================== */

static void LpbkApp_captCreate(LpbkApp_Obj *appObj);
static void LpbkApp_captDelete(LpbkApp_Obj *appObj);

static Int32 LpbkApp_captAllocQueueFrm(const LpbkApp_Obj *appObj,
                                       LpbkApp_InstObj   *instObj,
                                       UInt32             fieldMerged);
static Int32 LpbkApp_captFreeFrm(LpbkApp_Obj     *appObj,
                                 LpbkApp_InstObj *instObj,
                                 UInt32           fieldMerged);


static Int32 LpbkApp_createVidSensor(const LpbkApp_Obj *appObj,
                                     LpbkApp_InstObj   *instObj);



static void LpbkApp_initParams(LpbkApp_Obj *appObj);
static void LpbkApp_printStatistics(LpbkApp_Obj *appObj,
                                    UInt32       execTimeInMsecs);

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

#define MAX_CONTI_OF_INTERRUPTS     (2U)

/* Smaphore to sync between the reset and the callback function */
BspOsal_SemHandle overflowSem;

/* ========================================================================== */
/*                          Function Definitions                              */
/* ========================================================================== */

/**
 *  LpbkApp_runTest
 *  Lpbkure test function.
 */
void LpbkApp_runTest(LpbkApp_Obj *appObj, const LpbkApp_TestParams *testPrms)
{
    Int32            retVal = FVID2_SOK;
    UInt32           instCnt;
    UInt32           runTime, curTime, prevTime;
    UInt32           loopCount, prevFldCount, curFldCount, fldCount;
    LpbkApp_InstObj *instObj = NULL;

    loopCount = 0;
    curTime   = prevTime = curFldCount = prevFldCount = 0;
    BspUtils_memcpy(&appObj->testPrms, testPrms, sizeof (appObj->testPrms));

    /* Print test case information */
    GT_5trace(BspAppTrace, GT_INFO,
              APP_NAME ": CAPT HANDLES %d: DISP HANDLES %d: MODE %04x : "
              "RUN COUNT %d: OUTPUT:%d !!\r\n",
              appObj->testPrms.numCaptHandles,
              appObj->testPrms.numDispHandles,
              appObj->testPrms.captVideoIfMode,
              appObj->testPrms.runCount,
              appObj->testPrms.captOutDataFmt);

    /* Create driver */
    LpbkApp_initParams(appObj);
    LpbkApp_captCreate(appObj);


    /* Start decoder*/
    for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];
        if (NULL != instObj->decHandle)
        {
            Fvid2_start(instObj->decHandle, NULL);
        }
    }

    GT_0trace(BspAppTrace, GT_INFO,
              APP_NAME ": Starting loopback ... !!!\r\n");
    GT_0trace(BspAppTrace, GT_INFO,
              APP_NAME ": Loopback in progress ... DO NOT HALT !!!\r\n");

    appObj->startTime = BspOsal_getCurTimeInMsec();

    /* Start the load calculation */
    BspUtils_prfLoadCalcStart();


    /* Start capture driver */
    for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];

        retVal = Fvid2_start(instObj->captDrvHandle, NULL);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Capture Start Failed!!!\r\n");
            break;
        }

    }
    if (FVID2_SOK == retVal)
    {
        /* Capture frames, check status */
        while (loopCount < appObj->testPrms.runCount)
        {
            if (BSP_PLATFORM_ID_EVM == Bsp_platformGetId())
            {
                /* check status every sec */
                BspOsal_sleep(1000);
            }
            else
            {
                BspOsal_sleep(1);
            }

            /* Change current capture instance every few sec */
            if ((loopCount != 0) &&
                ((loopCount % LPBK_APP_CH_SWITCH_RATE) == 0))
            {
                if (appObj->curCaptInstance <
                    (appObj->testPrms.numCaptHandles - 1))
                {
                    appObj->curCaptInstance++;
                }
                else
                {
                    appObj->curCaptInstance = 0U;
                }
            }

            /* Get CPU load */
            appObj->totalCpuLoad += Load_getCPULoad();
            appObj->cpuLoadCount++;

            /* Do runtime print every 5 sec */
            if ((loopCount != 0) && ((loopCount % 5) == 0))
            {
                curTime     = BspOsal_getCurTimeInMsec() - appObj->startTime;
                curFldCount = appObj->totalCaptFldCount;

                runTime  = curTime - prevTime;
                fldCount = curFldCount - prevFldCount;

                prevTime     = curTime;
                prevFldCount = curFldCount;

                Bsp_rprintf(APP_NAME ": %5d.%3ds: Fields = %5d (%3d fps)\r\n",
                            (curTime / 1000), (curTime % 1000),
                            fldCount, ((fldCount * 1000) / runTime));
            }

            if (BSP_PLATFORM_ID_EVM == Bsp_platformGetId())
            {
                loopCount++;
            }
            else
            {
                if (loopCount != appObj->callBackCount)
                {
                    GT_1trace(BspAppTrace, GT_INFO,
                              APP_NAME ": Captured %d frames!!\r\n",
                              appObj->callBackCount);
                }
                loopCount = appObj->callBackCount;
            }
        }

        /* Stop capture driver */
        for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
        {
            instObj = &appObj->instObj[instCnt];
            retVal  = Fvid2_stop(instObj->captDrvHandle, NULL);
            if (retVal != FVID2_SOK)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Capture Stop Failed!!!\r\n");
                break;
            }
        }
    }

    if (FVID2_SOK == retVal)
    {
        /* Stop the load calculation */
        BspUtils_prfLoadCalcStop();
        runTime = (BspOsal_getCurTimeInMsec() - appObj->startTime);

        /* Stop decoder */
        for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
        {
            instObj = &appObj->instObj[instCnt];
            if (instObj->decHandle)
            {
                Fvid2_stop(instObj->decHandle, NULL);
            }
        }

        GT_0trace(BspAppTrace, GT_INFO,
                  APP_NAME ": Stopping loopback ... DONE !!!\r\n");

        LpbkApp_printStatistics(appObj, runTime);

        /* Delete driver */

        LpbkApp_captDelete(appObj);

        /* Print the load */
        BspUtils_prfLoadPrintAll(TRUE);
        /* Reset the load */
        BspUtils_prfLoadCalcReset();
    }

    return;
}

/*
 *  Callback function, recieved when overflow occues in VIP
 */
Int32 LpbkApp_overflowCbFxn(LpbkApp_InstObj *instObj)
{
    Int32 retVal = FVID2_SOK;
    /* Most of the time VIP recovers automatically. This is to give
     * some time to VIP to recover automatically
     */
    GT_assert(BspAppTrace, (instObj != NULL));

    instObj->overflowCount++;
    if (instObj->overflowCount > MAX_CONTI_OF_INTERRUPTS)
    {
        retVal = Fvid2_control(
            instObj->captDrvHandle,
            IOCTL_VPS_CAPT_RESET_VIP_PORT,
            NULL,
            NULL);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": VIP Reset Failed!!!\r\n");
        }
    }

    BspOsal_semPost(overflowSem);

    return retVal;
}

static void LpbkApp_captCreate(LpbkApp_Obj *appObj)
{
    Int32  retVal = FVID2_SOK;
    UInt32 instCnt;
    UInt32 streamId, chId;
    LpbkApp_InstObj       *instObj;
    Vps_VpdmaMaxSizeParams vipMaxSizePrms;
    Vps_CaptFrameSkip      frmSkipPrms;

    for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];
        instObj->captDrvHandle = Fvid2_create(
            FVID2_VPS_CAPT_VID_DRV,
            instObj->captInstId,
            &instObj->captCreatePrms,
            &instObj->captCreateStatus,
            &instObj->captCbPrms);
        if ((NULL == instObj->captDrvHandle) ||
            (instObj->captCreateStatus.retVal != FVID2_SOK))
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Capture Create Failed!!!\r\n");
            retVal = instObj->captCreateStatus.retVal;
        }
        if (FVID2_SOK == retVal)
        {
            retVal = Fvid2_control(
                instObj->captDrvHandle,
                IOCTL_VPS_CAPT_SET_VIP_PARAMS,
                &instObj->vipPrms,
                NULL);
            if (retVal != FVID2_SOK)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": VIP Set Params IOCTL Failed!!!\r\n");
            }
        }

        if (FVID2_SOK == retVal)
        {
            /* Set frame skip for each channel */
            VpsCaptFrameSkip_init(&frmSkipPrms);
            for (streamId = 0U; streamId < instObj->captCreatePrms.numStream;
                 streamId++)
            {
                for (chId = 0U; chId < instObj->captCreatePrms.numCh; chId++)
                {
                    /* This IOCTL is supported only for frame drop mode */
                    if (instObj->captCreatePrms.bufCaptMode ==
                        VPS_CAPT_BCM_FRM_DROP)
                    {
                        frmSkipPrms.chNum =
                            Vps_captMakeChNum(instObj->captInstId, streamId,
                                              chId);
                        frmSkipPrms.frmSkipMask = LPBK_APP_FRAME_SKIP_PATTERN;
                        retVal = Fvid2_control(
                            instObj->captDrvHandle,
                            IOCTL_VPS_CAPT_SET_FRAME_SKIP,
                            &frmSkipPrms,
                            NULL);
                        if (retVal != FVID2_SOK)
                        {
                            GT_0trace(
                                BspAppTrace, GT_ERR,
                                APP_NAME
                                ": VIP Set frame skip IOCTL Failed!!\r\n");
                            break;
                        }
                    }
                }
            }
        }

        if (!Bsp_platformIsTI814xFamilyBuild() && (FVID2_SOK == retVal))
        {
            VpsVpdmaMaxSizeParams_init(&vipMaxSizePrms);
            vipMaxSizePrms.instId =
                Vps_captGetVipId(appObj->testPrms.captInstId[instCnt]);
            vipMaxSizePrms.maxOutWidth[0U]  = instObj->maxOutWidth[0U];
            vipMaxSizePrms.maxOutHeight[0U] = instObj->maxOutHeight[0U];
            vipMaxSizePrms.maxOutWidth[1U]  = instObj->maxOutWidth[1U];
            vipMaxSizePrms.maxOutHeight[1U] = instObj->maxOutHeight[1U];
            vipMaxSizePrms.maxOutWidth[2U]  = instObj->maxOutWidth[2U];
            vipMaxSizePrms.maxOutHeight[2U] = instObj->maxOutHeight[2U];

            retVal = Fvid2_control(
                appObj->fvidHandleAll,
                IOCTL_VPS_CAPT_SET_VIP_MAX_SIZE,
                &vipMaxSizePrms,
                NULL);
            if (retVal != FVID2_SOK)
            {
                GT_0trace(
                    BspAppTrace, GT_ERR,
                    APP_NAME
                    ": VIP Set Max Frame Size Params IOCTL Failed!!!\r\n");
            }
        }
        if (FVID2_SOK == retVal)
        {
            retVal = LpbkApp_captAllocQueueFrm(appObj, instObj, FALSE);
            if (retVal != FVID2_SOK)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Capture Alloc and Queue Failed!!!\r\n");
            }
        }

        if (FVID2_SOK == retVal)
        {
            instObj->decHandle = NULL;
            if (BSP_PLATFORM_ID_EVM == Bsp_platformGetId())
            {
                if ((FVID2_VID_SENSOR_OV2659_DRV == instObj->decDrvId))
                {
   //                 LpbkApp_createVidSensor(appObj, instObj);
                }
            }
        }

        if (FVID2_SOK == retVal)
        {
            retVal = Bsp_boardSetPinMux(FVID2_VPS_CAPT_VID_DRV,
                                        instObj->captInstId,
                                        appObj->testPrms.captBoardMode);
            if (FVID2_SOK != retVal)
            {
                GT_0trace(BspAppTrace, GT_ERR, "Pin Muxing Failed \r\n");
            }
        }

        if (retVal == FVID2_SOK)
        {
            Vps_CaptOverflowParams overflowParams;
            overflowParams.overflowCbFxn =
                (Vps_CaptVipOverflowFxn) & LpbkApp_overflowCbFxn;
            overflowParams.retParams = (void *) instObj;
            retVal = Fvid2_control(
                instObj->captDrvHandle,
                IOCTL_VPS_CAPT_REGISTER_OVF_INTR,
                &overflowParams,
                NULL);
            if (retVal != FVID2_SOK)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": VIP Set Params IOCTL Failed!!!\r\n");
            }
        }
    }

    if (FVID2_SOK == retVal)
    {
        GT_0trace(BspAppTrace, GT_INFO,
                  APP_NAME ": Capture create complete!!\r\n");
    }

    return;
}

static void LpbkApp_captDelete(LpbkApp_Obj *appObj)
{
    Int32            retVal = FVID2_SOK;
    UInt32           instCnt, streamId, boardId, sensorInstId;
    LpbkApp_InstObj *instObj;
    Fvid2_FrameList  frmList;

    for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];

        for (streamId = 0U;
             streamId < instObj->captCreatePrms.numStream;
             streamId++)
        {
            /* Dequeue all the request from the driver */
            while (1U)
            {
                retVal = Fvid2_dequeue(
                    instObj->captDrvHandle,
                    &frmList,
                    streamId,
                    FVID2_TIMEOUT_NONE);
                if (FVID2_SOK != retVal)
                {
                    break;
                }
            }
        }

        /* Unregister from the overflow interrupt*/
        retVal = Fvid2_control(
            instObj->captDrvHandle,
            IOCTL_VPS_CAPT_UNREGISTER_OVF_INTR,
            NULL,
            NULL);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(
                BspAppTrace, GT_ERR,
                APP_NAME
                ": Unregister From Overflow Interrupt Failed!!!\r\n");
        }

        retVal = Fvid2_delete(instObj->captDrvHandle, NULL);
        if (FVID2_SOK != retVal)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Capture Delete Failed!!!\r\n");
        }

        /* free frame memory */
        retVal = LpbkApp_captFreeFrm(appObj, instObj, FALSE);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Capture Free Frames Failed!!!\r\n");
        }

        if (NULL != instObj->decHandle)
        {
            retVal = Fvid2_delete(instObj->decHandle, NULL);
            if (FVID2_SOK != retVal)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Decoder Delete Failed!!!\r\n");
            }
            instObj->decHandle = NULL;
        }

        /* DeConfiguring a particular instance of SerDeSer */
        boardId = Bsp_boardGetId();

        if ((BSP_BOARD_MULTIDES == boardId) &&
            ((FVID2_VID_SENSOR_MULDES_OV1063X_DRV == instObj->decDrvId) ||
             (FVID2_VID_SENSOR_GW4200_IMX290_DRV == instObj->decDrvId)))
        {
            sensorInstId = Bsp_boardGetVideoDeviceInstId(
                instObj->decDrvId,
                FVID2_VPS_CAPT_VID_DRV,
                instObj->captInstId);

            retVal = BspUtils_appDeConfSerDeSer(instObj->decDrvId,
                                                sensorInstId);
            if (retVal != FVID2_SOK)
            {
                GT_1trace(
                    BspAppTrace, GT_ERR,
                    APP_NAME
                    ": DeConfiguring MultiDes instance \
                      %d failed!!!\r\n",
                    sensorInstId);
            }
        }
    }

    GT_0trace(BspAppTrace, GT_INFO, APP_NAME ": Capture delete complete!!\r\n");

    return;
}


/**
 *  LpbkApp_captCbFxn
 *  \brief Driver callback function.
 */
static Int32 LpbkApp_captCbFxn(Fvid2_Handle handle, Ptr appData, Ptr reserved)
{
    Int32              retVal = FVID2_SOK;
    UInt32             streamId;
    LpbkApp_InstObj   *instObj = (LpbkApp_InstObj *) appData;
    Fvid2_FrameList    frmList;

    for (streamId = 0U;
         streamId < instObj->captCreatePrms.numStream;
         streamId++)
    {
        retVal = Fvid2_dequeue(
            instObj->captDrvHandle,
            &frmList,
            streamId,
            FVID2_TIMEOUT_NONE);
        if (FVID2_SOK == retVal)
        {

            /* Queue back to same capture instance */
            retVal = Fvid2_queue(
                instObj->captDrvHandle,
                &frmList,
                streamId);
            if (FVID2_SOK != retVal)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Capture Queue Failed!!!\r\n");
            }
        }
    }

    instObj->appObj->callBackCount++;

    return (retVal);
}


/**
 *  \brief Allocate and queue frames to driver
 */
static Int32 LpbkApp_captAllocQueueFrm(const LpbkApp_Obj *appObj,
                                       LpbkApp_InstObj   *instObj,
                                       UInt32             fieldMerged)
{
    Int32               retVal = FVID2_SOK;
    UInt32              streamId, chId, frmId, idx;
    UInt32              bufSize;
    UInt32              yFld1Offset, cbCrFld0Offset, cbCrFld1Offset;
    Fvid2_Format       *fmt;
    Fvid2_Frame        *frm;
    Fvid2_FrameList     frmList;
    Vps_CaptVipOutInfo *outInfo;
    Char fileStr[30U];

    /* for every stream and channel in a capture handle */
    Fvid2FrameList_init(&frmList);
    for (streamId = 0U; streamId < instObj->captCreatePrms.numStream;
         streamId++)
    {
        for (chId = 0U; chId < instObj->captCreatePrms.numCh; chId++)
        {
            outInfo = &instObj->vipPrms.outStreamInfo[streamId];
            fmt     = &instObj->captAllocFmt;
            Fvid2Format_init(fmt);

            /* base index for instObj->captFrames[] */
            idx =
                (VPS_CAPT_CH_PER_PORT_MAX * LPBK_APP_FRAMES_PER_CH
                 * streamId) + (LPBK_APP_FRAMES_PER_CH * chId);

            if (idx >= LPBK_APP_CAPT_MAX_FRM)
            {
                idx = 0U;
            }
            frm = &instObj->captFrames[idx];

            /* fill format with channel specific values  */
            fmt->chNum =
                Vps_captMakeChNum(instObj->captInstId, streamId, chId);
            fmt->width  = appObj->maxWidth;
            fmt->height = appObj->maxHeight + LPBK_APP_PADDING_LINES;
            if (fieldMerged)
            {
                fmt->height *= 2;
            }
            fmt->pitch[0U] = outInfo->outFmt.pitch[0U];
            fmt->pitch[1U] = outInfo->outFmt.pitch[1U];
            fmt->pitch[2U] = outInfo->outFmt.pitch[2U];
            if (FVID2_SF_INTERLACED == appObj->testPrms.captScanFormat)
            {
                /* Field merge capture - pitch is already doubled */
                fmt->pitch[0U] /= 2U;
                fmt->pitch[1U] /= 2U;
                fmt->pitch[2U] /= 2U;
            }
            fmt->fieldMerged[0U] = FALSE;
            fmt->fieldMerged[1U] = FALSE;
            fmt->fieldMerged[2U] = FALSE;
            fmt->dataFormat      = outInfo->outFmt.dataFormat;
            fmt->scanFormat      = FVID2_SF_PROGRESSIVE;
            fmt->bpp = FVID2_BPP_BITS8; /* ignored */

            Fvid2Frame_init(&instObj->captDropFrame);
            instObj->captDropFrame.chNum = fmt->chNum;
            instObj->isDropFrmWithDriver = FALSE;

            /*
             * alloc memory based on 'format'
             * Allocated frame info is put in frames[]
             * LPBK_APP_FRAMES_PER_CH is the number of buffers per channel to
             * allocate
             */
            retVal = BspUtils_memFrameAlloc(fmt, frm, LPBK_APP_FRAMES_PER_CH);
            if (FVID2_SOK != retVal)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Capture Frame Alloc Failed!!!\r\n");
                break;
            }

            snprintf(fileStr, 30U, "loopbackOption%uInst%uCh%uStr%u",
                     (unsigned int) appObj->testPrms.testId,
                     (unsigned int) instObj->captInstId,
                     (unsigned int) chId,
                     (unsigned int) streamId);
            BspUtils_memFrameGetSize(fmt, &bufSize, NULL);
            BspOsal_cacheInv(frm[0U].addr[0U][0U],
                             (LPBK_APP_FRAMES_PER_CH * bufSize),
                             BSP_OSAL_CT_ALL, BSP_OSAL_WAIT_FOREVER);
            BspUtils_appPrintSaveRawCmd(
                fileStr,
                frm[0U].addr[0U][0U],
                outInfo->outFmt.dataFormat,
                fmt->width,
                fmt->height,
                (LPBK_APP_FRAMES_PER_CH * bufSize));

            if (BSP_PLATFORM_ID_ZEBU != Bsp_platformGetId())
            {
                /* Fill with background color */
                BspUtils_memset(
                    frm[0U].addr[0U][0U],
                    0x10U,
                    (LPBK_APP_FRAMES_PER_CH * bufSize));
                /* Flush and invalidate the CPU write */
                BspOsal_cacheWbInv(
                    frm[0U].addr[0U][0U],
                    (LPBK_APP_FRAMES_PER_CH * bufSize),
                    BSP_OSAL_CT_ALL, BSP_OSAL_WAIT_FOREVER);
            }

            for (frmId = 0U; frmId < LPBK_APP_FRAMES_PER_CH; frmId++)
            {
                /* Fill frame info */
                instObj->captFrameInfo[frmId].bufFid  = FVID2_FID_TOP;
                instObj->captFrameInfo[frmId].bufPair =
                    &instObj->captFramesOdd[frmId];
                instObj->captFrameInfo[frmId].instObj = instObj;

                frm[frmId].fid     = FVID2_FID_FRAME;
                frm[frmId].appData = &instObj->captFrameInfo[frmId];

                /* Since BspUtils_memFrameAlloc is setting the address
                 * for only top field, set addresses for bottom fields. */
                if (Fvid2_isDataFmtYuv422I(fmt->dataFormat))
                {
                    yFld1Offset = (UInt32) frm[frmId].addr[0U][0U] +
                                  fmt->pitch[0U];
                    frm[frmId].addr[1U][0U] = (Ptr) yFld1Offset;
                }
                if (Fvid2_isDataFmtSemiPlanar(fmt->dataFormat))
                {
                    yFld1Offset =
                        (UInt32) frm[frmId].addr[0U][0U] + fmt->pitch[0U];
                    cbCrFld0Offset =
                        (UInt32) ((UInt32) frm[frmId].addr[0U][0U] +
                                  (fmt->pitch[0U] * appObj->maxHeight));
                    cbCrFld1Offset =
                        (UInt32) (cbCrFld0Offset + fmt->pitch[0U]);
                    frm[frmId].addr[0U][1U] = (Ptr) cbCrFld0Offset;
                    frm[frmId].addr[1U][0U] = (Ptr) yFld1Offset;
                    frm[frmId].addr[1U][1U] = (Ptr) cbCrFld1Offset;
                }

                /* Set number of frame in frame list */
                frmList.frames[0U] = &frm[frmId];
                frmList.numFrames  = 1U;

                /* In case of field capture give odd field frame */
                if (FVID2_SF_INTERLACED == appObj->testPrms.captScanFormat)
                {
                    BspUtils_memcpy(
                        &instObj->captFramesOdd[frmId],
                        &frm[frmId],
                        sizeof (Fvid2_Frame));

                    /* Fill frame info */
                    instObj->captFrameInfoOdd[frmId].bufFid  = FVID2_FID_BOTTOM;
                    instObj->captFrameInfoOdd[frmId].bufPair =
                        &instObj->captFrames[frmId];
                    instObj->captFrameInfoOdd[frmId].instObj = instObj;

                    /* Set the odd address and app data */
                    instObj->captFramesOdd[frmId].appData =
                        &instObj->captFrameInfoOdd[frmId];
                    instObj->captFramesOdd[frmId].addr[0U][0U] =
                        instObj->captFramesOdd[frmId].addr[1U][0U];
                    instObj->captFramesOdd[frmId].addr[0U][1U] =
                        instObj->captFramesOdd[frmId].addr[1U][1U];
                    instObj->captFramesOdd[frmId].addr[0U][2U] =
                        instObj->captFramesOdd[frmId].addr[1U][2U];

                    /* Queue the odd frame */
                    frmList.frames[1U] = &instObj->captFramesOdd[frmId];
                    frmList.numFrames++;
                }

                /*
                 * queue the frames in frmList
                 * All allocate frames are queued here as an example.
                 * In general atleast 2 frames per channel need to queued
                 * before starting capture,
                 * else frame will get dropped until frames are queued
                 */
                retVal =
                    Fvid2_queue(instObj->captDrvHandle, &frmList, streamId);
                if (FVID2_SOK != retVal)
                {
                    GT_0trace(BspAppTrace, GT_ERR,
                              APP_NAME ": Capture Queue Failed!!!\r\n");
                    break;
                }
            }

            if (FVID2_SOK != retVal)
            {
                break;
            }
        }

        if (FVID2_SOK != retVal)
        {
            break;
        }
    }

    return (retVal);
}

/**
 *  \brief Free frames
 */
static Int32 LpbkApp_captFreeFrm(LpbkApp_Obj     *appObj,
                                 LpbkApp_InstObj *instObj,
                                 UInt32           fieldMerged)
{
    Int32         retVal = FVID2_SOK;
    UInt32        streamId, chId, idx;
    Fvid2_Format *fmt;
    Fvid2_Frame  *frm;

    /* for every stream and channel in a capture handle */
    for (streamId = 0U; streamId < instObj->captCreatePrms.numStream;
         streamId++)
    {
        for (chId = 0U; chId < instObj->captCreatePrms.numCh; chId++)
        {
            fmt = &instObj->captAllocFmt;

            /* base index for instObj->captFrames[] */
            idx =
                (VPS_CAPT_CH_PER_PORT_MAX * LPBK_APP_FRAMES_PER_CH
                 * streamId) + (LPBK_APP_FRAMES_PER_CH * chId);

            if (idx >= LPBK_APP_CAPT_MAX_FRM)
            {
                idx = 0U;
            }
            frm = &instObj->captFrames[idx];

            retVal |= BspUtils_memFrameFree(fmt, frm, LPBK_APP_FRAMES_PER_CH);
            if (FVID2_SOK != retVal)
            {
                GT_0trace(BspAppTrace, GT_ERR,
                          APP_NAME ": Capture Frame Free Failed!!!\r\n");
            }
        }
    }

    return (retVal);
}

/**
 *  \brief Allocate and queue frames to driver
 */


/**
 *  \brief Create and configure video sensor.
 */
static Int32 LpbkApp_createVidSensor(const LpbkApp_Obj *appObj,
                                     LpbkApp_InstObj   *instObj)
{
    Int32  retVal;
    UInt32 sensorInstId, sensorI2cInstId, sensorI2cAddr;
    Bsp_VidSensorChipIdParams sensorChipIdPrms;
    Bsp_VidSensorChipIdStatus sensorChipIdStatus;
    Bsp_VidSensorCreateStatus sensorCreateStatus;

    sensorInstId = Bsp_boardGetVideoDeviceInstId(
        instObj->decDrvId,
        FVID2_VPS_CAPT_VID_DRV,
        instObj->captInstId);
    sensorI2cInstId = Bsp_boardGetVideoDeviceI2cInstId(
        instObj->decDrvId,
        FVID2_VPS_CAPT_VID_DRV,
        instObj->captInstId);
    sensorI2cAddr = Bsp_boardGetVideoDeviceI2cAddr(
        instObj->decDrvId,
        FVID2_VPS_CAPT_VID_DRV,
        instObj->captInstId);

    /* Power on video sensor at board level mux */
    retVal = Bsp_boardPowerOnDevice(instObj->decDrvId, sensorInstId, TRUE);
    if (FVID2_SOK != retVal)
    {
        GT_0trace(BspAppTrace, GT_ERR, "Device power-on failed\r\n");
    }
    else
    {
        /* select video sensor at board level mux */
        retVal = Bsp_boardSelectDevice(instObj->decDrvId, sensorInstId);
        if (FVID2_SOK != retVal)
        {
            GT_0trace(BspAppTrace, GT_ERR, "Device select failed\r\n");
        }
    }

    if (FVID2_SOK == retVal)
    {
        /* Select specific mode */
        retVal = Bsp_boardSelectMode(
            instObj->decDrvId,
            sensorInstId,
            appObj->testPrms.captBoardMode);
        if (FVID2_SOK != retVal)
        {
            GT_0trace(BspAppTrace, GT_ERR, "Device select mode failed\r\n");
        }
    }

    if (FVID2_SOK == retVal)
    {
        instObj->sensorCreatePrms.deviceI2cInstId     = sensorI2cInstId;
        instObj->sensorCreatePrms.numDevicesAtPort    = 1U;
        instObj->sensorCreatePrms.deviceI2cAddr[0U]   = sensorI2cAddr;
        instObj->sensorCreatePrms.deviceResetGpio[0U] =
            BSP_VID_SENSOR_GPIO_NONE;
        instObj->sensorCreatePrms.sensorCfg    = NULL;
        instObj->sensorCreatePrms.numSensorCfg = 0U;
        instObj->decHandle = Fvid2_create(
            instObj->decDrvId,
            sensorInstId,
            &instObj->sensorCreatePrms,
            &sensorCreateStatus,
            NULL);
        GT_assert(BspAppTrace, instObj->decHandle != NULL);

        sensorChipIdPrms.deviceNum = 0;
        retVal = Fvid2_control(
            instObj->decHandle,
            IOCTL_BSP_VID_SENSOR_GET_CHIP_ID,
            &sensorChipIdPrms,
            &sensorChipIdStatus);
        GT_assert(BspAppTrace, (retVal == FVID2_SOK));

        GT_6trace(
            BspAppTrace, GT_INFO,
            APP_NAME
            ": VIP %d: VID SENSOR %04x (0x%02x): %04x:%04x:%04x\r\n",
            instObj->captInstId,
            instObj->decDrvId,
            instObj->sensorCreatePrms.deviceI2cAddr[0U],
            sensorChipIdStatus.chipId,
            sensorChipIdStatus.chipRevision,
            sensorChipIdStatus.firmwareVersion);
    }

    /* Configure sensor in UYVY 8-bit 1280x720 @ 30FPS, OV only supports
     *   1280x720 @ 30FPS and VIP expects data in UYVY . */
    if ((FVID2_VID_SENSOR_OV2659_DRV == instObj->decDrvId)  &&
        (FVID2_SOK == retVal))
    {
        Bsp_VidSensorConfigParams configParams;
        configParams.videoIfWidth = appObj->testPrms.captVideoIfWidth;
        configParams.dataformat   = FVID2_DF_YUV422I_UYVY;
        configParams.standard     = appObj->testPrms.captStandard;
        configParams.fps          = FVID2_FPS_30;
        retVal =
            Fvid2_control(instObj->decHandle, IOCTL_BSP_VID_SENSOR_SET_CONFIG,
                          &configParams,
                          NULL);
        GT_assert(BspAppTrace, (retVal == FVID2_SOK));
    }

    /* start sensor, right now only OV sensor driver is implemented */
    if ((FVID2_VID_SENSOR_OV2659_DRV == instObj->decDrvId) &&
        (FVID2_SOK == retVal))
    {
        retVal = Fvid2_control(instObj->decHandle, FVID2_START, NULL, NULL);
        GT_assert(BspAppTrace, (retVal == FVID2_SOK));
    }

    return (retVal);
}

Int32 LpbkApp_configureVideoPllAndClkSrc(LpbkApp_Obj *appObj, UInt32 pixelClk)
{
    Bsp_PlatformSetPllFreq vPllCfg;
    Bsp_PlatformVencSrc    vencClkCfg;
    Int32 retVal = FVID2_SOK;

    vPllCfg.videoPll         = BSP_PLATFORM_PLL_VIDEO1;
    vPllCfg.pixelClk         = pixelClk;
    vPllCfg.chooseMaxorMinMN = (UInt32) BSP_PLATFORM_VIDEO_PLL_CALC_MAX_MN;

    if (TRUE == Bsp_platformIsTda3xxFamilyBuild())
    {
        vPllCfg.videoPll = BSP_PLATFORM_PLL_EVE_VID_DSP;
    }

    retVal = Bsp_platformSetPllFreq(&vPllCfg);
    if (retVal != FVID2_SOK)
    {
        GT_0trace(BspAppTrace, GT_ERR,
                  APP_NAME ": Configuring Pll Failed!!!\r\n");
    }

    if (FVID2_SOK == retVal)
    {
        if (appObj->testPrms.vencId[0U] == VPS_DCTRL_DSS_VENC_LCD3)
        {
            vencClkCfg.outputVenc = BSP_PLATFORM_VENC_LCD3;
            vencClkCfg.vencClkSrc = BSP_PLATFORM_CLKSRC_DPLL_VIDEO1_CLKOUT3;
        }
        else
        {
            vencClkCfg.outputVenc = BSP_PLATFORM_VENC_LCD1;
            vencClkCfg.vencClkSrc = BSP_PLATFORM_CLKSRC_DPLL_VIDEO1_CLKOUT1;
        }

        if (TRUE == Bsp_platformIsTda3xxFamilyBuild())
        {
            vencClkCfg.vencClkSrc = BSP_PLATFORM_CLKSRC_DPLL_EVE_VID_DSP;
        }

        retVal = Bsp_platformSetVencClkSrc(&vencClkCfg);
        if (retVal != FVID2_SOK)
        {
            GT_0trace(BspAppTrace, GT_ERR,
                      APP_NAME ": Setting Venc clock source Failed!!!\r\n");
        }
    }

    return retVal;
}

/**
 *  LpbkApp_initParams
 *  Initialize the global variables and frame pointers.
 */
static void LpbkApp_initParams(LpbkApp_Obj *appObj)
{
    UInt32 streamId, chId, instCnt, plId;
    Vps_CaptCreateParams *createPrms;
    Vps_CaptVipParams    *vipPrms;
    Vps_CaptVipScParams  *scPrms;
    Vps_CaptVipOutInfo   *outInfo;
    LpbkApp_InstObj      *instObj;
    UInt32 maxHandles;

    appObj->maxWidth          = LPBK_APP_BUF_WIDTH;
    appObj->maxHeight         = LPBK_APP_BUF_HEIGHT;
    appObj->totalCaptFldCount = 0U;
    appObj->totalDispCount    = 0U;
    appObj->totalCpuLoad      = 0U;
    appObj->cpuLoadCount      = 0U;
    appObj->callBackCount     = 0U;
    appObj->curCaptInstance   = 0U;

    maxHandles = appObj->testPrms.numCaptHandles;
    if (maxHandles < appObj->testPrms.numDispHandles)
    {
        maxHandles = appObj->testPrms.numDispHandles;
    }
    for (instCnt = 0U; instCnt < maxHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];
        instObj->captInstId    = appObj->testPrms.captInstId[instCnt];
        instObj->dispInstId    = appObj->testPrms.dispInstId[instCnt];
        instObj->decDrvId      = appObj->testPrms.decDrvId[instCnt];
        instObj->encDrvId      = appObj->testPrms.encDrvId[instCnt];
        instObj->captDrvHandle = NULL;
        instObj->dispDrvHandle = NULL;
        Fvid2CbParams_init(&instObj->captCbPrms);
        instObj->captCbPrms.cbFxn   = &LpbkApp_captCbFxn;
        instObj->captCbPrms.appData = instObj;
        instObj->appObj = appObj;

        createPrms = &instObj->captCreatePrms;
        VpsCaptCreateParams_init(createPrms);
//        createPrms->videoIfMode  = appObj->testPrms.captVideoIfMode;
        createPrms->videoIfMode  = FVID2_VIFM_SCH_DS_AVID_VSYNC;
        createPrms->videoIfWidth = appObj->testPrms.captVideoIfWidth;
        createPrms->bufCaptMode  = VPS_CAPT_BCM_LAST_FRM_REPEAT;
        createPrms->numCh        = LPBK_APP_NUM_CAPT_CH;
        createPrms->numStream    = LPBK_APP_NUM_CAPT_STREAM;

        vipPrms = &instObj->vipPrms;
        VpsCaptVipParams_init(vipPrms);
        vipPrms->inFmt.chNum           = 0U;
        vipPrms->inFmt.width           = 1000; //appObj->testPrms.captInWidth;
        vipPrms->inFmt.height          = 2; //appObj->testPrms.captInHeight;
        vipPrms->inFmt.pitch[0U]       = 0U;
        vipPrms->inFmt.pitch[1U]       = 0U;
        vipPrms->inFmt.pitch[2U]       = 0U;
        vipPrms->inFmt.fieldMerged[0U] = FALSE;
        vipPrms->inFmt.fieldMerged[1U] = FALSE;
        vipPrms->inFmt.fieldMerged[2U] = FALSE;
        vipPrms->inFmt.dataFormat      = FVID2_DF_RAW08; //appObj->testPrms.captInDataFmt;
        vipPrms->inFmt.scanFormat      = FVID2_SF_PROGRESSIVE;
        vipPrms->inFmt.bpp      = FVID2_BPP_BITS16;
        vipPrms->inFmt.reserved = NULL;

//        scPrms          = &instObj->vipScPrms;
//        vipPrms->scPrms = scPrms;
//        VpsCaptVipScParams_init(scPrms);
//        scPrms->inCropCfg.cropStartX   = 0U;
//        scPrms->inCropCfg.cropStartY   = 0U;
//        scPrms->inCropCfg.cropWidth    = vipPrms->inFmt.width;
//        scPrms->inCropCfg.cropHeight   = vipPrms->inFmt.height;
//        scPrms->scCfg.bypass           = FALSE;
//        scPrms->scCfg.nonLinear        = FALSE;
//        scPrms->scCfg.stripSize        = 0U;
//        scPrms->scCfg.enableEdgeDetect = TRUE;
//        scPrms->scCfg.enablePeaking    = TRUE;
//        scPrms->scCfg.advCfg           = NULL;
//        scPrms->scCoeffCfg      = NULL;
//        scPrms->enableCoeffLoad = FALSE;
        vipPrms->scPrms = NULL;
        vipPrms->vipPortCfg     = NULL;

        Bsp_platformSetVipClkInversion(instObj->captInstId, FALSE);
//        if (FVID2_VID_SENSOR_OV2659_DRV == instObj->decDrvId)
//        {
//            vipPrms->vipPortCfg = &instObj->vipPortCfg;
//            VpsVipPortConfig_init(&instObj->vipPortCfg);
//            instObj->vipPortCfg.disCfg.actvidPol = FVID2_POL_HIGH;
//            instObj->vipPortCfg.disCfg.vsyncPol  = FVID2_POL_HIGH;
//            instObj->vipPortCfg.disCfg.hsyncPol  = FVID2_POL_HIGH;
//
//            /* Enable VIP cropping */
//            instObj->vipPortCfg.actCropEnable     = TRUE;
//            instObj->vipPortCfg.actCropCfg.srcNum = 0U;
//            instObj->vipPortCfg.actCropCfg.cropCfg.cropStartX = 0U;
//            instObj->vipPortCfg.actCropCfg.cropCfg.cropStartY = 0U;
//            instObj->vipPortCfg.actCropCfg.cropCfg.cropWidth  = 1280U;
//            instObj->vipPortCfg.actCropCfg.cropCfg.cropHeight = 720U;
//        }

        vipPrms->vipPortCfg = &instObj->vipPortCfg;
        VpsVipPortConfig_init(&instObj->vipPortCfg);
        instObj->vipPortCfg.disCfg.actvidPol = FVID2_POL_HIGH;
        instObj->vipPortCfg.disCfg.vsyncPol  = FVID2_POL_HIGH;

        vipPrms->cscCfg = NULL;

        for (streamId = 0U; streamId < LPBK_APP_NUM_CAPT_STREAM; streamId++)
        {
            for (chId = 0U; chId < VPS_CAPT_CH_PER_PORT_MAX; chId++)
            {
                createPrms->chNumMap[streamId][chId] =
                    Vps_captMakeChNum(instObj->captInstId, streamId, chId);
            }
            outInfo = &vipPrms->outStreamInfo[streamId];
            outInfo->outFmt.chNum           = 0U;
            outInfo->outFmt.width           = 1000; //appObj->testPrms.captOutWidth;
            outInfo->outFmt.height          = 2; //appObj->testPrms.captOutHeight;
            outInfo->outFmt.fieldMerged[0U] = FALSE;
            outInfo->outFmt.fieldMerged[1U] = FALSE;
            outInfo->outFmt.fieldMerged[2U] = FALSE;
            outInfo->outFmt.dataFormat      = FVID2_DF_RAW08; //appObj->testPrms.captOutDataFmt;
            outInfo->outFmt.scanFormat      = FVID2_SF_PROGRESSIVE;
            outInfo->outFmt.bpp       = FVID2_BPP_BITS16;
            outInfo->outFmt.pitch[0U] = 0U;
            outInfo->outFmt.pitch[1U] = 0U;
            outInfo->outFmt.pitch[2U] = 0U;
            if (Fvid2_isDataFmtSemiPlanar(outInfo->outFmt.dataFormat))
            {
                outInfo->outFmt.pitch[FVID2_YUV_SP_Y_ADDR_IDX] =
                    VpsUtils_align(appObj->maxWidth, VPS_BUFFER_ALIGNMENT);
                outInfo->outFmt.pitch[FVID2_YUV_SP_CBCR_ADDR_IDX] =
                    outInfo->outFmt.pitch[FVID2_YUV_SP_Y_ADDR_IDX];
            }
            else if (Fvid2_isDataFmtYuv422I(outInfo->outFmt.dataFormat))
            {
                outInfo->outFmt.pitch[FVID2_YUV_INT_ADDR_IDX] =
                    VpsUtils_align(appObj->maxWidth * 2U, VPS_BUFFER_ALIGNMENT);
            }
            else if (Fvid2_isDataFmtRgb32bit(outInfo->outFmt.dataFormat))
            {
                outInfo->outFmt.pitch[FVID2_RGB_ADDR_IDX] =
                    VpsUtils_align(appObj->maxWidth * 4U, VPS_BUFFER_ALIGNMENT);
            }
            else
            {
                /* Align the pitch to BPP boundary as well since the pitch
                 * aligined to VPS_BUFFER_ALIGNMENT may not be multiple of 3
                 * bytes (1 pixel) */
                outInfo->outFmt.pitch[FVID2_RGB_ADDR_IDX] =
                    VpsUtils_align(
                        appObj->maxWidth * 3U, (VPS_BUFFER_ALIGNMENT * 3U));
            }

            if (FVID2_SF_INTERLACED == appObj->testPrms.captScanFormat)
            {
                /* Field merge capture - double the pitch */
                outInfo->outFmt.pitch[0U] *= 2U;
                outInfo->outFmt.pitch[1U] *= 2U;
                outInfo->outFmt.pitch[2U] *= 2U;
            }

            outInfo->bufFmt  = FVID2_BUF_FMT_FRAME;
            outInfo->memType = VPS_VPDMA_MT_NONTILEDMEM;
            if (!Bsp_platformIsTI814xFamilyBuild())
            {
                outInfo->maxOutWidth[0U]  = VPS_VPDMA_MAX_OUT_WIDTH_REG1;
                outInfo->maxOutHeight[0U] = VPS_VPDMA_MAX_OUT_HEIGHT_REG1;
                for (plId = 1U; plId < FVID2_MAX_PLANES; plId++)
                {
                    outInfo->maxOutWidth[plId]  = VPS_VPDMA_MAX_OUT_WIDTH_REG2;
                    outInfo->maxOutHeight[plId] =
                        VPS_VPDMA_MAX_OUT_HEIGHT_REG2;
                }
            }
            else
            {
                for (plId = 0U; plId < FVID2_MAX_PLANES; plId++)
                {
                    outInfo->maxOutWidth[plId] =
                        VPS_VPDMA_MAX_OUT_WIDTH_1920_PIXELS;
                    outInfo->maxOutHeight[plId] =
                        VPS_VPDMA_MAX_OUT_HEIGHT_1080_LINES;
                }
            }

            outInfo->scEnable = FALSE;
            if (LPBK_APP_SC_ENABLE_ALL == appObj->testPrms.captScEnable)
            {
                outInfo->scEnable = TRUE;
            }
            if ((0U == streamId) &&
                (LPBK_APP_SC_ENABLE_0 == appObj->testPrms.captScEnable))
            {
                outInfo->scEnable = TRUE;
            }
            if ((1U == streamId) &&
                (LPBK_APP_SC_ENABLE_1 == appObj->testPrms.captScEnable))
            {
                outInfo->scEnable = TRUE;
            }
            outInfo->subFrmPrms.subFrameEnable      = FALSE;
            outInfo->subFrmPrms.numLinesPerSubFrame = 0U;
            outInfo->subFrmPrms.subFrameCb          = NULL;
        }

        instObj->maxOutWidth[0U]  = LPBK_APP_MAXSIZE_1_WIDTH;
        instObj->maxOutHeight[0U] = LPBK_APP_MAXSIZE_1_HEIGHT;
        instObj->maxOutWidth[1U]  = LPBK_APP_MAXSIZE_2_WIDTH;
        instObj->maxOutHeight[1U] = LPBK_APP_MAXSIZE_2_HEIGHT;
        instObj->maxOutWidth[2U]  = LPBK_APP_MAXSIZE_3_WIDTH;
        instObj->maxOutHeight[2U] = LPBK_APP_MAXSIZE_3_HEIGHT;


    }

    return;
}

static void LpbkApp_printStatistics(LpbkApp_Obj *appObj,
                                    UInt32       execTimeInMsecs)
{
    UInt32           instCnt;
    LpbkApp_InstObj *instObj;
    UInt32           cpuLoad = 0U;

    GT_0trace(BspAppTrace, GT_INFO, " \r\n");
    GT_0trace(BspAppTrace, GT_INFO, " Execution Statistics \r\n");
    GT_0trace(BspAppTrace, GT_INFO, " ==================== \r\n");
    GT_0trace(BspAppTrace, GT_INFO, " \r\n");
    GT_2trace(BspAppTrace, GT_INFO,
              " Execution time    : %d.%d s \r\n",
              (execTimeInMsecs / 1000), (execTimeInMsecs % 1000));
    GT_2trace(BspAppTrace, GT_INFO,
              " Total field Count : %d (%d fields/sec) \r\n",
              appObj->totalCaptFldCount,
              (appObj->totalCaptFldCount * 1000) / execTimeInMsecs);
    if (appObj->cpuLoadCount != 0U)
    {
        /* Load count is non Zero, Valid CPU load value is updated. */
        cpuLoad = appObj->totalCpuLoad / appObj->cpuLoadCount;
    }
    GT_1trace(BspAppTrace, GT_INFO,
              " Avg CPU Load      : %d %% \r\n",
              cpuLoad);
    GT_0trace(BspAppTrace, GT_INFO, " \r\n");

    for (instCnt = 0U; instCnt < appObj->testPrms.numCaptHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];

        /* Print the capture driver status */
        BspUtils_appPrintCaptStatus(
            instObj->captDrvHandle,
            instObj->captInstId,
            instObj->captCreatePrms.numStream,
            instObj->captCreatePrms.numCh,
            execTimeInMsecs);
    }

    for (instCnt = 0U; instCnt < appObj->testPrms.numDispHandles; instCnt++)
    {
        instObj = &appObj->instObj[instCnt];

        /* Print the display driver status */
        BspUtils_appPrintDispStatus(
            instObj->dispDrvHandle,
            instObj->dispInstId,
            execTimeInMsecs);
    }

    /* Print the Display Controller error status */
    BspUtils_appPrintDctrlStatus(
        appObj->dctrlHandle);

    GT_0trace(BspAppTrace, GT_INFO, " \r\n");

    return;
}

