//###########################################################################
// Description:
//! \addtogroup f2806x_example_list
//! <h1>GPIO Setup (gpio_setup)</h1>
//!
//! This example Configures the 2806x GPIO into two different configurations
//! This code is verbose to illustrate how the GPIO could be setup.In a real
//! application, lines of code can be combined for improved code size and 
//! efficiency.
//!
//! This example only sets-up the GPIO.. nothing is actually done with
//! the pins after setup.
//!
//! In general:
//!   - All pullup resistors are enabled.  For ePWMs this may not be desired.
//!   - Input qual for communication ports (eCAN, SPI, SCI, I2C) is asynchronous
//!   - Input qual for Trip pins (TZ) is asynchronous
//!   - Input qual for eCAP and eQEP signals is synch to SYSCLKOUT
//!   - Input qual for some I/O's and interrupts may have a sampling window
//
//
//###########################################################################
// $TI Release: F2806x C/C++ Header Files and Peripheral Examples V151 $
// $Release Date: February  2, 2016 $
// $Copyright: Copyright (C) 2011-2016 Texas Instruments Incorporated -
//             http://www.ti.com/ ALL RIGHTS RESERVED $
//###########################################################################

#include "DSP28x_Project.h"     // Device Headerfile and Examples Include File
#include <math.h>

// Select the example to compile in.  Only one example should be set as 1
// the rest should be set as 0.

#define EXAMPLE1 1  // Basic pinout configuration example
#define EXAMPLE2 0  // Communication pinout example

// Prototype statements for functions found within this file.
void Gpio_setup1(void);
void Gpio_setup2(void);

static void vector_square (const float *restrict input, float *restrict output, int length)
{
    int index;

    for (index = 0; index < length; index++)
    {
        output[index] = input[index] * input[index];
    }
}

#define VECTOR_TEST_LENGTH 1000
static float input_vec[VECTOR_TEST_LENGTH];
static float output_vec[VECTOR_TEST_LENGTH];

void main(void)
{
    int index;

// Step 1. Initialize System Control:
// PLL, WatchDog, enable Peripheral Clocks
// This example function is found in the F2806x_SysCtrl.c file.
   InitSysCtrl();

// Step 2. Initalize GPIO:
// This example function is found in the F2806x_Gpio.c file and
// illustrates how to set the GPIO to it's default state.
   // InitGpio(); Skipped for this example

// Step 3. Clear all interrupts and initialize PIE vector table:
// Disable CPU interrupts
   DINT;

// Initialize PIE control registers to their default state.
// The default state is all PIE interrupts disabled and flags
// are cleared.
// This function is found in the F2806x_PieCtrl.c file.
   InitPieCtrl();

// Disable CPU interrupts and clear all CPU interrupt flags:
   IER = 0x0000;
   IFR = 0x0000;

// Initialize the PIE vector table with pointers to the shell Interrupt
// Service Routines (ISR).
// This will populate the entire table, even if the interrupt
// is not used in this example.  This is useful for debug purposes.
// The shell ISR routines are found in F2806x_DefaultIsr.c.
// This function is found in F2806x_PieVect.c.
   InitPieVectTable();


// Step 4. Initialize all the Device Peripherals:
// This function is found in F2806x_InitPeripherals.c
// InitPeripherals(); // Not required for this example

// Step 5. User specific code:

   // Enable an GPIO output on GPIO6, set it log
   EALLOW;
   GpioDataRegs.GPACLEAR.bit.GPIO6 = 1;   // Load output latch
   GpioCtrlRegs.GPAPUD.bit.GPIO6 = 1;   // Disable pullup on GPIO6
   GpioCtrlRegs.GPAMUX1.bit.GPIO6 = 0;  // GPIO6 = GPIO6
   GpioCtrlRegs.GPADIR.bit.GPIO6 = 1;   // GPIO6 = output
   EDIS;

   for (index = 0; index < VECTOR_TEST_LENGTH; index++)
   {
       input_vec[index] = (float) index  / (float) VECTOR_TEST_LENGTH;
   }
   for (;;)
   {
       asm ("pre_delay_calibration:");
       GpioDataRegs.GPASET.bit.GPIO6 = 1;
       DELAY_US (100);
       GpioDataRegs.GPACLEAR.bit.GPIO6 = 1;
       asm ("post_delay_calibration:");

       asm ("pre_vec_square_10:");
       GpioDataRegs.GPASET.bit.GPIO6 = 1;
       vector_square (input_vec, output_vec, 10);
       GpioDataRegs.GPACLEAR.bit.GPIO6 = 1;
       asm ("post_vec_square_10:");

       asm ("pre_vec_square_100:");
       GpioDataRegs.GPASET.bit.GPIO6 = 1;
       vector_square (input_vec, output_vec, 100);
       GpioDataRegs.GPACLEAR.bit.GPIO6 = 1;
       asm ("post_vec_square_100:");

       asm ("pre_vec_square_1000:");
       GpioDataRegs.GPASET.bit.GPIO6 = 1;
       vector_square (input_vec, output_vec, VECTOR_TEST_LENGTH);
       GpioDataRegs.GPACLEAR.bit.GPIO6 = 1;
       asm ("post_vec_square_1000:");
   }
}
