/**
 * @file   main_dsp.c
 *
 * @brief  This file tests the ICSS-EMAC driver APIs
 */
/*
 * Copyright (c) 2015-2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <xdc/std.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <xdc/runtime/Error.h>
#include <xdc/runtime/System.h>
#include <xdc/runtime/knl/Cache.h>
#include <ti/sysbios/BIOS.h>
#include <ti/sysbios/knl/Task.h>
#include <ti/sysbios/hal/Core.h>

#include <ti/csl/soc.h>



#include <ti/drv/uart/UART_stdio.h>
#include <ti/board/board.h>


#include <ti/drv/pruss/pruicss.h>
#include <ti/drv/pruss/soc/pruicss_v1.h>

#include <ti/drv/icss_emac/test/src/test_common_utils.h>

#include <ti/drv/icss_emac/test/src/tiemac_pruss_intc_mapping.h>

#ifdef SOC_AM572x
#include <ti/drv/icss_emac/firmware/am57x/v1_0/icss_emac_pru0_bin.h>
#include <ti/drv/icss_emac/firmware/am57x/v1_0/icss_emac_pru1_bin.h>
#include <ti/csl/soc/am572x/src/csl_device_xbar.h>
#define PRU0_FIRMWARE_V1_0_NAME      PRU0_FIRMWARE_V1_0        // name of the C struct in PRU header file for PG version 1.x
#define PRU1_FIRMWARE_V1_1_NAME      PRU1_FIRMWARE_V1_0        // name of the C struct in PRU header file for PG versin 1.x

#include <ti/drv/icss_emac/firmware/am57x/v2_1/icss_emac_pru0_bin.h>
#include <ti/drv/icss_emac/firmware/am57x/v2_1/icss_emac_pru1_bin.h>
#define PRU0_FIRMWARE_NAME      PRU0_FIRMWARE        // name of the C struct in PRU header file
#define PRU1_FIRMWARE_NAME      PRU1_FIRMWARE        // name of the C struct in PRU header file
#endif

#ifdef SOC_AM571x
#include <ti/drv/icss_emac/firmware/am57x/v2_1/icss_emac_pru0_bin.h>
#include <ti/drv/icss_emac/firmware/am57x/v2_1/icss_emac_pru1_bin.h>
#include <ti/csl/soc/am571x/src/csl_device_xbar.h>
#define PRU0_FIRMWARE_NAME      PRU0_FIRMWARE        // name of the C struct in PRU header file
#define PRU1_FIRMWARE_NAME      PRU1_FIRMWARE        // name of the C struct in PRU header file
#endif

#ifdef _TMS320C6X
#include <ti/sysbios/family/c66/Cache.h>
#endif

extern int ICSS_EMAC_testAM572xSetup(uint32_t boardType, uint32_t numPorts);

extern uint8_t ICSS_EMAC_testPkt[];
extern uint32_t ICSS_EMAC_testTotalPktRcvd;
/* number of links which actually came up to test the interface */
static uint32_t  ICSS_EMAC_testLinkUpCount = 0;

/* total number of packets received on all interfaces being tested, cummulative account */

PRUICSS_Handle ICSS_EMAC_testPruIcssHandle1 = NULL;
PRUICSS_Handle ICSS_EMAC_testPruIcssHandle2 = NULL;

extern ICSS_EmacHandle ICSS_EMAC_testHandle;
extern ICSS_EmacHandle ICSS_EMAC_testHandle1;
extern ICSS_EmacHandle ICSS_EMAC_testHandle2;
extern ICSS_EmacHandle ICSS_EMAC_testHandle3;


extern uint8_t ICSS_EMAC_testLclMac[];
extern uint8_t ICSS_EMAC_testLclMac1[];
extern uint8_t ICSS_EMAC_testLclMac2[];
extern uint8_t ICSS_EMAC_testLclMac3[];

extern uint32_t ICSS_EMAC_testPacketRcvdPort0;
extern uint32_t ICSS_EMAC_testPacketRcvdPort1;
extern uint32_t ICSS_EMAC_testPacketRcvdPort2;
extern uint32_t ICSS_EMAC_testPacketRcvdPort3;

extern uint32_t ICSS_EMAC_testLinkIsrPru1Eth0;
extern uint32_t ICSS_EMAC_testLinkIsrPru1Eth1;
extern uint32_t ICSS_EMAC_testLinkIsrPru2Eth0;
extern uint32_t ICSS_EMAC_testLinkIsrPru2Eth1;

extern uint8_t    ICSS_EMAC_testTtsModePort1;
extern uint8_t    ICSS_EMAC_testTtsModePort2;


/* either idkAM572x or idkAM571x */
uint8_t ICSS_EMAC_testEvmType = 0;

/* PG version of EVM */
uint32_t ICSS_EMAC_testPgVersion = 0;

static uint32_t ICSS_EMAC_testPruInstance2Done = 0;

/***********************************************************************/
/* local functions declaration                                         */
/***********************************************************************/


extern uint8_t pru_imem0_start;         /*start address for PRU0 binary array*/
extern uint8_t pru_imem0_end;           /*end address for PRU0 binary array*/
extern uint8_t pru_imem1_start;         /*start address for PRU1 binary array*/
extern uint8_t pru_imem1_end;           /*end address for PRU1 binary array*/

/*
 *    ---task to initialize PRU---
 */
Void ICSS_EMAC_testTaskPruss1(UArg a0, UArg a1)
{
    Uint8 firmwareLoad_done = FALSE;
    uint32_t count = 0;
    ICSS_EmacTxArgument txArgs;
    uint8_t ret = 0;

    memset(&txArgs, 0, sizeof(ICSS_EmacTxArgument));

    PRUICSS_pruDisable(ICSS_EMAC_testPruIcssHandle1, ICSS_EMAC_PORT_1-1);
    PRUICSS_pruDisable(ICSS_EMAC_testPruIcssHandle1, ICSS_EMAC_PORT_2-1);
    if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle1,PRU_ICSS_IRAM(0) ,0,
                          (uint32_t *) &pru_imem0_start,
                          &pru_imem0_end - &pru_imem0_start))
    {
        if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle1,PRU_ICSS_IRAM(1) ,0,
                              (uint32_t *) &pru_imem1_start,
                              &pru_imem1_end - &pru_imem1_start))
        {
            firmwareLoad_done = TRUE;
        }
    }

    if( firmwareLoad_done)
    {
        PRUICSS_pruEnable(ICSS_EMAC_testPruIcssHandle1, ICSS_EMAC_PORT_1-1);
        PRUICSS_pruEnable(ICSS_EMAC_testPruIcssHandle1, ICSS_EMAC_PORT_2-1);
    }


    while (!ICSS_EMAC_testPruInstance2Done)
    {
        /*    Do not print during time sensitive TTS testing    */
        if(!(ICSS_EMAC_testTtsModePort1 || ICSS_EMAC_testTtsModePort2))
        {
            UART_printf("\nICSS_EMAC_testTaskPruss1: Waiting for PRU2 Test to complete before testing PRU1\n");
        }
        Task_sleep(100);
    }

    if (!ICSS_EMAC_testLinkIsrPru1Eth0)
    {
        UART_printf("ICSS_EMAC_testTaskPruss1: LINK IS DOWN,skipping PRU1 ETH0\n");
        
    }
    else
    {
        UART_printf("ICSS_EMAC_testTaskPruss1: PRU1 ETH0: LINK IS UP, eth0 state: %d\n", ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->linkStatus[0]);
    
        ICSS_EMAC_testPacketRcvdPort0 = 1;
    
        txArgs.icssEmacHandle = ICSS_EMAC_testHandle;
        txArgs.lengthOfPacket = ICSS_EMAC_TEST_PKT_SIZE;
        txArgs.portNumber = 1;
        txArgs.queuePriority = 3;
        txArgs.srcAddress = &ICSS_EMAC_testPkt[0];

       for (count=0;count < ICSS_EMAC_TEST_PKT_TX_COUNT;count++)
        {
            if(((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->linkStatus[0] )
            {
                if(ICSS_EMAC_testPacketRcvdPort0 )
                {
                    ICSS_EmacTxPacket(&txArgs, NULL);
                    ICSS_EMAC_testPacketRcvdPort0 = 0;
                    while(!ICSS_EMAC_testPacketRcvdPort0)
                    {
                        Task_sleep(100);
                    }
                }
            }
        }
        ICSS_EMAC_testLinkUpCount++;
    }
    /* Verification on ETH1 for instance 1 only */
    if (!ICSS_EMAC_testLinkIsrPru1Eth1)
    {
        UART_printf("ICSS_EMAC_testTaskPruss1: LINK IS DOWN,skipping PRU1 ETH1\n");
    }
    else
    {
        UART_printf("ICSS_EMAC_testTaskPruss1: PRU1 ETH1: LINK IS UP, eth0 state: %d\n", ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->linkStatus[0]);
    
        ICSS_EMAC_testPacketRcvdPort1 = 1;
    
        txArgs.icssEmacHandle = ICSS_EMAC_testHandle1;
        txArgs.lengthOfPacket = ICSS_EMAC_TEST_PKT_SIZE;
        txArgs.portNumber = 2;
        txArgs.queuePriority = 3;
        txArgs.srcAddress = &ICSS_EMAC_testPkt[0];

        for (count=0;count < ICSS_EMAC_TEST_PKT_TX_COUNT;count++)
        {
            if(((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->linkStatus[0] )
            {
                if(ICSS_EMAC_testPacketRcvdPort1 )
                {
                    ICSS_EmacTxPacket(&txArgs, NULL);
                    ICSS_EMAC_testPacketRcvdPort1 = 0;
                    while(!ICSS_EMAC_testPacketRcvdPort1)
                    {
                        Task_sleep(100);
                    }
                }
            }
        }
        ICSS_EMAC_testLinkUpCount++;
    }

    if(ICSS_EMAC_testEvmType == ICSS_EMAC_TEST_BOARD_IDKAM572x)
    {
        UART_printf("ICSS_EMAC_testTaskPruss1: Skipping TTS Tests for PRUICSS Instance 1 for AM572x\n");
    }
    else
    {
        while(!ICSS_EMAC_testLinkIsrPru1Eth0 && !ICSS_EMAC_testLinkIsrPru1Eth1)
        {
            UART_printf("\nICSS_EMAC_testTaskPruss1: LINK IS DOWN, pluggin loopback cable for PRU1 ETH0 and PRU1 ETH1.\n");
        }

        UART_printf("\n============================================================");
        UART_printf("\nInitiating TTS tests on ICSS_EMAC_TEST_PRU1ETH0 and ICSS_EMAC_TEST_PRU1ETH1");

        /*    Create TTS related semaphores    */
        ret = ICSS_EMAC_testTtsSemCreate();
        assert(ret == 0);

        ICSS_EMAC_testTtsModePort1 = 1;
        ICSS_EMAC_testTtsModePort2 = 1;

        /*    Wait for TTS test to complete    */
        while(ICSS_EMAC_testTtsModePort1 || ICSS_EMAC_testTtsModePort2)
        {
            Task_sleep(10);
        }

        UART_printf("\n============================================================");
        UART_printf("\nTTS tests finished on ICSS_EMAC_TEST_PRU1ETH0 and ICSS_EMAC_TEST_PRU1ETH1");
        UART_printf("\n============================================================");
    }
    if (ICSS_EMAC_testTotalPktRcvd == (ICSS_EMAC_TEST_PKT_TX_COUNT * ICSS_EMAC_testLinkUpCount))
    {
        UART_printf("\nAll tests have passed\n");
    }
    else
        UART_printf("ICSS_EMAC_testTaskPruss1: total Pkts received: %d\n", ICSS_EMAC_testTotalPktRcvd);

    ICSS_EMAC_testInterruptDisable(ICSS_EMAC_testHandle);
    ICSS_EMAC_testInterruptDisable(ICSS_EMAC_testHandle1);

    ICSS_EmacDeInit(ICSS_EMAC_testHandle, ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    ICSS_EmacDeInit(ICSS_EMAC_testHandle1, ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
}

Void ICSS_EMAC_testTaskPruss2(UArg a0, UArg a1)
{
    Uint8 firmwareLoad_done = FALSE;
    uint32_t count = 0;
    int8_t ret = 0;

    ICSS_EmacTxArgument txArgs;
    memset(&txArgs, 0, sizeof(ICSS_EmacTxArgument));
    ICSS_EMAC_testPgVersion = (HW_RD_REG32(CSL_MPU_CTRL_MODULE_WKUP_CORE_REGISTERS_REGS + CTRL_WKUP_ID_CODE) & 0xf0000000) >> 28;


    UART_printf("ICSS_EMAC_testTaskPruss2: ICSS_EMAC_testPgVersion: 0x%x\n", ICSS_EMAC_testPgVersion);
    PRUICSS_pruDisable(ICSS_EMAC_testPruIcssHandle2, ICSS_EMAC_PORT_1-1);
    PRUICSS_pruDisable(ICSS_EMAC_testPruIcssHandle2, ICSS_EMAC_PORT_2-1);

#ifdef SOC_AM572x
    if (ICSS_EMAC_testPgVersion >= 2)
    {
        if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(0) ,0,
                              (uint32_t *) PRU0_FIRMWARE_NAME,
                              sizeof(PRU0_FIRMWARE_NAME)))
        {
            if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(1) ,0,
                                  (uint32_t *) PRU1_FIRMWARE_NAME,
                                  sizeof(PRU1_FIRMWARE_NAME)))
            {
                firmwareLoad_done = TRUE;
            }
        }
    }
    else
    {
        if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(0) ,0,
                              (uint32_t *) PRU0_FIRMWARE_V1_0_NAME,
                              sizeof(PRU0_FIRMWARE_V1_0_NAME)))
        {
            if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(1) ,0,
                                  (uint32_t *) PRU1_FIRMWARE_V1_1_NAME,
                                  sizeof(PRU1_FIRMWARE_V1_1_NAME)))
            {
                firmwareLoad_done = TRUE;
            }
        }
    }
#else
     if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(0) ,0,
                              (uint32_t *) PRU0_FIRMWARE_NAME,
                              sizeof(PRU0_FIRMWARE_NAME)))
    {
        if(PRUICSS_pruWriteMemory(ICSS_EMAC_testPruIcssHandle2,PRU_ICSS_IRAM(1) ,0,
                              (uint32_t *) PRU1_FIRMWARE_NAME,
                               sizeof(PRU1_FIRMWARE_NAME)))
        {
            firmwareLoad_done = TRUE;
        }
    }
#endif


    if( firmwareLoad_done)
    {
        PRUICSS_pruEnable(ICSS_EMAC_testPruIcssHandle2, ICSS_EMAC_PORT_1-1);
        PRUICSS_pruEnable(ICSS_EMAC_testPruIcssHandle2, ICSS_EMAC_PORT_2-1);
    }
    else
    {
        UART_printf("ICSS_EMAC_testTaskPruss2: firmware load failure\n");
        return;
    }

    while (!ICSS_EMAC_testLinkIsrPru2Eth0)
    {
        UART_printf("ICSS_EMAC_testTaskPruss2: LINK IS DOWN, pluggin loopback cable for PRU2 ETH0\n");
        Task_sleep(100);
    }
     ICSS_EMAC_testLinkUpCount++;
    UART_printf("ICSS_EMAC_testTaskPruss2: PRU2 ETH0: LINK IS UP, eth0 state: %d, link up count: %d\n", ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->linkStatus[0],
                      ICSS_EMAC_testLinkUpCount);

    ICSS_EMAC_testPacketRcvdPort2 = 1;

    txArgs.icssEmacHandle = ICSS_EMAC_testHandle2;
    txArgs.lengthOfPacket = ICSS_EMAC_TEST_PKT_SIZE;
    txArgs.portNumber = 1;
    txArgs.queuePriority = 3;
    txArgs.srcAddress = &ICSS_EMAC_testPkt[0];
    
    for (count=0;count < ICSS_EMAC_TEST_PKT_TX_COUNT;count++)
    {
        if(((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->linkStatus[0] )
        {
            if(ICSS_EMAC_testPacketRcvdPort2 )
            {
                ICSS_EmacTxPacket(&txArgs, NULL);
                ICSS_EMAC_testPacketRcvdPort2 = 0;
                while(!ICSS_EMAC_testPacketRcvdPort2)
                {
                    Task_sleep(100);
                }
            }
        }
    }
    if(ICSS_EMAC_testEvmType == ICSS_EMAC_TEST_BOARD_IDKAM571x)
    {
        UART_printf("ICSS_EMAC_testTaskPruss2: Skipping ICSS_EMAC_TEST_PRU2ETH1 for AM571x\n");
        UART_printf("ICSS_EMAC_testTaskPruss2: Skipping TTS Tests for PRUICSS Instance 2 for AM571x\n");
    }
    else
    {
        while (!ICSS_EMAC_testLinkIsrPru2Eth1)
        {
            UART_printf("ICSS_EMAC_testTaskPruss2: LINK IS DOWN,skipping PRU2 ETH1\n");
        }
        ICSS_EMAC_testLinkUpCount++;
        UART_printf("ICSS_EMAC_testTaskPruss2: PRU2 ETH1: LINK IS UP, eth0 state: %d, link up count: %d\n", ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->linkStatus[0],
                      ICSS_EMAC_testLinkUpCount);

        ICSS_EMAC_testPacketRcvdPort3 = 1;
        txArgs.icssEmacHandle = ICSS_EMAC_testHandle3;
        txArgs.lengthOfPacket = ICSS_EMAC_TEST_PKT_SIZE;
        txArgs.portNumber = 2;
        txArgs.queuePriority = 3;
        txArgs.srcAddress = &ICSS_EMAC_testPkt[0];
        for (count=0;count < ICSS_EMAC_TEST_PKT_TX_COUNT;count++)
        {
            if(((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->linkStatus[0] )
            {
                if(ICSS_EMAC_testPacketRcvdPort3 )
                {
                    ICSS_EmacTxPacket(&txArgs, NULL);
                    ICSS_EMAC_testPacketRcvdPort3 = 0;
                    while(!ICSS_EMAC_testPacketRcvdPort3)
                    {
                        Task_sleep(100);
                    }
                }
            }
         }

        while(!ICSS_EMAC_testLinkIsrPru2Eth0 && !ICSS_EMAC_testLinkIsrPru2Eth1)
        {
            UART_printf("\nICSS_EMAC_testTaskPruss2: LINK IS DOWN, pluggin loopback cable for PRU2 ETH0 and PRU2 ETH1.\n");
        }

        UART_printf("\n============================================================");
        UART_printf("\nInitiating TTS tests on ICSS_EMAC_TEST_PRU2ETH0 and ICSS_EMAC_TEST_PRU2ETH1");

        /*    Create TTS related semaphores    */
        ret = ICSS_EMAC_testTtsSemCreate();
        assert(ret == 0);

        ICSS_EMAC_testTtsModePort1 = 1;
        ICSS_EMAC_testTtsModePort2 = 1;

        /*    Wait for TTS test to complete    */
        while(ICSS_EMAC_testTtsModePort1 || ICSS_EMAC_testTtsModePort2)
        {
            Task_sleep(10);
        }

        UART_printf("\n============================================================");
        UART_printf("\nTTS tests finished on ICSS_EMAC_TEST_PRU2ETH0 and ICSS_EMAC_TEST_PRU2ETH1");
        UART_printf("\n============================================================");
    }

    ICSS_EMAC_testPruInstance2Done = 1;
    UART_printf("\nDone with PRU-ICSS Instance 2 Testing\n");

    ICSS_EMAC_testGetPruStats(1, ICSS_EMAC_testHandle2);
    ICSS_EMAC_testGetPruStats(2, ICSS_EMAC_testHandle3);
    if (ICSS_EMAC_testTotalPktRcvd == (ICSS_EMAC_TEST_PKT_TX_COUNT*ICSS_EMAC_testLinkUpCount))
    {
        UART_printf("All tests have passed\n");
    }

    ICSS_EMAC_testInterruptDisable(ICSS_EMAC_testHandle2);
    ICSS_EMAC_testInterruptDisable(ICSS_EMAC_testHandle3);
    ICSS_EmacDeInit(ICSS_EMAC_testHandle2, ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    ICSS_EmacDeInit(ICSS_EMAC_testHandle3, ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);

}


/*
 *  ======== main ========
 */

int32_t ICSS_EMAC_testPruIcssInstance1Setup(void)
{

    Task_Params taskParams;
    PRUICSS_IntcInitData pruss_intc_initdata = PRUSS_INTC_INITDATA;

    PRUICSS_Config  *cfg;

     int32_t ret  = PRUICSS_socGetInitCfg(&cfg);
    if (ret  != PRUICSS_RETURN_SUCCESS)
        return (ret);

    ICSS_EMAC_testPruIcssHandle1 = PRUICSS_create((PRUICSS_Config*)cfg,PRUICCSS_INSTANCE_ONE);


    /* For PRU2 Eth0 */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_43, CSL_XBAR_PRUSS1_IRQ_HOST8);  /* link ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_44, CSL_XBAR_PRUSS1_IRQ_HOST2);  /* RX PKT ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_45, CSL_XBAR_PRUSS1_IRQ_HOST4);  /* TX PKT ISR */


    /* For PRU1 Eth1 */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_48, CSL_XBAR_PRUSS1_IRQ_HOST9);  /* link ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_49, CSL_XBAR_PRUSS1_IRQ_HOST3);  /* RX PKT ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_50, CSL_XBAR_PRUSS1_IRQ_HOST5);  /* TX PKT ISR */
    /*PRU1ETH 0 initializations*/
    ICSS_EMAC_testHandle = (ICSS_EmacHandle)malloc(sizeof(ICSS_EmacConfig));

    ICSS_EmacInitConfig* icss_emacTestInitCfg0;
    icss_emacTestInitCfg0 = (ICSS_EmacInitConfig*)malloc(sizeof(ICSS_EmacInitConfig));
    icss_emacTestInitCfg0->phyAddr[0]=0;
    icss_emacTestInitCfg0->portMask = ICSS_EMAC_MODE_MAC1;
    icss_emacTestInitCfg0->ethPrioQueue = ICSS_EMAC_QUEUE1;
    icss_emacTestInitCfg0->halfDuplexEnable = 1;
    icss_emacTestInitCfg0->enableIntrPacing = ICSS_EMAC_ENABLE_PACING;
    icss_emacTestInitCfg0->ICSS_EmacIntrPacingMode = ICSS_EMAC_INTR_PACING_MODE1;
    icss_emacTestInitCfg0->pacingThreshold = 100;
    icss_emacTestInitCfg0->learningEn = 0;
    ICSS_EMAC_testSocCtrlGetPortMacAddr(0,ICSS_EMAC_testLclMac);
    icss_emacTestInitCfg0->macId = ICSS_EMAC_testLclMac;

    ICSS_EMAC_testDrvInit(ICSS_EMAC_testHandle,1);

    /*    Enable interrupt mode for TTS Cyclic Packet    */
    icss_emacTestInitCfg0->ICSS_EmacTTSEnableCycPktInterrupt = ICSS_EMAC_TTS_CYC_INTERRUPT_ENABLE;


    icss_emacTestInitCfg0->linkIntNum=43;
    icss_emacTestInitCfg0->rxIntNum = 44;
    icss_emacTestInitCfg0->txIntNum = 45;


    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->pruIcssHandle = ICSS_EMAC_testPruIcssHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->emacInitcfg = icss_emacTestInitCfg0;

    ICSS_EmacInit(ICSS_EMAC_testHandle,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    /* Test ICSS_EmacDeInit */
    ICSS_EmacDeInit(ICSS_EMAC_testHandle, ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    ICSS_EmacInit(ICSS_EMAC_testHandle,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);

    ICSS_EmacRegisterPort0ISRCallback(ICSS_EMAC_testHandle, (ICSS_EmacCallBack)ICSS_EMAC_testLinkIsrCb, (void*)ICSS_EMAC_TEST_PRU1ETH0);
    ICSS_EmacRegisterHwIntRx(ICSS_EMAC_testHandle, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackRxPacket);
    ICSS_EmacRegisterHwIntTx(ICSS_EMAC_testHandle, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackTxComplete);
    ICSS_EmacRegisterHwIntTTSCyc(ICSS_EMAC_testHandle, (ICSS_EmacCallBack)ICSS_EMAC_testTtsCycPort1Callback);

    /*PRU1 ETH1  Initializations*/
    ICSS_EMAC_testHandle1 = (ICSS_EmacHandle)malloc(sizeof(ICSS_EmacConfig));

    ICSS_EmacInitConfig* icss_emacTestInitCfg1;
    icss_emacTestInitCfg1 = (ICSS_EmacInitConfig*)malloc(sizeof(ICSS_EmacInitConfig));
    icss_emacTestInitCfg1->phyAddr[0]= 1;
    icss_emacTestInitCfg1->portMask = ICSS_EMAC_MODE_MAC2;
    icss_emacTestInitCfg1->ethPrioQueue = ICSS_EMAC_QUEUE3;
    icss_emacTestInitCfg1->enableIntrPacing = ICSS_EMAC_DISABLE_PACING;
    icss_emacTestInitCfg1->pacingThreshold = 100;
    icss_emacTestInitCfg1->learningEn = 0;
    ICSS_EMAC_testSocCtrlGetPortMacAddr(1,ICSS_EMAC_testLclMac1);
    icss_emacTestInitCfg1->macId = ICSS_EMAC_testLclMac1;

    ICSS_EMAC_testDrvInit(ICSS_EMAC_testHandle1, 1);

    /*    Enable interrupt mode for TTS Cyclic Packet    */
    icss_emacTestInitCfg1->ICSS_EmacTTSEnableCycPktInterrupt = ICSS_EMAC_TTS_CYC_INTERRUPT_ENABLE;

    icss_emacTestInitCfg1->linkIntNum=48;
    icss_emacTestInitCfg1->rxIntNum = 49;
    icss_emacTestInitCfg1->txIntNum = 50;

    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->pruIcssHandle = ICSS_EMAC_testPruIcssHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->emacInitcfg = icss_emacTestInitCfg1;

    ICSS_EmacInit(ICSS_EMAC_testHandle1,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
    /* Test ICSS_EmacDeInit */
    ICSS_EmacDeInit(ICSS_EMAC_testHandle1, ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
    ICSS_EmacInit(ICSS_EMAC_testHandle1,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);

    ICSS_EmacRegisterPort1ISRCallback(ICSS_EMAC_testHandle1, (ICSS_EmacCallBack)ICSS_EMAC_testLinkIsrCb, (void*)ICSS_EMAC_TEST_PRU1ETH1);
    ICSS_EmacRegisterHwIntRx(ICSS_EMAC_testHandle1,(ICSS_EmacCallBack) ICSS_EMAC_testCallbackRxPacket1);
    ICSS_EmacRegisterHwIntTx(ICSS_EMAC_testHandle1,(ICSS_EmacCallBack) ICSS_EMAC_testCallbackTxComplete);
    ICSS_EmacRegisterHwIntTTSCyc(ICSS_EMAC_testHandle1, (ICSS_EmacCallBack)ICSS_EMAC_testTtsCycPort2Callback);

    Task_Params_init(&taskParams);
    //taskParams.priority = 15;
    taskParams.instance->name = (char*)"port0_rxTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->rxTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsRxTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->rxTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    //taskParams.priority = 15;
    taskParams.instance->name = (char*)"port0_txTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->txTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTxTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->txTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port0_linkTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->linkTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsLinkTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->linkTaskHandle==NULL)
    {
        return -2;
    }
    Task_Params_init(&taskParams);
    //taskParams.priority = 15;
    taskParams.instance->name = (char*)"port0_ttsCycTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->ttsCycTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTTSCycTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle->object)->ttsCycTaskHandle==NULL)
    {
        return -2;
    }
    
    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port1_rxTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->rxTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsRxTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->rxTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port1_txTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->txTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTxTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->txTaskHandle==NULL)
    {
        return -2;
    }


    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port1_linkTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->linkTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsLinkTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->linkTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port1_ttsCycTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->ttsCycTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTTSCycTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle1->object)->ttsCycTaskHandle==NULL)
    {
        return -2;
    }

    ICSS_EMAC_testInterruptInit(ICSS_EMAC_testHandle);
    ICSS_EMAC_testInterruptInit(ICSS_EMAC_testHandle1);

    ICSS_EMAC_testInterruptEnable(ICSS_EMAC_testHandle1);
    ICSS_EMAC_testInterruptEnable(ICSS_EMAC_testHandle);
    return 0;
}

int32_t ICSS_EMAC_testPruIcssInstance2Setup(void)
{
    Task_Params taskParams;
    PRUICSS_IntcInitData pruss_intc_initdata = PRUSS_INTC_INITDATA;
     ICSS_EmacInitConfig* icss_emacTestInitCfg0;
     ICSS_EmacInitConfig* icss_emacTestInitCfg1;
    PRUICSS_Config  *cfg;

     int32_t ret  = PRUICSS_socGetInitCfg(&cfg);
    if (ret  != PRUICSS_RETURN_SUCCESS)
        return (ret);

    ICSS_EMAC_testPruIcssHandle2 = PRUICSS_create((PRUICSS_Config*)cfg,PRUICCSS_INSTANCE_TWO);


    /* For PRU2 Eth0 */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_76, CSL_XBAR_PRUSS2_IRQ_HOST8);  /* link ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_77, CSL_XBAR_PRUSS2_IRQ_HOST2);  /* RX PKT ISR */
    CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_78, CSL_XBAR_PRUSS2_IRQ_HOST4);  /* TX PKT ISR */

    if (ICSS_EMAC_testEvmType  == ICSS_EMAC_TEST_BOARD_IDKAM572x)
    {
        /* For PRU2 Eth1 */
        CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_89, CSL_XBAR_PRUSS2_IRQ_HOST9);  /* link ISR */
        CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_90, CSL_XBAR_PRUSS2_IRQ_HOST3);  /* RX PKT ISR */
        CSL_xbarDspIrqConfigure(1, CSL_XBAR_INST_DSP1_IRQ_93, CSL_XBAR_PRUSS2_IRQ_HOST5);  /* TX PKT ISR */
    }

    /*PRU2 ETH0 initializations*/
    ICSS_EMAC_testHandle2 = (ICSS_EmacHandle)malloc(sizeof(ICSS_EmacConfig));

    icss_emacTestInitCfg0 = (ICSS_EmacInitConfig*)malloc(sizeof(ICSS_EmacInitConfig));
    if ((ICSS_EMAC_testHandle2 == NULL) || (icss_emacTestInitCfg0 == NULL))
    {
        UART_printf("main: malloc returned null\n");
    }
    icss_emacTestInitCfg0->phyAddr[0]=0;
    icss_emacTestInitCfg0->portMask = ICSS_EMAC_MODE_MAC1;
    icss_emacTestInitCfg0->ethPrioQueue = ICSS_EMAC_QUEUE1;
    icss_emacTestInitCfg0->halfDuplexEnable = 1;
    icss_emacTestInitCfg0->enableIntrPacing = ICSS_EMAC_DISABLE_PACING;//ICSS_EMAC_ENABLE_PACING;
    icss_emacTestInitCfg0->ICSS_EmacIntrPacingMode = ICSS_EMAC_INTR_PACING_MODE1;
    icss_emacTestInitCfg0->pacingThreshold = 100;
    icss_emacTestInitCfg0->learningEn = 0;
    icss_emacTestInitCfg0->macId = ICSS_EMAC_testLclMac2;

    ICSS_EMAC_testDrvInit(ICSS_EMAC_testHandle2,2);

    /*Enable interrupt mode for TTS Cyclic Packet    */
    icss_emacTestInitCfg0->ICSS_EmacTTSEnableCycPktInterrupt = ICSS_EMAC_TTS_CYC_INTERRUPT_ENABLE;

    icss_emacTestInitCfg0->linkIntNum=76;
    icss_emacTestInitCfg0->rxIntNum = 77;
    icss_emacTestInitCfg0->txIntNum = 78;
    
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->pruIcssHandle = ICSS_EMAC_testPruIcssHandle2;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->emacInitcfg = icss_emacTestInitCfg0;



    ICSS_EmacInit(ICSS_EMAC_testHandle2,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    /* Test ICSS_EmacDeInit */
    ICSS_EmacDeInit(ICSS_EMAC_testHandle2, ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);
    ICSS_EmacInit(ICSS_EMAC_testHandle2,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC1|ICSS_EMAC_MODE_DUALMAC);

    ICSS_EmacRegisterPort0ISRCallback(ICSS_EMAC_testHandle2, (ICSS_EmacCallBack)ICSS_EMAC_testLinkIsrCb, (void*)ICSS_EMAC_TEST_PRU2ETH0);
    ICSS_EmacRegisterHwIntRx(ICSS_EMAC_testHandle2, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackRxPacket2);
    ICSS_EmacRegisterHwIntTx(ICSS_EMAC_testHandle2, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackTxComplete);
    ICSS_EmacRegisterHwIntTTSCyc(ICSS_EMAC_testHandle2, (ICSS_EmacCallBack)ICSS_EMAC_testTtsCycPort1Callback);

        /*PRU2 ETH1 initializations*/
        ICSS_EMAC_testHandle3 = (ICSS_EmacHandle)malloc(sizeof(ICSS_EmacConfig));
        
        icss_emacTestInitCfg1 = (ICSS_EmacInitConfig*)malloc(sizeof(ICSS_EmacInitConfig));
    if ((ICSS_EMAC_testHandle3 == NULL) || (icss_emacTestInitCfg1 == NULL))
    {
        UART_printf("main: malloc returned null\n");
    }
        icss_emacTestInitCfg1->phyAddr[0]= 1;
    
        icss_emacTestInitCfg1->portMask = ICSS_EMAC_MODE_MAC2;
        icss_emacTestInitCfg1->ethPrioQueue = ICSS_EMAC_QUEUE3;
        icss_emacTestInitCfg1->enableIntrPacing = ICSS_EMAC_DISABLE_PACING;
        icss_emacTestInitCfg1->pacingThreshold = 100;
    
        icss_emacTestInitCfg1->learningEn = 0;
    
    
       icss_emacTestInitCfg1->macId = ICSS_EMAC_testLclMac3;
    
        ICSS_EMAC_testDrvInit(ICSS_EMAC_testHandle3, 2);
    
        /*    Enable interrupt mode for TTS Cyclic Packet    */
        icss_emacTestInitCfg1->ICSS_EmacTTSEnableCycPktInterrupt = ICSS_EMAC_TTS_CYC_INTERRUPT_ENABLE;

        icss_emacTestInitCfg1->linkIntNum=89;
        icss_emacTestInitCfg1->rxIntNum = 90;
        icss_emacTestInitCfg1->txIntNum=93;

        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->pruIcssHandle = ICSS_EMAC_testPruIcssHandle2;
        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->emacInitcfg = icss_emacTestInitCfg1;
    
        ICSS_EmacInit(ICSS_EMAC_testHandle3,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
        /* Test ICSS_EmacDeInit */
        ICSS_EmacDeInit(ICSS_EMAC_testHandle3, ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
        ICSS_EmacInit(ICSS_EMAC_testHandle3,&pruss_intc_initdata,ICSS_EMAC_MODE_MAC2|ICSS_EMAC_MODE_DUALMAC);
    
        ICSS_EmacRegisterPort1ISRCallback(ICSS_EMAC_testHandle3, (ICSS_EmacCallBack)ICSS_EMAC_testLinkIsrCb, (void*)ICSS_EMAC_TEST_PRU2ETH1);
        ICSS_EmacRegisterHwIntRx(ICSS_EMAC_testHandle3, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackRxPacket3);
        ICSS_EmacRegisterHwIntTx(ICSS_EMAC_testHandle3, (ICSS_EmacCallBack)ICSS_EMAC_testCallbackTxComplete);
        ICSS_EmacRegisterHwIntTTSCyc(ICSS_EMAC_testHandle3, (ICSS_EmacCallBack)ICSS_EMAC_testTtsCycPort2Callback);
 
    
    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port2_rxTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle2;

    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->rxTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsRxTaskFnc, &taskParams, NULL);
    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->rxTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port2_txTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle2;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->txTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTxTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->txTaskHandle==NULL)
    {
        return -2;
    }


    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port2_linkTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle1;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->linkTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsLinkTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->linkTaskHandle==NULL)
    {
        return -2;
    }

    Task_Params_init(&taskParams);
    taskParams.instance->name = (char*)"port2_ttsCycTaskFnc";
    taskParams.stackSize = 0x1000;
    taskParams.arg0 = (UArg)ICSS_EMAC_testHandle2;
    ((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->ttsCycTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTTSCycTaskFnc, &taskParams, NULL);

    if(((ICSS_EmacObject*)ICSS_EMAC_testHandle2->object)->ttsCycTaskHandle==NULL)
    {
        return -2;
    }

    if (ICSS_EMAC_testEvmType  == ICSS_EMAC_TEST_BOARD_IDKAM572x)
    {
        Task_Params_init(&taskParams);
        taskParams.instance->name = (char*)"port3_rxTaskFnc";
        taskParams.stackSize = 0x1000;
        taskParams.arg0 = (UArg)ICSS_EMAC_testHandle3;
    
        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->rxTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsRxTaskFnc, &taskParams, NULL);
        if(((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->rxTaskHandle==NULL)
        {
            return -2;
        }

        Task_Params_init(&taskParams);
        taskParams.instance->name = (char*)"port3_txTaskFnc";
        taskParams.stackSize = 0x1000;
        taskParams.arg0 = (UArg)ICSS_EMAC_testHandle3;
        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->txTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTxTaskFnc, &taskParams, NULL);
    
    
        if(((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->txTaskHandle==NULL)
        {
            return -2;
        }


        Task_Params_init(&taskParams);
        taskParams.instance->name = (char*)"port3_linkTaskFnc";
        taskParams.stackSize = 0x1000;
        taskParams.arg0 = (UArg)ICSS_EMAC_testHandle3;
        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->linkTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsLinkTaskFnc, &taskParams, NULL);
    
        if(((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->linkTaskHandle==NULL)
        {
            return -2;
        }


        Task_Params_init(&taskParams);
        taskParams.instance->name = (char*)"port3_ttsCycTaskFnc";
        taskParams.stackSize = 0x1000;
        taskParams.arg0 = (UArg)ICSS_EMAC_testHandle3;
        ((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->ttsCycTaskHandle = Task_create((ti_sysbios_knl_Task_FuncPtr)ICSS_EMacOsTTSCycTaskFnc, &taskParams, NULL);

        if(((ICSS_EmacObject*)ICSS_EMAC_testHandle3->object)->ttsCycTaskHandle==NULL)
        {
            return -2;
        }
    }

    ICSS_EMAC_testInterruptInit(ICSS_EMAC_testHandle2);

    if (ICSS_EMAC_testEvmType  == ICSS_EMAC_TEST_BOARD_IDKAM572x)
    {
        ICSS_EMAC_testInterruptInit(ICSS_EMAC_testHandle3);
    }




    return 0;
}

    Error_Block eb;
    Task_Params taskParams;
int main()
{
    Board_IDInfo info;
    uint32_t numPorts = 4;
    Board_STATUS boardInitStatus =0;

    Board_initCfg cfg = BOARD_INIT_UART_STDIO | BOARD_INIT_PINMUX_CONFIG | BOARD_INIT_MODULE_CLOCK | BOARD_INIT_ICSS_ETH_PHY;

    Error_init(&eb);
    boardInitStatus  = Board_init(cfg);

    if (boardInitStatus !=0)
    {
        UART_printf("Board_init failure\n");
        return(0);
    }

#ifdef _TMS320C6X
    /* Clear pre-fetch bit in MAR to make OCMC un-cacheable */
    Cache_setMar((xdc_Ptr)0x4b000000U, 0x10000U, 0);
    Cache_setMar((xdc_Ptr)0x40000000U, 0x10000U, 0);
#endif

    memset(&info, 0,sizeof(Board_IDInfo));
    Board_getIDInfo(&info);
    UART_printf("boardName: %s\n", info.boardName);
    if (!(strcmp(info.boardName, "AM571IDK")))
    {
        ICSS_EMAC_testEvmType =ICSS_EMAC_TEST_BOARD_IDKAM571x;
        UART_printf("board type is AM571IDK, numPorts: %d\n", numPorts);
    }
    else
    {
        ICSS_EMAC_testEvmType =ICSS_EMAC_TEST_BOARD_IDKAM572x;
        UART_printf("board type is AM572IDK, numPorts: %d\n", numPorts);
     }

    Task_Params_init(&taskParams);
    taskParams.instance->name = "SwitchTask1";
    Task_create(ICSS_EMAC_testTaskPruss1, &taskParams, &eb);


    Task_Params_init(&taskParams);
    taskParams.instance->name = "SwitchTask2";
    Task_create(ICSS_EMAC_testTaskPruss2, &taskParams, &eb);


    Task_Params_init(&taskParams);
    taskParams.instance->name = "Port1TxTask";
    Task_create(ICSS_EMAC_testPort1TxTask, &taskParams, &eb);

    Task_Params_init(&taskParams);
    taskParams.instance->name = "Port2TxTask";
    Task_create(ICSS_EMAC_testPort2TxTask, &taskParams, &eb);

    ICSS_EMAC_testPruIcssInstance1Setup();
    ICSS_EMAC_testPruIcssInstance2Setup();

    BIOS_start();
    return(0);
}








