/*
 * Can.h
 *
 *	Created on: 04.02.2016
 *		Author: richterr
 *
 *	Interface of CAN (Common part)
 *
 *	This class based on "device_support\F2837xD\v160\F2837xD_common\driverlib\can.h"
 *
 * ###########################################################################
 * 
 *  FILE:   can.h
 * 
 *  TITLE:  Defines and Macros for the CAN controller.
 * 
 * ###########################################################################
 *  $TI Release: F2837xD Support Library v160 $
 *  $Release Date: Mon Jun 15 13:36:23 CDT 2015 $
 *  $Copyright: Copyright (C) 2013-2015 Texas Instruments Incorporated -
 *      http://www.ti.com/ ALL RIGHTS RESERVED $
 * ###########################################################################
 *
 */

#ifndef __can_h__
#define __can_h__

// *************************************************************************
// Includes
// *************************************************************************
#include "hw_can.h"
#include "hw_types.h"

// *************************************************************************
// Definitions
// *************************************************************************
//~ #define CAN_INDEX_TO_BASE(idx)     ((idx == 0) ? CAN_A_BASE : CAN_B_BASE)

//~ #define CAN_INDEX_TO_MSG_RAM_BASE(idx)     ((idx == 0) ? CAN_A_MSG_RAM : CAN_B_MSG_RAM)

//****************************************************************************
// These are the Defines to select CAN pin muxing when calling the functions
// ConfigCanPinMuxing(), ConfigGpioCanA() & ConfigGpioCanB() in F2837x_Can.c
//****************************************************************************
//~ #define CAN_A_GPIO4_GPIO5       1   //switch case 1
//~ #define CAN_A_GPIO19_GPIO18     2   //switch case 2
//~ #define CAN_A_GPIO31_GPIO30     3   //switch case 3
//~ #define CAN_A_GPIO37_GPIO36     4   //switch case 4
//~ #define CAN_A_GPIO63_GPIO62     5   //switch case 5
//~ #define CAN_A_GPIO71_GPIO70     6   //switch case 6

//~ #define CAN_B_GPIO6_GPIO7       1   //switch case 1
//~ #define CAN_B_GPIO8_GPIO10      2   //switch case 2
//~ #define CAN_B_GPIO12_GPIO13     3   //switch case 3
//~ #define CAN_B_GPIO16_GPIO17     4   //switch case 4
//~ #define CAN_B_GPIO20_GPIO21     5   //switch case 5
//~ #define CAN_B_GPIO38_GPIO39     6   //switch case 6
//~ #define CAN_B_GPIO72_GPIO73     7   //switch case 7

//*****************************************************************************
// Miscellaneous defines for Message ID Types
//*****************************************************************************

//! The structure used for encapsulating all the items associated with a CAN message object in the CAN controller.
typedef struct {
	//! The CAN message identifier used for 11 or 29 bit identifiers.
	Uint32 MsgID;
	//! The message identifier mask used when identifier filtering is enabled.
	Uint32 MsgIDMask;
	//! This value holds various status flags and settings specified by tCANObjFlags.
	Uint32 Flags;
	//! This value is the number of bytes of data in the message object.
	Uint32 MsgLen;
	//! This is a pointer to the message object's data.
	unsigned char* pMsgData;
} tCANMsgObject;

//! This structure is used for encapsulating the values associated with setting up the bit timing for a CAN controller.
//!  The structure is used when calling the CANGetBitTiming and CANSetBitTiming functions.
typedef struct {
	//! This value holds the sum of the Synchronization, Propagation, and Phase Buffer 1 segments, measured in time quanta.
	//!  The valid values for this setting range from 2 to 16.
	Uint16  SyncPropPhase1Seg;
	//! This value holds the Phase Buffer 2 segment in time quanta.  The valid values for this setting range from 1 to 8.
	Uint16  Phase2Seg;
	//! This value holds the Resynchronization Jump Width in time quanta.  The valid values for this setting range from 1 to 4.
	Uint16  SJW;
	//! This value holds the CAN_CLK divider used to determine time quanta.
	//!  The valid values for this setting range from 1 to 1023.
	Uint16  QuantumPrescaler;
} tCANBitClkParms;

//! This data type is used to identify the interrupt status register.  This is used when calling the GetInterruptStatus() function.
typedef enum {
	//! Read the CAN interrupt status information.
	IntStatusCause,
	//! Read a message object's interrupt status.
	IntStatusObject
} tCANIntStsReg;

//*****************************************************************************
//! This data type is used to identify which of several status registers to read when calling the GetStatus() function.
//*****************************************************************************
typedef enum {
	//! Read the full CAN controller status.
	ControllerStatus,
	//! Read the full 32-bit mask of message objects with a transmit request set.
	TxRequest,
	//! Read the full 32-bit mask of message objects with new data available.
	NewData,
	//! Read the full 32-bit mask of message objects that are enabled.
	MsgValue
} tCANStsReg;

//! This definition is used to determine the type of message object that will be set up via a call to the SetMessage() API.
typedef enum {
	//! Transmit message object.
	TypeTX,
	//! Transmit remote request message object
	TypeTXRemote,
	//! Receive message object.
	TypeRX,
	//! Receive remote request message object.
	TypeRXRemote,
	//! Remote frame receive remote, with auto-transmit message object.
	TypeRXTXRemote
} tMsgObjType;

// These are the flags used by the tCANMsgObject.Flags value when calling the SetMessage() and GetMessage() functions.
// -----------------------------------------------------------------------------
typedef enum {
	//! This definition is used with the tCANMsgObject Flags value and indicates that transmit interrupts
	//!  should be enabled, or are enabled.
	TxIntEnable = 0x00000001,
	//! This indicates that receive interrupts should be enabled, or are enabled.
	RxIntEnable = 0x00000002,
	//! This indicates that a message object will use or is using an extended identifier.
	ExtendedId = 0x00000004,
	//! This indicates that a message object will use or is using filtering based on the object's message identifier.
	UseIdFilter = 0x00000008,
	//! This indicates that new data was available in the message object.
	NewDataAvailable = 0x00000080,
	//! This indicates that data was lost since this message object was last read.
	DataLost = 0x00000100,
	//! This indicates that a message object will use or is using filtering based on the direction of the transfer.
	//!  If the direction filtering is used, then ID filtering must also be enabled.
	UseDirFilter = 0x00000010 | UseIdFilter,
	//! This indicates that a message object will use or is using message identifier filtering based on the extended identifier.
	//!  If the extended identifier filtering is used, then ID filtering must also be enabled.
	UseExtFilter = 0x00000020 | UseIdFilter,
	//! This indicates that a message object is a remote frame.
	RemoteFrame = 0x00000040,
	//! This indicates that this message object is part of a FIFO structure and not the final message object in a FIFO.
	Fifo = 0x00000200,
	//! This indicates that a message object has no flags set.
	NoFlags = 0x00000000,
	//! This define is used with the flag values to allow checking only status flags and not configuration flags.
	StatusMask = NewDataAvailable | DataLost
} tCanMsgObjFlagType;

// Define selectable CAN
typedef enum {
	// CANbus A
	CANbusA,
	// CANbus B
	CANbusB,
} tCANbusType;

// The following enumeration contains all error or status indicators that can be returned when calling the GetStatus() function.
// TODO: In eine Subklasse wandeln. Siehe SPI_Flash class FlashCommonCmds
// -----------------------------------------------------------------------------
typedef enum {
	//! CAN controller is in local power down mode.
	PowerDown = 0x00000400,
	//! CAN controller has initiated a system wakeup.
	WakeUp = 0x00000200,
	//! CAN controller has detected a parity error.
	ParityError = 0x00000100,
	//! CAN controller has entered a Bus Off state.
	CAN_STATUS_BUS_OFF = 0x00000080,
	//! CAN controller error level has reached warning level.
	ErrorLevelWarning = 0x00000040,
	//! CAN controller error level has reached error passive level.
	ErrorLevelPassive = 0x00000020,
	//! A message was received successfully since the last read of this status.
	RXSuccessful = 0x00000010,
	//! A message was transmitted successfully since the last read of this status.
	TXSuccessful = 0x00000008,
	//! This is the mask for the last error code field.
	MaskLastErrorCode = 0x00000007,
	//! There was no error.
	NoError = 0x00000000,
	//! A bit stuffing error has occurred.
	StuffingError = 0x00000001,
	//! A formatting error has occurred.
	FormattingError = 0x00000002,
	//! An acknowledge error has occurred.
	AcknowledgeError = 0x00000003,
	//! The bus remained a bit level of 1 for longer than is allowed.
	BitlLevel1Error = 0x00000004,
	//! The bus remained a bit level of 0 for longer than is allowed.
	BitlLevel0Error = 0x00000005,
	//! A CRC error has occurred.
	CRCError = 0x00000006
} tCanStatusType;

// *************************************************************************
// Declarations
// *************************************************************************
class CAN {
public:
	CAN(tCANbusType CANbus);
	virtual ~CAN();
	Uint32 GetBase(void);
	char GetBusType(void);
	void SelectClkSource(const Uint16 Source);
	void GetBitTiming( tCANBitClkParms* const pClkParms);
	void SetBitTiming(const tCANBitClkParms* const pClkParms);
	Uint32 SetBitRate(const Uint32 SourceClock, const Uint32 BitRate);
	void Disable(void);
	void Enable(void);
	bool GetErrorCounterLimit( Uint32* const pRxCount, Uint32* const pTxCount);
	void Init(void);
	void ClearInterrupt(const Uint32 IntClr);
	void DisableInterrupt(const Uint32 IntFlags);
	void EnableInterrupt(const Uint32 IntFlags);
	void RegisterInterrupt( unsigned char ucIntNumber, void (*pfnHandler)(void));
	Uint32 GetInterruptStatus(const tCANIntStsReg eIntStsReg);
	void UnregisterInterrupt(const unsigned char IntNumber);
	void ClearMessage(const Uint32 ObjID);
	void GetMessage(const Uint32 ObjID, tCANMsgObject* const pMsgObject, const bool bClrPendingInt);
	void SetMessage(const Uint32 ObjID, const tCANMsgObject* const pMsgObject, const tMsgObjType eMsgType);
	bool GetRetry(void);
	void SetRetry(const bool bAutoRetry);
	Uint32 GetStatus(const tCANStsReg eStatusReg);
	void EnableGlobalInt(const Uint32 IntFlags);
	void DisableGlobalInt(const Uint32 IntFlags);
	void ClearGlobalInt(const Uint32 IntFlags);
	bool CANGlobalIntstatusGet(const Uint32 IntFlags);

private:
	void WriteDataReg(unsigned char* const pData, Uint32* const pRegister, const int16 Size);
	int32 GetIntNumber(const unsigned char Number);
	bool GetBaseValid(void);

	Uint32 Base;		// Base address of the CAN controller

	// This is the maximum number that can be stored as an 11bit Message identifier.
	static const Uint16 Max11bitMsgID = 0x7ff;
	// Bit mask
	static const Uint32 RegisterMaskWord = 0xFFFFU;

	// This is used as the loop delay for accessing the CAN controller registers.
	// -----------------------------------------------------------------------------
	// The maximum CAN bit timing divisor is 13.
	static const Uint16 MaxBitDivisor = 13;
	// The minimum CAN bit timing divisor is 5.
	static const Uint16 MinBitDivisor = 5;
	// The maximum CAN pre-divisor is 1024.
	static const Uint16 MaxPreDivisor = 1024;
	// The minimum CAN pre-divisor is 1.
	static const Uint16 MinPreDivisor = 1;

	// These definitions are used to specify interrupt sources to EnableInterrupt() and DisableInterrupt().
	// -----------------------------------------------------------------------------
	//! This flag is used to allow a CAN controller to generate error interrupts.
	static const Uint32 EnableErrorIRQ = 0x00000008;
	//! This flag is used to allow a CAN controller to generate status interrupts.
	static const Uint32 EnableStatusIRQ = 0x00000004;
	//! This flag is used to allow a CAN controller to generate interrupts on interrupt line 0
	static const Uint32 EnableLine0IRQ = 0x00000002;
	//! This flag is used to allow a CAN controller to generate interrupts on interrupt line 1
	static const Uint32 EnableLine1IRQ = 0x00020000;
	// Defined to maintain compatibility with Stellaris Examples
	static const Uint32 MasterInterrupt = EnableLine0IRQ;

	// These definitions are used to specify the clock source to SelectClkSource()
	// -----------------------------------------------------------------------------
//~	//! This flag is used to clock the CAN controller Selected CPU SYSCLKOUT (CPU1.Sysclk or CPU2.Sysclk).
//~	static const Uint16 ClkCpuSysclkOut = 0;	// PERx.SYSCLK (default on reset)
//~	//! This flag is used to clock the CAN controller with the X1/X2 oscillator clock.
//~	static const Uint16 ClkExternalOsc = 1;		// External Oscillator (XTAL)
//~	//! This flag is used to clock the CAN controller with the clock from AUXCLKIN (from GPIO)
//~	static const Uint16 ClkAuxClkIn = 2;		// AUXCLKIN (from GPIO)

	// The following macros are added for the new Global Interrupt EN/FLG/CLR register
	// -----------------------------------------------------------------------------
//~	//CANINT0 global interrupt bit
//~	static const Uint32 CAN_GLOBAL_INT_CANINT0 = 0x00000001;
//~	//CANINT1 global interrupt bit
//~	static const Uint32 CAN_GLOBAL_INT_CANINT1 = 0x00000002;

	// The following macros are missing in hw_can.h because of scripting but driverlib can.c needs them
	// -----------------------------------------------------------------------------
	static const Uint16 CAN_INT_INT0ID_STATUS = 0x8000;
	static const Uint16 CAN_IF1ARB_STD_ID_S = 18;
	static const Uint32 CAN_IF1ARB_STD_ID_M = 0x1FFC0000;	// Standard Message Identifier
	static const Uint16 CAN_IF2ARB_STD_ID_S = 18;
	static const Uint32 CAN_IF2ARB_STD_ID_M = 0x1FFC0000;	// Standard Message Identifier

};

//*****************************************************************************
// Close the Doxygen group.
//! @}
//*****************************************************************************

// project global object of CANbus A and CANbus B
extern CAN goCanA;
extern CAN goCanB;

#endif //  __can_h__
