/*
 * FM25V05_Flash.cpp
 *
 *  Created on: 12.01.2016
 *      Author: richterr
 *
 *	Interface of Cypress FM25V05 SPI Flash
 *
 */

// *************************************************************************
// Includes
// *************************************************************************
#include <sstream>
#include <string>

#include <FM25V05_Flash.h>
#include "gpio.h"

// *************************************************************************
// Definitions of Serial FLASH Commands and FLASH Parameters
// *************************************************************************
// Special FM25V05 Serial Flash parameters
// -------------------------------------------------------------------------
// Type: FM25V05; 512-Kbit ferroelectric random access memory (F-RAM); Clock frequency up to 40 MHz
// Document: N:\pA_Datenblaetter\IC_s_allg\.. TODO ..\Cypress_FM25V05_SPI-FRAM_512Kbit.pdf (Revision F - December 2015)
//  512Kbit address range -> 512x1024/8 = 65536 Bytes
#define   FLASH_TOTAL_BYTES		0x10000

// CS timing parameters (in s)
// Document: N:\pA_Datenblaetter\IC_s_allg\.. TODO ..\Cypress_FM25V05_SPI-FRAM_512Kbit.pdf (Revision F - December 2015)
//	Chapter: AC Switching Characteristics
#define   CS_HIGH		0.010L		// [s]
#define   CS_HOLD		0.010L		// Chip hold setup t_CSH [s]
#define   CS_SETUP		0.010L		// Chip select setup t_CSU [s]

// FM25V05 SPI Aonfiguration Settings
// These are set in the InitSpi() function
//
// SPI Baud Rate Register
// Reference: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual (spruhm8e.pdf); Chapter 18.5.2.4
// LSPCLK = SYSCLK / LSP_CLK_DIV = 200MHz / 2 = 100.0MHz
// On SPI_BRR = 0, 1 or 2 -> BRR = LSPCLK / 4 = 100.0 MHz / 4 = 25.0 MHz
// On SPI_BRR = 3, 4, ..., 127 -> BRR = LSPCLK / (SPI_BRR + 1)
#define SPI_BAUD  0x0001		// SPI Baud Rate

// *************************************************************************
// Definitions of methods
// *************************************************************************
FM25V05_Flash::FM25V05_Flash() : SpiFlash(FLASH_TOTAL_BYTES, SPI_A, CS_HOLD, CS_HIGH, CS_SETUP)
{

}

FM25V05_Flash::~FM25V05_Flash() {
	// TODO Auto-generated destructor stub
}

/***************** doxygen like ******************************//**
 * Method: InitSPI
 * In: ---
 * Out: Serial FLASH API Status Messages
 * Description: Initialize SPI A.
// **************************************************************/
FLASH_API_STATUS_MESSAGES FM25V05_Flash::InitSPI(void)
{
	///   Logger_info0("Initialize SPI A for FM25V05 Flash.");
	   EALLOW;
	   CpuSysRegs.PCLKCR8.bit.SPI_C = 1;  // Enable the SPI A clock
	   //CpuSysRegs.LOSPCP.all = LSPCLK_DIV;

	   EDIS;

	   SpiaRegs.SPICCR.bit.SPISWRESET = 0;	// Perform SPI Software Reset
	   SpiaRegs.SPIFFTX.all = 0x8000;		// SPI Transmit FIFO enable
	   SpiaRegs.SPICCR.bit.SPICHAR = 0x7;	// Transfer 8 Bit
	   SpiaRegs.SPICCR.bit.SPILBK = 0;		// SPI loopback mode disabled
	   SpiaRegs.SPICTL.all = 0x000E;		// SPI ALK signal delayed one half-cycle, config as master
	   SpiaRegs.SPIBRR.bit.SPI_BIT_RATE = SPI_BAUD;		// Baudrate, for 0,1 or 2 = LSPCLK/4
	   SpiaRegs.SPICCR.bit.SPISWRESET = 1;	// Disable SPI Software Reset
	   SetChipSelectHigh();					// Set CSn high (inactive)

	   return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: VerifyDeviceId
 * In: Device ID from SPI device (see FM25V05 specific ReadDeviceId())
 * Out: Serial FLASH API Status Messages
 * Description: Verify if ID from Flash is same as in datasheet.
 *	 *
 *	|	Chip		|	Manufacturer	|	Manufacturer ID (last 2 bytes)	|	Produkt ID	|
 *	|_______________|___________________|___________________________________|_______________|
 *	|				|					|									|				|
 *	|	FM25V05		|	Cypress			|	0x7FC2							|	0x23		|
// **************************************************************/
FLASH_API_STATUS_MESSAGES FM25V05_Flash::VerifyDeviceId(const Uint32 DevId)
{
	if (0x7FC223 == DevId) {			// Cypress FM25V05
		return(SUCCESSFUL);
	}
	return(FAIL_DEVICE_ID);
}

/***************** doxygen like ******************************//**
 * Method: ReadDeviceId (FM25V05 specific)
 * In: ---
 * Out: Code of manufacturer ID and product ID
 * Description:
 *	Manufacturer ID is NOT JEDEC coded. The FM25V05 delivers a non standard device ID.
 *	See FM25V05 data sheet: N:\pA_Datenblaetter\IC_s_allg\.. TODO ..\Cypress_FM25V05_SPI-FRAM_512Kbit.pdf (Revision F - December 2015)
 *	Chapter: Device ID
 *	9 bytes overall: Byte 9 (MSB) - 3 fix pattern as manufacturer ID; byte 2 Product ID; byte 1 obsolete Sub/Revision
// **************************************************************/
Uint32 FM25V05_Flash::ReadDeviceId(void)
{
	Uint32 ManufacturerId1 = 0;
	Uint16 ManufacturerId2 = 0;
	Uint16 ProductID = 0;

	SetChipSelectLow();
	SendCommand(CmdReadManDevID);

	// Get non standard answer
	// Ignore byte 9 to 5 (only 0x7F)
	for(Uint16 i = 0; i < 5; i++) {
		ReceiveByte();
	}
	// Use only last 2 bytes of manufacturer ID as manufacturer ID for verification
	ManufacturerId1 = ReceiveByte();
	ManufacturerId2 = ReceiveByte();
	ProductID = ReceiveByte();
	ReceiveByte();							// Get Sub/Revision byte. Without information at the moment
	SetChipSelectHigh();
	return((ManufacturerId1 << 16) | (ManufacturerId2 << 8) | ProductID);
///return((ProductID << 16) | (ManufacturerId2 << 8) | ManufacturerId1);
}
