/*
 * FPGA.cpp
 *
 *	Created on: 01.12.2015
 *		Author: richterr
 *
 *	Interface of FPGA
 *
 */

#include <F2837xD_Device.h> // DSP2837xD header file peripheral address definitions
#include <F2837xD_GlobalPrototypes.h>
#include <F2837xD_Gpio.h>
#include <F2837xD_Gpio_defines.h>
#include <F2837xD_Emif_defines.h>
#include "GPIO_pin_assignment.h"
#include "FPGA.h"

//*****************************************************************************
// Explicit Declarations
//*****************************************************************************
#ifdef __cplusplus
extern "C" {
#endif

extern void Emif2Initialize(void);

#ifdef __cplusplus
}
#endif /* extern "C" */


//*****************************************************************************
// Class
//*****************************************************************************
// Construtor of class
FPGA::FPGA()
{
	// Initialize global variables
	poCOM1 = NULL;
}

FPGA::~FPGA() {
	// Auto-generated destructor stub
}

// Initialization of EMIF2 interface
Uint16 FPGA::InitEMIF2()
{
	Uint16  ErrorFlags = 0;

// *****************************************************************************
// Configure FPGA communication
// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
// *****************************************************************************
// EMIF2 Configuration Registers: Chapter 25.5.4

	// reset EMIF2
	Emif2Initialize();
	//Configure to run EMIF2 on half Rate (EMIF2CLK = CPU1SYSCLK/2)
	EALLOW;
	ClkCfgRegs.PERCLKDIVSEL.bit.EMIF2CLKDIV = 0x1;
	EDIS;
	if (ClkCfgRegs.PERCLKDIVSEL.bit.EMIF2CLKDIV != 0x1)
	{
		  ErrorFlags = (1 << 0);
	}

	//Disable any Access Protection (CPU_FETCH allowed = 0 / CPU_WR allowed = 0 / DMA_WR allowed = 0)
	EALLOW;
	Emif2ConfigRegs.EMIF2ACCPROT0.all = 0x0;
	EDIS;
	if (Emif2ConfigRegs.EMIF2ACCPROT0.all != 0x0)
	{
		  ErrorFlags |= (1 << 2);
	}

	// Commit the configuration related to protection. Till this bit remains set
	// content of EMIF2ACCPROT0 register can't be changed.
	EALLOW;
	Emif2ConfigRegs.EMIF2COMMIT.all = 0x1;
	EDIS;
	if(Emif2ConfigRegs.EMIF2COMMIT.all != 0x1)
	{
		  ErrorFlags |= (1 << 3);
	}

	// Lock the configuration so that EMIF2COMMIT register can't be changed any more.
	EALLOW;
	Emif2ConfigRegs.EMIF2LOCK.all = 0x1;
	EDIS;
	if (Emif2ConfigRegs.EMIF2LOCK.all != 1)
	{
		  ErrorFlags |= (1 << 4);
	}

	// Enable EMIF2 clock
	EALLOW;
	CpuSysRegs.PCLKCR1.bit.EMIF2 = 1;
	EDIS;
	if (CpuSysRegs.PCLKCR1.bit.EMIF2 != 1)
	{
		  ErrorFlags |= (1 << 5);
	}

	// Enable global Interrupts and higher priority real-time debug events:
	EINT;  // Enable Global interrupt INTM
	ERTM;  // Enable Global realtime interrupt DBGM

// *****************************************************************************
//Configure DSP - FPGA interface like a asynchronous RAM
// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
// *****************************************************************************
	// TMS320F2837xD ASYNC CS2 CR Register: Chapter 25.5.2.5
	Emif2Regs.ASYNC_CS2_CR.all =   (EMIF_ASYNC_ASIZE_16		| // 16Bit Memory Interface
									EMIF_ASYNC_TA_1			| // Turn Around time of 2 Emif Clock
									EMIF_ASYNC_RHOLD_1		| // Read Hold time of 1 Emif Clock
									EMIF_ASYNC_RSTROBE_10	| // Read Strobe time of 10 Emif Clock
									EMIF_ASYNC_RSETUP_2		| // Read Setup time of 1 Emif Clock
									EMIF_ASYNC_WHOLD_1		| // Write Hold time of 1 Emif Clock
									EMIF_ASYNC_WSTROBE_7	| // Write Strobe time of 7 Emif Clock
									EMIF_ASYNC_WSETUP_1		| // Write Setup time of 1 Emif Clock
									EMIF_ASYNC_EW_DISABLE	| // Extended Wait disabled
									EMIF_ASYNC_SS_ENABLE	  // Strobe Select Mode enabled
									);

	// TMS320F2837xD ASYNC WCCR Register: Chapter 25.5.2.2
	//  Only if Emif2Regs.ASYNC_CS2_CR.bit.EW is enabled (Extended Wait Mode)
	if(Emif2Regs.ASYNC_CS2_CR.bit.EW)
	{
		Emif2Regs.ASYNC_WCCR.all =	   (EMIF_ASYNC_WCCR_WP_HIGH	| // Wait if EMxWAIT port is high (default after reset)
										0x80					  // wait N+1 cycles (default after reset)
										);
	}

	//Add some delay
	Uint16 i = 0;
	for(i = 0; i < 123; i++) { }

	return(ErrorFlags);
}

// Get revision code and status of EMIF2
Uint32 FPGA::GetEMIF2Status(void)
{
	Uint32 status = 0;

	// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
	//	TMS320F2837xD RCSR Register: Chapter: 25.5.2.1
	status = Emif2Regs.RCSR.all;
	return(status);
}

// Read a integer (16 bit) from address (0x00 to FpgaSizeInBytes)
Uint16 FPGA::ReadInteger(const Uint32 Address)
{
	Uint16 result = 0;

	// Check on address range
	if(FpgaSizeInWords <= Address) {
// TODO Error handling: Assert, logging, ...
		return(0xFEDC);
	}

	// set address pointer; Data = 16 Bit => Address * OFFSET_EXTENDER
	result = __addr32_read_uint16(Address*OFFSET_EXTENDER + FpgaStartAddr);
	return(result);
}


// Write a integer (16 bit) to address (0x00 to FpgaSizeInBytes)
void FPGA::WriteInteger(const Uint32 Address, const Uint16 Value)
{
	// Check on address range
	if(FpgaSizeInWords <= Address) {
// TODO Error handling: Assert, logging, ...
		return;
	}

	// set address pointer; Data = 16 Bit => Address * OFFSET_EXTENDER
	__addr32_write_uint16(Address*OFFSET_EXTENDER + FpgaStartAddr, Value);
	return;
}

void FPGA::SetDebugCOM(UART_COM1* const COM)
{
	poCOM1 = COM;
	return;
}

unsigned char FPGA::GetJTAG_TDO(void)
{
	unsigned char returnvalue;

	returnvalue = GPIO_ReadPin(FPGA_TDO);

	return(returnvalue);
}

void FPGA::SetJTAG_TDI(const unsigned char aValue)
{
	if ( aValue )
	{
		GPIO_WritePin(FPGA_TDI, 1);
	} else {
		GPIO_WritePin(FPGA_TDI, 0);
	}

	return;
}

void FPGA::SetJTAG_TMS(const unsigned char aValue)
{
	if ( aValue )
	{
		GPIO_WritePin(FPGA_TMS, 1);
	} else {
		GPIO_WritePin(FPGA_TMS, 0);
	}

	return;
}

void FPGA::SetJTAG_TCK(const unsigned char aValue)
{
	if ( aValue )
	{
		GPIO_WritePin(FPGA_TCK, 1);
	} else {
		GPIO_WritePin(FPGA_TCK, 0);
	}

	return;
}

// Activate JTAG interface to the Lattice FPGA. Configure output pins.
void FPGA::EnableJTAG(void)
{
	// FPGA TDI pin:
	GPIO_SetupPinMux(FPGA_TDI, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(FPGA_TDI, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_WritePin(FPGA_TDI, 1);
	// FPGA TDO pin:
	GPIO_SetupPinMux(FPGA_TDO, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(FPGA_TDO, GPIO_INPUT, GPIO_PULLUP);
	// FPGA TCK pin:
	GPIO_SetupPinMux(FPGA_TCK, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(FPGA_TCK, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_WritePin(FPGA_TCK, 0);
	// FPGA TMS pin:
	GPIO_SetupPinMux(FPGA_TMS, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(FPGA_TMS, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_WritePin(FPGA_TMS, 1);

	return;
}

// Deactivate  JTAG interface to the Lattice FPGA. Set all pins as input
void FPGA::DisableJTAG(void)
{
	GPIO_SetupPinOptions(FPGA_TDI, GPIO_INPUT, GPIO_PULLUP);
	GPIO_SetupPinOptions(FPGA_TDO, GPIO_INPUT, GPIO_PULLUP);
	GPIO_SetupPinOptions(FPGA_TCK, GPIO_INPUT, GPIO_PULLUP);
	GPIO_SetupPinOptions(FPGA_TMS, GPIO_INPUT, GPIO_PULLUP);

	return;
}
