/*
 * SDRAM.cpp
 *
 *	Created on: 25.11.2015
 *		Author: richterr
 *
 *	Interface of SD-RAM
 *
 */

#include "compiler.h"
#include <F2837xD_Device.h> // DSP2837xD header file peripheral address definitions
#include <F2837xD_GlobalPrototypes.h>
#include "F2837xD_Emif_defines.h"
#include <SDRAM.h>

#ifdef __cplusplus
extern "C" {
#endif

/* *********************************************************************
* Explicit Declarations
********************************************************************* */
extern void Emif1Initialize(void);

#ifdef __cplusplus
}
#endif /* extern "C" */


// Construtor of class
SDRAM::SDRAM()
{
	// Initialize global variables
	poCOM1 = NULL;
}

SDRAM::~SDRAM() {
	// Auto-generated destructor stub
}

// Initialization of EMIF1 interface
Uint16 SDRAM::InitOnCPU1()
{
	Uint16  ErrorFlags = 0;

// *****************************************************************************
// Configure SD-RAM access
// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
// *****************************************************************************
// EMIF1 Configuration Registers: Chapter 25.5.3

	// reset EMIF1
	Emif1Initialize();
	//Configure to run EMIF1 on half Rate (EMIF1CLK = CPU1SYSCLK/2)
	EALLOW;
	ClkCfgRegs.PERCLKDIVSEL.bit.EMIF1CLKDIV = 0x1;
	EDIS;
	if (ClkCfgRegs.PERCLKDIVSEL.bit.EMIF1CLKDIV != 0x1)	// Check successful write operation
	{
		  ErrorFlags = (1 << 0);
	}

	// Grab EMIF1 For CPU1
	EALLOW;
	Emif1ConfigRegs.EMIF1MSEL.all = MSEL_EMIF1_CPU1;	// CPU1 is master
	EDIS;
	if (Emif1ConfigRegs.EMIF1MSEL.all != 0x1)			// Check successful write operation
	{
		  ErrorFlags |= (1 << 1);
	}

	//Disable any Access Protection (CPU_FETCH allowed = 0 / CPU_WR allowed = 0 / DMA_WR allowed = 0)
	EALLOW;
	Emif1ConfigRegs.EMIF1ACCPROT0.all = 0x0;
	EDIS;
	if (Emif1ConfigRegs.EMIF1ACCPROT0.all != 0x0)
	{
		  ErrorFlags |= (1 << 2);
	}

	// Commit the configuration related to protection. Till this bit remains set
	// content of EMIF1ACCPROT0 register can't be changed.
	EALLOW;
	Emif1ConfigRegs.EMIF1COMMIT.all = 0x1;
	EDIS;
	if(Emif1ConfigRegs.EMIF1COMMIT.all != 0x1)
	{
		  ErrorFlags |= (1 << 3);
	}

	// Lock the configuration so that EMIF1COMMIT register can't be changed any more.
	EALLOW;
	Emif1ConfigRegs.EMIF1LOCK.all = 0x1;
	EDIS;
	if (Emif1ConfigRegs.EMIF1LOCK.all != 1)
	{
		  ErrorFlags |= (1 << 4);
	}

// Enable global Interrupts and higher priority real-time debug events:
	EINT;  // Enable Global interrupt INTM
	ERTM;  // Enable Global realtime interrupt DBGM

// *****************************************************************************
//Configure SDRAM control registers: Need to be programmed based on SDRAM Data-Sheet
// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
// *****************************************************************************
	// TMS320F2837xD SDRAM Timing Register: Chapter 25.5.2.8
	// SDRAM: ISIS IS42S16160G-7xxA1 - Datasheet: H:\Projekte\TT3100\Datenbltter\IS42_45S83200G_16160G_ISSI.pdf (Dec 2015)
	//	Tables: AC ELECTRICAL CHARACTERISTICS; OPERATING FREQUENCY / LATENCY RELATIONSHIPS
	//	Parameter values for 100MHz (tCK = 10ns)	|
	Emif1Regs.SDRAM_TR.bit.T_RFC = 6;			//	| 7 <= 64ms = tREF (Type A1)
	Emif1Regs.SDRAM_TR.bit.T_RP  = 1;			//	| 2
	Emif1Regs.SDRAM_TR.bit.T_RCD = 1;			//	| 2
	Emif1Regs.SDRAM_TR.bit.T_WR  = 1;			//	| 2 (tDPL)
	Emif1Regs.SDRAM_TR.bit.T_RAS = 4;			//	| 5
	Emif1Regs.SDRAM_TR.bit.T_RC  = 5;			//	| 6
	Emif1Regs.SDRAM_TR.bit.T_RRD = 1;			//	| 2
///	Emif1Regs.SDRAM_TR.all = 0x31114510;


	// TMS320F2837xD SDRAM SR/PD Exit Timing Register: Chapter 25.5.2.11
	// SDRAM: ISIS IS42S16160G-7xxA1 - Datasheet: H:\Projekte\TT3100\Datenbltter\IS42_45S83200G_16160G_ISSI.pdf (Dec 2015)
	//	Tables: AC ELECTRICAL CHARACTERISTICS
	//	Parameter values for 100MHz (tCK = 10ns)	|
		// T_XS = 0x7								| 7 <= 70ms = tXSR
	Emif1Regs.SDR_EXT_TMNG.bit.T_XS = 7;

	// TMS320F2837xD SDRAM Refresh Control Register: Chapter 25.5.2.
	// SDRAM: ISIS IS42S16160G-7xxA1 - Datasheet: H:\Projekte\TT3100\Datenbltter\IS42_45S83200G_16160G_ISSI.pdf (Dec 2015)
	//	Tables: AC ELECTRICAL CHARACTERISTICS
		// tREF = 64ms (Type A1) for 8192 ROW, RR = 64000*100{tCK}/8192 = 781.25 => 0x30E
	Emif1Regs.SDRAM_RCR.bit.REFRESH_RATE = 0x30E;

	// TMS320F2837xD SDRAM (EMxCS0n) Config Register: Chapter 25.5.2.3
	// SDRAM: ISIS IS42S16160G-7xxA1 - Datasheet: H:\Projekte\TT3100\Datenbltter\IS42_45S83200G_16160G_ISSI.pdf (Dec 2015)
	//	Tables: AC ELECTRICAL CHARACTERISTICS; OPERATING FREQUENCY / LATENCY RELATIONSHIPS
	//	Parameters of IS45S16160G (4M x 16 x 4)		|
	Emif1Regs.SDRAM_CR.bit.SR = 0;				//	|
	Emif1Regs.SDRAM_CR.bit.PD = 0;				//	|
	Emif1Regs.SDRAM_CR.bit.PDWR = 0;			//	|
	Emif1Regs.SDRAM_CR.bit.NM = 1;				//	| Data bus between the EMIF and the attached SDRAM device (Data bus is 16-bits)
	Emif1Regs.SDRAM_CR.bit.CL = 2;				//	| 2 = /CAS Latency; Table: KEY TIMING PARAMETERS (100 MHz)
	Emif1Regs.SDRAM_CR.bit.BIT_11_9_LOCK = 1;	//	| Enable write to CL
	Emif1Regs.SDRAM_CR.bit.IBANK = 2;			//	| 2 <= 4 banks (4M x 16 x 4)
	Emif1Regs.SDRAM_CR.bit.PAGESIGE = 1;		//	| 1 <= 512 (512 columns by 16 bits)

	//Add some delay
	Uint16 i = 0;
	for(i = 0; i < 123; i++) { }

	return(ErrorFlags);
}

// Read a integer (16 bit) from address (0x00 to SDramSizeInBytes)
Uint16 SDRAM::ReadInteger(const Uint32 Address)
{
	Uint16 result = 0;

	// Check on address range
	if(SDramSizeInWords <= Address) {
// TODO Error handling: Assert, logging, ...
		return(0xFEDC);
	}

	// set address pointer; Data = 16 Bit => Address * 2 byte
	result = __addr32_read_uint16(Address*2 + SDramStartAddr);
/*	Uint16 *ReadAddress  = (Uint16*)(Address*2 + SDramStartAddr);

	EALLOW;
  	result = *(ReadAddress);
	EDIS;
// */
	return(result);
}


// Write a integer (16 bit) to address (0x00 to SDramSizeInBytes)
void SDRAM::WriteInteger(const Uint32 Address, const Uint16 Value)
{
	// Check on address range
	if(SDramSizeInWords <= Address) {
// TODO Error handling: Assert, logging, ...
		return;
	}

	// set address pointer; Data = 16 Bit => Address * 2 byte
	__addr32_write_uint16(Address*2 + SDramStartAddr, Value);
/*	Uint16 *WriteAddress  = (Uint16*)(Address*2 + SDramStartAddr);

	EALLOW;
  	*(WriteAddress) = Value;
	EDIS;
// */
	return;
}

void SDRAM::SetDebugCOM(UART_COM1* const COM)
{
	poCOM1 = COM;
	return;
}


// *****************************************************
// Quelle: controlSUITE\device_support\F2837xD\v160\F2837xD_examples_Cpu1\emif1_32bit_sdram\cpu01\emif1_32bit_sdram.c
// *****************************************************
char SDRAM::test_read_write(Uint32 start_addr, Uint32 mem_size)
{
    unsigned long mem_rdl;
    unsigned int mem_wdl;
    Uint32 XMEM_p;
    Uint32 i;

    //Write data
    XMEM_p = start_addr;
    //-------------------------------------------
    //Fill memory
    //-------------------------------------------
    mem_wdl = 0x0123;
    for (i=0; i < mem_size; i++)
    {
    	__addr32_write_uint16(XMEM_p, mem_wdl);
    	XMEM_p = XMEM_p+2;
    	mem_wdl += 0x1111;
    }

    //------------------------------------------
    //Verify memory
    //------------------------------------------
    mem_wdl = 0x0123;
    XMEM_p = start_addr;
    for (i=0; i < mem_size; i++)
    {
        mem_rdl = __addr32_read_uint16(XMEM_p);
        if( mem_rdl != mem_wdl)
        {
            return(1);
  	    }
        XMEM_p = XMEM_p+2;
  	    mem_wdl += 0x1111;
    }
    return(0);
}
