/*
 * SpiFlash.cpp
 *
 *	Created on: 09.12.2015
 *		Author: richterr
 *
 *	Interface of SpiFlash
 *
 *	This class based on "SPIFLASH28_SPI_Flashprog_Library.cpp" (TT3000 projects)
 *
 * ###########################################################################
 *  TI File $Revision:: 3    $
 * ###########################################################################
 *
 *  FILE:  SPIFLASH28_Flashprog_Library.h
 *
 *  TITLE: 28335 SPI FLASH Library
 *
 *  DESCRIPTION:
 *
 *        This file should be included in any project that uses the 28x SPI
 *        FLASH Library.
 *
 *        This program uses the DSP2834x header files that are included in the
 *        C/C++ Header Files and Peripheral Examples in C (SPRC876) which can
 *        be downloaded from the TI website.  This program currently uses V1.00
 *        of SPRC876 and the headerfiles used are included with this project.
 *
 * ###########################################################################
 *
 *   Ver | dd mmm yyyy  | Who				| Description of changes
 *  =====|==============|===================|=====================================
 *  1.00 | 02 Jun 2009  | TI				| Initial Release
 *  1.01 | 28 Feb 2011  | Rolf Schwesig		| Extensions for Standard 64MBit Flash
 * ###########################################################################
 *
 */

#include <sstream>
#include <string>

#include "compiler.h"
#include "application.h"
//- #include "F28x_Project.h"	// Device Headerfile and Examples Include File
#include <F2837xD_Device.h>		// DSP2837xD header file peripheral address definitions
#include <F2837xD_Examples.h>	// F2837xD Examples Include File
#include <F2837xD_Gpio.h>
#include <F2837xD_Gpio_defines.h>
#include <F2837xD_spi.h>
#include "Spi_Flash.h"
// Other module and project header
#include "UART_COM1.h"
#include "gpio.h"

using namespace std;

// *************************************************************************
// Global variables, structures, etc
// *************************************************************************
volatile struct FLASH_REGS FlashRegs;

// *************************************************************************
// Definitions
// *************************************************************************

// Construtor of class. Initialize device specific constants (Only flash size, CS times by default)
SpiFlash::SpiFlash(const Uint32 SpecificTotalNumOfBytes, const SPI_BUS SpiBus) :
		TotalNumOfBytes(SpecificTotalNumOfBytes),
		TotalNumOfWords(TotalNumOfBytes/2),
		TotalNumOfPages(TotalNumOfBytes/PageSizeInBytes),
		FlashLastAddr(TotalNumOfBytes-1),
		TotalNumOfBlocks(TotalNumOfBytes/BlockSizeInBytes),
		CSHoldTime(DEFAULT_CS_HOLD),
		CSHighTime(DEFAULT_CS_HIGH),
		CSSetupTime(DEFAULT_CS_SETUP)
{
	// Initialize global variables
	poDbgCom = NULL;
	// Set device specific SPI parameters
	switch(SpiBus) {
		case SPI_A:
			SpiBusOfDevice = SPI_A;
			DevCS = SPIEA_N;
			gpSpiRegs = &SpiaRegs;
			break;
		case SPI_C:
			SpiBusOfDevice = SPI_C;
			DevCS = SPIEC_N;
			gpSpiRegs = &SpicRegs;
			break;
		default:
///	TODO		Logger_info1("SPI of Flash device undefined");
			SpiBusOfDevice = SPI_UNDEF;
			DevCS = GPIO_0;
			gpSpiRegs = NULL;
			break;
	} // end of switch(SpiBus)
}

// Construtor of class. Initialize device specific constants (Only flash size and also CS times)
SpiFlash::SpiFlash(const Uint32 SpecificTotalNumOfBytes, const SPI_BUS SpiBus, const long double SpecificCSHoldTime,
		 const long double SpecificCSHighTime, const long double SpecificCSSetupTime) :
		TotalNumOfBytes(SpecificTotalNumOfBytes),
		TotalNumOfWords(TotalNumOfBytes/2),
		TotalNumOfPages(TotalNumOfBytes/PageSizeInBytes),
		FlashLastAddr(TotalNumOfBytes-1),
		TotalNumOfBlocks(TotalNumOfBytes/BlockSizeInBytes),
		CSHoldTime(SpecificCSHoldTime),
		CSHighTime(SpecificCSHighTime),
		CSSetupTime(SpecificCSSetupTime)
{
	// Initialize global variables
	poDbgCom = NULL;
	// Set device specific SPI parameters
	switch(SpiBus) {
		case SPI_A:
			SpiBusOfDevice = SPI_A;
			DevCS = SPIEA_N;
			gpSpiRegs = &SpiaRegs;
			break;
		case SPI_C:
			SpiBusOfDevice = SPI_C;
			DevCS = SPIEC_N;
			gpSpiRegs = &SpicRegs;
			break;
		default:
///	TODO		Logger_info1("SPI of Flash device undefined");
			SpiBusOfDevice = SPI_UNDEF;
			DevCS = GPIO_0;
			gpSpiRegs = NULL;
			break;
	} // end of switch(SpiBus)
}

SpiFlash::~SpiFlash() {
	// Auto-generated destructor stub
}

/***************** doxygen like ******************************//**
 * Method: InitSPI
 * In: ---
 * Out: Status is operation failed
 * Description: Common initialization of SPI interface.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::InitSPI(void)
{
	return(FAIL_INIT);
}

void SpiFlash::SetDebugCOM(UART_COM1* const COM)
{
	poDbgCom = COM;
	return;
}

/***************** doxygen like ******************************//**
 * Method: Reset
 * In: ---
 * Out: ---
 * Description: Perform Chip Reset.
// **************************************************************/
void SpiFlash::Reset(void)
{
///	Logger_info0("SPI flash reset.");
	SetChipSelectLow();
	SendCommand(CmdReset);				// Reset Command
	SendCommand(CmdConfirmReset);		// Security Confirmation
	SetChipSelectHigh();   				// SPI Flash will Reset at least 30us after CS return to high level
	return;
}

/***************** doxygen like ******************************//**
 * Method: ReadDeviceId (Default method for most devices)
 * In: ---
 * Out: Code of manufacturer Id and Flash capacity
 * Description:
 *	Manufacturer ID ist JEDEC coded and unique for every Manufacturer. Device Type/Id and/or Capacity are
 *	Manufacturer specific and can change for compatible but different device.
 *	Chip should return a 3-Byte code with following values :
 *
 *
 *	|	Chip		|	Manufacturer	|	JEDEC-ID	|	Device-Type	|	Capacity	|
 *	|_______________|___________________|_______________|_______________|_______________|
 *	|				|					|				|				|				|
 *	|	AT25DF641	|	Atmel			|	0x1F		|	0x48		|	0x00		|
 *	|	SST25VF064C	|	Microchip		|	0xBF		|	0x25		|	0x4B		|
 *	|	M25P64		|Micron Technologies|	0x20		|	0x20		|	0x17		|
 *	|	W25Q64CV	|	Winbond			|	0xEF		|	0x16		|	0x17		|
 *	|	S25FL064P	|	Spansion = AMD	|	0x01		|	0x02		|	0x16		|
 *	|	W25Q256FV	|	Winbond			|	0xEF		|	0x40		|	0x19		|
// **************************************************************/
Uint32 SpiFlash::ReadDeviceId(void)
{
	Uint16 manufacturerId,deviceId;
	Uint32 capacity;

	SetChipSelectLow();
	SendCommand(CmdReadManDevID);
	manufacturerId = ReceiveByte();
	deviceId = ReceiveByte();
	capacity = ReceiveByte();
	SetChipSelectHigh();
	return((capacity << 16) | (deviceId << 8) | manufacturerId);
}

/***************** doxygen like ******************************//**
 * Method: VerifyDeviceId
 * In: Device ID from SPI device
 * Out: Always status 'wrong ID'
 * Description: This is a stub method of the common SPI Flash.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::VerifyDeviceId(const Uint32 DevId)
{
	// return of stub method
	return(FAIL_DEVICE_ID);
}

/***************** doxygen like ******************************//**
 * Method: WritePage
 * In:
 * 	FlashAddr:	First address of the page to be written to in the FLASH
 * 	*BufAddr:	Pointer to the buffer of data to be programmed
 * 	Length:		Number of values in the buffer. If this is less then a full page, then missing data
 * 				will be read from the FLASH to fill in a full page
 * Out: Status of operation (STATUS_SUCCESS = Write completed - use the verify function to check the data for correctness;
 * 							 STATUS_FAIL_PROGRAM = Was unable to enable programming)
 * Description: Writes a page of data to the FLASH.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::WritePage(const Uint32 FlashAddr, const Uint16* const BufAddr, const Uint16 Length)
{
    Uint16 WordCount = 0;
    Uint32 tempFlashAddr = FlashAddr;
    Uint32 bytes = 0;
    Uint16 buffer[PageSizeInWords];
    Uint16 *pBuffer = buffer;
    Uint32 Addr = FlashAddr;
    FLASH_API_STATUS_MESSAGES result = SUCCESSFUL;

    if ( (Addr > (FlashFirstAddr - PageSizeInBytes)) || (Addr & 1) )
    {
       return(FAIL_PROGRAM);
    }

    else if(Length > PageSizeInWords - (Addr % PageSizeInWords)/2)
    {
       return(FAIL_PROGRAM);
    }

    else if(Length == 0)
    {
        // Nothing to do
        return(SUCCESSFUL);
    }

    else
    {
        // Fill beginning of page with data from the FLASH
        bytes = Addr % PageSizeInBytes;
        if (0 != bytes)
        {
            tempFlashAddr -= bytes;
            for(WordCount = 0; WordCount < (bytes/2); WordCount++)
            {
                *pBuffer = Read16bitWord(tempFlashAddr);
                pBuffer++;
                tempFlashAddr+=2;
            }
        }

        // Append new user data
        std::memcpy(pBuffer, BufAddr, Length);
        tempFlashAddr += 2 * Length;
        pBuffer += Length;

        // Fill remainder of page with data from the FLASH
        bytes = tempFlashAddr % PageSizeInBytes;
        if ( bytes != 0 )
        {
            bytes = PageSizeInBytes - bytes;
            for(WordCount = 0; WordCount < (bytes/2); WordCount++)
            {
                *pBuffer = Read16bitWord(tempFlashAddr);
                pBuffer++;
                tempFlashAddr+=2;
            }
        }

        // Adjust flash address to the start of the page.
        // The following operations will program a whole pate.
        Addr -= Addr % PageSizeInBytes;
    }

    // Clear protection bits in the status reg
    WriteStatReg(0x0000);

    result = WriteEnable();
    if(result != SUCCESSFUL) {
    	return(FAIL_PROGRAM);
    }

    SetChipSelectLow();
    SendCommand(CmdWriteMemory);
    SendFlashAddr(Addr);

    for(WordCount = 0; WordCount < PageSizeInWords; WordCount++)
    {
        Send16bitWord(buffer[WordCount]);
    }

    SetChipSelectHigh();

    // Wait for flash write to complete
    do
    {
    	FlashRegs.STATUS.all = ReadStatReg();
    } while( 1 == FlashRegs.STATUS.bit.BUSY_FLAG );

    return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: Erase4KBlock
 * In: Number of sector (0 .. TotalNumOfBlocks-1)
 * Out: Status of operation (STATUS_SUCCESS = Chip erased; STATUS_FAIL_PROGRAM = Was unable to erase chip)
 * Description: Erases one 4KByte block.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::Erase4KBlock(const Uint32 Sector)
{
	Uint32 Addr = 0;
	FLASH_API_STATUS_MESSAGES result = SUCCESSFUL;

	Addr = Sector * BlockSizeInBytes;
    // Clear protection bits in the status reg
    WriteStatReg(0x0000);

    result = WriteEnable();
    if(result != SUCCESSFUL) {
    	return(FAIL_ERASE);
    }

    SetChipSelectLow();

    SendCommand(Cmd4kBlockErase);
    SendFlashAddr(Addr);

    SetChipSelectHigh();

	// Wait for flash write to complete
	do
	{
		FlashRegs.STATUS.all = ReadStatReg();
	} while(1 == FlashRegs.STATUS.bit.BUSY_FLAG);

    return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: EraseTotalFlash
 * In: ---
 * Out: Status of operation (STATUS_SUCCESS = Chip erased; STATUS_FAIL_PROGRAM = Was unable to erase chip)
 * Description: Erases the entire SPI FLASH chip.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::EraseTotalFlash(void)
{
	FLASH_API_STATUS_MESSAGES result = SUCCESSFUL;

    // Clear protection bits in the status reg
    WriteStatReg(0x0000);

    result = WriteEnable();
    if(result != SUCCESSFUL) {
    	return(FAIL_ERASE);
    }

    SetChipSelectLow();
    SendCommand(CmdChipErase);
    SetChipSelectHigh();

	// Wait for flash write to complete
	do
	{
		FlashRegs.STATUS.all = ReadStatReg();
	} while(1 == FlashRegs.STATUS.bit.BUSY_FLAG);

    return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: SendFlashAddr
 * In: Address of the FLASH
 * Out: ---
 * Description: Sends a 24-bit FLASH Address.
// **************************************************************/
void SpiFlash::SendFlashAddr(const Uint32 FlashAddr)
{
	volatile Uint16 scratch = 0;

	gpSpiRegs->SPITXBUF = (Uint32)(FlashAddr>>8);	// Flash - high 8-bits of 24-bit address (shifted into MSB of TXBUF)
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1);	// Wait for byte to be shifted out
	scratch = gpSpiRegs->SPIRXBUF;					// Clear RX buffer

	gpSpiRegs->SPITXBUF = FlashAddr;				// Flash - middle 8-bits of 24-bit address (in MSB of TXBUF)
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1);	// Wait for byte to be shifted out
	scratch = gpSpiRegs->SPIRXBUF;					// Clear RX buffer

	gpSpiRegs->SPITXBUF = FlashAddr <<8;			// Flash - bottom 8-bits of 24-bit address (in MSB of TXBUF)
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1);	// Wait for byte to be shifted out
	scratch = gpSpiRegs->SPIRXBUF;					// Clear RX buffer

	return;
}

/***************** doxygen like ******************************//**
 * Method: WriteEnable
 * In: ---
 * Out: Status of operation (STATUS_SUCCESS = Chip erased; STATUS_FAIL_PROGRAM = Was unable to enable programming)
 * Description: This function sets the write enable bit in the FLASH status register.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::WriteEnable(void)
{
	SetChipSelectLow();
	SendCommand(CmdSetWriteEnableLatch);
	SetChipSelectHigh();

	FlashRegs.STATUS.all = ReadStatReg();
	if(0 == FlashRegs.STATUS.bit.WEL_FLAG) {
		return(FAIL_PROGRAM);
	}

	return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: SendCommand
 * In: The 8-bit command is passed in the LSByte of the parameter 'Command'
 * Out: ---
 * Description: Sends a command to the FLASH.
// **************************************************************/
void SpiFlash::SendCommand(const Uint16 Command)
{
    volatile Uint16 scratch = 0;

    gpSpiRegs->SPITXBUF = Command << 8;				//Send an FLASH Command
	while ( 0 == gpSpiRegs->SPISTS.bit.INT_FLAG ) ;	//Wait for byte to be shifted out
    scratch = gpSpiRegs->SPIRXBUF;					//Clear RX buffer
    return;
}

/***************** doxygen like ******************************//**
 * Method: Send16bitWord
 * In:
 * 	1.	For a write command, Word is the 16-bit value to be stored in the FLASH.
 * 		The Word is passed to the function as MSB:LSB and then stored in the FLASH as LSB:MSB
 * 		 as expected by the boot ROM.
 * 	2.	For a read command, Word can be garbage data and this function is used to shift out the
 * 		 value being read from the FLASH
 * Out:
 * 	1. garbage
 * 		if the function is used to send a value to be written ie it follows a write command,
 * 		then the return value is nonsense and is not used.
 * 	2. 16-bit value read
 * 		if the function is used to read a value, ie the parameter command is dummy data and the function
 * 		follows a read command, then the return value will be the 16-bit value read from the FLASH
 * Description: Sends a 16-bit value to the serial FLASH device.
// **************************************************************/
Uint16 SpiFlash::Send16bitWord(const Uint16 Word)
{
	Uint16 temp = 0x0000;

	gpSpiRegs->SPITXBUF = Word << 8;				//FLASH LSB 16 bit value
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1) ;	//Wait for byte to be shifted out
	temp = gpSpiRegs->SPIRXBUF;						//Clear RX buffer

	gpSpiRegs->SPITXBUF = Word;						//FLASH MSB 16 bit value
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1) ;	//Wait for byte to be shifted out
	temp |= gpSpiRegs->SPIRXBUF << 8;				//Clear RX buffer

	return(temp);
}

/***************** doxygen like ******************************//**
 * Method: ReceiveByte
 * In: ---
 * Out: Received 8 bit data
 * Description: Receives a byte of data.
// **************************************************************/
char SpiFlash::ReceiveByte(void)
{
    char result = 0;

	gpSpiRegs->SPITXBUF = 0x0000;					// Dummy data to sent
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1) ;	// Wait for byte to be shifted out
	result = gpSpiRegs->SPIRXBUF;					//Clear RX buffer

    return(result);
}

/***************** doxygen like ******************************//**
 * Method: Read16bitWord
 * In: FLASH Address
 * Out:	Unsigned 16-bit word from the FLASH (The value is read from the FLASH in order
 * 		 LSB:MSB and returned to the calling function as MSB:LSB)
 * Description: Reads a 16-bit value out of the serial FLASH device.
// **************************************************************/
Uint16 SpiFlash::Read16bitWord(const Uint32 FlashAddr)
{
	Uint16 WordVal = 0;

	SetChipSelectLow();
	SendCommand(CmdReadMemory);
	SendFlashAddr(FlashAddr);
	WordVal = Send16bitWord(0x0000);
	SetChipSelectHigh();
	return(WordVal);
}

/***************** doxygen like ******************************//**
 * Method: Verify
 * In:
 * 	FlashAddr:	First address of the page to be written to in the FLASH
 * 	*BufAddr:	Pointer to the buffer of golden data to be compared
 * 	Length:		Number of values to compare
 * Out: Status of operation (STATUS_SUCCESS = Write completed - use the verify function to check the data for correctness;
 * 							 FLASH_STATUS_FAIL_VERIFY = At least one word did not match)
 * Description: Compares 16-bit values.
// **************************************************************/
FLASH_API_STATUS_MESSAGES SpiFlash::Verify(const Uint32 FlashAddr, const Uint16* BufAddr, const Uint16 Length)
{
    Uint16 WordCount = 0;
    Uint16 CheckVal = 0;
    Uint16 failcounter = 0;
    Uint32 Addr = FlashAddr;

    for(WordCount = 0; WordCount < Length; WordCount++)
    {
        CheckVal = Read16bitWord(Addr);
        if( CheckVal != *BufAddr)
        {
        	failcounter++;
        }
        Addr +=2;
        BufAddr++;
    }
    if (failcounter == 0) {
        return(SUCCESSFUL);
    }
    return(FAIL_VERIFY);
}

/***************** doxygen like ******************************//**
 * Method: ReadStatReg
 * In: ---
 * Out: Status of Flash
 * Description: Reads the FLASH's status register.
// **************************************************************/
Uint16 SpiFlash::ReadStatReg(void)
{
	Uint16 StatReg = 0;

	SetChipSelectLow();
	SendCommand(CmdReadStatusReg);

	// Force the value of the status register to be shifted out
	gpSpiRegs->SPITXBUF = 0x0000;
	while ( gpSpiRegs->SPISTS.bit.INT_FLAG != 1) ;	// Wait for byte to be shifted out
	StatReg = gpSpiRegs->SPIRXBUF;					// Read Status Reg from buffer

	SetChipSelectHigh();

	return(StatReg);
}

/***************** doxygen like ******************************//**
 * Method: WriteStatReg
 * In: New status
 * Out: ---
 * Description: Writes new status to the FLASH's status register.
// **************************************************************/
void SpiFlash::WriteStatReg(const Uint16 StatReg)
{
	volatile Uint16 scratch = 0;

	WriteEnable();

	SetChipSelectLow();
	SendCommand(CmdWriteStatusReg);

	gpSpiRegs->SPITXBUF = StatReg << 8;				//Send the Status Register value
	while ( 1 != gpSpiRegs->SPISTS.bit.INT_FLAG ) ;	//Wait for byte to be shifted out
	scratch = gpSpiRegs->SPIRXBUF;					//Clear RX buffer

	SetChipSelectHigh();

	// Wait for flash write to complete
	do
	{
		FlashRegs.STATUS.all = ReadStatReg();
	} while(1 == FlashRegs.STATUS.bit.BUSY_FLAG);

	return;
}

/***************** doxygen like ******************************//**
 * Method: SetChipSelectHigh
 * In: ---
 * Out: ---
 * Description: This function sets the chip select signal (GPIO57) high.
 * 				Use it only on GPIO Index = 0 of 'SPIEC_N'. See GPIO definition of'SPIEC_N'
// **************************************************************/
void SpiFlash::SetChipSelectHigh(void)
{
	// Only active if GPIO Index = 0 of 'SPIEC_N'
	DELAY_US(CSHoldTime);
	//Set CSn High. Use it only on GPIO Index = 0 of 'SPIEn_N'. See GPIO definition of 'SPIEn_N'.
	GPIO_WritePin(DevCS, 1);	// SPIEC_N, 1);
    DELAY_US(CSHighTime);
    return;
}


/***************** doxygen like ******************************//**
 * Method: SetChipSelectLow
 * In: ---
 * Out: ---
 * Description: This function sets the chip select signal (GPIO57) low.
 * 				Use it only on GPIO Index = 0 of 'SPIEC_N'. See GPIO definition of'SPIEC_N'
// **************************************************************/
void SpiFlash::SetChipSelectLow(void)
{
	//Set CSn Low. Use it only on GPIO Index = 0 of 'SPIEn_N'. See GPIO definition of 'SPIEn_N'.
	GPIO_WritePin(DevCS, 0);	// SPIEC_N, 0);
    DELAY_US(CSSetupTime);
	return;
}

/***************** doxygen like ******************************//**
 * Method: Test
 * In:
 *  Number of blocks to test (if 0 test all blocks)
 *  Pointer to counter of failed blocks as result
 * Out: ---
 * Description: Do a complete erase, write and read test of Flash.
// **************************************************************/
//+ #ifdef SYSTEMTEST

FLASH_API_STATUS_MESSAGES SpiFlash::Test(const Uint32 NumOfBlocksToTest, Uint32* const pFailedBlocks)
{
    stringstream oDbg;
	Uint32 manDevId = 0;
	Uint16 i = 0;
	Uint16 j = 0;
	Uint16 Buf[PageSizeInBytes];
	Uint32 failcounter = 0;
	Uint32 NumOfBlocks = 0;
	FLASH_API_STATUS_MESSAGES result = SUCCESSFUL;
	FLASH_API_STATUS_MESSAGES readresult = SUCCESSFUL;

	// Verify number of testing blocks; If default value = 0 so test all blocks
	if(0 == NumOfBlocksToTest) {
		NumOfBlocks = TotalNumOfBlocks;
	}
	else if(TotalNumOfBlocks < NumOfBlocksToTest) {
		NumOfBlocks = TotalNumOfBlocks;
	}
	else {
		NumOfBlocks = NumOfBlocksToTest;
	}

	// Info
	oDbg.str("");
	oDbg << "\nTest device on SPI " << GetUsedSpiBus() << "\r\n";
	poDbgCom->Write(oDbg);

	oDbg.str("");
	oDbg << (hex);
	oDbg << "0x" << (Uint16)TotalNumOfBlocks << " blocks will erased\r\n";
	oDbg << "0x" << (Uint16)NumOfBlocks << " blocks will W/R and verified\r\n";
	oDbg << (dec) << endl;
	poDbgCom->Write(oDbg);

	// Populate Buffer
	for (i = 0; i < (PageSizeInBytes-1); i++) {
		Buf[i] = (i<<8) | i;
	}

	// Initialize return value with irregular value
	*pFailedBlocks = 0xFFFFFFFF;

	// Check on correct device
	manDevId = ReadDeviceId();
///		Logger_info1("device ID = 0x%x", manDevId);
	result = VerifyDeviceId(manDevId);
///		Logger_info1("Validation result of device ID: 0x%x", result);
	if(SUCCESSFUL == result) {
		goCOM1.Write("ID validation successful!\r\n");
	}
	else {
		goCOM1.Write("ID validation failed!\r\n");
		return(result);
	}

	// Erase flash blockwise
	goCOM1.Write("SPI Block erase:\r\n");
	for(j = 0; j < TotalNumOfBlocks; j++)
	{
		result = Erase4KBlock(j);
		if(SUCCESSFUL != result) {
			return(result);
		}
		if (0 == (j % PageSizeInWords))
		{
///			Log_info2("Result of block erase %d = 0x%x", j, result);
		goCOM1.Write('-');
		}
	}
	goCOM1.Write("\r\n");

	// Write buffer data to flash, readout again and verify data
	goCOM1.Write("SPI write page and verification:\r\n");
	for(j = 0; j < NumOfBlocks; j++)
		{
		WritePage(0x0 + j*PageSizeInBytes, Buf, PageSizeInWords);
		readresult = Verify(0x0 + j*PageSizeInBytes, Buf, PageSizeInWords);
		if (SUCCESSFUL != readresult) {
			failcounter++;
		}
		if (0 == (j % PageSizeInWords)) {
///			Logger_info2("Verify result of block %d = 0x%x", j, readresult);
		  goCOM1.Write('+');
		}
	}
	goCOM1.Write("\r\n");

	// How many blocks failed on verification
	*pFailedBlocks = failcounter;

	return(readresult);
}

//+ #endif

/***************** doxygen like ******************************//**
 * Method: GetTotalNumOfBlocks
 * In: ---
 * Out: Total number of bytes in flash (Capacity)
 * Description: Get devive specific parameter "TotalNumOfBytes"
// **************************************************************/
Uint32 SpiFlash::GetTotalNumOfBytes(void)
{
	return(TotalNumOfBytes);
}

/***************** doxygen like ******************************//**
 * Method: GetTotalNumOfBlocks
 * In: ---
 * Out: Total number of pages in flash
 * Description: Get devive specific parameter "TotalNumOfPages"
// **************************************************************/
Uint32 SpiFlash::GetTotalNumOfPages(void)
{
	return(TotalNumOfPages);
}

/***************** doxygen like ******************************//**
 * Method: GetTotalNumOfBlocks
 * In: ---
 * Out: Total number of blocks in flash
 * Description: Get devive specific parameter "TotalNumOfBlocks"
// **************************************************************/
Uint32 SpiFlash::GetTotalNumOfBlocks(void)
{
	return(TotalNumOfBlocks);
}

/***************** doxygen like ******************************//**
 * Method: GetChipSelectPin
 * In: ---
 * Out: Character of used SPI bus ('A', 'C' or 0 if not set)
 * Description: Get used SPI bus used by devive.
// **************************************************************/
char SpiFlash::GetUsedSpiBus(void)
{
	return((char)SpiBusOfDevice);
}

/***************** doxygen like ******************************//**
 * Method: ShowDeviceParameters
 * In: ---
 * Out: ---
 * Description: Show devive parameters like: Total number of bytes, total number of blocks, total number of pages, etc.
// **************************************************************/
void SpiFlash::ShowDeviceParameters(void)
{
	// Break, if COM device is not set
	if(NULL == poDbgCom) {
		return;
	}

	Uint32 FlashTotalNumOfX = 123456789;
    stringstream oDbg;

	FlashTotalNumOfX = GetTotalNumOfBytes();
	if(0 == FlashTotalNumOfX) {
		poDbgCom->Write("Invalid SPI Flash Capacity = 0x00!\n\r\n");
		return;
	}

	// SPI bus info

	oDbg.str("");
	if(SPI_UNDEF != SpiBusOfDevice) {
		oDbg << "SPI bus   : " << GetUsedSpiBus() << "\r" << endl;
	}
	else {
		oDbg << "SPI bus   : Undefined!\r" << endl;
	}
	poDbgCom->Write(oDbg);

	// Capacity parameters
	oDbg.str("");
	oDbg << hex;
	oDbg << "Bytes     : 0x" << FlashTotalNumOfX << "\r\n";
	oDbg << "Words     : 0x" << TotalNumOfWords << "\r\n";
	oDbg << "Pages     : 0x" << TotalNumOfPages << "\r\n";
	oDbg << "Blocks    : 0x" << TotalNumOfBlocks << "\r\n";
	oDbg << "Last Addr : 0x" << FlashLastAddr;
	oDbg << dec<< "\r" << endl;
	poDbgCom->Write(oDbg);

	// Chip Select times
	oDbg.str("");
	oDbg << "CS Hold  : " << (int)((float)(CSHoldTime*1000.0)) << "ns\r\n";
	oDbg << "CS High  : " << (int)((float)(CSHighTime*1000.0)) << "ns\r\n";
	oDbg << "CS Hold  : " << (int)((float)(CSSetupTime*1000.0)) << "ns\r" << endl;
	poDbgCom->Write(oDbg);

	return;
}
