/*
 * SpiFlash.h
 *
 *	Created on: 09.12.2015
 *		Author: richterr
 *
 *	Interface of SpiFlash (Common part)
 *
 *	Flash memory IC on SPI C
 *
 *	This class based on "SPIFLASH28_SPI_Flashprog_Library.h" (TT3000 projects)
 *
 * ###########################################################################
 *  TI File $Revision:: 3    $
 * ###########################################################################
 *
 *  FILE:  SPIFLASH28_Flashprog_Library.h
 *
 *  TITLE: 28335 SPI FLASH Library
 *
 *  DESCRIPTION:
 *
 *        This file should be included in any project that uses the 28x SPI
 *        FLASH Library.
 *
 *        This program uses the DSP2834x header files that are included in the
 *        C/C++ Header Files and Peripheral Examples in C (SPRC876) which can
 *        be downloaded from the TI website.  This program currently uses V1.00
 *        of SPRC876 and the headerfiles used are included with this project.
 *
 * ###########################################################################
 *
 *   Ver | dd mmm yyyy  | Who				| Description of changes
 *  =====|==============|===================|=====================================
 *  1.00 | 02 Jun 2009  | TI				| Initial Release
 *  1.01 | 28 Feb 2011  | Rolf Schwesig		| Extensions for Standard 64MBit Flash
 * ###########################################################################
 *
 */

#ifndef __spi_flash_h__
#define __spi_flash_h__

// *************************************************************************
// Includes
// *************************************************************************
// Include the C/C++ header file main include file.  This is part of the
// C/C++ Header Files and Peripheral Examples in C (SPRC097) which can
// be downloaded from the TI website.
#include "compiler.h"
#include "F2837xD_device.h"
#include "UART_COM1.h"

// *************************************************************************
// Definitions
// *************************************************************************
// Serial FLASH Commands and FLASH Parameters
// ---------------------------------------------------------------------------
// Common Serial Flash CS timing parameters (in us)
#define   DEFAULT_CS_HOLD		0.250L
#define   DEFAULT_CS_HIGH		0.250L
#define   DEFAULT_CS_SETUP		0.250L

// class of common serial flash commands
class FlashCommonCmds {
protected:
	static const Uint16 CmdSetWriteEnableLatch = 0x0006;
	static const Uint16 CmdResetWriteEnableLatch = 0x0004;
	static const Uint16 CmdReadStatusReg = 0x0005;
	static const Uint16 CmdWriteStatusReg = 0x0001;
	static const Uint16 CmdWriteMemory = 0x0002;
	static const Uint16 CmdReadMemory = 0x0003;
	static const Uint16 CmdChipErase = 0x0060;
	static const Uint16 Cmd4kBlockErase = 0x0020;
	static const Uint16 CmdReadManDevID = 0x009F;
	static const Uint16 CmdReset = 0x00F0;
	static const Uint16 CmdConfirmReset = 0x00D0;
};

// class of complete flash device commands (for common device the common commands only)
class FlashCmds : public FlashCommonCmds { };

/*---------------------------------------------------------------------------
 '28 Serial FLASH API Status Messages

 The following status values are returned from the SPI FLASH API to the
 calling program. These can be used to determine if the FLASH (FLASH Programmer)
 API function passed or failed.
---------------------------------------------------------------------------*/
typedef enum {
	SUCCESSFUL = 0,			// Operation passed, no errors were flagged
	FAIL_DEVICE_ID = 8,		// Unknown Manufactorer / DeviceID
	FAIL_INIT = 10,			// The CSM is preventing the function from performing its operation
	FAIL_ERASE = 20,		// Erase error
	FAIL_PROGRAM = 30,		// Write error
	FAIL_VERIFY = 40		// Verify error
} FLASH_API_STATUS_MESSAGES;

/*---------------------------------------------------------------------------
   Define different SPI busses: Device use SPI A or SPI B or SPI C ....
---------------------------------------------------------------------------*/
typedef enum {
	SPI_UNDEF = 0,		// Not defined
	SPI_A = 'A',		// SPI A
	SPI_B = 'B',		// SPI B
	SPI_C = 'C'			// SPI C
} SPI_BUS;

/*---------------------------------------------------------------------------
   Serial FLASH Registers
---------------------------------------------------------------------------*/
// Flash Status Register Bits
struct FLASH_STATUS_BITS {			// bits description
    Uint16 BUSY_FLAG:1;				// 0  Busy (also 'Ready')
    Uint16 WEL_FLAG:1;				// 1 Write Enable Latch (also 'WEN')
    Uint16 BP:4;					// 5:2 Block protection bits
    Uint16 TB_FLAG:1;				// 6 Top/Bottom protection (used by some devices, else unused)
    Uint16 WPEN_FLAG:1;				// 7 Write Protect Enable / Sector Protection Registers Locked (SPRL) / SRP (Status Register Protect)
    Uint16 rsvd:8;					// 15:8 Reserved
};

union FLASH_STATUS_REG {
    Uint16  all;
    struct  FLASH_STATUS_BITS  bit;
};

struct FLASH_REGS {
    union   FLASH_STATUS_REG			STATUS;				// Status Register
};

// *************************************************************************
// Declarations
// *************************************************************************
class SpiFlash : public virtual FlashCmds {
public:
	SpiFlash(const Uint32 SpecificTotalNumOfBytes, const SPI_BUS SpiBus);
	SpiFlash(const Uint32 SpecificTotalNumOfBytes, const SPI_BUS SpiBus, const long double SpecificCSHoldTime,
			 const long double SpecificCSHighTime, const long double SpecificCSSetupTime);
	virtual ~SpiFlash();

	virtual FLASH_API_STATUS_MESSAGES InitSPI(void);
	void SetDebugCOM(UART_COM1* const COM);

// -----------------------------------
	FLASH_API_STATUS_MESSAGES WritePage(const Uint32 FlashAddr, const Uint16* const BufAddr, const Uint16 Length);
	FLASH_API_STATUS_MESSAGES Verify(const Uint32 FlashAddr, const Uint16* BufAddr, const Uint16 Length);
	FLASH_API_STATUS_MESSAGES EraseTotalFlash(void);
	FLASH_API_STATUS_MESSAGES Erase4KBlock(const Uint32 Sector);
	Uint16 Read16bitWord(const Uint32 FlashAddr);
	Uint16 ReadStatReg(void);
	void WriteStatReg(const Uint16 StatReg);
//+ #ifdef SYSTEMTEST
	FLASH_API_STATUS_MESSAGES Test(const Uint32 NumOfBlocksToTest, Uint32* const pFailedBlocks);
//+ #endif

	FLASH_API_STATUS_MESSAGES WriteEnable(void);
	void SendCommand(const Uint16 Command);
	Uint16 Send16bitWord(const Uint16 Value);
	void SendFlashAddr(const Uint32 FlashAddr);

	Uint32 GetTotalNumOfBytes(void);
	Uint32 GetTotalNumOfPages(void);
	Uint32 GetTotalNumOfBlocks(void);
	char GetUsedSpiBus(void);
	void ShowDeviceParameters(void);
	virtual Uint32 ReadDeviceId(void);		// Default method for most devices
	virtual FLASH_API_STATUS_MESSAGES VerifyDeviceId(const Uint32 DevId);
	void Reset(void);

protected:
	static const Uint32 FlashFirstAddr = 0x00000000;

	UART_COM1* poDbgCom;
	// SPI bus specific parameters:
	SPI_BUS SpiBusOfDevice;						// Remember selected SPI of device
	Uint16 DevCS;								// Chip Select pin of SPI n (Default: undefined = GPIO_0)
	volatile struct SPI_REGS* gpSpiRegs;		// Global pointer to SPI Control Registers of SPI n

	char ReceiveByte(void);
	void SetChipSelectHigh(void);
	void SetChipSelectLow(void);

private:
	// Busy is set by each API function before it determines a pass or fail condition for that operation.
	// The calling function will will not receive this status condition back from the API.
	static const Uint16 ApiBusy = 999;

	// Common Serial Flash parameters (device independent)
	static const Uint32 PageSizeInBytes = 256;						// Size in bytes per page
	static const Uint32 PageSizeInWords = PageSizeInBytes/2;		// Size in words per page
	static const Uint32 BlockSizeInBytes = 0x1000;					// Size of the smallest erasable block unit in bytes
	static const Uint32 BlockSizeInWords = BlockSizeInBytes/2;		// Size of the smallest erasable block unit in words
	// Common Serial Flash parameters (based on device specific 'TotalNumOfBytes')
	const Uint32 TotalNumOfBytes;									// Capacity in bytes (set by specific device class)
	const Uint32 FlashLastAddr;
	const Uint32 TotalNumOfWords;
	const Uint32 TotalNumOfPages;
	const Uint32 TotalNumOfBlocks;
	// CS Times
	const long double CSHoldTime;
	const long double CSHighTime;
	const long double CSSetupTime;
};

#endif // __spi_flash_h__
