/*
 * UART_COM1.cpp
 *
 *  Created on: 22.09.2015
 *      Author: richterr
 *
 *  Serial interface COM1 handling
 *
 */

#include <sstream>
#include "application.h"
#include <F2837xD_Device.h> // DSP2837xD header file peripheral address definitions
#include <F2837xD_sci.h>
//r#include <sysctl.h>
#include "UART_COM1.h"

using namespace std;

// *************************************************************************
// Defines
// *************************************************************************
// according to SYSTEM_CLOCK_SPEED (define of 'SYSTEM_CLOCK_SPEED' in driverlib/sysctl.h)
#define SYSCLK	200000000

#ifndef LSP_CLK_DIV
 #define LSP_CLK_DIV		4	//!< LSPCLK = SYSCLK / 4
#endif

// *************************************************************************
// Global variables, structures, etc
// *************************************************************************
volatile uint16_t gRxChar = 0;
volatile char gReceiveBuffer[RECEIVE_BUFFER_SIZE];		// Last character is a constant '\0' as end termination

// *************************************************************************
// C-style Definitions
// *************************************************************************
#ifdef __cplusplus
extern "C" {

/***************** doxygen like ******************************//**
 * Method: COM1_RX_ISR
 * In: ---
 * Out: ---
 * Description: interrupt service routine of receive data.
 * WARNING: Function is a stub. Extend funtionality!
// **************************************************************/
interrupt void COM1_RX_ISR(void)
{
	static uint16_t RxBufIdx = 0;

// Incomplete function: Only basic funtionality!!!!!
	gReceiveBuffer[RxBufIdx] = SciaRegs.SCIRXBUF.all & 0x00FF;	// Get data

	RxBufIdx++;
	// Reset buffer (Last character is a constant '\0' as end termination)
	if(RECEIVE_BUFFER_SIZE - 1 == RxBufIdx) {
		RxBufIdx = 0;
	}

	SciaRegs.SCIFFRX.bit.RXFFOVRCLR = 1;		// Clear Overflow flag
	SciaRegs.SCIFFRX.bit.RXFFINTCLR = 1;		// Clear Interrupt flag

	PieCtrlRegs.PIEACK.all |= 0x100;			// Issue PIE ack
}

/***************** doxygen like ******************************//**
 * Method: COM1_TX_ISR
 * In: ---
 * Out: ---
 * Description: interrupt service routine of transmit data.
// **************************************************************/
interrupt void COM1_TX_ISR(void)
{
	SciaRegs.SCIFFTX.bit.TXFFINTCLR = 1;		// Clear SCI Interrupt flag
	PieCtrlRegs.PIEACK.all |= 0x100;			// Issue PIE ACK
}

}	// end of 'extern "C"'
#endif

// *************************************************************************
// C++ Definitions
// *************************************************************************
UART_COM1::UART_COM1() {
	// Initialize receive buffer
	memcpy((void*)gReceiveBuffer, 0, RECEIVE_BUFFER_SIZE + 1);
}

UART_COM1::~UART_COM1() {
	// Auto-generated destructor stub
}

/***************** doxygen like ******************************//**
 * Method: Open
 * In: ---
 * Out: ---
 * Description: Initiallize the COM interface.
// **************************************************************/
void UART_COM1::Open(uint32_t const Baudrate, uint16_t const DataBytes, uint16_t const Parity, uint16_t const StopBits)
{
	uint32_t BAUD = ((SYSCLK / LSP_CLK_DIV) / (Baudrate * 8)) -1;
	uint16_t NumDataBytes = DataBytes - 1;
	uint16_t ParityValue = Parity;
	uint16_t NumStopBits = StopBits - 1;

	// Check parameters
	if(7 < NumDataBytes) {
		NumDataBytes = 7;
	}
	if(1 < ParityValue) {
		ParityValue = 1;
	}
	if(1 < NumStopBits) {
		NumStopBits = 1;
	}

	// Set interrupt service routines
	EALLOW;
	PieVectTable.SCIA_RX_INT = &COM1_RX_ISR;
	PieVectTable.SCIA_TX_INT = &COM1_TX_ISR;
	EDIS;

 	// Enable interrupts on SCI A
	PieCtrlRegs.PIECTRL.bit.ENPIE = 1;		// Enable the PIE block
	PieCtrlRegs.PIEIER9.bit.INTx1 = 1;		// PIE Group 9, INT1 (9.1 - SCIA Receive Interrupt)
	PieCtrlRegs.PIEIER9.bit.INTx2 = 1;		// PIE Group 9, INT2 (9.2 - SCIA Transmit Interrupt)

	// Reference: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual (spruhm8e.pdf); Chapter 19.14ff
    // Note: Clocks were turned on to the SCIA peripheral in the InitSysCtrl() function

	//  SCICCR: Chapter 19.14.2.1
	// Data format
	SciaRegs.SCICCR.all = 0x0000;			// No loopback, async mode, idle-line protocol
	SciaRegs.SCICCR.bit.SCICHAR = NumDataBytes;
	SciaRegs.SCICCR.bit.STOPBITS = NumStopBits;
	SciaRegs.SCICCR.bit.PARITYENA = ParityValue;

	// Baudrate registers: Chapter 19.14.2.3 and 19.14.2.4
	// LSPCLK = SysPLL / 4 (Chapter 2.7.3.4)
	// BAUD = (LSPCLK / (baud rate * 8)) - 1
	// @SysPLL = 200 MHz -> @LSPCLK = 50 MHz
	//	baud rate = 9600 baud:		BAUD = 0x028A => HBAUD = 0x02 and LBAUD = 0x8B.
	//	baud rate = 115200 baud:	BAUD = 0x0035 => HBAUD = 0x00 and LBAUD = 0x35.
	//
	SciaRegs.SCIHBAUD.all = (BAUD >> 8) & 0x000000FF;
	SciaRegs.SCILBAUD.all = BAUD & 0x000000FF;

	//  SCICTL2: Chapter 19.14.2.5
	SciaRegs.SCICTL2.all = 0x0000;
	SciaRegs.SCICTL2.bit.TXINTENA = 0;		// SCITXBUF-register interrupt not enabled
	SciaRegs.SCICTL2.bit.RXBKINTENA = 1;	// enable Receiver-buffer/break interrupt

	//  SCIFFTX: Chapter 19.14.2.10
	SciaRegs.SCIFFTX.all = 0x0000;
	SciaRegs.SCIFFTX.bit.TXFFIL = 0x01;		// TXFFIL4-0 Transmit FIFO interrupt level bits (request on every single data in FIFO)
	SciaRegs.SCIFFTX.bit.TXFFIENA = 0;		// Transmit FIFO interrrupt enable
	SciaRegs.SCIFFTX.bit.TXFFINTCLR = 1;	// Transmit FIFO interrupt clear
	SciaRegs.SCIFFTX.bit.TXFIFORESET = 1;	// Transmit FIFO reset
	SciaRegs.SCIFFTX.bit.SCIFFENA = 1;		// SCI FIFO enable
	SciaRegs.SCIFFTX.bit.SCIRST = 1;		// SCI Reset

	//  SCIFFRX: Chapter 19.14.2.11
	SciaRegs.SCIFFRX.all = 0x0000;
	SciaRegs.SCIFFRX.bit.RXFFIL = 0x01;		// Receive FIFO interrupt level bits (request on every single received data in FIFO)
	SciaRegs.SCIFFRX.bit.RXFFIENA = 1;		// Receive FIFO interrupt enable
	SciaRegs.SCIFFRX.bit.RXFFINTCLR = 1;	// Receive FIFO interrupt clear
	SciaRegs.SCIFFRX.bit.RXFIFORESET = 1;	// Receive FIFO reset
	SciaRegs.SCIFFRX.bit.RXFFOVRCLR = 0;	// RXFFOVF clear
	SciaRegs.SCIFFRX.bit.RXFFOVF = 0;		// Receive FIFO overflow

	//  SCIFFCT: Chapter 19.14.2.12
	SciaRegs.SCIFFCT.all = 0x0000;
	SciaRegs.SCIFFCT.bit.FFTXDLY = 0x00;	// No FIFO transfer delay
	SciaRegs.SCIFFCT.bit.CDC = 0;			// CDC calibrate A-detect bit (No auto-baudrate detection)
	SciaRegs.SCIFFCT.bit.ABDCLR = 0;		// ABD-clear bit

	//  SCICTL2: Chapter 19.14.2.2
	SciaRegs.SCICTL1.all = 0x0000;			// internal SCICLK, Disable RX ERR, SLEEP, TXWAKE
	SciaRegs.SCICTL1.bit.RXENA = 1;
	SciaRegs.SCICTL1.bit.TXENA = 1;
	SciaRegs.SCICTL1.bit.SWRESET = 1;

	SciaRegs.SCIFFTX.bit.TXFIFORESET = 1;	// Re-enable transmit FIFO operation
	SciaRegs.SCIFFRX.bit.RXFIFORESET = 1;	// Re-enable receive FIFO operation

	return;
}

/***************** doxygen like ******************************//**
 * Method: Close
 * In: ---
 * Out: ---
 * Description: Disable RX, TX and interrupts.
// **************************************************************/
void UART_COM1::Close(void)
{
	SciaRegs.SCICTL1.bit.RXENA = 0;
	SciaRegs.SCICTL1.bit.TXENA = 0;

	SciaRegs.SCICTL2.bit.TXINTENA = 0;
	SciaRegs.SCICTL2.bit.RXBKINTENA = 0;

	SciaRegs.SCIFFRX.bit.RXFFIENA = 0;
	SciaRegs.SCIFFRX.bit.RXFFINTCLR = 1;
	SciaRegs.SCIFFRX.bit.RXFIFORESET = 0;
	SciaRegs.SCIFFRX.bit.RXFFOVRCLR = 0;

	SciaRegs.SCIFFTX.bit.TXFFIENA = 0;
	SciaRegs.SCIFFTX.bit.TXFFINTCLR = 1;
	SciaRegs.SCIFFTX.bit.TXFIFORESET = 0;
	SciaRegs.SCIFFTX.bit.SCIFFENA = 0;
	SciaRegs.SCIFFTX.bit.SCIRST = 1;		// SCI Reset

	return;
}

/***************** doxygen like ******************************//**
 * Method: Write
 * In: character to send
 * Out: ---
 * Description: Transmit a character.
// **************************************************************/
void UART_COM1::Write(int const a)
{
    while (SciaRegs.SCIFFTX.bit.TXFFST != 0) {}
    SciaRegs.SCITXBUF.all =a;
    return;
}

/***************** doxygen like ******************************//**
 * Method: Write
 * In: pointer to character array (null terminated)
 * Out: ---
 * Description: Transmit a character array.
// **************************************************************/
void UART_COM1::Write(const char* const msg)
{
    int i;
    i = 0;
    while(msg[i] != '\0')
    {
        Write(msg[i]);
        i++;
    }
    return;
}

/***************** doxygen like ******************************//**
 * Method: Write (stringstream)
 * In: reference to the object
 * Out: ---
 * Description: Wrapper function to transmit a stringstream object.
// **************************************************************/
void UART_COM1::Write(const stringstream& msg)
{
	Write(msg.str().c_str());
    return;
}

/***************** doxygen like ******************************//**
 * Method: Write (ostringstream)
 * In: reference to the object
 * Out: ---
 * Description: Wrapper function to transmit a ostringstream object.
// **************************************************************/
void UART_COM1::Write(const ostringstream& msg)
{
	Write(msg.str().c_str());
    return;
}

/***************** doxygen like ******************************//**
 * Method: Read
 * In: Pointer to target buffer
 * Out: Number of data byte in buffer. NULL = no data
 * Description: Get received data bytes from RX buffer.
// **************************************************************/
uint16_t UART_COM1::Read(const char* pBuf)
{
	uint16_t NumBytes = 0;

	pBuf = (char*)gReceiveBuffer;

	return(NumBytes);
}

/***************** doxygen like ******************************//**
 * Method: ResetTerminal
 * In: ---
 * Out: ---
 * Description: Set a terminal to default settings (No colour, not bold, etc).
// **************************************************************/
void UART_COM1::ResetTerminal(void)
{
	Write("\e[0m");
	return;
}
