/*
 * W25Q256FV_Flash.cpp
 *
 *  Created on: 23.12.2015
 *      Author: richterr
 *
 *	Interface of Winbond W25Q256FV SPI Flash
 *
 */

// *************************************************************************
// Includes
// *************************************************************************
#include <sstream>
#include <string>

#include <W25Q256FV_Flash.h>
#include "gpio.h"

// *************************************************************************
// Definitions of Serial FLASH Commands and FLASH Parameters
// *************************************************************************
// Special W25Q256FV Serial Flash parameters
// -------------------------------------------------------------------------
// Type: W25Q256FV; 256Mbit Serial flash memory; Clock frequency 50 MHz
// Document: N:\pA_Datenblaetter\IC_s_allg\winbond\W25Q256FV_Winbond.pdf (Revision H - February 2015)
//	Chapter: TODO
//  256Mbit address range -> 256x1024x1024/16 = 16777216 Words
#define   FLASH_TOTAL_BYTES		0x2000000

// CS timing parameters (in us)
// Document: N:\pA_Datenblaetter\IC_s_allg\winbond\W25Q256FV_Winbond.pdf (Revision H - February 2015)
//	Chapter: 9.6 AC Electrical Charateristics
#define   CS_HIGH		0.008L		// Clock High for Read Data (03h) [s]
#define   CS_HOLD		0.003L		// Data in Hold Time [s]
#define   CS_SETUP		0.002L		// Data in Setup Time [s]

// W25Q256FV SPI Configuration Settings
// These are set in the InitSpi() function
//
// SPI Baud Rate Register
// Reference: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual (spruhm8e.pdf); Chapter 18.5.2.4
// LSPCLK = SYSCLK / LSP_CLK_DIV = 200MHz / 2 = 100.0MHz
// On SPI_BRR = 0, 1 or 2 -> BRR = LSPCLK / 4 = 100.0 MHz / 4 = 25.0 MHz
// On SPI_BRR = 3, 4, ..., 127 -> BRR = LSPCLK / (SPI_BRR + 1)
#define SPI_BAUD  0x0001		// SPI Baud Rate

// *************************************************************************
// Definitions of methods
// *************************************************************************
W25Q256FV_Flash::W25Q256FV_Flash() : SpiFlash(FLASH_TOTAL_BYTES, SPI_C, CS_HOLD, CS_HIGH, CS_SETUP)
{

}

W25Q256FV_Flash::~W25Q256FV_Flash() {
	// TODO Auto-generated destructor stub
}

/***************** doxygen like ******************************//**
 * Method: InitSPI
 * In: ---
 * Out: Serial FLASH API Status Messages
 * Description: Initialize SPI C.
// **************************************************************/
FLASH_API_STATUS_MESSAGES W25Q256FV_Flash::InitSPI(void)
{
	///   Logger_info0("Initialize SPI C for W25Q256FV Flash.");
	   EALLOW;
	   CpuSysRegs.PCLKCR8.bit.SPI_C = 1;  // Enable the SPI C clock
	   //CpuSysRegs.LOSPCP.all = LSPCLK_DIV;

	   EDIS;

	   SpicRegs.SPICCR.bit.SPISWRESET = 0;	// Perform SPI Software Reset
	   SpicRegs.SPIFFTX.all = 0x8000;		// SPI Transmit FIFO enable
	   SpicRegs.SPICCR.bit.SPICHAR = 0x7;	// Transfer 8 Bit
	   SpicRegs.SPICCR.bit.SPILBK = 0;		// SPI loopback mode disabled
	   SpicRegs.SPICTL.all = 0x000E;		// SPI CLK signal delayed one half-cycle, config as master
	   SpicRegs.SPIBRR.bit.SPI_BIT_RATE = SPI_BAUD;		// Baudrate, for 0,1 or 2 = LSPCLK/4
	   SpicRegs.SPICCR.bit.SPISWRESET = 1;	// Disable SPI Software Reset
	   SetChipSelectHigh();					// Set CSn high (inactive)

	   return(SUCCESSFUL);
}

/***************** doxygen like ******************************//**
 * Method: VerifyDeviceId
 * In: Device ID from SPI device
 * Out: Serial FLASH API Status Messages
 * Description: Verify if ID from Flash is same as in datasheet.
 *	Manufacturer ID ist JEDEC coded and unique for every Manufacturer. Device Type/Id and/or Capacity are
 *	Manufacturer specific and can change for compatible but different device.
 *	Chip should return a 3-Byte code with following values :
 *
 *
 *	|	Chip		|	Manufacturer	|	JEDEC-ID	|	Device-Type	|	Capacity	|
 *	|_______________|___________________|_______________|_______________|_______________|
 *	|				|					|				|				|				|
 *	|	W25Q256FV	|	Winbond			|	0xEF		|	0x40		|	0x19		|
// **************************************************************/
FLASH_API_STATUS_MESSAGES W25Q256FV_Flash::VerifyDeviceId(const Uint32 DevId)
{
	if (0x1940EF == DevId) {			// Winbond W25Q256FV
		return(SUCCESSFUL);
	}
	return(FAIL_DEVICE_ID);
}
