/* *********************************************************************
* Devices: TMS320F2837xD
* Author: Rudi Richter
* Description: Configures all GPIOS for the TT3100 board
********************************************************************* */

// Logger usage
#define  __gpio_cpp__
/// #include "logger_setup.h"

#include <xdc/std.h>
#include "compiler.h"
#include <F2837xD_Device.h> // DSP2837xD header file peripheral address definitions
#include "F2837xD_Examples.h"
#include <F2837xD_GlobalPrototypes.h>
#include <F2837xD_Gpio.h>
#include <F2837xD_Gpio_defines.h>
#include "gpio.h" // Module header

#ifdef __cplusplus
extern "C" {
#endif

/* *********************************************************************
* Explicit Declarations
********************************************************************* */
extern void setup_emif1_pinmux_sdram_16bit(Uint16);
extern void setup_emif2_pinmux_async_16bit(Uint16);

/* *********************************************************************
* Description: Initializes the shared GPIO pins on the F2837xD
********************************************************************* */
void Init_TT3100_A_gpio(void)
{
	EALLOW;		 							// Enable EALLOW protected register access

// TODO Check --- Group A pins - GPIO0 to GPIO31
	GpioCtrlRegs.GPACTRL.all  = 0x00000000;		// QUALPRD = SYSCLKOUT for all group A GPIO
	GpioCtrlRegs.GPAQSEL1.all = 0x00000000;		// No qualification for all group A GPIO 0-15
	GpioCtrlRegs.GPAQSEL2.all = 0x00000000;		// No qualification for all group A GPIO 16-31
	GpioCtrlRegs.GPADIR.all   = 0x00000000;		// All group A GPIO are inputs
	GpioCtrlRegs.GPAPUD.all   = 0x00000000;		// all Inputs Pulled up

// TODO Check --- Group B pins - GPIO32 to GPIO63
	GpioCtrlRegs.GPBCTRL.all  = 0x00000000;		// QUALPRD = SYSCLKOUT for all group B GPIO
	GpioCtrlRegs.GPBQSEL1.all = 0x00000000;		// No qualification for all group B GPIO 32-47
	GpioCtrlRegs.GPBQSEL2.all = 0x00000000;		// No qualification for all group B GPIO 48-63
	GpioCtrlRegs.GPBDIR.all   = 0x00000000;		// All group B GPIO are inputs
	GpioCtrlRegs.GPBPUD.all   = 0x0000FFFF;		// GPIO64-48 pull up, 47-32 disabled (Addressbus)

// TODO Check --- Group C pins - GPIO64 to GPIO87 (Address Bus Interface)
	GpioCtrlRegs.GPCDIR.all   = 0x00000000;		// All group C GPIO are inputs
	GpioCtrlRegs.GPCPUD.all   = 0xFFFFFFFF;		// All group C pullups disabled

// EMIF1 module for external RAM. Do first (TI usual configuration inside) then TT3100 specific configuration.
	setup_emif1_pinmux_sdram_16bit(GPIO_MUX_CPU1);

// EMIF2 module for FPGA. Do first (TI usual configuration inside) then TT3100 specific configuration.
	setup_emif2_pinmux_async_16bit(GPIO_MUX_CPU1);
	// Chip control FPGA
	GPIO_SetupPinMux(CS0_FPGA_N, GPIO_MUX_CPU1, 3);	GPIO_WritePin(CS0_FPGA_N, 1);	// Inactive by default
	GPIO_SetupPinMux(CS2_FPGA_N, GPIO_MUX_CPU1, 3);	GPIO_WritePin(CS2_FPGA_N, 1);	// Inactive by default
	GPIO_SetupPinMux(FRW_N, GPIO_MUX_CPU1, 3);		GPIO_WritePin(FRW_N, 1);		// Inactive by default
	GPIO_SetupPinMux(FWE_N, GPIO_MUX_CPU1, 3);		GPIO_WritePin(FWE_N, 1);		// Inactive by default
	GPIO_SetupPinMux(FOE_N, GPIO_MUX_CPU1, 3);		GPIO_WritePin(FOE_N, 1);		// Inactive by default
	// Clock FPGA
	GPIO_SetupPinMux(EM2CLK, GPIO_MUX_CPU1, 3);

// TODO Defined, but not used
	GPIO_SetupPinMux(BOOT0, GPIO_MUX_CPU1, 0);			GPIO_SetupPinOptions(BOOT0, GPIO_INPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(DSP_FPGA1, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(DSP_FPGA1, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_WritePin(DSP_FPGA1, 0);

// PWM Outputs 1A to 12B
	GPIO_SetupPinMux(PWM1A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM1A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM1B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM1B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM2A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM2A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM2B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM2B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM3A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM3A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM3B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM3B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM4A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM4A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM4B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM4B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM5A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM5A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM5B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM5B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM6A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM6A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM6B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM6B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM7A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM7A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM7B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM7B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM8A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM8A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM8B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM8B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM9A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM9A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM9B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM9B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM10A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM10A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM10B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM10B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM11A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM11A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM11B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM11B_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM12A_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM12A_DSP, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(PWM12B_DSP, GPIO_MUX_CPU1, 1);		// GPIO_SetupPinOptions(PWM12B_DSP, GPIO_OUTPUT, GPIO_PULLUP);

// Hardware Reference Inputs (GPIO 27, 26, 110)
	// HWREF2
	GPIO_SetupPinMux(HWREF2, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(HWREF2, GPIO_INPUT, GPIO_PULLUP);
	// HWREF3
	GPIO_SetupPinMux(HWREF3, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(HWREF3, GPIO_INPUT, GPIO_PULLUP);
	// HWREF1
	GPIO_SetupPinMux(HWREF1, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(HWREF1, GPIO_INPUT, GPIO_PULLUP);

// FPGA to DSP test pins
	GPIO_SetupPinMux(TEST_FPGA1, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(TEST_FPGA1, GPIO_INPUT, GPIO_PULLUP);
	GPIO_SetupPinMux(TEST_FPGA2, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(TEST_FPGA2, GPIO_INPUT, GPIO_PULLUP);

// JTAG interface to the Lattice FPGA. Default is 'Inactive'. All pins set as inputs.
	// FPGA TDI pin:
	GPIO_SetupPinMux(FPGA_TDI, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(FPGA_TDI, GPIO_INPUT, GPIO_PUSHPULL);
	// FPGA TDO pin:
	GPIO_SetupPinMux(FPGA_TDO, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(FPGA_TDO, GPIO_INPUT, GPIO_PUSHPULL);
	// FPGA TCK pin:
	GPIO_SetupPinMux(FPGA_TCK, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(FPGA_TCK, GPIO_INPUT, GPIO_PUSHPULL);
	// FPGA TMS pin:
	GPIO_SetupPinMux(FPGA_TMS, GPIO_MUX_CPU1, 0);	GPIO_SetupPinOptions(FPGA_TMS, GPIO_INPUT, GPIO_PUSHPULL);

// CAN Interface Module A
	// CANA RX
	GPIO_SetupPinMux(CAN1_TXD, GPIO_MUX_CPU1, 6);		// GPIO_SetupPinOptions(CAN1_TXD, GPIO_OUTPUT, GPIO_PULLUP);
	// CANA TX
	GPIO_SetupPinMux(CAN1_RXD, GPIO_MUX_CPU1, 6);		// GPIO_SetupPinOptions(CAN1_RXD, GPIO_INPUT, GPIO_PULLUP);

// CAN Interface Module B
	// CANB TX
	GPIO_SetupPinMux(CAN2_TXD, GPIO_MUX_CPU1, 6);		// GPIO_SetupPinOptions(CAN2_TXD, GPIO_OUTPUT, GPIO_PULLUP);
	// CANB RX
	GPIO_SetupPinMux(CAN2_RXD, GPIO_MUX_CPU1, 6);		// GPIO_SetupPinOptions(CAN2_RXD, GPIO_INPUT, GPIO_PULLUP);

// user UART (SCIB/COM1)
	// SCIB TX
	GPIO_SetupPinMux(COM1_TX, GPIO_MUX_CPU1, 6);		GPIO_SetupPinOptions(COM1_TX, GPIO_INPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SCIB RX
	GPIO_SetupPinMux(COM1_RX, GPIO_MUX_CPU1, 6);		GPIO_SetupPinOptions(COM1_RX, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);

// diagnostic UART (SCIC/COM2)
	// SCIC TX
	GPIO_SetupPinMux(COM2_TX, GPIO_MUX_CPU1, 2);		GPIO_SetupPinOptions(COM2_TX, GPIO_INPUT, GPIO_PULLUP);
	// SCIC RX
	GPIO_SetupPinMux(COM2_RX, GPIO_MUX_CPU1, 2);		GPIO_SetupPinOptions(COM2_RX, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);

// SPI FRAM (SPI A)
	// SPI Serial IN
	GPIO_SetupPinMux(SPISOMIA, GPIO_MUX_CPU1, 1);		GPIO_SetupPinOptions(SPISOMIA, GPIO_INPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Serial Out
	GPIO_SetupPinMux(SPISIMOA, GPIO_MUX_CPU1, 1);		GPIO_SetupPinOptions(SPISIMOA, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Clock
	GPIO_SetupPinMux(SPICLKA, GPIO_MUX_CPU1, 1);		GPIO_SetupPinOptions(SPICLKA, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Chip Select
	//  See Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
	//  SPI Module Signals: Chapter 18.2.1
	//  GPIO Index = 1: Use SPI interface signal |SPISTEA as signal |CS, managed automatically by SPI interface
	//  GPIO Index = 0: Use common GPIO function as signal |CS, managed explicit by SPI functions
	GPIO_SetupPinMux(SPIEA_N, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(SPIEA_N, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(SPIEA_N, 1);			// Inactive by default; without effect on GPIO Index = 1

// SPI Data Flash (SPI C)
	// SPI Serial IN
	GPIO_SetupPinMux(SPISOMIC, GPIO_MUX_CPU1, 6);		GPIO_SetupPinOptions(SPISOMIC, GPIO_INPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Serial Out
	GPIO_SetupPinMux(SPISIMOC, GPIO_MUX_CPU1, 6);		GPIO_SetupPinOptions(SPISIMOC, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Clock
	GPIO_SetupPinMux(SPICLKC, GPIO_MUX_CPU1, 6);		GPIO_SetupPinOptions(SPICLKC, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	// SPI Chip Select
	//  See Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
	//  SPI Module Signals: Chapter 18.2.1
	//  GPIO Index = 6: Use SPI interface signal |SPISTEC as signal |CS, managed automatically by SPI interface
	//  GPIO Index = 0: Use common GPIO function as signal |CS, managed explicit by SPI functions
	GPIO_SetupPinMux(SPIEC_N, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(SPIEC_N, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(SPIEC_N, 1);			// Inactive by default; without effect on GPIO Index = 6

// DSP Test Outputs
	// TEST_DSP1
	GPIO_SetupPinMux(TEST_DSP1, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(TEST_DSP1, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	// TEST_DSP2
	GPIO_SetupPinMux(TEST_DSP2, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(TEST_DSP2, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);

// DSP Jumper 1 and 2
	// JMP_DSP1
	GPIO_SetupPinMux(JMP_DSP1, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(JMP_DSP1, GPIO_INPUT, GPIO_PULLUP);
	// JMP_DSP2
	GPIO_SetupPinMux(JMP_DSP2, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(JMP_DSP2, GPIO_INPUT, GPIO_PULLUP);

// Enable Inputs for Driver Stages Status
	// ENABLE_DRV12
	GPIO_SetupPinMux(ENABLE_DRV12, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV12, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV12, 0);
	// ENABLE_DRV34
	GPIO_SetupPinMux(ENABLE_DRV34, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV34, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV34, 0);
	// ENABLE_DRV56
	GPIO_SetupPinMux(ENABLE_DRV56, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV56, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV56, 0);
	// ENABLE_DRV78
	GPIO_SetupPinMux(ENABLE_DRV78, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV78, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV78, 0);
	// ENABLE_DRV910
	GPIO_SetupPinMux(ENABLE_DRV910, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV910, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV910, 0);
	// ENABLE_DRV1112
	GPIO_SetupPinMux(ENABLE_DRV1112, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(ENABLE_DRV1112, GPIO_OUTPUT, GPIO_PULLUP | GPIO_ASYNC);
	GPIO_WritePin(ENABLE_DRV1112, 0);

// DSP LED Output
	// DSP_LED
	GPIO_SetupPinMux(DSP_LED, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(DSP_LED, GPIO_OUTPUT, GPIO_PULLUP);
	GPIO_WritePin(DSP_LED, 0); 		// set the port to low

/*/ TODO Check --- External interrupt selection
// IRQ from FPGA
	// Interrupt 0
	GPIO_SetupPinMux(INT0, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(INT0, GPIO_INPUT, GPIO_PULLUP);
	// Interrupt 1
	GPIO_SetupPinMux(INT1, GPIO_MUX_CPU1, 0);		GPIO_SetupPinOptions(INT1, GPIO_INPUT, GPIO_PULLUP);

	GpioIntRegs.GPIOXINT1SEL.all = 0x0000;		// GPIO0 is XINT1 source
	GpioIntRegs.GPIOXINT2SEL.all = 0x0000;		// GPIO0 is XINT2 source
	GpioIntRegs.GPIOXINT3SEL.all = 0x0000;		// GPIO32 is XINT3 source
	GpioIntRegs.GPIOXINT4SEL.all = 0x0000;		// GPIO32 is XINT4 source
	GpioIntRegs.GPIOXINT5SEL.all = 0x0000;		// GPIO32 is XINT5 source
	GpioIntRegs.GPIOXINT6SEL.all = 0x0000;		// GPIO32 is XINT6 source
	GpioIntRegs.GPIOXINT7SEL.all = 0x0000;		// GPIO32 is XINT7 source
	GpioIntRegs.GPIOXNMISEL.all = 0x0000;		// GPIO0 is XNMI source
	XIntruptRegs.XINT1CR.all = 0x0000;			// XINT1 disabled
	XIntruptRegs.XINT2CR.all = 0x0000;			// XINT2 disabled
	XIntruptRegs.XINT3CR.all = 0x0000;			// XINT3 disabled
	XIntruptRegs.XINT4CR.all = 0x0000;			// XINT4 disabled
	XIntruptRegs.XINT5CR.all = 0x0000;			// XINT5 disabled
	XIntruptRegs.XINT6CR.all = 0x0000;			// XINT6 disabled
	XIntruptRegs.XINT7CR.all = 0x0000;			// XINT7 disabled
	XIntruptRegs.XNMICR.all = 0x0000;			// XNMI disabled
// */

// TODO Check --- Low-power mode selection
/// todo	GpioIntRegs.GPIOLPMSEL.all = 0x00000000;	// No pin selected for HALT and STANDBY wakeup (reset default)

//--- Finish up
	EDIS;										// Disable EALLOW protected register access
}


#ifdef __cplusplus
}
#endif /* extern "C" */



/*********************************************************************************************************
 * High Level Functions, which are C++ compatible
 * ******************************************************************************************************/
GPIO_Class::GPIO_Class()
{
	// Auto-generated constructor stub
}

GPIO_Class::~GPIO_Class() {
	// Auto-generated destructor stub
}

void GPIO_Class::InitializeGpio(void)
{
	// Default initialization: Sets all pins to be muxed to GPIO in input mode with pull-ups enabled.
	InitGpio();
	// TT3100 module specific configuration
	Init_TT3100_A_gpio();
	return;
}

// Switches the DSP LED on or off.
void GPIO_Class::SwitchDspLedOn(const bool on)
{
	if (on)
	{
		GPIO_WritePin(DSP_LED, 1);
	}
	else
	{
		GPIO_WritePin(DSP_LED, 0);
	}

	return;
}


// Toggles the DSP LED on or off.
void GPIO_Class::ToggleDspLed(void)
{
	GPIO_TogglePin(DSP_LED);
	return;
}

bool GPIO_Class::IsDriverEnabled(const unsigned int IndexPairOfDrivers)
{
	bool bValue = false;

	switch (IndexPairOfDrivers)
	{
		case IndexDriverPair12:
			bValue = GPIO_ReadPin(ENABLE_DRV12);
			break;
		case IndexDriverPair34:
			bValue = GPIO_ReadPin(ENABLE_DRV34);
			break;
		case IndexDriverPair56:
			bValue = GPIO_ReadPin(ENABLE_DRV56);
			break;
		case IndexDriverPair78:
			bValue = GPIO_ReadPin(ENABLE_DRV78);
			break;
		case IndexDriverPair910:
			bValue = GPIO_ReadPin(ENABLE_DRV910);
			break;
		case IndexDriverPair1112:
			bValue = GPIO_ReadPin(ENABLE_DRV1112);
			break;
		default:
/// todo			Logger_error1("Pair index %d is out of range.", channel);
			break;
	}
	return(bValue);
}

void GPIO_Class::EnableDriver(const unsigned int IndexPairOfDrivers)
{
	switch (IndexPairOfDrivers)
	{
		case IndexDriverPair12:
			GPIO_WritePin(ENABLE_DRV12, DriverEnable);
			break;
		case IndexDriverPair34:
			GPIO_WritePin(ENABLE_DRV34, DriverEnable);
			break;
		case IndexDriverPair56:
			GPIO_WritePin(ENABLE_DRV56, DriverEnable);
			break;
		case IndexDriverPair78:
			GPIO_WritePin(ENABLE_DRV78, DriverEnable);
			break;
		case IndexDriverPair910:
			GPIO_WritePin(ENABLE_DRV910, DriverEnable);
			break;
		case IndexDriverPair1112:
			GPIO_WritePin(ENABLE_DRV1112, DriverEnable);
			break;
		default:
/// todo			Logger_error1("Pair index %d is out of range.", channel);
			break;
	}
	return;
}

void GPIO_Class::EnableAllDrivers(void)
{
	GPIO_WritePin(ENABLE_DRV12, DriverEnable);
	GPIO_WritePin(ENABLE_DRV34, DriverEnable);
	GPIO_WritePin(ENABLE_DRV56, DriverEnable);
	GPIO_WritePin(ENABLE_DRV78, DriverEnable);
	GPIO_WritePin(ENABLE_DRV910, DriverEnable);
	GPIO_WritePin(ENABLE_DRV1112, DriverEnable);
	return;
}

void GPIO_Class::DisableDriver(const unsigned int IndexPairOfDrivers)
{
	switch (IndexPairOfDrivers)
	{
		case IndexDriverPair12:
			GPIO_WritePin(ENABLE_DRV12, DriverDisable);
			break;
		case IndexDriverPair34:
			GPIO_WritePin(ENABLE_DRV34, DriverDisable);
			break;
		case IndexDriverPair56:
			GPIO_WritePin(ENABLE_DRV56, DriverDisable);
			break;
		case IndexDriverPair78:
			GPIO_WritePin(ENABLE_DRV78, DriverDisable);
			break;
		case IndexDriverPair910:
			GPIO_WritePin(ENABLE_DRV910, DriverDisable);
			break;
		case IndexDriverPair1112:
			GPIO_WritePin(ENABLE_DRV1112, DriverDisable);
			break;
		default:
/// todo			Logger_error1("Pair index %d is out of range.", channel);
			break;
	}
	return;
}

void GPIO_Class::DisableAllDrivers(void)
{
	GPIO_WritePin(ENABLE_DRV12, DriverDisable);
	GPIO_WritePin(ENABLE_DRV34, DriverDisable);
	GPIO_WritePin(ENABLE_DRV56, DriverDisable);
	GPIO_WritePin(ENABLE_DRV78, DriverDisable);
	GPIO_WritePin(ENABLE_DRV910, DriverDisable);
	GPIO_WritePin(ENABLE_DRV1112, DriverDisable);
	return;
}

int GPIO_Class::GetBootloaderJumperGPIO(void)
{
	bool aJumperStatus = false;

	aJumperStatus = !GPIO_ReadPin(JMP_DSP2);

	return(aJumperStatus);
}

int GPIO_Class::GetJumperDSP1(void)
{
	bool aJumperStatus = false;

	aJumperStatus = !GPIO_ReadPin(JMP_DSP1);

	return(aJumperStatus);
}

int GPIO_Class::GetInput(const int channel)
{
	bool aDigitalValue = false;

	switch(channel)
	{

	default:
		aDigitalValue = false;
		break;
	}

	return(aDigitalValue);
}
