//###########################################################################
// FILE:   main.c
// TITLE:  System test of TT3100 module.
//
//!
//
//###########################################################################

#include <stdlib.h>
#include <sstream>
#include <string>
#include "compiler.h"
#include "application.h"
#include "F2837xD_device.h"        // F2837xD Headerfile Include File
#include "F2837xD_Examples.h"    // F2837xD Examples Include File
#include "gpio.h"
#include "UART_COM1.h"
#include "SDRAM.h"
#include "FPGA.h"
#include "W25Q256FV_Flash.h"
#include "FM25V05_Flash.h"
#include "CAN.h"

//Include Flash API example header file
#include "flash_programming_c28.h"

#ifdef TI_controlCARD
 #include "F2837xD_Gpio_defines.h"
#endif

using namespace std;

//*****************************************************************************
// FILE Flash API include file
//*****************************************************************************
extern "C" {
 #include "F021_F2837xD_C28x.h"
} // end extern "C"

//*****************************************************************************
// Defines
//*****************************************************************************
// SD-RAM test block size
#define SIZE_SDRAM_TEST_BLOCK	32
// Output colours
#define TERM_COLOUR_BEGIN	31
#define TERM_COLOUR_END		35
// terminal codes
#define TERM_CLEAR_LINE		"\033[K"
#define TERM_LINE_UP		"\033[1A"
//Data/Program Buffer used for testing the flash API functions
#define  WORDS_IN_FLASH_BUFFER    0xFF  // Programming data buffer, words

// Debugging addresses of FPGA (FPGA test program only)
#define FPGA_ADDR_DEBUG_READ_WRITE			200
#define FPGA_ADDR_DEBUG_READ_INCREMENT		201
#define FPGA_ADDR_DEBUG_WRITE_INCREMENT		202
#define FPGA_ADDR_DEBUG_READ_SHIFT			203
#define FPGA_ADDR_DEBUG_WRITE_SHIFT			204

// DEBUG ONLY - Temporary additional LEDs to test interrupts (Special routing in FPGA)
#define LED_RUN		132		// FPGA: TEST_FPGA1 -> LED_RUN
#define LED_READY	134		// FPGA: TEST_FPGA2 -> LED_READY
#define LED_ERROR	36		// FPGA: DSP_FPGA1 -> LED_ERROR

//*****************************************************************************
// Type Defines, Enums, ...
//*****************************************************************************
//! This data type is used to set the prescaler of the WDCR register. Configure the watchdog counter clock (WDCLK) rate relatice to INTOSC1/512.
//*****************************************************************************
typedef enum {
	//! Watchdog periode in ms.
	WDPeriode_14ms5 = 1,
	WDPeriode_29ms,
	WDPeriode_58ms,
	WDPeriode_116ms,
	WDPeriode_232ms,
	WDPeriode_464ms,
	WDPeriode_926ms
} tWDPrescaler;


//*****************************************************************************
// Global variables
//*****************************************************************************
class GPIO_Class goGPIO;				// General GPIO
class UART_COM1 goCOM1;					// Serial COM port 1
class SDRAM goSdram;					// SD-RAM on EMIF1
class FPGA goFpga;						// FPGA on EMIF2
class W25Q256FV_Flash goSpiFlash;		// W25Q256FV Flash on SPI C
class FM25V05_Flash goSpiFRAM;			// FM25V05 Flash on SPI A
class CAN goCanA(CANbusA);				// CAN A interface

//*****************************************************************************
// Prototype of the functions used in this example
//*****************************************************************************
#ifdef FLASH
 void CallFlashAPI(void);
#endif
void HandleError(Fapi_StatusType status);
void TestOfCAN(void);
void Watchdog_Enable(const tWDPrescaler Prescaler, const bool NoResetPerformIRQ);
void Watchdog_Trigger(void);
bool Watchdog_IsWatchdogResetOccured(void);
void PrintDetailedDspInfo(const Uint16 RegH, const Uint32 RegL);
void DoFinaleLoop(void);

//*****************************************************************************
// This is an example code demonstrating F021 Flash API usage.
// This code is in Flash
//*****************************************************************************
interrupt void WD_IRQ(void)
{
	static uint16_t Count = 0;

	GPIO_TogglePin(LED_ERROR);

	goCOM1.Write("Watchdog IRQ\r\n\0");
	Count++;
	if(5 == Count) {
		// Enable the watchdog in RESET mode
		Watchdog_Enable(WDPeriode_29ms, false);
		Count = 0;
	}

	// Acknowledge this interrupt to get more from group 1
	PieCtrlRegs.PIEACK.all = PIEACK_GROUP1;
}

// **************************************************************/
void main(void)
{
	Uint16 Result = 0;
	Uint16 loop = 0;
    static stringstream oDbgOutput;

	// *****************************************************************************
	// Get DSP information
	// Document: TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual; ID: SPRUHM8E
	// *****************************************************************************
	// Get 'Device Configuration Registers': PartID Low: Chapter 2.14.9.2; PartID High: Chapter 2.14.9.3
    Uint32 PartIdL = DevCfgRegs.PARTIDL.all;
    Uint16 PartIdH = DevCfgRegs.PARTIDH.all;
    // Get 'Reset Cause Register': Chapter 2.14.11.25
	Uint32 ResetCauses = CpuSysRegs.RESC.all;

// Step 1. Initialize System Control:
	// Enable Peripheral Clocks
	// This example function is found in the F2837xD_SysCtrl.c file.
    InitSysCtrl();

	uint16_t LspClkDiv = ClkCfgRegs.LOSPCP.bit.LSPCLKDIV;

//  Unlock CSM
//
//  If the API functions are going to run in unsecured RAM
//  then the CSM must be unlocked in order for the flash
//  API functions to access the flash.
//  If the flash API functions are executed from secure memory
//  then this step is not required.
    //DcsmZ1Unlock();

// Step 2. Initialize GPIO:
    //	TT3100 GPIO initialiazation
	goGPIO.InitializeGpio();
#ifdef TI_controlCARD
	GPIO_SetupPinMux(31, GPIO_MUX_CPU1, 0);		// LED LD2 via CPU1
	GPIO_SetupPinOptions(31, GPIO_OUTPUT, GPIO_PUSHPULL);
#endif
	// Turn DSP LED on and toggle 20 times
	goGPIO.SwitchDspLedOn(true);
	for(loop = 0; loop < 20; loop++) {
#ifdef TI_controlCARD
	GpioDataRegs.GPATOGGLE.bit.GPIO31 = 1;
#endif
		goGPIO.ToggleDspLed();
		DELAY_US(1000*100);
	}

// Step 3. Clear all interrupts and initialize PIE vector table:
    // Disable CPU interrupts
    DINT;

	// Initialize the PIE control registers to their default state.
	// The default state is all PIE interrupts disabled and flags
	// are cleared.
    InitPieCtrl();

    // Disable CPU interrupts and clear all CPU interrupt flags:
	EALLOW;			// This is needed to write to EALLOW protected registers
	IER = 0x0000;
	IFR = 0x0000;
	EDIS;			// This is needed to disable write to EALLOW protected registers

	// Initialize the PIE vector table with pointers to the shell Interrupt
	// Service Routines (ISR).
	// This will populate the entire table, even if the interrupt
	// is not used in this example.  This is useful for debug purposes.
	// The shell ISR routines are found in DSP2802x_DefaultIsr.c.
	InitPieVectTable();

	// Initialize SCI A (COM1)
	// Initialize COM1: 115200, 8N1, Hardware control NONE
	goCOM1.Open(115200, 8, 0, 1);

 	// Enable interrupts
	IER = 0x100; // Enable CPU INT
   // Enable global Interrupts and higher priority real-time debug events:
	EINT;  // Enable Global interrupt INTM
	ERTM;  // Enable Global realtime interrupt DBGM

	// Reset terminal to default output attributes
	goCOM1.ResetTerminal();
	// Test message to send
	goCOM1.Write("\r\nStart!\r\n\0");

// Copy time critical code and Flash setup code to RAM
// This includes InitFlash(), Flash API functions and any functions that are
// assigned to ramfuncs section.
// The  RamfuncsLoadStart, RamfuncsLoadEnd, and RamfuncsRunStart
// symbols are created by the linker. Refer to the device .cmd file.
///	memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (size_t)&RamfuncsLoadSize);

#ifdef FLASH
	// Call Flash Initialization to setup flash waitstates
	// This function must reside in RAM
	InitFlash();

    // Gain pump semaphore
	SeizeFlashPump();

	// Jump to RAM and call the Flash API functions
	CallFlashAPI();
#endif

	// Print DSP info (Device Configuration Registers)
	oDbgOutput.str("");
	goCOM1.Write("DSP Config: H 0x\0");
	oDbgOutput << (hex) << PartIdH << " | L 0x";
	oDbgOutput << PartIdL << (dec) << "\r\n" << endl;
	goCOM1.Write(oDbgOutput);
	PrintDetailedDspInfo(PartIdH, PartIdL);

	oDbgOutput.str("");
	goCOM1.Write("CPU1 system register RESC: 0x\0");
	oDbgOutput << (hex) << ResetCauses << dec << "\r\n" << endl;
	goCOM1.Write(oDbgOutput);

	oDbgOutput.str("");
	goCOM1.Write("CPU1 Low Speed Clock Prescaler register LOSPCP: 0x\0");
	oDbgOutput << LspClkDiv << "\r\n" << endl;
	goCOM1.Write(oDbgOutput);

	// External RAM reset test
	const uint32_t cRAMSignatureAdressLow =		0x00FFFFFC;
	const uint32_t cRAMSignatureAdressHigh =	0x00FFFFFE;
	const uint16_t cRAMSignatureLow =	0xCAFE;
	const uint16_t cRAMSignatureHigh =	0xF369;
	// Test Poweron/Reset signature in SD-RAM
	uint16_t SignatureLow = goSdram.ReadInteger(cRAMSignatureAdressLow);
	uint16_t SignatureHigh = goSdram.ReadInteger(cRAMSignatureAdressHigh);
	// If correct signature, so a reset has occured
	if((cRAMSignatureHigh == SignatureHigh) && (cRAMSignatureLow == SignatureLow)) {
		goCOM1.Write("Correct RAM signature 0xCAFEF369\r\n\n");
	}
	else {	// PowerOn Startup
		oDbgOutput.str("");
		oDbgOutput << "Wrong RAM signature: 0x"<< (hex) << SignatureHigh << SignatureLow << dec << "\r\n\n" << endl;
		goCOM1.Write(oDbgOutput);
		// Set RAM signature
		goSdram.WriteInteger(cRAMSignatureAdressHigh, cRAMSignatureHigh);
		goSdram.WriteInteger(cRAMSignatureAdressLow, cRAMSignatureLow);
	}

	// Initialize EMIF2 interface to FPGA. Enable clock to FPGA
	Result = goFpga.InitEMIF2();
	oDbgOutput.str("");
	oDbgOutput << "Fpga InitEMIF2: " << Result << "\r" << endl;
	goCOM1.Write(oDbgOutput);
	// Get status of EMIF2
	Result = goFpga.GetEMIF2Status();
	oDbgOutput.str("");
	oDbgOutput << "Fpga GetEMIF2Status: " << hex << Result << dec << "\r" << endl;
	goCOM1.Write(oDbgOutput);

	// Initialize EMIF1 (CPU1 access) interface to SD-RAM
	Result = goSdram.InitOnCPU1();
	oDbgOutput.str("");
	oDbgOutput << "Sdram InitOnCPU1: " << Result << "\r" << endl;
	goCOM1.Write(oDbgOutput);

	Uint32 FlashID = 0;
	// Initialize W25Q256FV SPI Flash on SPI C
	goSpiFlash.InitSPI();
	goCOM1.Write("\nW25Q256FV on SPI C Init done\r\n");
	// Set debug COM device
	goSpiFlash.SetDebugCOM(&goCOM1);
	// Print parameters
	goSpiFlash.ShowDeviceParameters();
	// Get device ID
	FlashID = goSpiFlash.ReadDeviceId();
	oDbgOutput.str("");
	oDbgOutput << "Device ID: 0x" << (hex) << FlashID << (dec) << "\r" << endl;
	goCOM1.Write(oDbgOutput);
	// Verify device ID
	Result = goSpiFlash.VerifyDeviceId(FlashID);
	oDbgOutput.str("");
	oDbgOutput << "ID Verification: " << Result << "\r" << endl;
	goCOM1.Write(oDbgOutput);

	// Initialize FM25V05 SPI F-RAM Flash on SPI A
	goSpiFRAM.InitSPI();
	goCOM1.Write("\nFM25V05 on SPI A Init done\r\n");
	// Set debug COM device
	goSpiFRAM.SetDebugCOM(&goCOM1);
	// Print parameters
	goSpiFRAM.ShowDeviceParameters();
	// Get device ID
	FlashID = goSpiFRAM.ReadDeviceId();
	oDbgOutput.str("");
	oDbgOutput << "Device ID: 0x" << (hex) << FlashID << (dec) << "\r" << endl;
	goCOM1.Write(oDbgOutput);
	// Verify device ID
	Result = goSpiFRAM.VerifyDeviceId(FlashID);
	oDbgOutput.str("");
	oDbgOutput << "ID Verification: " << Result << "\r\n" << endl;
	goCOM1.Write(oDbgOutput);

	/*/ Only if reset before was not a reset by watchdog, activate watchdog IRQ (Inside: After N WD IRQ enable WD reset)
	if(false == Watchdog_IsWatchdogResetOccured()) {
		// Enable the watchdog in interrupt mode
		Watchdog_Enable(WDPeriode_926ms, true);
		goCOM1.Write("Enable WD IRQ mode\r\n\n");
	} // */

	// First test of CAN
	TestOfCAN();

	/*/ Do TI defined flash test
	Uint32 NumOfFailedBlocks = 0;
	FLASH_API_STATUS_MESSAGES SpiFlashResult = SUCCESSFUL;
	SpiFlashResult = goSpiFlash.Test(0x400, &NumOfFailedBlocks);
	oDbgOutput.str("");
	oDbgOutput << "SPI " << goSpiFlash.GetUsedSpiBus() << " test - ";
	switch(SpiFlashResult) {
		case SUCCESSFUL:
			oDbgOutput << "OK!" << "\r" << endl;
			break;
		case FAIL_DEVICE_ID:
			oDbgOutput << "Error: Fail Device ID!" << "\r" << endl;
			break;
		case FAIL_ERASE:
			oDbgOutput << "Error: Status Fail Erase!" << "\r" << endl;
			break;
		case FAIL_INIT:
			oDbgOutput << "Error: Status Fail Init!" << "\r" << endl;
			break;
		case FAIL_PROGRAM:
			oDbgOutput << "Error: Status Fail Program!" << "\r" << endl;
			break;
		case FAIL_VERIFY:
			oDbgOutput << "Error: Status Fail Verify! Failed Blocks: " << NumOfFailedBlocks << "\r" << endl;
			break;
		default:
			oDbgOutput << "Undefined result: " << Result << "\r" << endl;
			break;
	}	// end of of switch
	goCOM1.Write(oDbgOutput);
// */

    // One time tests are done here
	DoFinaleLoop();
}

#ifdef FLASH
//*****************************************************************************
//  CallFlashAPI
//
//  This function will interface to the flash API.
//  Flash API functions used in this function are executed from RAM
//*****************************************************************************
void CallFlashAPI(void)
{
    uint32 u32Index = 0;
    uint16 i = 0;
    static uint16 Buffer[WORDS_IN_FLASH_BUFFER + 1];
    static uint32 *Buffer32 = (uint32 *)Buffer;

    Fapi_StatusType            oReturnCheck;
    volatile Fapi_FlashStatusType       oFlashStatus;
    Fapi_FlashStatusWordType   oFlashStatusWord;

    // Disable ECC.  ECC does not have to be disabled to do FSM operations like
    // program and erase.
    // However, on Sonata Rev. 0 silicon, due to an OTP ECC errata,
    // disable ECC to avoid ECC errors while using Flash API functions that
    // read TI-OTP
    EALLOW;
    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0x0;
    EDIS;

    EALLOW;

    // This function is required to initialize the Flash API based on System
    // frequency before any other Flash API operation can be performed

	oReturnCheck = Fapi_initializeAPI(F021_CPU0_BASE_ADDRESS, 200);//for now keeping it out

    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        HandleError(oReturnCheck);
    }

    // Fapi_setActiveFlashBank function sets the Flash bank and FMC for further
    // Flash operations to be performed on the bank
    oReturnCheck = Fapi_setActiveFlashBank(Fapi_FlashBank0);
    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        HandleError(oReturnCheck);
    }

    // Erase Sector C
    oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_EraseSector,
                   (uint32 *)Bzero_SectorC_start);

    // Wait until FSM is done with erase sector operation
    while (Fapi_checkFsmForReady() != Fapi_Status_FsmReady){}

    // Verify that SectorL is erased.  The Erase step itself does a
    // verify as it goes.  This verify is a 2nd verification that can be done.
    oReturnCheck = Fapi_doBlankCheck((uint32 *)Bzero_SectorC_start,
                   Bzero_16KSector_u32length,
                   &oFlashStatusWord);

    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        // If Erase command fails, use Fapi_getFsmStatus() function to get the
        // FMSTAT register contents to see if any of the EV bit, ESUSP bit,
        // CSTAT bit or VOLTSTAT bit is set (Refer to API documentation for
        // more details)
        HandleError(oReturnCheck);
    }

    // Erase Sector B
    oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_EraseSector,
                   (uint32 *)Bzero_SectorB_start);

    // Wait until FSM is done with erase sector operation
    while (Fapi_checkFsmForReady() != Fapi_Status_FsmReady){}
    // Verify that SectorB is erased.  The Erase step itself does a verify as
    // it goes.  This verify is a 2nd verification that can be done.
    oReturnCheck = Fapi_doBlankCheck((uint32 *)Bzero_SectorB_start,
                   Bzero_16KSector_u32length,
                   &oFlashStatusWord);

    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        // If Erase command fails, use Fapi_getFsmStatus() function
        // to get the FMSTAT register contents
        // to see if any of the EV bit, ESUSP bit, CSTAT bit or VOLTSTAT
        // bit is set (Refer to API documentation for more details)
        HandleError(oReturnCheck);
    }


    // A data buffer of max 8 words can be supplied to the program function.
    // Each word is programmed until the whole buffer is programmed or a
    // problem is found. However to program a buffer that has more than 8
    // words, program function can be called in a loop to program 8 words for
    // each loop iteration until the whole buffer is programmed


    // Example: Program 0xFF bytes in Flash Sector C along with auto-
    // generated ECC

    // In this case just fill a buffer with data to program into the flash.
    for(i=0;i<=WORDS_IN_FLASH_BUFFER;i++)
    {
        Buffer[i] = i;
    }

    for(i=0, u32Index = Bzero_SectorC_start;
       (u32Index < (Bzero_SectorC_start + WORDS_IN_FLASH_BUFFER))
       && (oReturnCheck == Fapi_Status_Success); i+= 8, u32Index+= 8)
    {
        oReturnCheck = Fapi_issueProgrammingCommand((uint32 *)u32Index,Buffer+i,
                       8,
                       0,
                       0,
                       Fapi_AutoEccGeneration);

        while(Fapi_checkFsmForReady() == Fapi_Status_FsmBusy);

        if(oReturnCheck != Fapi_Status_Success)
        {
        // Check Flash API documentation for possible errors
        HandleError(oReturnCheck);
        }

        // Read FMSTAT register contents to know the status of FSM after
        // program command for any debug
        oFlashStatus = Fapi_getFsmStatus();

        // Verify the values programmed.  The Program step itself does a verify
        // as it goes.  This verify is a 2nd verification that can be done.
        oReturnCheck = Fapi_doVerify((uint32 *)u32Index,
                       4,
                       Buffer32+(i/2),
                       &oFlashStatusWord);
        if(oReturnCheck != Fapi_Status_Success)
        {
            // Check Flash API documentation for possible errors
            HandleError(oReturnCheck);
        }
    }

    // Example:  Program 0xFF bytes in Flash Sector B with out ECC
    // Disable ECC so that error is not generated when reading Flash contents
    // without ECC
    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0x0;

    for(i=0, u32Index = Bzero_SectorB_start;
       (u32Index < (Bzero_SectorB_start + WORDS_IN_FLASH_BUFFER))
       && (oReturnCheck == Fapi_Status_Success); i+= 8, u32Index+= 8)
    {
        oReturnCheck = Fapi_issueProgrammingCommand((uint32 *)u32Index,
                       Buffer+i,
                       8,
                       0,
                       0,
                       Fapi_DataOnly);
        while(Fapi_checkFsmForReady() == Fapi_Status_FsmBusy);

        if(oReturnCheck != Fapi_Status_Success)
        {
            // Check Flash API documentation for possible errors
            HandleError(oReturnCheck);
        }

        // Read FMSTAT register contents to know the status of FSM
        // after program command for any debug
        oFlashStatus = Fapi_getFsmStatus();

        // Verify the values programmed.  The Program step itself does a verify
        // as it goes.  This verify is a 2nd verification that can be done.
        oReturnCheck = Fapi_doVerify((uint32 *)u32Index,
                       4,
                       Buffer32+(i/2),
                       &oFlashStatusWord);
        if(oReturnCheck != Fapi_Status_Success)
        {
            // Check Flash API documentation for possible errors
            HandleError(oReturnCheck);
        }
    }

    // Erase the sectors that we have programmed above
    // Erase Sector C
    oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_EraseSector,
                   (uint32 *)Bzero_SectorC_start);

    // Wait until FSM is done with erase sector operation
    while (Fapi_checkFsmForReady() != Fapi_Status_FsmReady){}

    // Verify that SectorC is erased.  The Erase step itself does a verify as
    // it goes.
    // This verify is a 2nd verification that can be done.
    oReturnCheck = Fapi_doBlankCheck((uint32 *)Bzero_SectorC_start,
                   Bzero_16KSector_u32length,
                   &oFlashStatusWord);

    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        // If Erase command fails, use Fapi_getFsmStatus() function to get the
        // FMSTAT register contents
        // to see if any of the EV bit, ESUSP bit, CSTAT bit or VOLTSTAT bit is
        // set (Refer to API documentation for more details)
        HandleError(oReturnCheck);
    }

    // Erase Sector B
    oReturnCheck = Fapi_issueAsyncCommandWithAddress(Fapi_EraseSector,
                   (uint32 *)Bzero_SectorB_start);
    // Wait until FSM is done with erase sector operation
    while (Fapi_checkFsmForReady() != Fapi_Status_FsmReady){}
    // Verify that SectorB is erased.  The Erase step itself does a verify
    // as it goes.  This verify is a 2nd verification that can be done.
    oReturnCheck = Fapi_doBlankCheck((uint32 *)Bzero_SectorB_start,
                   Bzero_16KSector_u32length,
                   &oFlashStatusWord);

    if(oReturnCheck != Fapi_Status_Success)
    {
        // Check Flash API documentation for possible errors
        // If Erase command fails, use Fapi_getFsmStatus() function to get
        // the FMSTAT register contents to see if any of the EV bit, ESUSP bit,
        // CSTAT bit or VOLTSTAT bit is set (Refer to API documentation
        // for more details)
        HandleError(oReturnCheck);
    }

    // Enable ECC
    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0xA;

    EDIS;

    // Leave control over flash pump
    ReleaseFlashPump();

    return;
}
#endif // FLASH

//******************************************************************************
// If an error is found just stop here
//******************************************************************************
#ifdef FLASH
 #ifdef __cplusplus
  #pragma CODE_SECTION("ramfuncs")
 #else
  #pragma CODE_SECTION(DoFinaleLoop,"ramfuncs");
 #endif
#endif
void HandleError(Fapi_StatusType status)
{
    //  Error code will be in the status parameter
	__asm("    ESTOP0");
}

//******************************************************************************
//  Test of CAN interfaces
//	 A copy of TI example 'can_loopback_cpu01' (controlSUITE\device_support\F2837xD\v160\F2837xD_examples_Cpu1\can_loopback\cpu01)
//******************************************************************************
#ifdef FLASH
 #ifdef __cplusplus
  #pragma CODE_SECTION("ramfuncs")
 #else
  #pragma CODE_SECTION(DoFinaleLoop,"ramfuncs");
 #endif
#endif
void TestOfCAN(void)
{
	// Define test data
	const Uint16 SizeOfCANMsg = 4;			// Range: 1, 2, 3, ..., 8
	const Uint32 NumOfMainLoops = 1024;
	const Uint16 StepSize = 4;
	const Uint32 TestData14 = 0x00F0FAEC;	// Test data of data byte 1 to 4
	const Uint32 TestData58 = 0x12345678;	// Test data of data byte 5 to 8

	Uint32 MainLoop = 0;
	Uint16 SecondLoop = 0;
	Uint16 Temp = 0;
	bool VerificationOK = true;						// Flag if verification loop failed
    unsigned char TXMsgData[SizeOfCANMsg], RXMsgData[SizeOfCANMsg];
    tCANMsgObject TXCANMessage, RXCANMessage;
    static stringstream oDebug;

	oDebug.str("");
	oDebug << "\r\nTest CANbus " << goCanA.GetBusType() << "\r" << endl;
	goCOM1.Write(oDebug);

	// Initialize the CAN controller
    goCanA.Init();
    // Setup CAN to be clocked off the M3/Master subsystem clock
    goCanA.SelectClkSource(0);

    // Set up the bit rate for the CAN bus: 500kBit/s
    goCanA.SetBitRate(200000000, 500000);

    // Enable test mode and select external loopback
    HWREG(goCanA.GetBase() + CAN_O_CTL) |= CAN_CTL_TEST;
    HWREG(goCanA.GetBase() + CAN_O_TEST) = CAN_TEST_EXL;

    // Enable the CAN for operation.
    goCanA.Enable();

    // Prepare test data
    // Byte 1 to 4 of TX test data
    Temp = SizeOfCANMsg;
    if(4 < SizeOfCANMsg) {
    	Temp = 4;
    }
	for(MainLoop = 0; MainLoop < Temp; MainLoop++) {
		TXMsgData[MainLoop] = (TestData14 >> ((3-MainLoop) * 8)) & 0x000000FF;
	}
    // Byte 5 to 8 of TX test data
    if(4 < SizeOfCANMsg) {
    	for(MainLoop = 4; MainLoop < SizeOfCANMsg; MainLoop++) {
    		TXMsgData[MainLoop] = (TestData58 >> ((7-MainLoop) * 8)) & 0x000000FF;
    	}
    }
    // RX data message
	for(MainLoop = 0; MainLoop < SizeOfCANMsg; MainLoop++) {
		RXMsgData[MainLoop] = 0;
	}

	// Init CAN Messages
    TXCANMessage.MsgID = 1;							// CAN message ID - use 1
    TXCANMessage.MsgIDMask = 0;						// no mask needed for TX
    TXCANMessage.Flags = TxIntEnable;				// enable interrupt on TX
    TXCANMessage.MsgLen = sizeof(TXMsgData);		// size of message is 4
    TXCANMessage.pMsgData = TXMsgData;				// ptr to message content

    RXCANMessage.MsgID = 1;							// CAN message ID - use 1
    RXCANMessage.MsgIDMask = 0;						// no mask needed for TX
    RXCANMessage.Flags = NoFlags;
    RXCANMessage.MsgLen = sizeof(RXMsgData);		// size of message is 4
    RXCANMessage.pMsgData = RXMsgData;				// ptr to message content

    // Setup the message object being used to receive messages
    goCanA.SetMessage(2, &RXCANMessage, TypeRX);

    // Enter loop to send messages. A new message will be sent with a delay. The 4 bytes of message content
    //  will be treated as an unsigned long and incremented by one each time.
    for(MainLoop = 0; MainLoop < NumOfMainLoops; MainLoop++)
    {
        // Send the CAN message using object number 1 (not the same thing as CAN ID, which is also 1 in this example).
    	//  This function will cause the message to be transmitted right away.
    	goCanA.SetMessage(1, &TXCANMessage, TypeTX);

        // Now wait N ms before continuing
        DELAY_US(1000*10);

        // Get the receive message
        goCanA.GetMessage(2, &RXCANMessage, true);

        // Debug CAN messages
    	oDebug.str("");
    	oDebug << "CAN TX data: ";
    	for(SecondLoop = 0; SecondLoop < SizeOfCANMsg; SecondLoop++) {
    		oDebug << ((Uint16)TXMsgData[SecondLoop]&0x00FF) << " ";
    	}
    	oDebug << "| RX data: ";
    	for(SecondLoop = 0; SecondLoop < SizeOfCANMsg; SecondLoop++) {
    		oDebug << ((Uint16)RXMsgData[SecondLoop]&0x00FF) << " ";
    	}
    	oDebug << "\r" << endl;
    	goCOM1.Write(oDebug);	// */

        // Ensure the received data matches the transmitted data
    	// Attention: a 'unsigned char' is also a 16bit value, so verify the complete memory range of MsgData
    	if(0 != memcmp(TXMsgData, RXMsgData, sizeof(Uint16) * SizeOfCANMsg)) {
        	VerificationOK = false;
        	oDebug.str("");
        	oDebug << " CAN verify failed!\r" << endl;
        	goCOM1.Write(oDebug);
            // Debug CAN messages
        	oDebug.str("");
        	oDebug << "CAN TX data: ";
        	for(SecondLoop = 0; SecondLoop < SizeOfCANMsg; SecondLoop++) {
        		oDebug << ((Uint16)TXMsgData[SecondLoop]&0x00FF) << " ";
        	}
        	oDebug << "| RX data: ";
        	for(SecondLoop = 0; SecondLoop < SizeOfCANMsg; SecondLoop++) {
        		oDebug << ((Uint16)RXMsgData[SecondLoop]&0x00FF) << " ";
        	}
        	oDebug << "\r" << endl;
        	goCOM1.Write(oDebug);
        }

		// Overflow
		Temp = (TXMsgData[0] + StepSize) & 0xFF00;
        // Increment the value in the transmitted message data.
		TXMsgData[0] = (TXMsgData[0] + StepSize) & 0x00FF;
		// for each data byte
    	for(SecondLoop = 1; SecondLoop < SizeOfCANMsg; SecondLoop++) {
    		// test on overflow
    		if(Temp) {
    			// Test on overflow by actual byte and increment actual byte by previous overflow
    			Temp = (TXMsgData[SecondLoop] + StepSize) & 0xFF00;
    			TXMsgData[SecondLoop] = (TXMsgData[SecondLoop] + StepSize) & 0x00FF;
    		}
    	}
    } // end of for(MainLoop)

	if(VerificationOK) {
		oDebug.str("");
		oDebug << "CAN verification loop successfully done with " << MainLoop << " loops!\r\n" << endl;
		goCOM1.Write(oDebug);
	}

	return;
}

//******************************************************************************
//  Print information of 'Device PART Identification' registers 'PARTIDH' and 'PARTIDL' in human readable text
//	 See chapters 2.14.9.2+3
//******************************************************************************
#ifdef FLASH
 #ifdef __cplusplus
  #pragma CODE_SECTION("ramfuncs")
 #else
  #pragma CODE_SECTION(DoFinaleLoop,"ramfuncs");
 #endif
#endif
void PrintDetailedDspInfo(const Uint16 RegH, const Uint32 RegL)
{
	// DSP Family
	switch((RegH >> 8) & 0x07)
	{
	case 0x03:
		goCOM1.Write("Delfino Dual Core\0");
		break;
	case 0x04:
		goCOM1.Write("Delfino Single Core\0");
		break;
	case 0x05:
		goCOM1.Write("Piccolo Single Core\0");
		break;
	default:
		goCOM1.Write("Undefined family type\0");
		break;
	}
	goCOM1.Write("\r\n\0");

	// DSP Quality
	switch((RegL >> 6) & 0x03)
	{
	case 0x00:
		goCOM1.Write("Engineering sample (TMX)\0");
		break;
	case 0x01:
		goCOM1.Write("Pilot production (TMP)\0");
		break;
	case 0x02:
		goCOM1.Write("Fully qualified (TMS)\0");
		break;
	default:
		goCOM1.Write("Undefined quality\0");
		break;
	}
	goCOM1.Write("\r\n\0");

	// DSP Flash size
	goCOM1.Write("Flash size: \0");
	switch((RegL >> 16) & 0xFF)
	{
	case 0x07:
		goCOM1.Write("512KB\0");
		break;
	case 0x06:
		goCOM1.Write("256KB\0");
		break;
	default:
		goCOM1.Write("Undefined\0");
		break;
	}
	goCOM1.Write("\r\n\n\0");

	return;
}

//******************************************************************************
//  Do all tests, first single one, then all cyclic tests
//******************************************************************************
#ifdef FLASH
 #ifdef __cplusplus
  #pragma CODE_SECTION("ramfuncs")
 #else
  #pragma CODE_SECTION(DoFinaleLoop,"ramfuncs");
 #endif
#endif
void DoFinaleLoop(void)
{
	Uint32 SdAddress = 0;		// Actual SD-RAM read/write address
	Uint32 SdramLoop = 1;		// Counter for a complete SD-RAM test
	Uint16 res = 0;
	Uint16 ResultFpgaInc = 0;
	Uint16 i = 0;
	Uint16 colour = TERM_COLOUR_BEGIN;	// Output colour
	Uint16 aPattern[SIZE_SDRAM_TEST_BLOCK];
	char RcvBuf[RECEIVE_BUFFER_SIZE];
    stringstream oDebug;

    memcpy(RcvBuf, 0 , RECEIVE_BUFFER_SIZE);

	// FPGA read version information
	oDebug.str("");
	oDebug << "FPGA Ver: ";
	for(i = 0; i < 3; i++) {
		res = goFpga.ReadInteger(FPGA_ADDR_VERSION_BOARD + i);
		oDebug << res << ".";
	}
//		oDebug << "\r" << endl;
	goCOM1.Write(oDebug);
	res = goFpga.ReadInteger(FPGA_ADDR_VERSION_STATE);
	goCOM1.Write((int)res);
	goCOM1.Write("\r\n");

	// SD-RAM test sequence from controlSUITE example
	oDebug.str("");
	res = goSdram.test_read_write(0x80000000, 0x300000);
	oDebug << "SD-RAM test sequence read/write: ";
	if(res == 0) {
		oDebug << "OK!" << endl;
	}
	else {
		oDebug << "FAILED!" << endl;
	}
	goCOM1.Write(oDebug);
	oDebug.str("");

	// Generate SDRAM test pattern
	goCOM1.Write("\r\nSD-RAM test pattern: ");
	for(i = 0; i < SIZE_SDRAM_TEST_BLOCK; i++) {
		aPattern[i] = i + 'A';
		goCOM1.Write((int)aPattern[i]);
	}
	goCOM1.Write("\r\n");

	// Delay for a bit.
	DELAY_US(1000*500);

	while(1)
	{
		// Trigger watchdog
		Watchdog_Trigger();

		// SD-RAM write test
		for(i = 0; i < SIZE_SDRAM_TEST_BLOCK; i++) {
			goSdram.WriteInteger(SdAddress+i, aPattern[i]);
		}

		//FPGA read increment register
		oDebug.str("");
		ResultFpgaInc = goFpga.ReadInteger(FPGA_ADDR_DEBUG_READ_INCREMENT);
		oDebug << TERM_CLEAR_LINE << "F inc: " << ResultFpgaInc << "\r" << endl;
		goCOM1.Write(oDebug);

		// FPGA write/read test (write result of read operation before)
		oDebug.str("");
		res = goFpga.ReadInteger(FPGA_ADDR_DEBUG_READ_WRITE);
		oDebug << TERM_CLEAR_LINE << "F  wr: " << res << "\r" << endl;
		goCOM1.Write(oDebug);
		goFpga.WriteInteger(FPGA_ADDR_DEBUG_READ_WRITE, ResultFpgaInc);

		// Toggle DSP LED
		goGPIO.ToggleDspLed();
#ifdef TI_controlCARD
	GpioDataRegs.GPATOGGLE.bit.GPIO31 = 1;
#endif
		DELAY_US(1000*50);

		// SD-RAM read test
///		goCOM1.Write("\r\n");
		for(i = 0; i < SIZE_SDRAM_TEST_BLOCK; i++) {
			res = goSdram.ReadInteger(SdAddress+i);
			if(res != aPattern[i]) {
				oDebug.str("");
				oDebug << "\r\nSD-RAM error on address: 0x" << hex << SdAddress << dec << "\r\nSTOPPED! \r" << endl;
				goCOM1.Write(oDebug);
				// stop on error
				__asm("    ESTOP0");
			}
		}
		oDebug.str("");
		oDebug << TERM_CLEAR_LINE << "S Adr: 0x" << hex << SdAddress << dec << " | Loop: " << SdramLoop << "\r" << endl;
		goCOM1.Write(oDebug);

		// Overwrite lines
		oDebug.str("");
		oDebug << TERM_LINE_UP << TERM_LINE_UP << TERM_LINE_UP << TERM_LINE_UP << endl;
		goCOM1.Write(oDebug);

		// Next SD-RAM address range
		SdAddress += SIZE_SDRAM_TEST_BLOCK;
		// Reset address on end of SD-RAM, increment counter and change output colour
		if(goSdram.SDramSizeInWords <= SdAddress) {
			SdAddress = 0;
			SdramLoop++;
			oDebug.str("");
			// Set next output colour
			oDebug << "\e[" << colour++ << "m" << endl;
			goCOM1.Write(oDebug);
			// Reset output colour
			if(TERM_COLOUR_END == colour) {
				colour = TERM_COLOUR_BEGIN;
			}
		}	// end of if(goSdram. ...)

    	// Get received data from COM1
		res = goCOM1.Read(RcvBuf);
	}	// end of 'while()
}

// **************************************************************/
void Watchdog_Enable(const tWDPrescaler Prescaler, const bool NoResetPerformIRQ)
{
	// Enable IRQ
	if(true == NoResetPerformIRQ)
	{
		// Set watchdog ISR function
		EALLOW;
		PieVectTable.WAKE_INT = &WD_IRQ;
		EDIS;

		// Connect the watchdog to the WAKEINT interrupt of the PIE
		// Write to the whole SCSR register to avoid clearing WDOVERRIDE bit
		//  See "TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual (spruhm8); Chapter 2.14.4.1"
		//  System Control & Status Register
		EALLOW;
		WdRegs.SCSR.all = BIT1;
		EDIS;

		// Enable WAKEINT in the PIE: Group 1 interrupt 8
		// Enable INT1 which is connected to WAKEINT:
		PieCtrlRegs.PIECTRL.bit.ENPIE = 1;   // Enable the PIE block
		PieCtrlRegs.PIEIER1.bit.INTx8 = 1;   // Enable PIE Group 1 INT8
		IER |= M_INT1;                       // Enable CPU INT1
		EINT;                                // Enable Global Interrupts
	}
	else {	// Enable RESET
		EALLOW;
		WdRegs.SCSR.bit.WDENINT = 0;
		EDIS;

		// Disable INT1 which is connected to WAKEINT:
		PieCtrlRegs.PIEIER1.bit.INTx8 = 0;   // Disable PIE Group 1 INT8
	}

	// Reset the counters
    ServiceDog();

    // Enable the watchdog and set prescaler to 1 ... 64
	//  See "TMS320F2837xD Dual-Core Delfino Microcontrollers - Technical Reference Manual (spruhm8); Chapter 2.14.4.4"
	//  Watchdog Control Register
    //  INTOSC1 (10MHz): 1/(10MHz / 512 [income divider] / 64) * 256 [8 bit counter] = 838.86ms Calculated! Measurement 926.0ms!?
	EALLOW;
	WdRegs.WDCR.all = 0x0028 | Prescaler;
	EDIS;

	return;
}

// **************************************************************/
void Watchdog_Trigger(void)

{
    ServiceDog();
	return;
}

// **************************************************************/
bool Watchdog_IsWatchdogResetOccured(void)
{
	uint16_t ResetflagSet = CpuSysRegs.RESC.bit.WDRSn;
	if (ResetflagSet == 0) {
		return(false);		// Resetflag (WDRSn) = 0 ? => Other reason
	}
	else {
		// Clear reset cause 'WDRSn'
		CpuSysRegs.RESC.bit.WDRSn = 1;
		return(true);		// Watchdog reset has occured
	}
}

