/**
 * @file MSP432P401R_link_binary_blob.c
 * @author Chester Gillon
 * @date 8-May-2020
 * @details
 *  Program which demonstrates how to get the TI CGT armobjcopy to create an object containing the contents of binary blob
 *  into an object file which can be linked using the TI ARM linker.
 *  The source code of this file is used as a blob which is linked into the program so the program can display its source.
 *
 *  The project setup steps were:
 *  1. Add the following pre-build step:
 *       ${CG_TOOL_ROOT}/bin/armobjcopy --input-target=binary --output-target=elf32-littlearm ../MSP432P401R_link_binary_blob.c MSP432P401R_link_binary_blob_elf.o --rename-section .data=.const,readonly
 *
 *     By default, when converting from a binary to ELF file armobjcopy creates a .data section.
 *     Use of a .data section means the binary contents linked to the program can be modified at run time, but results
 *     in increased memory usage as the binary file has be copied to RAM by the run time start up code.
 *
 *     The addition of "--rename-section .data=.const,readonly" to the above command causes the binary file to be placed in
 *     a read-only .const section which can then be placed into FLASH by the linker.
 *
 *  2. Add the following linker libraries:
 *        MSP432P401R_link_binary_blob_elf.o
 *
 *  3. Under CCS Build -> ARM Linker -> Advanced Options -> Diagnostics
 *     add 16006 to "Treat diagnostic <id> as warning (--diag_warning)"
 *
 *     This is because the armobjcopy command generates the ELF output file with an ABI version of "0", which by default the
 *     linker considers incompatible with the ABI version of "5" generated by the TI ARM compiler.
 *
 *     The above setting changes diagnostic 16006 from a linker error to a warning, to allow the program to link.
 *     I have been unable to find how to make armobjcopy insert an ABI version of "5" in the generated object file.
 */

#include <stdint.h>
#include <stdio.h>

#include <msp.h>

/* The symbols generated by the conversion of this source file into an object file */
extern char _binary____MSP432P401R_link_binary_blob_c_start;
extern char _binary____MSP432P401R_link_binary_blob_c_end;

int main(void)
{
    WDT_A->CTL = WDT_A_CTL_PW | WDT_A_CTL_HOLD;     // stop watchdog timer

    const size_t length = &_binary____MSP432P401R_link_binary_blob_c_end - &_binary____MSP432P401R_link_binary_blob_c_start;
    printf ("binary blob start address=0x%p\n", &_binary____MSP432P401R_link_binary_blob_c_start);
    printf ("binary blob end address=0x%p\n", &_binary____MSP432P401R_link_binary_blob_c_end);
    printf ("binary blob length=%u\n", length);
    printf ("binary blob contents:\n");
    fwrite (&_binary____MSP432P401R_link_binary_blob_c_start, 1, length, stdout);
    return 0;
}
