/* ============================================================================
 * Copyright (c) 2008-2012 Texas Instruments Incorporated.
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
*/


/** @file csl_sar_PollExample.c
 *
 *  @brief Test code to verify the SAR functionality in polled mode
 *
 *
 * \page    page16  CSL SAR EXAMPLE DOCUMENTATION
 *
 * \section SAR1   SAR EXAMPLE1 - POLL MODE TEST
 *
 * \subsection SAR1x    TEST DESCRIPTION:
 *		This test verifies the functionality of CSL SAR(Successive Approximation
 * Register) in polled mode. SAR module on C5515/C5517 DSP is used to convert
 * the voltage generated by the key pad available on the C5515/C5517 EVM to
 * digital values. This key pad is having 10 button with each button having
 * a different digital value corresponding to the voltage generated when it is
 * pressed.
 *
 * This test is having two parts. In the first part of the test C5515/C5517
 * key pad button voltage is read and the converted digital value is displayed
 * on the CCS "stdout". This digital value corresponds to the voltage of the
 * button pressed while running the test. SAR module is having six channels to
 * read the data. This test configures channel four for reading the button
 * voltage in polled mode. After successful configuration of the SAR, voltage
 * conversion is started using the CSL API SAR_startConversion(). Digital value
 * of the button voltage is read for 40 times and is displayed in the CCS
 * "stdout" window. After reading 40 digital values SAR data conversion is
 * stopped and CSL SAR module will be closed.
 *
 * In the second part of the test SAR is configured using SAR_chanConfig() API.
 * Configured values are read back using SAR_getConfig() API and are compared
 * with the original values. This part of the test also verifies setting the GPO
 * direction using SAR_GPODirSet() API. Success of this API call can be
 * verified using the SAR GPO control register(SARGPOCTRL) value displayed in
 * the CCS "stdout" window.
 *
 * @verbatim
   Digital values corresponding to the button voltage will vary slightly
   depending on the system clock value. Below are the range of values for
   each button available on the C5515/C5517 EVM key pad.

    No button pushed: 0x3fb - 0x3ff
    SHIFT:    0x230 - 0x238
    STOP:     0x0
    RWD:      0x20f - 0x216
    MENU:     0x367 - 0x374
    DN:       0x2bc - 0x2c6
    SEL/PLAY: 0x1d3 - 0x1d8
    UP:       0x24f - 0x258
    REC:      0xd5  - 0xd8
    FWD:      0x163 - 0x167
    MODE:     0x30b - 0x316

   @endverbatim
 *
 *
 * NOTE: THIS TEST HAS BEEN DEVELOPED TO WORK WITH CHIP VERSIONS C5515 AND
 * C5517. MAKE SURE THAT PROPER CHIP VERSION MACRO IS DEFINED IN THE FILE
 * c55xx_csl\inc\csl_general.h.
 *
 * \subsection SAR1y    TEST PROCEDURE:
 *  @li Open the CCS and connect the target (C5515/C5517 EVM)
 *  @li Open the project "CSL_SAR_PollExample.pjt" and build it
 *  @li Load the program on to the target
 *  @li Set the PLL frequency to 12.288MHz
 *  @li Press any one of the buttons present on C5515/C5517 EVM key pad.
 *  @li Run the program and observe the test result
 *      C5515: 60MHz and 100MHz
 *      C5517: 60MHz, 100MHz, 150MHz and 200MHz
 *  @li Repeat the test in Release mode
 *
 * \subsection SAR1z    TEST RESULT:
 *  @li All the CSL APIs should return success
 *  @li Digital value corresponding to a button voltage should be within the
 *      range specified in the test procedure.
 *  @li SAR configuration values should be read successfully and should match
 *      with original value.
 *  @li Value of the "SARGPOCTRL" register displayed after calling SAR config
 *      API should be 0x80 indicating that GPO direction set is successful.
 *
 * ============================================================================
 */

/* ============================================================================
 * Revision History
 * ================
 * 20-Sep-2008 created
 * 20-Jul-2012 Added C5517 compatibility
 * ============================================================================
 */

/* Inclusion of header files */
#include <csl_sar.h>
#include <csl_general.h>
#include <stdio.h>

/* CSL example test passed */
#define CSL_TEST_FAILED         (1)
/* CSL example test failed */
#define CSL_TEST_PASSED         (0)
/* SAR object Structure    */
CSL_SarHandleObj SarObj;
/* For loop                */
int 			 i;

/* To test Servo Test Mode */
int  sar_servo_test_mode();

/* To test config API's */
int sar_api_test();

/* To test Keypad voltage Measurement */
int  sar_test_keypad_voltage();

   /////INSTRUMENTATION FOR BATCH TESTING -- Part 1 --
   /////  Define PaSs_StAtE variable for catching errors as program executes.
   /////  Define PaSs flag for holding final pass/fail result at program completion.
        volatile Int16 PaSs_StAtE = 0x0001; // Init to 1. Reset to 0 at any monitored execution error.
        volatile Int16 PaSs = 0x0000; // Init to 0.  Updated later with PaSs_StAtE when and if
   /////                                  program flow reaches expected exit point(s).
   /////
void main(void)
{
    int    result;


	printf("CSL SAR Poll Mode Tests\n\n");

	printf("CSL SAR Voltage Read Test\n");
	printf("Press Any Button in the EVM Keypad\n\n");

    /* To Test Keypad Voltage Measurement - It covers
    SAR_A2DMeasParamSet API call				*/

    result = sar_test_keypad_voltage();
    if(result != CSL_TEST_PASSED)
    {
        printf("\nCSL SAR Voltage Read Test Failed!!\n");
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
   /////
    }
    else
    {
        printf("\nCSL SAR Voltage Read Test Passed!!\n");
    }


	printf("\nCSL SAR Configuration Test\n");
    /* To Test SAR_chanConfig and SAR_getconfig API */
    result = sar_api_test();
    if(result != CSL_TEST_PASSED)
    {
        printf("CSL SAR Configuration Test Failed!!\n");
    /////INSTRUMENTATION FOR BATCH TESTING -- Part 2 --
   /////  Reseting PaSs_StAtE to 0 if error detected here.
        PaSs_StAtE = 0x0000; // Was intialized to 1 at declaration.
   /////
   }
    else
    {
        printf("CSL SAR Configuration Test Passed!!\n");
    }

	printf("\nCSL SAR Poll Mode Tests Completed!!\n\n");
   /////INSTRUMENTATION FOR BATCH TESTING -- Part 3 --
   /////  At program exit, copy "PaSs_StAtE" into "PaSs".
        PaSs = PaSs_StAtE; //If flow gets here, override PaSs' initial 0 with
   /////                   // pass/fail value determined during program execution.
   /////  Note:  Program should next exit to C$$EXIT and halt, where DSS, under
   /////   control of a host PC script, will read and record the PaSs' value.
   /////
}

/* Testing of SAR A/D Keypad Voltage Measurement */
int  sar_test_keypad_voltage(void)
{
    CSL_Status    status;
    CSL_SarHandleObj *SarHandle;
    CSL_SarChSetup param;
    int result;
    int chanNo;
    Uint16 readBuffer;
    result = CSL_TEST_FAILED;

    printf("Testing SAR in polling mode\n");

    /* Initialize the SAR module */
    status = SAR_init();
    if(status != CSL_SOK)
    {
        printf("SAR Init Failed!!\n");
        return(result);
    }
    /* Open SAR channel */
    status = SAR_chanOpen(&SarObj,CSL_SAR_CHAN_3);
    SarHandle = &SarObj;
    if(status != CSL_SOK)
    {
        printf("SAR_chanOpen Failed!!\n");
        return result;
    }
    /* Initialize channel */
    status = SAR_chanInit(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanInit Failed!!\n");
        return(result);
    }
    param.OpMode = CSL_SAR_POLLING;
    param.MultiCh = CSL_SAR_NO_DISCHARGE;
    param.RefVoltage = CSL_SAR_REF_VIN;
    param.SysClkDiv = 0x0b ;
    /* Configuration for SAR module */
    status = SAR_chanSetup(SarHandle,&param);
    if(status != CSL_SOK)
    {
        printf("SAR_chanConfig Failed!!\n");
        return(result);
    }
    /* Set channel cycle set */
    status = SAR_chanCycSet(SarHandle,CSL_SAR_CONTINUOUS_CONVERSION);
    if(status != CSL_SOK)
    {
        printf("SAR_chanCycSet Failed!!\n");
        return(result);
    }
    /* set ADC Measurement parameters */
    status = SAR_A2DMeasParamSet(SarHandle,CSL_KEYPAD_MEAS,&chanNo);
    if(status != CSL_SOK)
    {
        printf("SAR_A2DMeasParamSet Failed!!\n");
        return(result);
    }
    printf("Channel Number selected %d\n",chanNo);
    /* start the conversion */
    status = SAR_startConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_startConversion Failed!!\n");
        return(result);
    }
    i = 0;
    /* Read the ADC data continously 40 times */
    while(i < 40)
    {
    /* Check whether the ADC data is available or not */
        while(CSL_SAR_DATA_AVAILABLE !=
        SAR_getStatus(SarHandle,&status));

        status = SAR_readData(SarHandle, &readBuffer);
        if(status != CSL_SOK)
        {
        printf("SAR_readData Failed!!\n");
        return(result);
        }
        i++;
        printf("SAR ADC read data 0x%x\n",readBuffer);
    }
    /* Stop the conversion */
    status = SAR_stopConversion(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_stopConversion Failed!!\n");
        return(result);
    }
    /* Close the channel */
    status = SAR_chanClose(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanClose Failed!!\n");
        return(result);
    }
    /* Deinit */
    status = SAR_deInit();
    if(status != CSL_SOK)
    {
        printf("SAR_deInit Failed!!\n");
        return(result);
    }
    result = CSL_TEST_PASSED;
    return(result);
}

/* Testing of SAR API's */
int sar_api_test()
{

    CSL_Status    status;
    CSL_SarHandleObj *SarHandle;
    CSL_SarChConfig param;
    int result;
    result = CSL_TEST_FAILED;

	printf("calling sar_api_test\n");

    /* Initialize the SAR module */
    status = SAR_init();
    if(status != CSL_SOK)
    {
        printf("SAR Init Failed!!\n");
        return(result);
    }
    /* Open SAR channel */
    status = SAR_chanOpen(&SarObj,CSL_SAR_CHAN_3);
    SarHandle = &SarObj;
    if(status != CSL_SOK)
    {
        printf("SAR_chanOpen Failed!!\n");
        return result;
    }
    /* Initialize channel */
    status = SAR_chanInit(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanInit Failed!!\n");
        return(result);
    }
    printf("Before calling SAR_chanConfig API\n");
    /* Retrieve SAR A/D Registers */
    SAR_getConfig(SarHandle, &param);
    printf(" SARCTRL 0x%x\n",param.SARCTRL);
    printf(" SARCLKCTR 0x%x\n",param.SARCLKCTRL);
    printf(" SARPINCTR 0x%x\n",param.SARPINCTRL);
    printf(" SARGPOCTRL 0x%x\n",param.SARGPOCTRL);

    param.SARCTRL = 0x3800;
    param.SARCLKCTRL = 0x000b;
    param.SARPINCTRL = 0x0100;
    /* Configure SAR A/D Registers */
    status = SAR_chanConfig(SarHandle, &param);
    if(status != CSL_SOK)
    {
    printf("SAR_chanConfig Failed\n");
    return(result);
    }

   /* Test SAR_GPODirSet API */
    status = SAR_GPODirSet(SarHandle,CSL_SAR_GPO_3,CSL_SAR_GPO_OUT);
    if(CSL_SOK != status)
    {
        printf("SAR_GPODirSet test Failed\n");
        return (result);
    }


    printf("After calling SAR_chanConfig API\n");

    /* Get config */
    SAR_getConfig(SarHandle, &param);
    printf(" SARCTRL 0x%x\n",param.SARCTRL);
    printf(" SARCLKCTRL 0x%x\n",param.SARCLKCTRL);
    printf(" SARPINCTRL 0x%x\n",param.SARPINCTRL);
    printf(" SARGPOCTRL 0x%x\n",param.SARGPOCTRL);

   /* Close the channel */
    status = SAR_chanClose(SarHandle);
    if(status != CSL_SOK)
    {
        printf("SAR_chanClose Failed!!\n");
        return(result);
    }
    /* Deinit */
    status = SAR_deInit();
    if(status != CSL_SOK)
    {
        printf("SAR_deInit Failed!!\n");
        return(result);
    }
    result = CSL_TEST_PASSED;
    return(result);
}

