/*
 * Description:
 *  Opens a SAR connection to an ADC to poll for data
 *  Performs an FFT on the polled data.
 *  Prepares to send that data to the Bluetooth module [Future implementation]
 *
 */

// included libraries
#include <stdio.h>

// external referenced functions
int SAR_initialize();
unsigned short * SAR_ReadValues(unsigned short num);
void FFT_PerformTransform(short* x, unsigned short num);
int UART_Initialize();
void UART_Close();
int UART_Write(char* string);

// defined constants
const unsigned short num_samples = 2048;
const char DEBUG = 1;

// Function headers
unsigned short * DoFFT();


// Global variables
char IsExitting = 0;
unsigned short * FFTBuffer;
int i;

/**
 * main.c
 */
int main(void)
{
    int status = 0;

    printf("Setting up SAR ADC controller...\n");
    status = SAR_initialize();
    if(status != 0)
    {
        return(status);
    }

    // Starts up the UART controller
    if(DEBUG)
    {
        printf("Setting up UART controller...\n");
        status = UART_Initialize();
        if(status != 0)
        {
            return(status);
        }
    }

    // Begins collecting data
    printf("Entering iteration routine...\n");
    while(IsExitting != 255)
    {
        printf("Collecting samples...\n");
        FFTBuffer = DoFFT(); // Collects num_samples samples
        printf("Finished FFT collection\n");

        if(DEBUG)
        {
            // Transmits the FFT data to the UART
            /*
             * Transmits in format $num0,num1,num2,...,numN$
             */
            UART_Write("$");
            for(i = 0; i < num_samples; i++)
            {
                UART_Write((char*)FFTBuffer[i]);
                if(i == (num_samples - 1))
                    UART_Write(",");
            }
            UART_Write("$");
        }
    }

    if(DEBUG)
    {
        // Shuts down the UART controller
        UART_Close();
    }

	return 0;
}

unsigned short * DoFFT()
{
    printf("Beginning an FFt data set...");

    // Reads samples
    unsigned short * readBuffer;
    readBuffer = SAR_ReadValues(num_samples);

    // Performs FFT
    FFT_PerformTransform((short *)readBuffer, num_samples);

    return readBuffer;
}
