/*
 * uart.c
 *
 * Description:
 *  Handles communicating with the host computer during program debugging
 *
 *  Created on: Jan 5, 2018
 *      Author: aaron.jencks
 */

#include <stdio.h>
#include "csl_uart.h"
#include "csl_uartAux.h"
#include "csl_intc.h"
#include "csl_general.h"
#include "csl_sysctrl.h"

/* Global constants */
#define CSL_UART_BUF_LEN        (4U)

/* String length to be received and transmitted */
#define WR_STR_LEN        80
#define RD_STR_LEN        10

#define CSL_PLL_DIV_000    (0)
#define CSL_PLL_DIV_001    (1u)
#define CSL_PLL_DIV_002    (2u)
#define CSL_PLL_DIV_003    (3u)
#define CSL_PLL_DIV_004    (4u)
#define CSL_PLL_DIV_005    (5u)
#define CSL_PLL_DIV_006    (6u)
#define CSL_PLL_DIV_007    (7u)

#define CSL_PLL_CLOCKIN    (32768u)

#define PLL_CNTL1        *(ioport volatile unsigned *)0x1C20
#define PLL_CNTL2        *(ioport volatile unsigned *)0x1C21
#define PLL_CNTL3        *(ioport volatile unsigned *)0x1C22
#define PLL_CNTL4        *(ioport volatile unsigned *)0x1C23

/* Global data definition */
/* UART setup structure */
CSL_UartSetup mySetup =
{
    /* Input clock freq in MHz */
    60000000,
    /* Baud rate */
    2400,
    /* Word length of 8 */
    CSL_UART_WORD8,
    /* To generate 1 stop bit */
    0,
    /* Disable the parity */
    CSL_UART_DISABLE_PARITY,
    /* Enable trigger 14 fifo */
    CSL_UART_FIFO_DMA1_DISABLE_TRIG14,
    /* Loop Back enable */
    CSL_UART_NO_LOOPBACK,
    /* No auto flow control*/
    CSL_UART_NO_AFE ,
    /* No RTS */
    CSL_UART_NO_RTS ,
};

/* CSL UART data structures */
CSL_UartObj uartObj;
CSL_UartHandle hUart;

/* CSL UART data buffers */
Uint16    uartIntcWriteBuff[CSL_UART_BUF_LEN ];
Uint16    uartIntcReadBuff[CSL_UART_BUF_LEN ];
char      buffer1[30] = "Hello World!";
char      buffer2[10] = {0};
char      buffer3[30] = "\r\n";
Uint16    gcount = 16;
Uint16    mutex = 0;
char      *ptr = buffer1;
Uint16    setbit;

// Function declarations
int UART_Initialize();
void UART_Close();
int UART_Write(char* string);

/* Interrupt vector start address */
extern void VECSTART(void);

/**
 *  \brief  Function to calculate the system clock
 *
 *  \param    none
 *
 *  \return   System clock value in Hz
 */
Uint32 getSysClk(void);

/**
 *  \brief  Interrupt Service Routine to handle UART Line Status Interrupt
 *
 *  \param  none
 *
 *  \return none
 */
void uart_lsiIsr(void)
{
    Uint16 reg;
    reg=hUart->uartRegs->LSR;
    reg= reg;
}

/**
 *  \brief  Interrupt Service Routine to handle UART Character Timeout Interrupt
 *
 *  \param  none
 *
 *  \return none
 */
void uart_ctoIsr(void)
{
    UART_read( hUart,buffer2,0,0);
    ptr = buffer2;
    gcount = 1;
    UART_eventEnable(hUart,CSL_UART_XMITOR_REG_EMPTY_INTERRUPT);
}

/**
 *  \brief  Interrupt Service Routine to handle UART Transmit Interrupt
 *
 *  \param  none
 *
 *  \return none
 */
void uart_txIsr(void)
{
    if(gcount == 1)
    {
        UART_write(hUart, buffer3, 2, 0);
        UART_write(hUart, ptr, gcount, 0);
        gcount = 13;
    }

    UART_write( hUart,buffer1,gcount,0);

    UART_eventDisable(hUart, CSL_UART_XMITOR_REG_EMPTY_INTERRUPT);
}

/**
 *  \brief  Interrupt Service Routine to handle UART Receive Interrupt
 *
 *  \param  none
 *
 *  \return none
 */
void uart_rxIsr(void)
{
    UART_read( hUart,buffer2,0,0);
    gcount = 1;
    UART_eventEnable(hUart,CSL_UART_XMITOR_REG_EMPTY_INTERRUPT);
}

/**
 *  \brief  Interrupt Dispatcher to identify interrupt source
 *
 *  This function identify the type of UART interrupt generated and
 *  calls concerned ISR to handle the interrupt
 *
 *  \param  none
 *
 *  \return none
 */
interrupt void UART_intrDispatch(void)
{
    Uint16 eventId = 0;

    IRQ_disable(UART_EVENT);

    /* Get the event Id which caused interrupt */
    eventId = UART_getEventId(hUart);

    if (((void (*)(void))(hUart->UART_isrDispatchTable[eventId])))
    {
        ((void (*)(void))(hUart->UART_isrDispatchTable[eventId]))();
    }

    IRQ_enable(UART_EVENT);

    return;
}

Uint32 getSysClk(void)
{
    Bool      pllRDBypass;
    Bool      pllOutDiv;
    Uint32    sysClk;
    Uint16    pllM;
    Uint16    pllRD;
    Uint16    pllOD;

    pllM = CSL_FEXT(CSL_SYSCTRL_REGS->CGCR1, SYS_CGCR1_M);

    pllRD = CSL_FEXT(CSL_SYSCTRL_REGS->CGCR2, SYS_CGCR2_RDRATIO);
    pllOD = CSL_FEXT(CSL_SYSCTRL_REGS->CGCR4, SYS_CGCR4_ODRATIO);

    pllRDBypass = CSL_FEXT(CSL_SYSCTRL_REGS->CGCR2, SYS_CGCR2_RDBYPASS);
    pllOutDiv   = CSL_FEXT(CSL_SYSCTRL_REGS->CGCR4, SYS_CGCR4_OUTDIVEN);

    sysClk = CSL_PLL_CLOCKIN;

    if (0 == pllRDBypass)
    {
        sysClk = sysClk/(pllRD + 4);
    }

    sysClk = (sysClk * (pllM + 4));

    if (1 == pllOutDiv)
    {
        sysClk = sysClk/(pllOD + 1);
    }

    /* Return the value of system clock in KHz */
    return(sysClk);
}

// -------------------------------------------------------------------MY FUNCTIONS!!!---------------------------------------------------------------------------------------------------

int UART_Initialize()
{
    CSL_UartIsrAddr    isrAddr;
    CSL_Status         status;
    Uint32            sysClk;

    sysClk = getSysClk();

    mySetup.clkInput = sysClk;

    status = SYS_setEBSR(CSL_EBSR_FIELD_PPMODE,
                             CSL_EBSR_PPMODE_1);
    if(CSL_SOK != status)
    {
        printf("SYS_setEBSR failed\n");
        return(1);
    }

    /* Loop counter and error flag */
    status = UART_init(&uartObj,CSL_UART_INST_0,UART_INTERRUPT);
    if(CSL_SOK != status)
    {
        printf("UART_init failed error code %d\n",status);
        return(2);
    }
    else
    {
        printf("UART_init Successful\n");
    }

    status = SYS_setEBSR(CSL_EBSR_FIELD_PPMODE,
                             CSL_EBSR_PPMODE_1);
    if(CSL_SOK != status)
    {
        printf("SYS_setEBSR failed\n");
        return (3);
    }

    /* Handle created */
    hUart = (CSL_UartHandle)(&uartObj);

    /* Configure UART registers using setup structure */
    status = UART_setup(hUart,&mySetup);
    if(CSL_SOK != status)
    {
        printf("UART_setup failed error code %d\n",status);
        return(4);
    }
    else
    {
        printf("UART_setup Successful\n");
    }

    /* Configure and Register the UART interrupts */
    isrAddr.rbiAddr  = uart_rxIsr;
    isrAddr.tbeiAddr = uart_txIsr;
    isrAddr.ctoi     = uart_ctoIsr;
    isrAddr.lsiAddr  = uart_lsiIsr;

    /* Disable interrupt */
    IRQ_globalDisable();

    /* Clear any pending interrupts */
    IRQ_clearAll();

    /* Disable all the interrupts */
    IRQ_disableAll();

    IRQ_setVecs((Uint32)(&VECSTART));

    /* Configuring Interrupt */
    IRQ_plug (UART_EVENT, &UART_intrDispatch);

    /* Enabling Interrupt */
    IRQ_enable(UART_EVENT);
    IRQ_globalEnable();

    /* Set the UART callback function */
    status = UART_setCallback(hUart,&isrAddr);
    if(status != CSL_SOK)
    {
        printf("UART_setCallback Failed\n");
        return(5);
    }

    /* Enable the UART Events */
    status = UART_eventEnable(hUart, CSL_UART_XMITOR_REG_EMPTY_INTERRUPT);
    if(status != CSL_SOK)
    {
        printf("UART_eventEnable Failed\n");
        return(6);
    }

    status = UART_eventEnable(hUart, CSL_UART_RECVOR_REG_DATA_INTERRUPT);
    if(status != CSL_SOK)
    {
        printf("UART_eventEnable Failed\n");
        return(7);
    }

    status = UART_eventEnable(hUart, CSL_UART_RECVOR_LINE_STATUS_INTERRUPT);
    if(status != CSL_SOK)
    {
        printf("UART_eventEnable Failed\n");
        return(8);
    }

    printf("UART Initialization complete!");
    return(0);

}

void UART_Close()
{
    printf("Closing UART Events...");

    /* Disable UART interrupts */
    IRQ_disable(UART_EVENT);

    /* Disable GLobal Interrupts */
    IRQ_globalDisable();

    /* Disable interrupt */
    IRQ_globalDisable();

    /* Clear any pending interrupts */
    IRQ_clearAll();

    /* Disable all the interrupts */
    IRQ_disableAll();

    printf("UART closed.");
}

int UART_Write(char* string)
{
    CSL_Status         status;
    status = UART_fputs(hUart, string, 0);
    return status;
}
