/******************************************************************************
 *
 * Linker script for Texas Instruments TM4C129ENCPDT
 *
 * Derived from Chester Gillon's tm4c1294ncpdt.lds posted to e2e.ti.com on
 * 2017-04-23
 *
 *****************************************************************************/

/*
    Using this linker script in CSS:
    1. Toolchain must be GCC
    2. Project options must specify STACKSIZE and HEAPSIZE (or the options
       -Wl,--defsym,STACKSIZE=4096 -Wl,--defsym,HEAPSIZE=4096 (with desired
       stack and heap sizes) must be passed to GCC
    3. Entry point is defined as ResetISR in this linker script. Therefore
       no other entry point should be passed to the linker (do not use the
       -e or --entry options; if these options must be used, specify ResetISR
       as the entry point.
    4. The C startup file (tm4c129encpdt_startup_css_gcc.c or similar) must
       meet the following requirements:
       a. Do not declare pui32Stack[] array for the stack, as is default under
          CSS. That should be replaced with extern char *__stack; The __stack
          identifier is created in this linker script.
       b. The 0th element in g_pfnVectors[] should be (void (*)(void))&__stack
       c. Remove extern declarations of __bss_start__ and __bss_end__.
       d. Remove the __asm code to zero fill the bss segment. This will be
          zeroed by _start()
       e. Enable floating point math with the same line used in TI's example:
          HWREG(0xE000ED88) = ((HWREG (0xE000ED88) & ~0x00F00000) | 0x00F00000);
       f. Do not call main(); Call _start(), which will perform additional
          setup of the C/C++ environment and in turn call main(). The _start()
          function never returns.

    When properly configured, the program's execution will be as follows:
    1. ResetISR() runs and calls _start()
    2. _start() runs and calls main()
    3. main() runs

    Example linker command line options (Debug):
    -DPART_TM4C129ENCPDT -DTARGET_IS_TM4C129_RA0 -ffunction-sections
    -fdata-sections -g -gdwarf-3 -gstrict-dwarf -Wall -MD -std=c99
    -march=armv7e-m -mthumb -mfloat-abi=hard -mfpu=fpv4-sp-d16
    -Wl,-Map,"MasterHardware.map"
    -L"C:/ti/TivaWare_C_Series-2.1.4.178/driverlib/gcc"
    -Wl,--defsym,STACKSIZE=4096 -Wl,--defsym,HEAPSIZE=4096

    Example linker command line options (Release):
    -DPART_TM4C129ENCPDT -DTARGET_IS_TM4C129_RA0 -O3 -Os -ffunction-sections
    -fdata-sections -gdwarf-3 -Wall -MD -std=c99 -march=armv7e-m -mthumb
    -mfloat-abi=hard -mfpu=fpv4-sp-d16 -Wl,-Map,"MasterHardware.map"
    -L"C:/ti/TivaWare_C_Series-2.1.4.178/driverlib/gcc"
    -Wl,--defsym,STACKSIZE=4096 -Wl,--defsym,HEAPSIZE=4096
*/


MEMORY
{
    /* TM4C129ENCPDT has:
       1024 KB = 0x00100000 bytes of Flash starting at address 0
       256  KB = 0x00040000 bytes of SRAM  starting at address 0x20000000
    */
    FLASH (RX) : ORIGIN = 0x00000000, LENGTH = 0x00100000
    SRAM (WX)  : ORIGIN = 0x20000000, LENGTH = 0x00040000
}


REGION_ALIAS("REGION_TEXT", FLASH);
REGION_ALIAS("REGION_BSS", SRAM);
REGION_ALIAS("REGION_DATA", SRAM);
REGION_ALIAS("REGION_STACK", SRAM);
REGION_ALIAS("REGION_HEAP", SRAM);
REGION_ALIAS("REGION_ARM_EXIDX", FLASH);
REGION_ALIAS("REGION_ARM_EXTAB", FLASH);


/* Set the program entry point to the ResetISR() function */
ENTRY(ResetISR)


SECTIONS {

    PROVIDE (_intvecs_base_address = 0x0);

    .intvecs (_intvecs_base_address) : AT (_intvecs_base_address) {
        KEEP (*(.intvecs))
    } > REGION_TEXT

    PROVIDE (_vtable_base_address = 0x20000000);

    .vtable (_vtable_base_address) : AT (_vtable_base_address) {
        KEEP (*(.vtable))
    } > REGION_DATA


    .text : {
        CREATE_OBJECT_SYMBOLS
        *(.text)
        *(.text.*)

        KEEP(*(.init))
        KEEP(*(.fini))

        /* .ctors */
        *crtbegin.o(.ctors)
        *crtbegin?.o(.ctors)
        *(EXCLUDE_FILE(*crtend?.o *crtend.o) .ctors)
        *(SORT(.ctors.*))
        *(.ctors)

        /* .dtors */
         *crtbegin.o(.dtors)
         *crtbegin?.o(.dtors)
         *(EXCLUDE_FILE(*crtend?.o *crtend.o) .dtors)
         *(SORT(.dtors.*))
         *(.dtors)

        . = ALIGN(4);
        /* preinit data */
        PROVIDE_HIDDEN (__preinit_array_start = .);
        KEEP(*(.preinit_array))
        PROVIDE_HIDDEN (__preinit_array_end = .);

        . = ALIGN(4);
        /* init data */
        PROVIDE_HIDDEN (__init_array_start = .);
        KEEP(*(SORT(.init_array.*)))
        KEEP(*(.init_array))
        PROVIDE_HIDDEN (__init_array_end = .);


        . = ALIGN(4);
        /* finit data */
        PROVIDE_HIDDEN (__fini_array_start = .);
        KEEP(*(SORT(.fini_array.*)))
        KEEP(*(.fini_array))
        PROVIDE_HIDDEN (__fini_array_end = .);
    } > REGION_TEXT


    .rodata : {
        *(.rodata)
        *(.rodata*)
    } > REGION_TEXT


    /* The ResetISR function uses the __data_load__, __data_start__ and __data_end__ symbols
       to copy the .data segment initialisers from flash to SRAM.
    */
    .data : ALIGN (4) {
        __data_load__ = LOADADDR (.data);
        __data_start__ = .;
        *(.data)
        *(.data*)
        . = ALIGN (4);
        __data_end__ = .;
    } > REGION_DATA AT> REGION_TEXT

    .ARM.exidx : {
        __exidx_start = .;
        *(.ARM.exidx* .gnu.linkonce.armexidx.*)
        __exidx_end = .;
    } > REGION_ARM_EXIDX

    .ARM.extab : {
        *(.ARM.extab* .gnu.linkonce.armextab.*)
    } > REGION_ARM_EXTAB

    /* The GCC run time library _start() start-up function zeros the .bss area
       identified by the range __bss_start__ ... (__bss_end__ - 1)
    */
    .bss : {
        __bss_start__ = .;
        *(.shbss)
        *(.bss)
        *(.bss.*)
        *(COMMON)
        . = ALIGN (4);
        __bss_end__ = .;
    } > REGION_BSS

    /* The GCC run time library assumes the the following memory layout for the heap and stack:
        end // Base address of heap
         |
         |  // Heap expands at increasing addresses
        \_/

         _
        / \
         |  // Stack expands at decreasing addresses
         |
         |  // Initial stack pointer

       The heap and stack need to be adjacent in memory, with the heap at a lower address than the stack.
       The symbol end defines the base address of the heap used by the GCC run time library.

       The GCC run time library doesn't reference any symbol which identifies the end address allocated
       for the heap. Instead, the heap is allowed to grow up to the current stack pointer (at the time
       of an allocation request).

       The linker symbol HEAPSIZE defined in the project properties specifies the
       number of bytes allocated for the heap.
    */
    .heap : {
        end = .;
        . = . + HEAPSIZE;
        KEEP(*(.heap))
    } > REGION_HEAP

    /* The symbol _stack gives the top of the stack which is referenced by both:
       - The initial stack pointer set in the vector table, which is the stack pointer
         at the start of the ResetISR() function.

       - The GCC run time library _start() start-up function, which is used to
         re-initialise the stack pointer.

       The linker symbol STACKSIZE defined in the project properties specifies the
       number of bytes allocated for the stack.
    */
    .stack : ALIGN(0x8) {
        . = ORIGIN(REGION_STACK) + LENGTH(REGION_STACK);
        __stack = .;
        KEEP(*(.stack))
    } > REGION_STACK

    /*
    .stack : ALIGN(0x8) {
        . = . + STACKSIZE;
        __stack = .;
        KEEP(*(.stack))
    } > REGION_STACK
    */
}
