
#include "Variabili.h"

//###########################################################################
//
// FILE:   F2837xD_SysCtrl.c
//
// TITLE:  F2837xD Device System Control Initialization & Support Functions.
//
// DESCRIPTION:
//
//         Example initialization of system resources.
//
//###########################################################################
// $TI Release: F2837xD Support Library v130 $
// $Release Date: Mon Oct 20 10:15:06 CDT 2014 $
//###########################################################################

#include "F2807x_device.h"     // Headerfile Include File
#include "F2807x_Examples.h"   // Examples Include File

// Functions that will be run from RAM need to be assigned to
// a different section.  This section will then be mapped to a load and
// run address using the linker cmd file.
//
//  *IMPORTANT*
//  IF RUNNING FROM FLASH, PLEASE COPY OVER THE SECTION "ramfuncs"  FROM FLASH
//  TO RAM PRIOR TO CALLING InitSysCtrl(). THIS PREVENTS THE MCU FROM THROWING
//  AN EXCEPTION WHEN A CALL TO DELAY_US() IS MADE.
//
#pragma CODE_SECTION(InitFlash, "ramfuncs");




extern void inline System_Tick_Init();

void DisablePeripheralClocks()
{
	EALLOW;

	CpuSysRegs.PCLKCR0.all = 0;
	CpuSysRegs.PCLKCR1.all = 0;
	CpuSysRegs.PCLKCR2.all = 0;
	CpuSysRegs.PCLKCR3.all = 0;
	CpuSysRegs.PCLKCR4.all = 0;
	CpuSysRegs.PCLKCR6.all = 0;
	CpuSysRegs.PCLKCR7.all = 0;
	CpuSysRegs.PCLKCR8.all = 0;
	CpuSysRegs.PCLKCR9.all = 0;
	CpuSysRegs.PCLKCR10.all = 0;
	CpuSysRegs.PCLKCR11.all = 0;
	CpuSysRegs.PCLKCR12.all = 0;
	CpuSysRegs.PCLKCR13.all = 0;
	CpuSysRegs.PCLKCR14.all = 0;
	CpuSysRegs.PCLKCR16.all = 0;

	EDIS;
}

//---------------------------------------------------------------------------
// Example: InitFlash:
//---------------------------------------------------------------------------
// This function initializes the Flash Control registers

//                   CAUTION
// This function MUST be executed out of RAM. Executing it
// out of OTP/Flash will yield unpredictable results


//---------------------------------------------------------------------------
// Example: ServiceDog:
//---------------------------------------------------------------------------
// This function resets the watchdog timer.
// Enable this function for using ServiceDog in the application

void ServiceDog(void)
{
    EALLOW;
    WdRegs.WDKEY.bit.WDKEY = 0x0055;
    WdRegs.WDKEY.bit.WDKEY = 0x00AA;
    EDIS;
}

//---------------------------------------------------------------------------
// Example: InitPll2:
//---------------------------------------------------------------------------
// This function initializes the PLL2 registers.
//
// Note: The internal oscillator CANNOT be used as the PLL source if the
// PLLSYSCLK is configured to frequencies above 194 MHz.

void InitAuxPll(Uint16 clock_source, Uint16 imult, Uint16 fmult, Uint16 divsel)
{
	Uint16 temp_divsel;

	if((clock_source == ClkCfgRegs.CLKSRCCTL2.bit.AUXOSCCLKSRCSEL)   &&
	  (imult		 == ClkCfgRegs.AUXPLLMULT.bit.IMULT) 	      &&
	  (fmult 	     == ClkCfgRegs.AUXPLLMULT.bit.FMULT)          &&
	  (divsel        == ClkCfgRegs.AUXCLKDIVSEL.bit.AUXPLLDIV))
    {
	    //everything is set as required, so just return
	    return;
	}

	switch (clock_source)
	{

		case INT_OSC2:
			AuxIntOsc2Sel();
			break;

		case XTAL_OSC:
			AuxXtalOscSel();
			break;

		case AUXCLKIN:
			AuxAuxClkSel();
			break;

	}

   // Change the SYSPLL Integer Multiplier (or) SYSPLL Fractional Multiplier
   if(ClkCfgRegs.AUXPLLMULT.bit.IMULT != imult || ClkCfgRegs.AUXPLLMULT.bit.FMULT !=fmult)
   {
	   EALLOW;
	   ClkCfgRegs.AUXCLKDIVSEL.bit.AUXPLLDIV = AUXPLLRAWCLK_BY_8;
	   ClkCfgRegs.AUXPLLMULT.bit.IMULT = imult;		//Setting integer multiplier
	   ClkCfgRegs.AUXPLLMULT.bit.FMULT = fmult;     //Setting fractional multiplier
	   ClkCfgRegs.AUXPLLCTL1.bit.PLLEN = 1;			//Enable AUXPLL
	   EDIS;

	   //Wait for the AUXPLL lock
	   while(ClkCfgRegs.AUXPLLSTS.bit.LOCKS != 1)
	   {
	        // Uncomment to service the watchdog
	        // ServiceDog();
	   }
   }

	 //increase the freq. of operation in steps to avoid any VDD fluctuations
	 temp_divsel = AUXPLLRAWCLK_BY_8;
	 while(ClkCfgRegs.AUXCLKDIVSEL.bit.AUXPLLDIV != divsel)
	 {
		 EALLOW;
		 ClkCfgRegs.AUXCLKDIVSEL.bit.AUXPLLDIV = temp_divsel - 1;
		 EDIS;

		 temp_divsel = temp_divsel - 1;
		 if(ClkCfgRegs.AUXCLKDIVSEL.bit.AUXPLLDIV != divsel)
		 {
			 DELAY_US(15L);
		 }
	 }

   EALLOW;
   ClkCfgRegs.AUXPLLCTL1.bit.PLLCLKEN = 1;			//Enable AUXPLLCLK is fed from AUX PLL
   EDIS;
}


//---------------------------------------------------------------------------
// Example: SysIntOsc1Sel:
//---------------------------------------------------------------------------
// This function switches to Internal Oscillator 1 and turns off all other clock
// sources to minimize power consumption

void SysIntOsc1Sel (void) {
    EALLOW;
    ClkCfgRegs.CLKSRCCTL1.bit.OSCCLKSRCSEL = 2; // Clk Src = INTOSC1
    EDIS;
}

//---------------------------------------------------------------------------
// Example: SysIntOsc2Sel:
//---------------------------------------------------------------------------
// This function switches to Internal oscillator 2 from External Oscillator
// and turns off all other clock sources to minimize power consumption
// NOTE: If there is no external clock connection, when switching from
//       INTOSC1 to INTOSC2, EXTOSC and XLCKIN must be turned OFF prior
//       to switching to internal oscillator 1

void SysIntOsc2Sel (void) {

    EALLOW;
    ClkCfgRegs.CLKSRCCTL1.bit.INTOSC2OFF=0;     // Turn on INTOSC2
    ClkCfgRegs.CLKSRCCTL1.bit.OSCCLKSRCSEL = 0; // Clk Src = INTOSC2
    EDIS;

}

//---------------------------------------------------------------------------
// Example: SysXtalOscSel:
//---------------------------------------------------------------------------
// This function switches to External CRYSTAL oscillator and turns off all other clock
// sources to minimize power consumption. This option may not be available on all
// device packages

void SysXtalOscSel (void)  {

    EALLOW;
    ClkCfgRegs.CLKSRCCTL1.bit.XTALOFF=0;        // Turn on XTALOSC
    ClkCfgRegs.CLKSRCCTL1.bit.OSCCLKSRCSEL = 1; // Clk Src = XTAL
    EDIS;

}

//---------------------------------------------------------------------------
// Example: AuxIntOsc2Sel:
//---------------------------------------------------------------------------
// This function switches to Internal oscillator 2 from External Oscillator
// and turns off all other clock sources to minimize power consumption
// NOTE: If there is no external clock connection, when switching from
//       INTOSC1 to INTOSC2, EXTOSC and XLCKIN must be turned OFF prior
//       to switching to internal oscillator 1

void AuxIntOsc2Sel (void) {

    EALLOW;
    ClkCfgRegs.CLKSRCCTL1.bit.INTOSC2OFF=0;     // Turn on INTOSC2
    ClkCfgRegs.CLKSRCCTL2.bit.AUXOSCCLKSRCSEL = 0; // Clk Src = INTOSC2
    EDIS;

}

//---------------------------------------------------------------------------
// Example: AuxXtalOscSel:
//---------------------------------------------------------------------------
// This function switches to External CRYSTAL oscillator and turns off all other clock
// sources to minimize power consumption. This option may not be available on all
// device packages

void AuxXtalOscSel (void)  {

    EALLOW;
    ClkCfgRegs.CLKSRCCTL1.bit.XTALOFF=0;        // Turn on XTALOSC
    ClkCfgRegs.CLKSRCCTL2.bit.AUXOSCCLKSRCSEL = 1; // Clk Src = XTAL
    EDIS;

}

//---------------------------------------------------------------------------
// Example: AuxAUXCLKOscSel:
//---------------------------------------------------------------------------
// This function switches to External CRYSTAL oscillator and turns off all other clock
// sources to minimize power consumption. This option may not be available on all
// device packages

void AuxAuxClkSel (void)  {

    EALLOW;
    ClkCfgRegs.CLKSRCCTL2.bit.AUXOSCCLKSRCSEL = 2; // Clk Src = XTAL
    EDIS;

}


//Enter IDLE mode (single CPU)
void IDLE()
{
	EALLOW;
	CpuSysRegs.LPMCR.bit.LPM = LPM_IDLE;
	EDIS;
	asm(" IDLE");
}

//Enter STANDBY mode (single CPU)
void STANDBY()
{
	EALLOW;
	CpuSysRegs.LPMCR.bit.LPM = LPM_STANDBY;
	EDIS;
	asm(" IDLE");
}

//Enter HALT mode (dual CPU). Puts CPU2 in IDLE mode first.
void HALT()
{
    EALLOW;
    CpuSysRegs.LPMCR.bit.LPM = LPM_HALT;
    ClkCfgRegs.SYSPLLCTL1.bit.PLLCLKEN = 0;
    ClkCfgRegs.SYSPLLCTL1.bit.PLLEN = 0;
    EDIS;
    asm(" IDLE");
}

//Enter HIB mode (dual CPU). Puts CPU2 in STANDBY first. Alternately,
//CPU2 may be in reset.
void HIB()
{
    EALLOW;
    CpuSysRegs.LPMCR.bit.LPM = LPM_HIB;
    EDIS;
    DisablePeripheralClocks();
    EALLOW;
    ClkCfgRegs.SYSPLLCTL1.bit.PLLCLKEN = 0;
    ClkCfgRegs.SYSPLLCTL1.bit.PLLEN = 0;
    EDIS;
    asm(" IDLE");
}

// ----------------------------------------------------

void System_WatchDog_Service()
{
    EALLOW;
    WdRegs.WDKEY.bit.WDKEY = 0x0055;
    WdRegs.WDKEY.bit.WDKEY = 0x00AA;
    EDIS;
}

// ----------------------------------------------------

void System_WatchDog_Disable()
{
	volatile Uint16 temp;
    EALLOW;
//    WdRegs.SCSR.bit.WDENINT = 0;

    //Grab the clock config so we don't clobber it
//	WdRegs.SCSR.bit.WDOVERRIDE = 1;
    temp = WdRegs.WDCR.all & 0x0007;
    WdRegs.WDCR.all = 0x0068 | temp;

    temp = WdRegs.SCSR.all;
    temp &= ~(0x0003);
    WdRegs.SCSR.all = 0;

    EDIS;
}

// ----------------------------------------------------

void System_WatchDog_Enable()
{
	EALLOW;
	WdRegs.WDCR.all = 0x0028;
//	WdRegs.SCSR.bit.WDOVERRIDE = 1;
	EDIS;
}

// ----------------------------------------------------
void System_Flash_Configuration(void)
{
    EALLOW;

    //At reset bank and pump are in sleep
    //A Flash access will power up the bank and pump automatically
    //After a Flash access, bank and pump go to low power mode (configurable in FBFALLBACK/FPAC1 registers)-
    //if there is no further access to flash
    //Power up Flash bank and pump and this also sets the fall back mode of flash and pump as active
    Flash0CtrlRegs.FPAC1.bit.PMPPWR = 0x1;
    Flash0CtrlRegs.FBFALLBACK.bit.BNKPWR0 = 0x3;

    //Disable Cache and prefetch mechanism before changing wait states
    Flash0CtrlRegs.FRD_INTF_CTRL.bit.DATA_CACHE_EN = 0;
    Flash0CtrlRegs.FRD_INTF_CTRL.bit.PREFETCH_EN = 0;

    //Set waitstates according to frequency
    //                CAUTION
    //Minimum waitstates required for the flash operating
    //at a given CPU rate must be characterized by TI.
    //Refer to the datasheet for the latest information.
    #if CPU_FRQ_200MHZ
    Flash0CtrlRegs.FRDCNTL.bit.RWAIT = 0x3;
    #endif

    #if CPU_FRQ_150MHZ
    Flash0CtrlRegs.FRDCNTL.bit.RWAIT = 0x2;
    #endif

    #if CPU_FRQ_120MHZ
    Flash0CtrlRegs.FRDCNTL.bit.RWAIT = 0x2;
    #endif

    //Enable Cache and prefetch mechanism to improve performance
    //of code executed from Flash.
    Flash0CtrlRegs.FRD_INTF_CTRL.bit.DATA_CACHE_EN = 1;
    Flash0CtrlRegs.FRD_INTF_CTRL.bit.PREFETCH_EN = 1;

    //At reset, ECC is enabled
    //If it is disabled by application software and if application again wants to enable ECC
    Flash0EccRegs.ERR_THRESHOLD.bit.ERR_THRESHOLD = 2;
    Flash0EccRegs.ECC_ENABLE.bit.ENABLE = 0xA;

    EDIS;

    //Force a pipeline flush to ensure that the write to
    //the last register configured occurs before returning.

    __asm(" RPT #7 || NOP");

}
// ----------------------------------------------------
void inline System_Flash_Init(void)
{
#ifdef _FLASH
    memcpy(&RamfuncsRunStart, &RamfuncsLoadStart, (size_t)&RamfuncsLoadSize);
    System_Flash_Configuration();
#endif
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
interrupt void NON_GESTITO_ISR(void)
{
    // Insert ISR Code here

    // Next two lines for debug only to halt the processor here
    // Remove after inserting ISR Code
    asm ("      ESTOP0");
    for(;;);
}
//---------------------------------------------------------------------------
void System_Interrupt_PieCtrl(void)
{
    // Disable Interrupts at the CPU level:
    DINT;

    // Disable the PIE
    PieCtrlRegs.PIECTRL.bit.ENPIE = 0;

	// Clear all PIEIER registers:
	PieCtrlRegs.PIEIER1.all = 0;
	PieCtrlRegs.PIEIER2.all = 0;
	PieCtrlRegs.PIEIER3.all = 0;
	PieCtrlRegs.PIEIER4.all = 0;
	PieCtrlRegs.PIEIER5.all = 0;
	PieCtrlRegs.PIEIER6.all = 0;
	PieCtrlRegs.PIEIER7.all = 0;
	PieCtrlRegs.PIEIER8.all = 0;
	PieCtrlRegs.PIEIER9.all = 0;
	PieCtrlRegs.PIEIER10.all = 0;
	PieCtrlRegs.PIEIER11.all = 0;
	PieCtrlRegs.PIEIER12.all = 0;

	// Clear all PIEIFR registers:
	PieCtrlRegs.PIEIFR1.all = 0;
	PieCtrlRegs.PIEIFR2.all = 0;
	PieCtrlRegs.PIEIFR3.all = 0;
	PieCtrlRegs.PIEIFR4.all = 0;
	PieCtrlRegs.PIEIFR5.all = 0;
	PieCtrlRegs.PIEIFR6.all = 0;
	PieCtrlRegs.PIEIFR7.all = 0;
	PieCtrlRegs.PIEIFR8.all = 0;
	PieCtrlRegs.PIEIFR9.all = 0;
	PieCtrlRegs.PIEIFR10.all = 0;
	PieCtrlRegs.PIEIFR11.all = 0;
	PieCtrlRegs.PIEIFR12.all = 0;
}

//---------------------------------------------------------------------------

void System_Interrupt_PieVectTable(void)
{
    Uint16  i;
    Uint32  *Dest  =  (void  *)  &PieVectTable;

    // Reset di Tutti Vettori Interrupt
    Dest  =  Dest  +  3;
    EALLOW;
    for(i  =  0;  i  <  221;  i++)
    	*Dest++  =  (Uint32) (void *) &NON_GESTITO_ISR;		//*Source++;
    EDIS;

    // Imposta Interrupt Necessari
    EALLOW;
    PieVectTable.EPWM7_INT = &Interrupt_Servizio;
    PieVectTable.EPWM6_INT = &Interrupt_MAIN;
    PieVectTable.SCIA_RX_INT = &Interrupt_SerialePC485;
    PieVectTable.TIMER2_INT = &Interrupt_Tick;
    EDIS;

    // Enable the PIE Vector Table
    PieCtrlRegs.PIECTRL.bit.ENPIE  =  1;
}

//---------------------------------------------------------------------------



//---------------------------------------------------------------------------

void System_Interrupt_Init()
{
	DINT;									// Disable and clear all CPU interrupts:

	IER = 0x0000;
	IFR = 0x0000;

	System_Interrupt_PieCtrl();

	System_Interrupt_PieVectTable();

	System_Tick_Init();

	//PieCtrlRegs.PIEIER1.all =  M_INT1;		// Abilita Interrupt Main Loop
	PieCtrlRegs.PIEIER3.all =  M_INT6 | M_INT7;		// Abilita Interrupt di Servizio
    PieCtrlRegs.PIEIER9.all =  M_INT1;		// Abilita Interrupt Seriale
    PieCtrlRegs.PIEIER11.all = M_INT1;

}


//---------------------------------------------------------------------------

void System_GPIO_DisablePullups()
{
	EALLOW;

	if (DevCfgRegs.PARTIDL.bit.PIN_COUNT != 5)
	{
	    asm ("      ESTOP0");
	    for(;;);
	}

	//System_GPIO_EnableUnbondedIOPullupsFor100Pin();
    GpioCtrlRegs.GPAPUD.all = ~0xFFC003E3;  //GPIOs 0-1, 5-9, 22-31
    GpioCtrlRegs.GPBPUD.all = ~0x03FFF1FF;  //GPIOs 32-40, 44-57
    GpioCtrlRegs.GPCPUD.all = ~0xE10FBC18;  //GPIOs 67-68, 74-77, 79-83, 93-95
    GpioCtrlRegs.GPDPUD.all = ~0xFFFFFFF7;  //GPIOs 96-127
    GpioCtrlRegs.GPEPUD.all = ~0xFFFFFFFF;  //GPIOs 128-159
    GpioCtrlRegs.GPFPUD.all = ~0x000001FF;  //GPIOs 160-168


    EDIS;
}

//---------------------------------------------------------------------------

void System_Gpio_Init()
{
	volatile Uint32 *gpioBaseAddr;
	Uint16 regOffset;

	//Disable pin locks
	EALLOW;
	GpioCtrlRegs.GPALOCK.all = 0x00000000;
	GpioCtrlRegs.GPBLOCK.all = 0x00000000;
	GpioCtrlRegs.GPCLOCK.all = 0x00000000;
	GpioCtrlRegs.GPDLOCK.all = 0x00000000;
	GpioCtrlRegs.GPELOCK.all = 0x00000000;
	GpioCtrlRegs.GPFLOCK.all = 0x00000000;

	//Fill all registers with zeros. Writing to each register separately
	//for six GPIO modules would make this function *very* long. Fortunately,
	//we'd be writing them all with zeros anyway, so this saves a lot of space.
	gpioBaseAddr = (Uint32 *)&GpioCtrlRegs;
	for (regOffset = 0; regOffset < sizeof(GpioCtrlRegs)/2; regOffset++)
	{
		//Hack to avoid enabling pull-ups on all pins. GPyPUD is offset
		//0x0C in each register group of 0x40 words. Since this is a
		//32-bit pointer, the addresses must be divided by 2.
		if (regOffset % (0x40/2) != (0x0C/2))
			gpioBaseAddr[regOffset] = 0x00000000;
	}

	gpioBaseAddr = (Uint32 *)&GpioDataRegs;
	for (regOffset = 0; regOffset < sizeof(GpioDataRegs)/2; regOffset++)
	{
		gpioBaseAddr[regOffset] = 0x00000000;
	}

	EDIS;


//						// Servono solo x scheda Debug
//						GPIO_SetupPinMux(31, GPIO_MUX_CPU1, 0);
//						GPIO_SetupPinOptions(31, GPIO_OUTPUT, GPIO_PUSHPULL);


    // Inizializzazione GPIO
    EALLOW;

    // -------------------------------------

    GpioCtrlRegs.GPAPUD.bit.GPIO2 = 1;
    GpioCtrlRegs.GPADIR.bit.GPIO2 = 1;
    GpioDataRegs.GPADAT.bit.GPIO2 = 0;

//    GpioCtrlRegs.GPAGMUX1.bit.GPIO2 = 1;
//    GpioCtrlRegs.GPAMUX1.bit.GPIO2 = 2;

    GpioCtrlRegs.GPAPUD.bit.GPIO3 = 1;
    GpioCtrlRegs.GPADIR.bit.GPIO3 = 1;
    GpioDataRegs.GPADAT.bit.GPIO3 = 0;

//    GpioCtrlRegs.GPAGMUX1.bit.GPIO3 = 1;
//    GpioCtrlRegs.GPAMUX1.bit.GPIO3 = 2;

    // -------------------------------------

    GpioCtrlRegs.GPAPUD.bit.GPIO4 = 1;
    GpioCtrlRegs.GPAGMUX1.bit.GPIO4 = 0;
    GpioCtrlRegs.GPAMUX1.bit.GPIO4 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPAPUD.bit.GPIO10 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO10 = 1;
    GpioDataRegs.GPADAT.bit.GPIO10 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO11 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO11 = 1;
    GpioDataRegs.GPADAT.bit.GPIO11 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO12 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO12 = 1;
    GpioDataRegs.GPADAT.bit.GPIO12 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO13 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO13 = 1;
    GpioDataRegs.GPADAT.bit.GPIO13 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO14 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO14 = 1;
    GpioDataRegs.GPADAT.bit.GPIO14 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO15 = 1;
    GpioCtrlRegs.GPAMUX1.bit.GPIO15 = 1;
    GpioDataRegs.GPADAT.bit.GPIO15 = 0;

//			xMOTOR_CLA.PWM_1.ptr = &PWM_MOTORE_PHASE_A;
//			xMOTOR_CLA.PWM_2.ptr = &PWM_MOTORE_PHASE_B;
//			xMOTOR_CLA.PWM_3.ptr = &PWM_MOTORE_PHASE_C;

    // -------------------------------------

    GpioCtrlRegs.GPAPUD.bit.GPIO16 = 1;
    GpioCtrlRegs.GPAGMUX2.bit.GPIO16 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO16 = 1;

    GpioCtrlRegs.GPAPUD.bit.GPIO17 = 1;
    GpioCtrlRegs.GPAGMUX2.bit.GPIO17 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO17 = 1;

    GpioCtrlRegs.GPAPUD.bit.GPIO18 = 1;
    GpioCtrlRegs.GPAGMUX2.bit.GPIO18 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO18 = 1;

    GpioCtrlRegs.GPAPUD.bit.GPIO19 = 1;
    GpioCtrlRegs.GPAGMUX2.bit.GPIO19 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO19 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPAPUD.bit.GPIO20 = 1;
    GpioCtrlRegs.GPADIR.bit.GPIO20 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO20 = 0;

    GpioCtrlRegs.GPAPUD.bit.GPIO21 = 1;
    GpioCtrlRegs.GPADIR.bit.GPIO21 = 1;
    GpioCtrlRegs.GPAMUX2.bit.GPIO21 = 0;

    // -------------------------------------

    GpioCtrlRegs.GPBPUD.bit.GPIO41 = 1;
    GpioCtrlRegs.GPBDIR.bit.GPIO41 = 1;
    GpioDataRegs.GPBDAT.bit.GPIO41 = 0;

    GpioCtrlRegs.GPBPUD.bit.GPIO42 = 1;
    GpioCtrlRegs.GPBMUX1.bit.GPIO42 = 3;
    GpioCtrlRegs.GPBGMUX1.bit.GPIO42 = 3;

    GpioCtrlRegs.GPBPUD.bit.GPIO43 = 1;
    GpioCtrlRegs.GPBMUX1.bit.GPIO43 = 3;
    GpioCtrlRegs.GPBGMUX1.bit.GPIO43 = 3;

   // -------------------------------------

    GpioCtrlRegs.GPBPUD.bit.GPIO58 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO58 = 0x03;
    GpioCtrlRegs.GPBMUX2.bit.GPIO58 = 0x03;

    GpioCtrlRegs.GPBPUD.bit.GPIO59 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO59 = 0x03;
    GpioCtrlRegs.GPBMUX2.bit.GPIO59 = 0x03;

    GpioCtrlRegs.GPBPUD.bit.GPIO60 = 1;
    GpioCtrlRegs.GPBDIR.bit.GPIO60 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO60 = 0x03;
    GpioCtrlRegs.GPBMUX2.bit.GPIO60 = 0x03;

    GpioCtrlRegs.GPBPUD.bit.GPIO61 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO61 = 0x03;
    GpioCtrlRegs.GPBMUX2.bit.GPIO61 = 0x03;

    // -------------------------------------

    GpioCtrlRegs.GPBPUD.bit.GPIO62 = 1;
    GpioCtrlRegs.GPBDIR.bit.GPIO62 = 0;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO62 = 0x01;
    GpioCtrlRegs.GPBMUX2.bit.GPIO62 = 0x02;

    GpioCtrlRegs.GPBPUD.bit.GPIO63 = 1;
    GpioCtrlRegs.GPBGMUX2.bit.GPIO63 = 0x01;
    GpioCtrlRegs.GPBMUX2.bit.GPIO63 = 0x02;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO64 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO64 = 1;
    GpioDataRegs.GPCDAT.bit.GPIO64 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO65 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO65 = 0;
    //GpioCtrlRegs.GPCINV.bit.GPIO65 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO66 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO66 = 1;
    GpioDataRegs.GPCDAT.bit.GPIO66 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO69 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO69 = 1;
    GpioDataRegs.GPCDAT.bit.GPIO69 = 0;

    GpioCtrlRegs.GPCPUD.bit.GPIO70 = 1;
    GpioCtrlRegs.GPCMUX1.bit.GPIO70 = 2;
    GpioCtrlRegs.GPCGMUX1.bit.GPIO70 = 1;

    GpioCtrlRegs.GPCPUD.bit.GPIO71 = 1;
    GpioCtrlRegs.GPCMUX1.bit.GPIO71 = 2;
    GpioCtrlRegs.GPCGMUX1.bit.GPIO71 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO72 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO72 = 0;
    GpioCtrlRegs.GPCGMUX1.bit.GPIO72 = 0x01;
    GpioCtrlRegs.GPCMUX1.bit.GPIO72 = 0x01;

    GpioCtrlRegs.GPCPUD.bit.GPIO73 = 1;
    GpioCtrlRegs.GPCGMUX1.bit.GPIO73 = 0x01;
    GpioCtrlRegs.GPCMUX1.bit.GPIO73 = 0x01;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO78 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO78 = 0;


    // -------------------------------------


    //    GpioCtrlRegs.GPCPUD.bit.GPIO84 = 1;    				// ?????
    //    GpioCtrlRegs.GPCDIR.bit.GPIO84 = 1;
    //    GpioDataRegs.GPCDAT.bit.GPIO84 = 1;

    //    GpioCtrlRegs.GPCPUD.bit.GPIO85 = 1;    				// ?????
    //    GpioCtrlRegs.GPCDIR.bit.GPIO85 = 1;
    //    GpioDataRegs.GPCDAT.bit.GPIO85 = 1;

        // -------------------------------------

    GpioCtrlRegs.GPCMUX2.bit.GPIO86 = 0;
	GpioCtrlRegs.GPCDIR.bit.GPIO86 = 0;
//	GpioDataRegs.GPCSET.bit.GPIO86 = 1;

    // -------------------------------------

//  GpioCtrlRegs.GPCMUX2.bit.GPIO87 = 0;
//	GpioCtrlRegs.GPCDIR.bit.GPIO87 = 1;
//	GpioDataRegs.GPCSET.bit.GPIO87 = 1;

    // -------------------------------------

	GpioCtrlRegs.GPCPUD.bit.GPIO89 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO89 = 1;
    GpioDataRegs.GPCDAT.bit.GPIO89 = 0;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO90 = 1;
    GpioCtrlRegs.GPCDIR.bit.GPIO90 = 1;
    GpioDataRegs.GPCDAT.bit.GPIO90 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPCPUD.bit.GPIO91 = 1;

    GpioCtrlRegs.GPCPUD.bit.GPIO92 = 1;

    // -------------------------------------

    GpioCtrlRegs.GPDPUD.bit.GPIO99 = 1;
    GpioCtrlRegs.GPDDIR.bit.GPIO99 = 1;
    GpioDataRegs.GPDDAT.bit.GPIO99 = 0;

    // -------------------------------------

	EDIS;

}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
void System_Clock_Peripheral_Init()
{
	EALLOW;

	CpuSysRegs.PCLKCR0.bit.CLA1 = 1;
//	CpuSysRegs.PCLKCR0.bit.DMA = 1;
//	CpuSysRegs.PCLKCR0.bit.CPUTIMER0 = 1;
//	CpuSysRegs.PCLKCR0.bit.CPUTIMER1 = 1;
	CpuSysRegs.PCLKCR0.bit.CPUTIMER2 = 1;
//	CpuSysRegs.PCLKCR0.bit.HRPWM = 1;
	CpuSysRegs.PCLKCR0.bit.TBCLKSYNC = 1;

	CpuSysRegs.PCLKCR1.bit.EMIF1 = 1;
	CpuSysRegs.PCLKCR1.bit.EMIF2 = 1;

//	CpuSysRegs.PCLKCR2.bit.EPWM1 = 1;
//	CpuSysRegs.PCLKCR2.bit.EPWM2 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM3 = 1;
//	CpuSysRegs.PCLKCR2.bit.EPWM4 = 1;
//	CpuSysRegs.PCLKCR2.bit.EPWM5 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM6 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM7 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM8 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM9 = 1;
	CpuSysRegs.PCLKCR2.bit.EPWM10 = 1;
//	CpuSysRegs.PCLKCR2.bit.EPWM11 = 1;
//	CpuSysRegs.PCLKCR2.bit.EPWM12 = 1;

//	CpuSysRegs.PCLKCR3.bit.ECAP1 = 1;
//	CpuSysRegs.PCLKCR3.bit.ECAP2 = 1;
//	CpuSysRegs.PCLKCR3.bit.ECAP3 = 1;
//	CpuSysRegs.PCLKCR3.bit.ECAP4 = 1;
//	CpuSysRegs.PCLKCR3.bit.ECAP5 = 1;
//	CpuSysRegs.PCLKCR3.bit.ECAP6 = 1;

//	CpuSysRegs.PCLKCR4.bit.EQEP1 = 1;
//	CpuSysRegs.PCLKCR4.bit.EQEP2 = 1;
//	CpuSysRegs.PCLKCR4.bit.EQEP3 = 1;

//	CpuSysRegs.PCLKCR6.bit.SD1 = 1;
//	CpuSysRegs.PCLKCR6.bit.SD2 = 1;

	CpuSysRegs.PCLKCR7.bit.SCI_A = 1;
	CpuSysRegs.PCLKCR7.bit.SCI_B = 1;
//	CpuSysRegs.PCLKCR7.bit.SCI_C = 1;
//	CpuSysRegs.PCLKCR7.bit.SCI_D = 1;

	CpuSysRegs.PCLKCR8.bit.SPI_A = 1;
//	CpuSysRegs.PCLKCR8.bit.SPI_B = 1;
//	CpuSysRegs.PCLKCR8.bit.SPI_C = 1;

//	CpuSysRegs.PCLKCR9.bit.I2C_A = 1;
	CpuSysRegs.PCLKCR9.bit.I2C_B = 1;

	CpuSysRegs.PCLKCR10.bit.CAN_A = 1;
	CpuSysRegs.PCLKCR10.bit.CAN_B = 1;

//	CpuSysRegs.PCLKCR11.bit.McBSP_A = 1;
//	CpuSysRegs.PCLKCR11.bit.McBSP_B = 1;
//	CpuSysRegs.PCLKCR11.bit.USB_A = 1;

	CpuSysRegs.PCLKCR12.bit.uPP_A = 1;

	CpuSysRegs.PCLKCR13.bit.ADC_A = 1;
	CpuSysRegs.PCLKCR13.bit.ADC_B = 1;
	//CpuSysRegs.PCLKCR13.bit.ADC_C = 1;
	//CpuSysRegs.PCLKCR13.bit.ADC_D = 1;

//	CpuSysRegs.PCLKCR14.bit.CMPSS1 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS2 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS3 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS4 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS5 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS6 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS7 = 1;
//	CpuSysRegs.PCLKCR14.bit.CMPSS8 = 1;
//
//	CpuSysRegs.PCLKCR16.bit.DAC_A = 1;
//	CpuSysRegs.PCLKCR16.bit.DAC_B = 1;
//	CpuSysRegs.PCLKCR16.bit.DAC_C = 1;

	EDIS;
}

// ----------------------------------------------------


	// Vecchia Routine di Inizializzazione

void System_Clock_Init()
{
	// Parametrizzare e VErificare Funzionamento ....
	// Serve Generare LOCK


	EALLOW;

	ClkCfgRegs.SYSPLLCTL1.bit.PLLEN = 0;

	// Wait for the SYSPLL lock
    while(ClkCfgRegs.SYSPLLSTS.bit.LOCKS != 0);

	// Small 100 cycle delay
    asm(" RPT #100 || NOP");


    // Configurazione clock principale con quarzo esterno
	ClkCfgRegs.CLKSRCCTL1.bit.OSCCLKSRCSEL = 0x01;


	// Disabilita PLL
	ClkCfgRegs.SYSPLLCTL1.bit.PLLCLKEN = 0;
    ClkCfgRegs.SYSCLKDIVSEL.bit.PLLSYSCLKDIV = 0;


    // Configura PLL principale
	ClkCfgRegs.SYSPLLMULT.bit.FMULT = 0;
	ClkCfgRegs.SYSPLLMULT.bit.IMULT = (int16) (2 * Frequenza_DSP)/Quarzo;			//10;
	ClkCfgRegs.SYSCLKDIVSEL.bit.PLLSYSCLKDIV = PLL_Doppia_Freq;
	ClkCfgRegs.SYSPLLCTL1.bit.PLLEN = 1;

	// Wait for the SYSPLL lock
    while(ClkCfgRegs.SYSPLLSTS.bit.LOCKS != 1);

    // Seconda Scrittura per certezza e Wait for the SYSPLL lock
    ClkCfgRegs.SYSPLLMULT.bit.FMULT = 0;
    while(ClkCfgRegs.SYSPLLSTS.bit.LOCKS != 1);

    // Abilita Circuito PLL principale
	ClkCfgRegs.SYSPLLCTL1.bit.PLLCLKEN = 1;

	// Small 100 cycle delay
    asm(" RPT #100 || NOP");

	// Definisce Low Speed Clock
	ClkCfgRegs.LOSPCP.bit.LSPCLKDIV = (int16) (((float32) Frequenza_DSP / (2 * Frequenza_Low)));


	// Configurazione clock CAN con quarzo esterno
	ClkCfgRegs.CLKSRCCTL2.bit.CANABCLKSEL = 0x00;
	ClkCfgRegs.CLKSRCCTL2.bit.CANBBCLKSEL = 0x00;


	// PWM Alla massima frequenza
	ClkCfgRegs.PERCLKDIVSEL.bit.EPWMCLKDIV = 0;


//	// Configurazione clock x USB con quarzo esterno
//	ClkCfgRegs.CLKSRCCTL2.bit.AUXOSCCLKSRCSEL = 0x01;
//
//	// Configura PLL Secondaria
//	ClkCfgRegs.AUXPLLCTL1.bit.PLLCLKEN = 0;
//	ClkCfgRegs.AUXPLLCTL1.bit.PLLEN = 0;
//	ClkCfgRegs.AUXCLKDIVSEL = 0;




	// Spegne Oscillatore Interno 2
	ClkCfgRegs.CLKSRCCTL1.bit.INTOSC2OFF = 1;

	// ???????????????????? SERVE LOCK
	//ClkCfgRegs.CLKCFGLOCK1


	EDIS;
}


// ----------------------------------------------------

void inline System_Tick_Init()
{
	// Inizializza Interrupt a 1 ms

	xTICK.RegsAddr = &CpuTimer2Regs;
	xTICK.PeriodInUSec = (float32) Frequenza_Tick * 1000;

	xTICK.RegsAddr->PRD.all  = 0xFFFFFFFF;

	xTICK.RegsAddr->TPR.all  = 0;
	xTICK.RegsAddr->TPRH.all = 0;

	xTICK.RegsAddr->TCR.bit.TSS = 1;

	xTICK.RegsAddr->TCR.bit.TRB = 1;

	xTICK.RegsAddr->PRD.all = (long) (Frequenza_DSP * xTICK.PeriodInUSec);;

    // Set pre-scale counter to divide by 1 (SYSCLKOUT):
	xTICK.RegsAddr->TPR.all  = 0;
	xTICK.RegsAddr->TPRH.all  = 0;

	xTICK.RegsAddr->TCR.bit.TSS = 1;           // 1 = Stop timer, 0 = Start/Restart
                                               // Timer
	xTICK.RegsAddr->TCR.bit.TRB = 1;           // 1 = reload timer
	xTICK.RegsAddr->TCR.bit.SOFT = 0;
	xTICK.RegsAddr->TCR.bit.FREE = 0;          // Timer Free Run Disabled
	xTICK.RegsAddr->TCR.bit.TIE = 1;      	   // 0 = Disable/ 1 = Enable Timer
                                           	   // Interrupt

	xTICK.RegsAddr->TCR.all = 0x4000; 		   // Use write-only instruction to set TSS bit = 0

}

// ----------------------------------------------------
// ----------------------------------------------------
// ----------------------------------------------------

void inline System_Interrupt_Start()
{
    IER |= M_INT1; 			//Enable Interrupt AD
    IER |= M_INT3; 			//Enable Interrupt PWM
    IER |= M_INT9; 			//Enable Interrupt Seriale
    IER |= M_INT11; 		//Enable Interrupt CLA
    IER |= M_INT14;			//Enable Interrupt Timer 2

	EINT;   				// Enable Global interrupt INTM
   	ERTM;   				// Enable Global realtime interrupt DBGM
}

// ----------------------------------------------------

void System_Start()
{

	System_WatchDog_Enable();

	System_Interrupt_Start();

}

// ----------------------------------------------------

void System_RAM_Init()
{
//	EALLOW;
//	MemCfgRegs.DxINIT.all = 0x0F;
//	while (MemCfgRegs.DxINITDONE.all != 0x0F);
//	EDIS;
}

// ----------------------------------------------------
unsigned long XAR0_Fuffa;
void System_DCSM_Unlock()
{
	int i;

//    volatile long int *ECSL = (volatile long int *)0x5F000;

	unsigned long LinkPointer;
	unsigned long *Z1_ZoneSelBlockPtr, *Z1_Save_ZoneSelBlockPtr;
	unsigned long *Z2_ZoneSelBlockPtr;
	int bitpos = 28;
	int zerofound = 0;

	XAR0_Fuffa = *(unsigned long *)0x703F0;         /* Dummy read of SECDC REGISTER   */

	XAR0_Fuffa = *(unsigned long *)0x78000;         /* Dummy read of Z1 - LinkPointer1 */
	XAR0_Fuffa = *(unsigned long *)0x78004;         /* Dummy read of Z1 - LinkPointer2 */
	XAR0_Fuffa = *(unsigned long *)0x78008;         /* Dummy read of Z1 - LinkPointer3 */

	XAR0_Fuffa = *(unsigned long *)0x78200;         /* Dummy read of Z2 - LinkPointer1 */
	XAR0_Fuffa = *(unsigned long *)0x78204;         /* Dummy read of Z2 - LinkPointer2 */
	XAR0_Fuffa = *(unsigned long *)0x78208;         /* Dummy read of Z2 - LinkPointer3 */

	XAR0_Fuffa = *(unsigned long *)0x78010;         /* Dummy read of Z1 - PSWDLOCK  */
	XAR0_Fuffa = *(unsigned long *)0x78210;         /* Dummy read of Z2 - PSWDLOCK  */

	XAR0_Fuffa = *(unsigned long *)0x78014;         /* Dummy read of Z1 - CRCLOCK  */
	XAR0_Fuffa = *(unsigned long *)0x78214;         /* Dummy read of Z2 - CRCLOCK  */

	XAR0_Fuffa = *(unsigned long *)0x78018;         /* Dummy read of Z1 - JTAGLOCK  */
	XAR0_Fuffa = *(unsigned long *)0x78218;         /* Dummy read of Z2 - JTAGLOCK  */

	XAR0_Fuffa = *(unsigned long *)0x7801E;         /* Dummy read of Z1 - BOOTCTRL  */
	XAR0_Fuffa = *(unsigned long *)0x7821E;         /* Dummy read of Z2 - BOOTCTRL  */

	LinkPointer = *(unsigned long *)0x5F000;         /* Read Z1-Linkpointer out of Z1-LINKPOINTER register */
	LinkPointer = LinkPointer << 3;                              /* Bits 31,30 and 29 as most-sigificant 0 are invalid LinkPointer options */
	while ((zerofound == 0) && (bitpos > -1))
	{
	    if ((LinkPointer & 0x80000000) == 0)
		{
		    zerofound = 1;
			Z1_ZoneSelBlockPtr = (unsigned long *)(0x78000 + ((bitpos + 3)*16));
		}
		else
		{
		    bitpos--;
			LinkPointer = LinkPointer << 1;
		}
	}
	if (zerofound == 0)
	{
	    Z1_ZoneSelBlockPtr = (unsigned long *)0x78020;
	}
	Z1_Save_ZoneSelBlockPtr = Z1_ZoneSelBlockPtr;

	bitpos = 28;
	zerofound = 0;
	LinkPointer = *(unsigned long *)0x5F040;         /* Read Z2-Linkpointer out of Z1-LINKPOINTER register */
	LinkPointer = LinkPointer << 3;                              /* Bits 31 and 30 as most-sigificant 0 are invalid LinkPointer options */
	while ((zerofound == 0) && (bitpos > -1))
	{
	    if ((LinkPointer & 0x80000000) == 0)
		{
		    zerofound = 1;
			Z2_ZoneSelBlockPtr = (unsigned long *)(0x78200 + ((bitpos + 3)*16));
		}
		else
		{
		    bitpos--;
			LinkPointer = LinkPointer << 1;
		}
	}
	if (zerofound == 0)
	{
	    Z2_ZoneSelBlockPtr = (unsigned long *)0x78220;
	}

	/* Perform dummy reads of the Zone Select Block locations */
	for (i = 0; i < 8; i++)
	{
		XAR0_Fuffa =  *Z1_ZoneSelBlockPtr;
		XAR0_Fuffa =  *Z2_ZoneSelBlockPtr;
		Z1_ZoneSelBlockPtr++;
		Z2_ZoneSelBlockPtr++;
	}

	volatile long int *CSM = (volatile long int *)0x5F010; //CSM register file
	//volatile long int *CSMPWL = (volatile long int *)0x78078; //CSM Password location (assuming default Zone sel block)

	Z1_Save_ZoneSelBlockPtr += 0x04;			// Aggiunge Offset x PSW ....
												// Scrivo 4 invece di 8 perch puntatore a unsigned long

	volatile long int tmp;
	int I;
	for (I=0;I<4; I++) tmp = *Z1_Save_ZoneSelBlockPtr++;

	*CSM++ = 0x12345678;
	*CSM++ = 0x12345678;
	*CSM++ = 0x12345678;
	*CSM++ = 0x12345678;

}
// ----------------------------------------------------

void System_DCSM()
{

	//System_DCSM_Unlock();

}

// ----------------------------------------------------
// ----------------------------------------------------

void System_CLA_Memory_Init()
{
    extern uint32_t Cla1funcsRunStart, Cla1funcsLoadStart, Cla1funcsLoadSize;
    EALLOW;

    // Copia Programma CLA in RAM
    memcpy((uint32_t *)&Cla1funcsRunStart, (uint32_t *)&Cla1funcsLoadStart,
           (uint32_t)&Cla1funcsLoadSize);

    // Initialize and wait for CLA1ToCPUMsgRAM
    MemCfgRegs.MSGxINIT.bit.INIT_CLA1TOCPU = 1;
    while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CLA1TOCPU != 1){};
    // Initialize and wait for CPUToCLA1MsgRAM
    MemCfgRegs.MSGxINIT.bit.INIT_CPUTOCLA1 = 1;
    while(MemCfgRegs.MSGxINITDONE.bit.INITDONE_CPUTOCLA1 != 1){};

    // Inizializza Memoria Programma CLA			ATTENZIONE: Bisogna modificare anche routine System_CLA_Memory_Release()
    //												------------------------------------------------------------------------
    MemCfgRegs.LSxMSEL.bit.MSEL_LS2 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS2 = 1;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS3 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS3 = 1;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS4 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS4 = 1;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS5 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS5 = 1;

    // Inizializza Memoria Dati CLA					ATTENZIONE: Bisogna modificare anche routine System_CLA_Memory_Release()
    //												------------------------------------------------------------------------
    MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = 0;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = 1;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = 0;

    EDIS;
}

// ----------------------------------------------------

void System_CLA_Memory_Release()
{
	// Reset della CLA
	EALLOW;
	Cla1Regs.MCTL.bit.HARDRESET = 1;
	EDIS;

	// Rilascia Memoria Programma
	EALLOW;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS2 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS2 = 0;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS3 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS3 = 0;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS4 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS4 = 0;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS5 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS5 = 0;

    // Rilascia Memoria Dati
    MemCfgRegs.LSxMSEL.bit.MSEL_LS0 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS0 = 0;
    MemCfgRegs.LSxMSEL.bit.MSEL_LS1 = 0;
    MemCfgRegs.LSxCLAPGM.bit.CLAPGM_LS1 = 0;

    EDIS;

}

// ----------------------------------------------------

extern interrupt void Cla1Task1 ( void );
extern interrupt void Cla1Task2 ( void );
extern interrupt void Cla1Task3 ( void );
extern interrupt void Cla1Task4 ( void );
extern interrupt void Cla1Task5 ( void );
extern interrupt void Cla1Task6 ( void );
extern interrupt void Cla1Task7 ( void );
extern interrupt void Cla1Task8 ( void );

interrupt void cla1Isr_Nop(void)
{
    // Insert ISR Code here

    // Next two lines for debug only to halt the processor here
    // Remove after inserting ISR Code
    asm ("      ESTOP0");
    for(;;);
}

interrupt void cla1Isr1 ()
{
    //
    // Acknowledge the end-of-task interrupt for task 1
    //
    PieCtrlRegs.PIEACK.all = M_INT11;

    //
    // Uncomment to have debugger halt here
    //
//    asm(" ESTOP0");
}

void System_CLA_Cpu_Init()
{
    //
    // Compute all CLA task vectors
    // On Type-1 CLAs the MVECT registers accept full 16-bit task addresses as
    // opposed to offsets used on older Type-0 CLAs
    //
    EALLOW;
    Cla1Regs.MVECT1 = (uint16_t)(&Interrupt_CLA_PWM);
    Cla1Regs.MVECT2 = (uint16_t)(&Cla1Task2);
    Cla1Regs.MVECT3 = (uint16_t)(&Cla1Task3);
    Cla1Regs.MVECT4 = (uint16_t)(&Cla1Task4);
    Cla1Regs.MVECT5 = (uint16_t)(&Cla1Task5);
    Cla1Regs.MVECT6 = (uint16_t)(&Cla1Task6);
    Cla1Regs.MVECT7 = (uint16_t)(&Cla1Task7);
    Cla1Regs.MVECT8 = (uint16_t)(&Cla1Task8);

    //
    // Enable the IACK instruction to start a task on CLA in software
    // for all  8 CLA tasks. Also, globally enable all 8 tasks (or a
    // subset of tasks) by writing to their respective bits in the
    // MIER register
    //
    Cla1Regs.MCTL.bit.IACKE = 1;
    Cla1Regs.MIER.all = 0x0001;

    // Seleziona Interrrupt AD per avere interrupt CLA
    DmaClaSrcSelRegs.CLA1TASKSRCSEL1.bit.TASK1 = 1;

    //
    // Configure the vectors for the end-of-task interrupt for all
    // 8 tasks
    //
    PieVectTable.CLA1_1_INT = &cla1Isr1;
    PieVectTable.CLA1_2_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_3_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_4_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_5_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_6_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_7_INT = &cla1Isr_Nop;
    PieVectTable.CLA1_8_INT = &cla1Isr_Nop;

//    //
//    // Enable CLA interrupts at the group and subgroup levels
//    //
//    PieCtrlRegs.PIEIER11.all = 0xFFFF;
//    IER |= (M_INT11 );

}

// ----------------------------------------------------

void System_CLA_Init()
{
	System_CLA_Memory_Init();

	System_CLA_Cpu_Init();

}

// ----------------------------------------------------

void System_Init(void)
{

	System_WatchDog_Disable();

	System_RAM_Init();							// Inzializza RAM

    System_Flash_Init();						// Da Attivare

    System_GPIO_DisablePullups();				// Da Verificare di avere settato quello corretto ....

    System_Clock_Init();						// Da verificare che sia corretto !!!!

    System_Clock_Peripheral_Init();

    System_Gpio_Init();

    System_Interrupt_Init();

    System_CLA_Init();

    System_DCSM();

}
// ----------------------------------------------------
// ----------------------------------------------------
// ----------------------------------------------------
