//*****************************************************************************
//
// usbhcdc.c - This file contains the USB CDC host class driver.
//
// Copyright (c) 2008-2019 Texas Instruments Incorporated.  All rights reserved.
// Software License Agreement
// 
// Texas Instruments (TI) is supplying this software for use solely and
// exclusively on TI's microcontroller products. The software is owned by
// TI and/or its suppliers, and is protected under applicable copyright
// laws. You may not combine this software with "viral" open-source
// software in order to form a larger program.
// 
// THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
// NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
// NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. TI SHALL NOT, UNDER ANY
// CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
// DAMAGES, FOR ANY REASON WHATSOEVER.
// 
// This is part of revision 2.2.0.295 of the Tiva USB Library.
//
//*****************************************************************************

#include <string.h>
#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "driverlib/usb.h"
#include "usblib/usblib.h"
#include "usblib/usblibpriv.h"
#include "usblib/usbcdc.h"
#include "usblib/host/usbhost.h"
#include "usblib/host/usbhostpriv.h"
#include "usblib/host/usbhcdc.h"

static void *CDCDriverOpen(tUSBHostDevice *psDevice);
static void CDCDriverClose(void *pvInstance);

//*****************************************************************************
//
//! \addtogroup usblib_host_class
//! @{
//
//*****************************************************************************

//*****************************************************************************
//
// Prototypes for local functions.
//
//*****************************************************************************
static void USBHCDCCallback(uint32_t ui32Pipe, uint32_t ui32Event);

//*****************************************************************************
//
// The size of a USB CDC serial data size packets.
//
//*****************************************************************************
#define USBHCDC_DATA_SIZE    64

//*****************************************************************************
//
// This is the structure that holds all of the data for a given instance of
// a CDC device.
//
//*****************************************************************************
struct tCDCInstance
{
    //
    // Save the device instance.
    //
    tUSBHostDevice *psDevice;

    //
    // Used to save the callback.
    //
    tUSBCallback pfnCallback;

    //
    // Callback data provided by caller.
    //
    void *pvCBData;

    //
    // Used to remember what type of device was registered.
    //
    tCDCSubClassProtocol iDeviceType;

    //
    // State Flags for the CDC device.
    //
    uint32_t ui32CDCFlags;

    //
    // Bulk IN pipe.
    //
    uint32_t ui32CDCBulkInPipe;
    uint16_t ui16PipeSizeIn;
    tUSBCallback pfnInCallback;
    void *pvInBuffer;

    //
    // Bulk OUT pipe.
    //
    uint32_t ui32CDCBulkOutPipe;
    uint16_t ui16PipeSizeOut;
    tUSBCallback pfnOutCallback;
    void *pvOutBuffer;

    //
    // Interrupt IN pipe.
    //
    uint32_t ui32CDCIntInPipe;
    uint16_t ui16PipeSizeIntIn;
    tUSBCallback pfnIntInCallback;
    void *pvIntInBuffer;
};

//*****************************************************************************
//
// The instance data storage for attached CDC devices.
//
//*****************************************************************************
static tCDCInstance g_psCDCDevice =
{
 0
};

//*****************************************************************************
//
//! This constant global structure defines the CDC Class Driver that is
//! provided with the USB library.
//
//*****************************************************************************
const tUSBHostClassDriver g_sUSBCDCClassDriver =
{
    USB_CLASS_CDC,
    CDCDriverOpen,
    CDCDriverClose,
    0
};

//*****************************************************************************
//
//! This function handles event callbacks from the USB CDC driver layer.
//!
//! \param pvCdc is the pointer that was passed in to the USBHCDCOpen()
//! call.
//! \param ui32Event is the event that has been passed up from the CDC driver.
//! \param ui32MsgParam has meaning related to the \e ui32Event that occurred.
//! \param pvMsgData has meaning related to the \e ui32Event that occurred.
//!
//! This function will receive all event updates from the CDC driver layer.
//!
//! \return Non-zero values should be assumed to indicate an error condition.
//
//*****************************************************************************
static void USBHCDCCallback(uint32_t ui32Pipe, uint32_t ui32Event)
{
    switch (ui32Event)
    {
        //
        // New CDC serial device has been connected so notify the application.
        //
        case USB_EVENT_CONNECTED:
        {
            //
            // Remember that a CDC serial device is present.
            //
            g_psCDCDevice.ui32CDCFlags |= (uint32_t)USBHCDC_DEVICE_PRESENT;

            //
            // Notify the application about the connected device.
            //
            g_psCDCDevice.pfnCallback(0, ui32Event,
                                       g_psCDCDevice.ui32CDCFlags,
                                       g_psCDCDevice.pvCBData);

            break;
        }
        case USB_EVENT_DISCONNECTED:
        {
            //
            // No CDC is present.
            //
            g_psCDCDevice.ui32CDCFlags &= ~USBHCDC_DEVICE_PRESENT;

            //
            // Notify the application about the connected device.
            //
            g_psCDCDevice.pfnCallback(0, ui32Event,
                                       g_psCDCDevice.ui32CDCFlags,
                                       g_psCDCDevice.pvCBData);

            break;
        }
        //
        // Called when new data is available on the interrupt IN pipe.
        //
        case USB_EVENT_RX_AVAILABLE:
        {
            //
            // If the callback exists then call it.
            //
            if(g_psCDCDevice.pfnCallback != 0)
            {
                //
                // Check for how much data has been received.
                //
                g_psCDCDevice.ui16PipeSizeIn = USBHCDPipeCurrentSizeGet(ui32Pipe);

                //
                // Read out the data into the USB IN buffer.
                //
                if (g_psCDCDevice.ui16PipeSizeIn > 0)
                {
                    USBHCDCReadData(&g_psCDCDevice,
                                    g_psCDCDevice.pvInBuffer,
                                    g_psCDCDevice.ui16PipeSizeIn);
                }
            }
        }
        //
        // Called when data has finished transferring on the interrupt OUT pipe.
        //
        case USB_EVENT_TX_COMPLETE:
        {
            //
            // If the callback exists then call it.
            //
            if(g_psCDCDevice.pfnCallback != 0)
            {
                //
                // Notify the application that the TX Complete occurred.
                //
                g_psCDCDevice.pfnCallback(&g_psCDCDevice, ui32Event,
                                          ui32Pipe, 0);
            }
        }
    }
}

//*****************************************************************************
//
// This function handles callbacks for the Interrupt IN endpoint.
//
//*****************************************************************************

static void USBHCDCIntINCallback(uint32_t ulPipe, uint32_t ulEvent)
{
    //
    // Handles a request to schedule a new request on the interrupt IN
    // pipe.
    //
    if(ulEvent == USB_EVENT_SCHEDULER)
    {
        //
        // Schedule packet to send on IN pipe for triggering device to send
        // data when available.
        //
        // TODO: TI Comments 9/1/2020:
        // To work with usb_dev_cdcserial, this is hard coded to the
        // Bulk IN pipe, but that may be an issue limited to the
        // usb_dev_cdcserial example project.  Further investigation is
        // needed to assess if it should always be on the Bulk IN pipe or
        // on the Interrupt IN pipe.  The commented out line of code will
        // send the trigger on the Interrupt IN pipe so if data is not being
        // received from the device, try using that instead.
        //
        USBHCDPipeSchedule(g_psCDCDevice.ui32CDCBulkInPipe, 0, 1);
//        USBHCDPipeSchedule(ulPipe, 0, 1);
    }

    //
    // Called when new data is available on the interrupt IN pipe.
    //
    if(ulEvent == USB_EVENT_RX_AVAILABLE)
    {
        //
        // If the callback exists then call it.
        //
        if(g_psCDCDevice.pfnCallback != 0)
        {
            //
            // Check for how much data was received.
            //
            g_psCDCDevice.ui16PipeSizeIntIn = USBHCDPipeCurrentSizeGet(ulPipe);

            //
            // Read the data out.
            //
            if (g_psCDCDevice.ui16PipeSizeIntIn > 0)
            {
                USBHCDCReadData(&g_psCDCDevice,
                                g_psCDCDevice.pvIntInBuffer,
                                g_psCDCDevice.ui16PipeSizeIntIn);
            }
        }
    }
}

//*****************************************************************************
//
//! This function is used to open an instance of a CDC device.
//!
//! \param pfnCallback is the function that will be called whenever changes
//! are detected for this device.
//! \param pvCBData is the data that will be returned in when the
//! \e pfnCallback function is called.
//!
//! This function creates an instance of an specific type of CDC device.  The
//! \e iDeviceType parameter is one subclass/protocol values of the types
//! specified in enumerated types tCDCSubClassProtocol.  Only devices that
//! enumerate with this type will be called back via the \e pfnCallback
//! function.  The \e pfnCallback parameter is the callback function for any
//! events that occur for this device type.  The \e pfnCallback function must
//! point to a valid function of type \e tUSBCallback for this call to complete
//! successfully.  To release this device instance the caller of USBHCDCOpen()
//! should call USBHCDCClose() and pass in the value returned from the
//! USBHCDCOpen() call.
//!
//! \return This function returns and instance value that should be used with
//! any other APIs that require an instance value.  If a value of 0 is returned
//! then the device instance could not be created.
//
//*****************************************************************************
tCDCInstance *
USBHostCDCOpen(tUSBCallback pfnCallback, void *pvRxBuffer)
{
    //
    // Save the instance data for this device.
    //
    g_psCDCDevice.pfnCallback = pfnCallback;
    g_psCDCDevice.iDeviceType = eUSBHCDCClassDirectLineControl;
    g_psCDCDevice.pvInBuffer = pvRxBuffer;

    //
    // Return the device instance pointer.
    //
    return(&g_psCDCDevice);
}

//*****************************************************************************
//
//! This function is used to release an instance of a CDC device.
//!
//! \param psCDCInstance is the instance value for a CDC device to release.
//!
//! This function releases an instance of a CDC device that was created by a
//! call to USBHCDCOpen().  This call is required to allow other CDC devices
//! to be enumerated after another CDC device has been disconnected.  The
//! \e psCDCInstance parameter should hold the value that was returned from
//! the previous call to USBHCDCOpen().
//!
//! \return None.
//
//*****************************************************************************
void
USBHostCDCClose(tCDCInstance *psCDCInstance)
{
    //
    // Disable any more notifications from the CDC layer.
    //
    psCDCInstance->pfnCallback = 0;

    //
    // Mark this device slot as free.
    //
    psCDCInstance->iDeviceType = eUSBHCDCClassNone;
}

//*****************************************************************************
//
//! This function is used to open an instance of the CDC driver.
//!
//! \param psDevice is a pointer to the device information structure.
//!
//! This function will attempt to open an instance of the CDC driver based on
//! the information contained in the \e psDevice structure.  This call can fail
//! if there are not sufficient resources to open the device.  The function
//! returns a value that should be passed back into USBCDCClose() when the
//! driver is no longer needed.
//!
//! \return The function will return a pointer to a CDC driver instance.
//
//*****************************************************************************
static void *
CDCDriverOpen(tUSBHostDevice *psDevice)
{
    int32_t i32Idx;
    uint8_t NumOfInterfaces, i;
    tEndpointDescriptor *psEndpointDescriptor;
    tInterfaceDescriptor *psInterface;

    NumOfInterfaces = psDevice->psConfigDescriptor->bNumInterfaces;

    psInterface = USBDescGetInterface(psDevice->psConfigDescriptor, 0, 0);
    //
    // Search the currently open instances for one that supports the protocol
    // of this device.
    //
    if(g_psCDCDevice.iDeviceType ==
            psInterface->bInterfaceProtocol)
    {
        //
        // Save the device pointer.
        //
        g_psCDCDevice.psDevice = psDevice;

        for (i = 0; i < NumOfInterfaces; i++)
        {
            //
            // Get the interface descriptor for each interface
            //
            psInterface = USBDescGetInterface(psDevice->psConfigDescriptor, i, 0);

            //
            // Loop through the endpoints of the device.
            //
            for(i32Idx = 0; i32Idx < 3; i32Idx++)
            {
                //
                // Get the interrupt endpoint descriptor
                //
                psEndpointDescriptor =
                        USBDescGetInterfaceEndpoint(psInterface, i32Idx,
                                                    psDevice->ui32ConfigDescriptorSize);

                //
                // If no more endpoints then break out.
                //
                if(psEndpointDescriptor == 0)
                {
                    break;
                }

                //
                // See if this is a bulk endpoint.
                //
                if((psEndpointDescriptor->bmAttributes & USB_EP_ATTR_TYPE_M) ==
                        USB_EP_ATTR_BULK)
                {
                    //
                    // See if this is bulk IN or bulk OUT.
                    //
                    if(psEndpointDescriptor->bEndpointAddress & USB_EP_DESC_IN)
                    {
                        //
                        // Allocate the USB Pipe for this Bulk IN endpoint.
                        //
                        g_psCDCDevice.ui32CDCBulkInPipe =
                                USBHCDPipeAllocSize(0, USBHCD_PIPE_BULK_IN,
                                                    psDevice,
                                                    psEndpointDescriptor->wMaxPacketSize,
                                                    USBHCDCCallback);
                        //
                        // Configure the USB pipe as a Bulk IN endpoint.
                        //
                        USBHCDPipeConfig(g_psCDCDevice.ui32CDCBulkInPipe,
                                         psEndpointDescriptor->wMaxPacketSize,
                                         0,
                                         ((psEndpointDescriptor->bEndpointAddress) &
                                                 USB_EP_DESC_NUM_M));
                    }
                    else
                    {
                        //
                        // Allocate the USB Pipe for this Bulk OUT endpoint.
                        //
                        g_psCDCDevice.ui32CDCBulkOutPipe =
                                USBHCDPipeAllocSize(0, USBHCD_PIPE_BULK_OUT,
                                                    psDevice,
                                                    psEndpointDescriptor->wMaxPacketSize,
                                                    USBHCDCCallback);
                        //
                        // Configure the USB pipe as a Bulk OUT endpoint.
                        //
                        USBHCDPipeConfig(g_psCDCDevice.ui32CDCBulkOutPipe,
                                         psEndpointDescriptor->wMaxPacketSize,
                                         0, (psEndpointDescriptor->bEndpointAddress &
                                                 USB_EP_DESC_NUM_M));
                    }
                }

                //
                // See if this is an interrupt endpoint.
                //
                if((psEndpointDescriptor->bmAttributes & USB_EP_ATTR_TYPE_M) ==
                        USB_EP_ATTR_INT)
                {
                    //
                    // See if this is interrupt IN endpoint.
                    //
                    if(psEndpointDescriptor->bEndpointAddress & USB_EP_DESC_IN)
                    {
                        //
                        // Allocate the USB Pipe for this Interrupt IN endpoint.
                        //
                        g_psCDCDevice.ui32CDCIntInPipe =
                                USBHCDPipeAlloc(0, USBHCD_PIPE_INTR_IN,
                                                psDevice, USBHCDCIntINCallback);

                        //
                        // Configure the USB pipe as a Interrupt IN endpoint.
                        //
                        USBHCDPipeConfig(g_psCDCDevice.ui32CDCIntInPipe,
                                         psEndpointDescriptor->wMaxPacketSize,
                                         psEndpointDescriptor->bInterval,
                                         (psEndpointDescriptor->bEndpointAddress &
                                                 USB_EP_DESC_NUM_M));
                    }
                }
            }
        }
        //
        // If there is a callback function call it to inform the application
        // that the device has been enumerated.
        //
        if(g_psCDCDevice.pfnCallback != 0)
        {
            g_psCDCDevice.pfnCallback(
                    g_psCDCDevice.pvCBData,
                    USB_EVENT_CONNECTED,
                    (uint32_t)&g_psCDCDevice, 0);
        }

        //
        // Save the device pointer.
        //
        g_psCDCDevice.psDevice = psDevice;
        return (&g_psCDCDevice);
    }

    //
    // If we get here, no user has registered an interest in this particular
    // CDC device so we return an error.
    //
    return(0);
}


//*****************************************************************************
//
//! This function is used to release an instance of the CDC driver.
//!
//! \param pvInstance is an instance pointer that needs to be released.
//!
//! This function will free up any resources in use by the CDC driver instance
//! that is passed in.  The \e pvInstance pointer should be a valid value that
//! was returned from a call to USBCDCOpen().
//!
//! \return None.
//
//*****************************************************************************
static void
CDCDriverClose(void *pvInstance)
{
    tCDCInstance *psInst;

    //
    // Get our instance pointer.
    //
    psInst = (tCDCInstance *)pvInstance;

    //
    // Reset the device pointer.
    //
    psInst->psDevice = 0;

    //
    // Free the Interrupt IN pipe.
    //
    if(psInst->ui32CDCIntInPipe != 0)
    {
        USBHCDPipeFree(psInst->ui32CDCIntInPipe);
    }

    //
    // Free the Bulk IN pipe.
    //
    if(psInst->ui32CDCBulkInPipe != 0)
    {
        USBHCDPipeFree(psInst->ui32CDCBulkInPipe);
    }

    //
    // Free the Bulk OUT pipe.
    //
    if(psInst->ui32CDCBulkOutPipe != 0)
    {
        USBHCDPipeFree(psInst->ui32CDCBulkOutPipe);
    }

    //
    // If the callback exists, call it with a DISCONNECTED event.
    //
    if(psInst->pfnCallback != 0)
    {
        psInst->pfnCallback(psInst->pvCBData, USB_EVENT_DISCONNECTED,
                            (uint32_t)pvInstance, 0);
    }
}


//*****************************************************************************
//
//! This function is used to get line coding information for a CDC device.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//!
//! This function is used to retrieve line parameter information from a device.
//! The parameters retrieved are DTR rate, stop bit, partity and data bits
//! This request is sent when a CDC device is connected to the host.
//!
//! \return Returns the number of bytes read into the \e pui8Buffer.
//
//*****************************************************************************
uint32_t
USBHCDCGetLineCoding(tCDCInstance *psCDCInstance, uint8_t *pui8Buffer,
                     uint32_t ui32Size)
{
    tUSBRequest sSetupPacket;
    uint32_t ui32Bytes;

    //
    // This is a Class specific interface OUT request.
    //
    sSetupPacket.bmRequestType = USB_RTYPE_DIR_IN | USB_RTYPE_CLASS |
                                 USB_RTYPE_INTERFACE;

    //
    // Request a Device Descriptor.
    //
    sSetupPacket.bRequest = USBREQ_GET_LINE_CODING;
    sSetupPacket.wValue = 0;

    //
    // Set this on interface 0.
    //
    sSetupPacket.wIndex = 0;

    //
    // This is always 7 for this request.
    //
    sSetupPacket.wLength = 0x07;

    //
    // Put the setup packet in the buffer.
    //
    ui32Bytes = (USBHCDControlTransfer(0, &sSetupPacket,
                                       psCDCInstance->psDevice,
                                       pui8Buffer, 0x07, MAX_PACKET_SIZE_EP0));

    return ui32Bytes;  
}


//*****************************************************************************
//
//! This function is used to set the line coding information for a CDC device.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//! \param pui8Data is the line parameter values
//!
//! This function is used to set the line parameter information from a device.
//! The parameters are baud rate, stop bit, partiy, and data bits.
//! This request is sent when a CDC device is connected to the host.
//!
//! \return Returns the number of bytes sent to the device.
//
//*****************************************************************************
uint32_t
USBHCDCSetLineCoding(tCDCInstance *psCDCInstance,  uint8_t *pui8Data)
{
    tUSBRequest sSetupPacket;

    //
    // This is a Class specific interface OUT request.
    //
    sSetupPacket.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS |
                                 USB_RTYPE_INTERFACE;

    //
    // Request a Device Descriptor.
    //
    sSetupPacket.bRequest = USBREQ_SET_LINE_CODING;
    sSetupPacket.wValue = 0;

    //
    // Set this on interface 0.
    //
    sSetupPacket.wIndex = 0;

    //
    // This is always 7 for this request.
    //
    sSetupPacket.wLength = 0x07;  

    //
    // Put the setup packet in the buffer.
    // This request includes an OUT transaction and an IN transaction.
    // The OUT transaction is the line coding structure of length 7 bytes.
    //
    USBHCDControlTransfer(0, &sSetupPacket, psCDCInstance->psDevice,
                                 pui8Data, 0x07, MAX_PACKET_SIZE_EP0);

    return (0);  
}

//*****************************************************************************
//
//! This function is used to set the line state of a CDC device.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//!
//! This function is used to set the line parameter information from a device.
//! The parameters are baud rate, stop bit, partiy and data bits.
//! This request is sent when a CDC device is connected to the host.
//!
//! \return Returns 0 since this is a status message.
//
//*****************************************************************************
uint32_t
USBHCDCSetControlLineState(tCDCInstance *psCDCInstance, uint16_t carrierValue)
{
    tUSBRequest sSetupPacket;

    //
    // This is a Class specific interface OUT request (from host to device).
    //
    sSetupPacket.bmRequestType = USB_RTYPE_DIR_OUT | USB_RTYPE_CLASS |
                                 USB_RTYPE_INTERFACE;

    //
    // Request a Device Descriptor.
    //
    sSetupPacket.bRequest = USBREQ_SET_CONTROL_LINE_STATE;
    if(carrierValue == 0x03)
    {
      sSetupPacket.wValue = 0x03; // Activate carrier, DTE present.
    }
    else
    {
        sSetupPacket.wValue = 0x00; // Deactivate carrier, DTE not present.
    }

    //
    // Set this on interface 0.
    //
    sSetupPacket.wIndex = 0;

    //
    // This is always 0 for this request.
    //
    sSetupPacket.wLength = 0x0;

    //
    // Put the setup packet in the buffer.
    // This request includes an OUT transaction and an IN transaction.
    // The OUT transaction is the line coding structure of length 7 bytes.
    //
    USBHCDControlTransfer(0, &sSetupPacket, psCDCInstance->psDevice,
                                 0, 0, MAX_PACKET_SIZE_EP0);

    return (0);  
}

//*****************************************************************************
//
//! This function is used to read data from the bulk IN endpoint.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//! \param pui8Data is the memory buffer to use to store the data.
//! \param ui32Size is the size in bytes of the buffer pointed to by
//! \e pui8Buffer.
//!
//! This function is used to read data sent from a connected device.  The
//! function specifically reads data from a USB Bulk IN pipe.
//!
//! \return Returns the number of bytes read from buffer.
//
//*****************************************************************************
uint32_t
USBHCDCReadData(tCDCInstance *psCDCInstance, uint8_t *pui8Data,
                uint32_t ui32Size)
{
    //
    // Read data from bulk IN pipe.
    //
    ui32Size = USBHCDPipeReadNonBlocking(psCDCInstance->ui32CDCBulkInPipe,
                                         pui8Data, ui32Size);

    //
    // Return the number of bytes read from the IN pipe.
    //
    return(ui32Size);
}

//*****************************************************************************
//
//! This function writes data to the bulk OUT endpoint.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//! \param pui8Data is the memory buffer storing the data.
//! \param ui32Size is the size in bytes of the buffer pointed to by
//! \e pui8Buffer.
//!
//! This function is used to send data to a connected device by writing the
//! data stored in a memory buffer to a USB Bulk OUT pipe.
//!
//! \return Returns the number of bytes read from buffer.
//
//*****************************************************************************
uint32_t
USBHCDCWriteData(tCDCInstance *psCDCInstance, uint8_t *pui8Data,
                 uint32_t ui32Size)
{
    //
    // Send data to device via bulk OUT pipe.
    //
    ui32Size = USBHCDPipeWrite(psCDCInstance->ui32CDCBulkOutPipe,
                                             pui8Data, ui32Size);

    //
    // Return the number of bytes sent to OUT pipe.
    //
    return(ui32Size);
}

//*****************************************************************************
//
//! This function schedules the bulk OUT endpoint to send data.
//!
//! \param psCDCInstance is the value that was returned from the call to
//! USBHCDCOpen().
//! \param pui8Data is the memory buffer storing the data.
//! \param ui32Size is how many bytes of data from \e pui8Buffer should be
//! written to the endpoint.
//!
//! This function will not block for sending but will load the USB FIFO with
//! the provided data.  The amount of data loaded will be limited to what will
//! fit in the FIFO for provided Pipe which in this case is the CDC Bulk Out
//! pipe.
//!
//! \return None
//
//*****************************************************************************
void
USBHCDCScheduleWrite(tCDCInstance *psCDCInstance, uint8_t *pui8Data,
                 uint32_t ui32Size)
{
    //
    // Schedule the next OUT Pipe transaction.
    //
    USBHCDPipeSchedule(psCDCInstance->ui32CDCBulkOutPipe, pui8Data, ui32Size);
}

//*****************************************************************************
//
//! This function returns the current count of bytes received on the USB
//! IN pipe.
//!
//! \return Returns the number of bytes read from buffer.
//
//*****************************************************************************
uint32_t
USBHCDCInPipeDataCount(void)
{
    //
    // Return the number of bytes received on the IN pipe.
    //
    return(g_psCDCDevice.ui16PipeSizeIn);
}

//*****************************************************************************
//
//! This function is used to initialize a CDC interface after a CDC device
//! is detected.
//!
//! \param psCdcInstance is the instance value for this device.
//!
//! This function should be called after receiving a \b STATE_CDC_DEVICE_INIT
//! event in the callback function provided by USBHCDCOpen(), however this
//! function should only be called outside the callback function.  This will
//! initialize the CDC interface.  The \e psCdcInstance value is the value
//! that was returned when the application called USBHCDCOpen().
//! This function only needs to be called once per connection event but it
//! should be called every time a \b STATE_CDC_DEVICE_INIT event occurs.
//!
//! \return This function returns 0 to indicate success any non-zero value
//! indicates an error condition.
//
//*****************************************************************************
uint32_t
USBHCDCInitNewDevice(tCDCInstance *psCdcInstance)
{
    uint8_t dataArray[USB_GET_LINE_CODING_SIZE];
    uint8_t parmArray[7] = {0x00, 0xC2, 01, 00, 00, 00, 0x08};

    //
    // Get Line coding data.  size of data is 7 bytes
    //
    USBHCDCGetLineCoding(psCdcInstance,
                         dataArray, USB_GET_LINE_CODING_SIZE);

    //
    // Set Control line state - deactivate the carrier if it is active.
    //
    USBHCDCSetControlLineState(psCdcInstance,
                               CDC_DEACTIVATE_CARRIER);

    //
    // Set the parameters of the line:
    // [Stop bits = 1, Partity = none, Data bits = 8]
    //
    USBHCDCSetLineCoding(psCdcInstance, parmArray);

    //
    // Get Line coding data.  size of data is 7 bytes.
    //
    USBHCDCGetLineCoding(psCdcInstance,
                         dataArray, USB_GET_LINE_CODING_SIZE);

    return(0);
}

//*****************************************************************************
//
// Close the Doxygen group.
//! @}
//
//*****************************************************************************

