/**
 * These definitions are shared between the boot loader and the application.
 * The header in the application must match the expectations of the boot loader.
 * The two of them must also agree on flash memory usage.
 */

#ifndef BOOT_LOADER_CONFIG_H_
#define BOOT_LOADER_CONFIG_H_

#include <sysctl.h>


/******************************************************************************
 * Constant and Macro Definitions using #define
 ******************************************************************************/

/**
 * Application: The first byte of FLASH memory that should be erased by the EraseAppl command.
 * Boot Loader: This is used during the search for the application header.
 */
#define FLASH_APPLICATIONS_START  APP_BASE // from ami_system_cfg.h (typically 0x02000)

/** 
 * Application: One byte after the last byte of FLASH memory that should be erased by the EraseAppl command.
 * Boot Loader: This is used during the search for the application header.
 */
#define FLASH_MEMORY_TOP CALIBRATIONS_BASE

/** 
 * Application: Not used
 * Boot Loader: This is used to limit the flash locations erased by the bootloader
 * 				Ensures that the calibrations are retained across a flash.
 */
#define FLASH_RSVD_SPACE (END_OF_FLASH_MEMORY - FLASH_MEMORY_TOP)

//*****************************************************************************
//
// The frequency of the crystal used to clock the microcontroller.
//
// This defines the crystal frequency used by the microcontroller running the
// boot loader.  If this is unknown at the time of production, then use the
// UART_AUTOBAUD feature to properly configure the UART.
//
// Depends on: None
// Exclusive of: None
// Requires: None
//
//*****************************************************************************
#if (AMIHW_TARGET == AMIHW_TARGET_LM3S9B92_EVB)
	#define CRYSTAL_FREQ            16000000 // The EVB uses 16 MHz
// I've compared to the values used in SYS_Initialize() in sys.c about line 150.
#elif (AMIHW_TARGET == AMIHW_TARGET_LM3S2B93_PRODUCTION)
	#define CRYSTAL_FREQ            10000000 // The production hardware uses a 10 MHz oscillator
#elif (AMIHW_TARGET == AMIHW_TARGET_TM4C129)
   /*
    * The Royale III production system utilizes a 25MHz external "Main"
    * Oscillator/Crystal, required by the Ethernet core.
    */
	#define CRYSTAL_FREQ            SYSCTL_XTAL_25MHZ
#elif (AMIHW_TARGET == AMIHW_TARGET_TM4C1290_ALTEX)
   #define CRYSTAL_FREQ            10000000 // The production hardware uses a 10 MHz oscillator
#endif

#ifndef FLASH_APPLICATIONS_START
	#error MIA6
#endif


/**
 * Check for required conditions.
 * 
 */
#ifdef APP_START_ADDRESS
	#if APP_START_ADDRESS == FLASH_APPLICATIONS_START
	// Good, the boot loader can transfer control to the application.
	#else
		#if CUSTOM_BOOT_LOADER_ENABLED >= 0
			#error Can't compile/link the boot loader because the CUSTOM_BOOT_LOADER_ENABLED switch is turned off.
		#endif /* CUSTOM_BOOT_LOADER_ENABLED > 0 */
		#error The EraseAppl command must erase all of the application.
	#endif
#endif

#ifdef VTABLE_START_ADDRESS
	#if VTABLE_START_ADDRESS == FLASH_APPLICATIONS_START
		// Good
	#else
		#if CUSTOM_BOOT_LOADER_ENABLED <= 0
			#error Can't compile/link the boot loader because the CUSTOM_BOOT_LOADER_ENABLED switch is turned off.
		#endif /* CUSTOM_BOOT_LOADER_ENABLED > 0 */
		#error The vector table is expected to be at the start of the application.  See boot loader comments near VTABLE_START_ADDRESS.
	#endif
#endif



/*
 * 
 * This section of #define's will set up the UART port(s) used by 
 * the custom boot loader.
 * 
 * 
 * When executing the boot loader, allow communication on one or more of the UART ports.
 * Select one or more UART ports; the custom boot loader will listen on all the
 * selected ports at all times.
 *   
 * When would you want to disable a port? If unsolicited input will be sent into the
 * fuel cell system when you're trying to do an update, then the unsolicited
 * bytes will mingle with the protocol for the update -- causing errors.  So
 * disabling the port will keep the custom boot loader from listening on that port.
 * 
 */
 
/** Port 2 is usually the Customer Serial Interface on the fuel cell system
 * (typically set to 9600 baud).  
 * It is also the accessible UART on the evaluation board -- typically set to 
 * 115200 baud and connected to the command prompt. 
 * By defining this symbol, the custom boot loader will use the port for updates
 * and synchronize to the PC serial baud rate -- typically 115200 baud. 
 */
//#define UART_ENABLE_PORT2

/** Port x is not used because it is usually connected to LIN. */
//#define UART_ENABLE_PORTx

/** Port 1 is the accessible debug UART on the fuel cell -- typically set to
 * 115200 baud and connected to the command prompt.
 * By defining this symbol, the custom boot loader will use the port for updates
 * and synchronize to the PC serial baud rate -- typically 115200 baud. 
 */
#define UART_ENABLE_PORT1


/* I've got a little calculation going on here with temporary results. */
#ifdef UART_ENABLE_PORT0
#define NUMBER_UARTS_TEMP0 1
#else
#define NUMBER_UARTS_TEMP0 0
#endif

#ifdef UART_ENABLE_PORT1
#define NUMBER_UARTS_TEMP1 1
#else
#define NUMBER_UARTS_TEMP1 0
#endif

#ifdef UART_ENABLE_PORT2
#define NUMBER_UARTS_TEMP2 1
#else
#define NUMBER_UARTS_TEMP2 0
#endif

/** How many UART ports are enabled.  This is used in the timing estimator in bl_uart.c */
#define NUMBER_UARTS (NUMBER_UARTS_TEMP0+NUMBER_UARTS_TEMP1+NUMBER_UARTS_TEMP2)



/******************************************************************************
 * Enumerations, Structures and Typedefs
 ******************************************************************************/

/**
 * This is the custom header stored in binary images of Stellaris boot loaders and applications programs.
 * The sflash.exe program on the PC will fill in the checksum and totalBytes fields.
 */
struct customBootLoaderHeader {
    long code;              //! 0xFEEDC0DE used to identify this header
    char filler1[4];        //! "HERE" or such for visual ID
    
    /* these two are filled by the PC utility */
    unsigned long checksum; //! location for checksum of binary image to be stored
    long totalBytes;        //! total bytes in the binary image
    
    
    long filler2;           //! spare location left 0xFFFFFFFF
    char identity[252];     //! text returned by boot loader to PC
    long zero;              //! convenience to guarantee \0 terminated string.

	// The boot loader has a serial number, but the application does not.
    long serialno[4];		//! left 0xFFFFFFFFUL for boot loader or 0 for application
};



#endif /*BOOT_LOADER_CONFIG_H_*/

