/*****************************************************************************
 * File Name : ECGSystemFunctions.c 
 *
 * Brief	 : The file holds all the functions implementation and th
 *  			Algorithm implementation for the ECG system. The file   
 *  			includes the following functions:-                      
 *                                                                      
 * 	- ADS1258 Channel Data Read                                  

 *
 * Copyright (C) 2009 -2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
******************************************************************************/


#include "ECGSystemFunctions.h"
#include "oled.h"
#include "lcd_command.h"

#define MAX_PEAK_TO_SEARCH 				5
#define THREE_SEC_SAMPLES  				1500

#define BUFFER_SHIFT 0
#define ASSEMBLY_IMP 1


extern ECG_FilterProcess(Int16 *,Int16 *,Int16*);

//#pragma DATA_SETCTION(WorkingBuff,".WorkBuffer");
/* Working Buffer Used for Filtering*/
Int16 WorkingBuff[MAXCHAN][2 * FILTERORDER];
/* System Information structure*/
extern ECG_System_Info ECG_Info;
/* Multi band FIR LPF coefficients*/
extern Int16 CoeffBuf[];

/* structure which stores the UART configuration values*/
PSP_UartConfig uartCfg;
/* Instance no used for UART */
Uint8 instNum = 0;
/* Handle for UART driver*/
PSP_Handle hUart = 0;


/*  Pointer which points to the index in B4 buffer where the processed data*/
/*  has to be filled */
static Uint16 QRS_B4_Buffer_ptr = 0 ;
/* Variable which will hold the calculated heart rate */
Uint16 QRS_Heart_Rate = 0 ;

/* 	Variable which holds the threshold value to calculate the maxima*/
Uint16 QRS_Threshold_New = 0;

/* Variables to hold the sample data for calculating the 1st and 2nd */
/* differentiation                                                   */


Int16 QRS_Second_Prev_Sample = 0 ;
Int16 QRS_Prev_Sample = 0 ;
Int16 QRS_Current_Sample = 0 ;
Int16 QRS_Next_Sample = 0 ;
Int16 QRS_Second_Next_Sample = 0 ;
Int16 j = 0;

Int16 cs_flag=0;
Uint8 col = 0;

/*Flag which identifies the duration for which sample count has to be incremented*/
Uint8 Start_Sample_Count_Flag = 0;
Uint8 first_peak_detect = FALSE ;
Uint16 sample_count = 0 ;
Uint16 sample_index[MAX_PEAK_TO_SEARCH+2] = {0};

/* indicates the Zoom-in/ Zoom-out status of the waveform displayed on LCD */
Uint8 Zoom_Flag = 0;
/* variable to hold the status of the lead to be displayed on a key-press */
Uint8 Key_Count = 0;
/* variable to store the zooming value */
Uint16 Zoom_Factor = MEDIUM;
Uint16 FB[128][3];

/*-------------------------------------------------------------------------**
** Function Name 	: ECG_UARTTxCurrSamples()                              **
** Description		: 	- The function transmits the current row () of the **
** 					LeadInfo buffer i.e., current samples of all the       **
** 					leads through UART At every 1 sec, the UART Packet     **
** 					Header has to be transmitted. Also the LeadOff         **
** 					Status and the HR value has to be transmitted The      **
** 					Packet No is transmitted along with all the packets.   **
** 					The Packet no has to start from 0 at every 1 sec.      **
**                                                                         **
** Parameters	  : CurrLead 		- In - Current processing lead         **
** 				    LeadInfoBufRow  - In - Currently filled row            **
** Return			: None                                                 **
**-------------------------------------------------------------------------*/

	Uint8 UARTPacket[256] = {0};

	unsigned char UARTstart = 0;
	unsigned char UARTend =0;
	unsigned UARTdataRdy = 0;
	Uint8 LCD_pacer =0;
	extern Uint16 Pacer_detected;

void ECG_UARTTxCurrSamples(Int16 * CurSample)
{
	extern Uint8 LeadOff_Flag ;
	static Uint8 UART_PacketNo = 0;
	Uint8 Count1 = 0, pacer;
	Uint8 ECG_UART_Header[5] = {0x00,0x80,0x00,0x80,0x00};
	extern Uint16 LeadOffStatus;

	static Uint8 PacerArry[FILTERORDER] = {0}, start =0;
	int i;

	static Uint8 even_flag = 0;

	UARTstart = 0;
	even_flag++;
	if(even_flag == 2)
	{
			even_flag =0;

			pacer = PacerArry[start];
			LCD_pacer |= pacer;
			PacerArry[start++] = (Uint8) Pacer_detected;

			if ( start == ((FILTERORDER/4)-1 )) start = 0;

			if(UART_PacketNo == 0)
			{
				/*  Set the UART Header for Tx*/
				for(Count1=0;Count1<5;Count1++)
				UARTPacket[UARTstart++] = ECG_UART_Header[Count1];

				/* Set the packet No for Tx*/
				UARTPacket[UARTstart++] = UART_PacketNo;

				/* Set the Current Heart rate*/
				UARTPacket[UARTstart++] = ECG_Info.ECG_HeartRate;

				/* Set the current Leadoff status*/
				UARTPacket[UARTstart++] = LeadOffStatus & 0x00FF;
				UARTPacket[UARTstart++] = (LeadOffStatus & 0xFF00)>>8 ;


			}
			else
			{
				/* Set the packet No for Tx*/
				UARTPacket[UARTstart++] = UART_PacketNo;
			}
			/* Formation 8 channel ECG data */

			for ( i =0 ; i < 8; i++)
			{
				if ( pacer == 1 )
				{
				 	CurSample[i] |=1;
				}
				else 
				{
					CurSample[i] &= 0xFFFE;
				}

				UARTPacket[UARTstart++]= (CurSample[i] & 0x00FF);
				UARTPacket[UARTstart++]= (CurSample[i] & 0xFF00) >> 8 ;
			}
			
			Pacer_detected = 0;

			/* Reset PacketNo at every 1 second*/

			UART_PacketNo++;

			if(UART_PacketNo == 255)
			{
				UART_PacketNo = 0;
			}
			/* Send ECG data to UART */
			dda_uartWrite(hUart,(const char *)UARTPacket, UARTstart, -1, &uartCfg);
	}


	return;
}

#if RAW_DATA
extern Uint16 ChannelNo;
extern Int16 Raw_ecg[];
extern Int16 bank_flag;
extern Int16 Sample_Data[];

void ECG_UARTTxCurrSamples_debug(Int16 * CurSample)
{

	static Uint16 UART_PacketNo = 0;
	Uint8 Count1 = 0;
	Uint8 ECG_UART_Header[5] = {0x00,0x80,0x00,0x80,0x00};

	int i;
	Int32 ptr =0;
	while ( bank_flag)
	{
		UARTstart = 0;

		if(UART_PacketNo == 0)
		{
			/*  Set the UART Header for Tx*/
			for(Count1=0;Count1<5;Count1++)
			UARTPacket[UARTstart++] = ECG_UART_Header[Count1];

			UARTPacket[UARTstart++] = ChannelNo;

		}
		/* Formation 8 channel ECG data */
		
		for ( i =0 ; i < 16; i++)
		{
			UARTPacket[UARTstart++]= (Sample_Data[ptr] & 0x00FF);
			UARTPacket[UARTstart++]= (Sample_Data[ptr++] & 0xFF00) >> 8 ;
		}
	
		/* Reset PacketNo at every 1 second*/

		UART_PacketNo++;

		if(UART_PacketNo == 250)
		{
			UART_PacketNo = 0;
		}
		/* Send ECG data to UART */
		dda_uartWrite(hUart,(const char*)UARTPacket, UARTstart, -1, &uartCfg);
		
		if ( ptr >= 80000)
		{
			bank_flag =0;
		}
	}
	return;
}

#endif
/*-----------------------------------------------------------------------------**
** Function Name : ECG_LCDPlotCurrSample()                                     **
** Description	  : The function plots the current processed sample in the LCD **
** Parameters	  : CurrLead 		- In - Current processing lead             **
** 				    LeadInfoBufRow  - In - Currently filled row                **
** Returns		  : void                                                       **
**-----------------------------------------------------------------------------*/
void ECG_LCDPlotCurrSample(Int16 *CurSample, Uint16 DispLead)
{

	static Int16 row_amplitude_old = 40;
	Int16 Amplitude = 0;
	Int16 i, row_amplitude = 0;

	Zoom_Factor = 1;
	/* check for Zoom key_press status*/
	if(Zoom_Flag == 1)
	{
		Zoom_Factor = HIGH;
	}
	else if(Zoom_Flag == 2)
	{
		Zoom_Factor = LOW;
	}
	else
	{
		Zoom_Factor = MEDIUM;
	}
	switch (DispLead)
	{

		case 0:
			
		/*Display Lead I*/
					row_amplitude = 70 -
									(((double)(CurSample[0]) /
									Zoom_Factor) * 40);
				break;
		case 1:
				/*Display Lead II*/
				row_amplitude = 70 -
								(((double)(CurSample[1])/
								Zoom_Factor) * 40);
				break;

		case 2:
				/*Display Lead III*/
				row_amplitude = 70 -
								(((double)(CurSample[1] - CurSample[0])/
								Zoom_Factor) * 40);
				break;

		case 3:
				/*Display Lead aVR*/
				row_amplitude = 70 -
								(((double)(-0.5 * (CurSample[1] + CurSample[0])))/
								Zoom_Factor) * 40;
		    	break;

		case 4:
				/*Display Lead aVL*/
				row_amplitude = 70 -
								(((double)CurSample[0] -(0.5 * CurSample[1]))/
								Zoom_Factor) * 40;

				break;

		case 5:
				/*Display Lead aVF*/
				row_amplitude = 70 -
								(((double) CurSample[1] - ( 0.5 * CurSample[0]))/
								(Zoom_Factor) * 40);

				break;

		case 6:
				/*Display Lead V1*/
				row_amplitude = 70 -
								(((double) (CurSample[2] )/
								Zoom_Factor) * 40);
				break;

		case 7:
				/*Display Lead V2*/
					row_amplitude = 70 -
								(((double) (CurSample[3] )/
								Zoom_Factor) * 40);
				break;

		case 8:
				/*Display Lead V3*/
				row_amplitude = 70 -
								(((double) (CurSample[4] )/
								Zoom_Factor) * 40);
				break;

		case 9:
				/*Display Lead V4*/
				row_amplitude = 70 -
								(((double) (CurSample[5] )/
								Zoom_Factor) * 40);
				break;

		case 10:
				/*Display Lead V5*/
				row_amplitude = 70 -
								(((double) (CurSample[6] )/
								Zoom_Factor) * 40);
				break;

		case 11:
				/*Display Lead V6*/
				row_amplitude = 70 -
								(((double) (CurSample[7] )/
								Zoom_Factor) * 40);
				break;
		default:
				/*Default condition-Display Lead II*/
				row_amplitude = 70 -
								(((double)(CurSample[1])/
								Zoom_Factor) * 40);
			break;
	}

	/* limitting the display to the available LCD screen.*/
	if(row_amplitude < 2)
	{
		row_amplitude = 2;
	}
	else
	{
		//if(row_amplitude > 106)
		//{
		//	row_amplitude = 106;
		//}
				if(row_amplitude > 100) // EVM 5515
				{
					row_amplitude = 100;
				}
	}

	/* Calculating the absolute row value */
	Amplitude = abs(row_amplitude_old - row_amplitude);

            // save the position
			FB[col][0] = Amplitude;
			FB[col][1] = row_amplitude_old;
			FB[col][2] = row_amplitude;


	if(Amplitude >= 2 )
	{
#if 1			 
// save time for C5515 EVM   
				cmdWrite(SET_COL_ADD); /* row address */
				dataWrite(col);
				dataWrite(col);
				cmdWrite(SET_ROW_ADD);/* col. address */
				if(row_amplitude > row_amplitude_old)
				{
			        dataWrite(22+row_amplitude_old);
			        dataWrite(22+row_amplitude_old+Amplitude);

			    }
			    else
			    {
			        dataWrite(22+row_amplitude_old-Amplitude);
			        dataWrite(22+row_amplitude_old);

			    }
			    cmdWrite(WRITE_COMMAND);

				for(j = 0 ;j < (Amplitude - 1);j++)
	    		{
					dataWrite(0);
					dataWrite(0x3F);
					dataWrite(0);
	    		}			    
#else
		for(j = 0 ;j < (Amplitude - 1);j++)
		{
            cmdWrite(SET_COL_ADD);
	        dataWrite(col);
	        dataWrite(col);        
            
	        cmdWrite(SET_ROW_ADD); 
			if(row_amplitude > row_amplitude_old)
			{
				dataWrite(22+(row_amplitude_old+j));
				dataWrite(22+(row_amplitude_old+j));
			}
			else
			{
				dataWrite(22+(row_amplitude_old-j));
				dataWrite(22+(row_amplitude_old-j));
			}
            
	        cmdWrite(WRITE_COMMAND);            // Enable MCU to Write into RAM	 
			/* glow the pixel corresponding to that row */
			dataWrite(0);
			dataWrite(0x3F);
			dataWrite(0);
		}
#endif

	}
	else
	{
    cmdWrite(SET_COL_ADD);
	dataWrite(col);
	dataWrite(col);

	cmdWrite(SET_ROW_ADD);
	dataWrite(22+row_amplitude);
	dataWrite(22+row_amplitude);

	cmdWrite(WRITE_COMMAND);
	/* glow the pixel corresponding to that row */
	dataWrite(0);
	dataWrite(0x3F);
	dataWrite(0);
	}

	row_amplitude_old = row_amplitude;

	if ( LCD_pacer )
	{
		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(23);
		dataWrite(23);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0x3F);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(24);
		dataWrite(24);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0x3F);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(25);
		dataWrite(25);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0x3F);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(26);
		dataWrite(26);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0x3F);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(27);
		dataWrite(27);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0x3F);
	   	dataWrite(0);
	   	dataWrite(0);
		LCD_pacer = 0;
    }
    else
	{
		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(23);
		dataWrite(23);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(24);
		dataWrite(24);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(25);
		dataWrite(25);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(26);
		dataWrite(26);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0);
	   	dataWrite(0);
	   	dataWrite(0);

		cmdWrite(SET_COL_ADD);
		dataWrite(col);
		dataWrite(col);

		cmdWrite(SET_ROW_ADD);
		dataWrite(27);
		dataWrite(27);

		cmdWrite(WRITE_COMMAND);
		/* glow the pixel corresponding to that row */
	   	dataWrite(0);
	   	dataWrite(0);
	   	dataWrite(0);
		LCD_pacer = 0;

	}

	col ++;
	if(col >= 128)
	{
		col = 0;
	}
#if 1
	        LCD_clear_window(col,23,col,27);
            lcd_remove_waveform(col,FB[col][0],FB[col][1],FB[col][2]);
#else
	/* Clear Window for next 10 pixels to give a scrolling appearence */
	LCD_clear_window(col,22,col+10,128);
//	cmdWrite(0x8E);
//	dataWrite(col);
//	dataWrite(22);
//	dataWrite(col+10);
//	dataWrite(128);
#endif

}

/*--------------------------------------------------------------------------**
** Function Name : ECG_ProcessCurrSample()                                  **
** Description	  :                                                         **
** 				The function process one sample of data at a time and       **
** 				which stores the filtered out sample in the Leadinfobuff.   **
** 				The function does the following :-                          **
**                                                                          **
** 				- DC Removal of the current sample                          **
** 				- Multi band FIR LPF with Notch at 50Hz filtering           **
** Parameters	  :                                                         **
** 				- CurrAqsSample		- In - AqsnBuf Row count                **
** 				- FirstFlag			- In - Flag to idenftify the first exec **
** 				- CurrFilterChan	- In - Current Row of filtering         **
** 				- CurrLeadInfoRow	- In - Current row of lead info buff    **
** Return 		  : None                                                    **
**--------------------------------------------------------------------------*/
void ECG_ProcessCurrSample(Int16 *CurrAqsSample, Int16 *FilteredOut)
{

 	static Uint16 bufStart=0, bufCur = FILTERORDER-1, FirstFlag = 1;

	/* Count variable*/
	Uint16 Cur_Chan;
	Int16 FiltOut;
	Int32 FilterOut[2];

	if  ( FirstFlag )
	{
		for ( Cur_Chan =0 ; Cur_Chan < FILTERORDER; Cur_Chan++)
		{
			WorkingBuff[0][Cur_Chan] = 0;
			WorkingBuff[1][Cur_Chan] = 0;
			WorkingBuff[2][Cur_Chan] = 0;
			WorkingBuff[3][Cur_Chan] = 0;
			WorkingBuff[4][Cur_Chan] = 0;
			WorkingBuff[5][Cur_Chan] = 0;
			WorkingBuff[6][Cur_Chan] = 0;
			WorkingBuff[7][Cur_Chan] = 0;
		} 
		FirstFlag = 0;
	}
	for (Cur_Chan = 0; Cur_Chan < 8; Cur_Chan++)
	{

		/* Store the DC removed value in Working buffer in millivolts range*/
		WorkingBuff[Cur_Chan][bufCur] = CurrAqsSample[Cur_Chan];
		ECG_FilterProcess(&WorkingBuff[Cur_Chan][bufCur],CoeffBuf,(Int16*)&FilterOut);
		/* Store the DC removed value in Working buffer in millivolts range*/
		WorkingBuff[Cur_Chan][bufStart] = CurrAqsSample[Cur_Chan];


		FiltOut = (Int16)(((FilterOut[0] >> 15) & (0x0000FFFF)));

		/* Store the filtered out sample to the LeadInfo buffer*/
		FilteredOut[Cur_Chan] = FiltOut ;//(CurrOut);

	}

	bufCur++;
	bufStart++;
	if ( bufStart  == (FILTERORDER-1))
	{
		bufStart=0; 
		bufCur = FILTERORDER-1;
	}

	return ;
}
/*-------------------------------------------------------------------------------**
** 	Function Name : Wait()                                                       **
** 	Description  : 	- The function gives a delay of n CPU cycles where n is the  **
** 					  parameter Count                                            **
** 	Parameters	:	- Count - In - no of delay cycles                            **
** 	Return		: none                                                           **
**-------------------------------------------------------------------------------*/

void wait(Uint32 count)
{
	Int32 i;

	for(i=0;i<count;i++)
	asm("	nop");
}


/*-------------------------------------------------------------------**
** 	Function Name : QRS_Get_Heart_Rate()                             **
** 	Description - This function gives the HR value when called       **
** 	Parameter -   Nil                                                **
** 	Return - QRS_Heart_Rate                                          **
**-------------------------------------------------------------------*/

Uint16 QRS_Get_Heart_Rate( )
{
	return ( QRS_Heart_Rate ) ;
}



static void QRS_check_sample_crossing_threshold( Uint16 scaled_result )
{
	/* array to hold the sample indexes S1,S2,S3 etc */
	extern Uint16 LeadOffStatus;
	
	static Uint16 s_array_index = 0 ;
	static Uint16 m_array_index = 0 ;
	
	static Uint8 threshold_crossed = FALSE ;
	static Uint16 maxima_search = 0 ;
	static Uint8 peak_detected = FALSE ;
	static Uint16 skip_window = 0 ;
	static Uint16 peak = 0;
	static Uint32 sample_sum = 0;
	static Uint16 nopeak=0;
	Uint16 HRAvg;

	extern Uint8 HR_flag;
	extern Uint8 Initial_HR_Flag;

	sample_count ++ ;
	if( TRUE == threshold_crossed  )
	{
		/*
		Once the sample value crosses the threshold check for the
		maxima value till MAXIMA_SEARCH_WINDOW samples are received
		*/
		maxima_search ++ ;

		if( scaled_result > peak )
		{
			peak = scaled_result ;
		}

		if( maxima_search >= MAXIMA_SEARCH_WINDOW )
		{
			// Store the maxima values for each peak
			threshold_crossed = FALSE ;
			peak_detected = TRUE ;
		}

	}
	else if( TRUE == peak_detected )
	{
		/*
		Once the sample value goes below the threshold
		skip the samples untill the SKIP WINDOW criteria is meet
		*/
		skip_window ++ ;

		if( skip_window >= MINIMUM_SKIP_WINDOW )
		{
			peak_detected = FALSE ;
		}

		if( m_array_index == MAX_PEAK_TO_SEARCH )
		{
			
			sample_sum =  (  sample_sum / (MAX_PEAK_TO_SEARCH - 1) ) ;
			HRAvg = (Uint16) sample_sum;
			if(((LeadOffStatus & 0x0007)== 0x0007)||((LeadOffStatus & 0x01FF)== ELECTRODE_LA))
			{
				
				QRS_Heart_Rate = (Uint16) 60 *  SAMPLING_RATE / HRAvg ;
				if(QRS_Heart_Rate > 250)
					QRS_Heart_Rate = 250 ;
			}
			else
			{
				QRS_Heart_Rate = 0;
			}
			/* Setting the Current HR value in the ECG_Info structure*/
			ECG_Info.ECG_HeartRate = QRS_Heart_Rate;
			HR_flag = 1;
			Initial_HR_Flag = 1;
			skip_window = 0 ;

	 		sample_count = 0 ;
	 		s_array_index = 0 ;
	 		m_array_index = 0 ;
			sample_index[0] = 0 ;
			sample_index[1] = 0 ;
			sample_index[2] = 0 ;
			sample_index[3] = 0 ;
			sample_index[4] = 0 ;
			Start_Sample_Count_Flag = 1;
			sample_sum = 0;
		}
	}
	else if( scaled_result > QRS_Threshold_New )
	{
		/*
			If the sample value crosses the threshold then store the sample index
		*/
		Start_Sample_Count_Flag = 1;
		m_array_index++;
		threshold_crossed = TRUE ;
		peak = scaled_result ;
		nopeak = 0;
		maxima_search = 0 ;
		skip_window = 0 ;
		/*	storing sample index*/
	   	sample_index[ s_array_index ] = sample_count ;
		if( s_array_index >= 1 )
		{
			sample_sum += sample_index[ s_array_index ] - sample_index[ s_array_index - 1 ] ;

		}
		s_array_index ++ ;
	}

	else if(( scaled_result < QRS_Threshold_New ) && (Start_Sample_Count_Flag == 1))
	{
        nopeak++;	
        if (nopeak > (3 * SAMPLING_RATE))
        { 
        	sample_count = 0 ;
	 		s_array_index = 0 ;
	 		m_array_index = 0 ;
			sample_index[0] = 0 ;
			sample_index[1] = 0 ;
			sample_index[2] = 0 ;
			sample_index[3] = 0 ;
			sample_index[4] = 0 ;
			Start_Sample_Count_Flag = 0;
			peak_detected = FALSE ;
			sample_sum = 0;
        	    	
        	first_peak_detect = FALSE;
	      	nopeak=0;

			QRS_Heart_Rate = 0;
			ECG_Info.ECG_HeartRate = QRS_Heart_Rate;
			HR_flag = 1;
			Initial_HR_Flag = 1;

        }
	}

   else
   {
    nopeak++;	
   	if (nopeak > (3 * SAMPLING_RATE))
     { 
     	sample_count = 0 ;
 		s_array_index = 0 ;
 		m_array_index = 0 ;
		sample_index[0] = 0 ;
		sample_index[1] = 0 ;
		sample_index[2] = 0 ;
		sample_index[3] = 0 ;
		sample_index[4] = 0 ;
		Start_Sample_Count_Flag = 0;
		peak_detected = FALSE ;
		sample_sum = 0;
     	
     	first_peak_detect = FALSE;
	 	nopeak = 0;
		QRS_Heart_Rate = 0;
		ECG_Info.ECG_HeartRate = QRS_Heart_Rate;
		HR_flag = 1;
		Initial_HR_Flag = 1;

     }
   }

}

/*-----------------------------------------------------------------------**
** 	Function Name : QRS_process_buffer()                                 **
** 	Description -                                                        **
** 	                                                                     **
** 					This function will be doing the first and second     **
** 					order differentiation for 	input sample,            **
** 					QRS_Current_Sample.After we process the data we can  **
** 					fill the 	QRS_Proc_Data_Buffer which is the input  **
** 					for HR calculation Algorithm. This function is       **
** 					called for each n sample.Once we have received 6s of **
** 					processed 	data(i.e.Sampling rate*6) in the B4      **
** 					buffer we will start the heart rate calculation for  **
** 					first time and later we will do heart rate           **
** 					calculations once we receive the defined 	number   **
** 					of samples for the expected number of refresh        **
** 					seconds.                                             **
** 	                                                                     **
** 	Parameters  - Nil                                                    **
** 	Return 		- None                                                   **
**-----------------------------------------------------------------------*/
static void QRS_process_buffer( void )
{

	extern Uint16 LeadOffStatus;
	Int16 first_derivative = 0 ;
	Uint16 scaled_result = 0 ;

	static Uint16 max = 0;

	/* calculating first derivative*/
	first_derivative = QRS_Next_Sample - QRS_Prev_Sample  ;

	/*taking the absolute value*/

	if(first_derivative < 0)
	{
		first_derivative = -(first_derivative);
	}

	scaled_result = first_derivative;

	if( scaled_result > max )
	{
		max = scaled_result ;
	}

	QRS_B4_Buffer_ptr++;
	if (QRS_B4_Buffer_ptr ==  THREE_SEC_SAMPLES)
	{
		QRS_Threshold_New = (Uint16)( QRS_THRESHOLD_FRACTION * max ) ;
		if(max > 10)
		first_peak_detect = TRUE ;
		QRS_B4_Buffer_ptr = 0;
		max = 0;
	}


	if( TRUE == first_peak_detect )
	{
		QRS_check_sample_crossing_threshold( scaled_result ) ;
	}
}

/*-----------------------------------------------------------------------**
**                                                                       **
** 	Function Name : QRS_Algorithm_Interface                              **
** 	Description -   This function is called by the main acquisition      **
** 					thread when (N/3) samples are filled in				 **
** 					the Lead_Info_buffer for Lead II.Here N = 351 and    **
** 					thus it is called for a fill of every 117			 **
** 					samples												 **
**                  Before calling the process_buffer() the below check  **
** 					has to be done. i.e. We have always received +2      **
** 					samples before starting the processing  for each     **
** 					samples. This function basically checks the          **
** 					difference between the QRS_Lead_Process_Ptr and      **
** 					Main_Index_Ptr and picks the data from               **
** 					Lead_Info_Buffer for processing.  This function also **
** 					checks for the roll back and fill the variables      **
** 					accordingly which will be used in the 1st and 2nd    **
** 					differentiation calculations.                        **
**                                                                       **
** 	Parameters  : - data - array having eight channel read data          **
** 	Return		: None                                                   **
**-----------------------------------------------------------------------*/
void QRS_Algorithm_Interface(Int16 CurrSample)
{

	static short prev_data[32] ={0};
	short i;
	Int32 Mac=0;
	prev_data[0] = CurrSample;
	for ( i=31; i > 0; i--)
	{
		Mac += prev_data[i];
		prev_data[i] = prev_data[i-1];

	}
	Mac += CurrSample;
	Mac = Mac >> 2;
	CurrSample = (short) Mac;
	QRS_Second_Prev_Sample = QRS_Prev_Sample ;
	QRS_Prev_Sample = QRS_Current_Sample ;
	QRS_Current_Sample = QRS_Next_Sample ;
	QRS_Next_Sample = QRS_Second_Next_Sample ;
	QRS_Second_Next_Sample = CurrSample ;

	QRS_process_buffer();
}

