/*****************************************************************************
 * File Name : ECGSystemInit.c 
 *
 * Brief	 : The file does all the initialization regarding the ECG system.
 *                                                                      
 *
 * Copyright (C) 2009 -2010 Texas Instruments Incorporated - http://www.ti.com/ 
 * 
 * 
 *  Redistribution and use in source and binary forms, with or without 
 *  modification, are permitted provided that the following conditions 
 *  are met:
 *
 *    Redistributions of source code must retain the above copyright 
 *    notice, this list of conditions and the following disclaimer.
 *
 *    Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the 
 *    documentation and/or other materials provided with the   
 *    distribution.
 *
 *    Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS 
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT 
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT 
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT 
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT 
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE 
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
******************************************************************************/

#include "ECGSystemInit.h"
#include "tistdtypes.h"

#define INIT_SPI_FREQ  1000000;
#define READ_SPI_FREQ 25000000;

extern ECG_System_Info ECG_Info;

/* structure which stores the UART configuration values*/
extern PSP_UartConfig uartCfg;
/* Instance no used for UART */
extern Uint8 instNum;
/* Handle for UART driver*/
extern PSP_Handle hUart;

/*--------------------------------------------------------------------------------------**
** 	Function Name : ECG_C5505Init()                                                     **
** 	Description : 	- The function initializes all the peripherals of                   **
** 					  the C5505 as required for the ECG system. Also it                 **
** 					  initializes the Front End board components as                     **
** 					  required by the system Has the following                          **
** 					  initialization                                                    **
**                                                                                      **
** 					 - LCD                                                              **
** 					 - GPIO                                                             **
** 					 - SPI                                                              **
** 					 - I2C                                                              **
** 					 - Front End(Consistes of ADS1258, PCA9535)                         **
** 					 - SAR                                                              **
** 					 - DMA                                                              **
**                                                                                      **
** 	Parameters	:	- ECG_Status - Output : return the status ECG_OK on successful Init **
** 	Return		: Status                                                                **
**--------------------------------------------------------------------------------------*/

ECG_Status ECG_C5505Init()
{
	ECG_Status Status=ECG_OK;
    Uint32 SpiClkRate;
    Uint16 WordLen;
    Uint16 FrameLen;

	/*making Serial Port 0 and Serial Port 1 to mode 10 
	so that serial port 0 pins are configured as GPIO*/
	*PERIPHSEL0_ADDR = *PERIPHSEL0_ADDR & 0xFAFF; //clearing bits 8 and 10
	*PERIPHSEL0_ADDR = *PERIPHSEL0_ADDR | 0x0A00; // setting bits 9 and 11

	/*Disable  interrupt for reading ADC - INT1 of CPU, SAR interrupt for key press */
	*CPU_IER0_ADDR = *CPU_IER0_ADDR & 0x0000; 

	asm("\tBIT (ST1, #ST1_INTM) = #1");	  		/*Disable GLobal Int.Mask*/   

	/*  Call GPIO Init*/
	ECG_GPIO_INIT();

	wait(1000);

	//Setting SPI clock to 1 MHz for ADS1298 initialization
	SpiClkRate = INIT_SPI_FREQ;
	WordLen = 8;
	FrameLen = 1;

	/* Call SPI Init*/
	Status = ECG_SPI_INIT(SpiClkRate, WordLen, FrameLen);
	/*If SPI initialization fails*/
	if(Status != ECG_OK)
	{
		while(1);
	}
	wait(1000);

	/*  Call Front End Init*/
	Status = ECG_FE_INIT();
	if(Status != ECG_OK)
	{
		while(1);
	}
	wait(1000);	

	//Setting SPI clock to 4 MHz for ADS1298 read conversion data
	SpiClkRate = READ_SPI_FREQ;
	WordLen = 24;
	FrameLen = 9;

	Status = ECG_SPI_INIT(SpiClkRate, WordLen, FrameLen);

#if DISP_LCD

	/* Call LCD initialization*/
	ECG_LCD_INIT();

#endif	
	wait(1000);

	
	/* Call UART initialisation*/
	ECG_UART_INIT();

	wait(1000);

	/*   Call SAR Init*/
	Status = ECG_SAR_INIT();
	if(Status != ECG_OK)
	{
		while(1);
	}
	wait(1000);

	/*   Call DMA Init*/
	Status = ECG_DMA_INIT();
	if(Status != ECG_OK)
	{
		while(1);
	}

	wait(1000);
	
	/*	Call Timer Initialization*/
	ECG_TIMER_INIT();	 
//	ECG_GPIO_INIT();
	/*Enable  interrupt for reading ADC - TINT of CPU, SAR interrupt for key press
	and LCD interrupt */
	//*CPU_IER0_ADDR = *CPU_IER0_ADDR | 0x3008;
	//*CPU_IER0_ADDR = *CPU_IER0_ADDR | 0x3018;

	return Status;
}

/*----------------------------------------------------------------------**
** Function Name 	: ECG_UART_INIT()                                   **
** Description 	: 	- The function initializes the UART module so as to **
** 					transmit the Lead Information at a rate of 115200   **
** 					bps                                                 **
** Parameters		: -	None                                            **
** Return			: - void                                            **
**----------------------------------------------------------------------*/


void ECG_UART_INIT()
{

   uartCfg.inputFreq = 100000000; //UART_INPUT_CLK; /* Input Clock to UART */
   uartCfg.opMode = PSP_OPMODE_POLLED; /* Supporting only Polled Mode */
   uartCfg.syncMode = True;
/*	uartCfg.baudRate = 115200;
//	uartCfg.charLen = 8;
	uartCfg.opmode = PSP_OPMODE_POLLED;
	uartCfg.parity = 1;
*/

    instNum = 0;

	*(ioport volatile int *)0x1C02 &= 0x7FFB;

	/* changing parallel port mode to 5 - supports UART */
	*(ioport volatile unsigned *)0x1C00=0x5A00;
	

	wait(500);

	hUart = dda_uartInit(instNum, &uartCfg);
		
	/* To override the driver setting for baud rate so as to set 115.2 kbps */
				
	*(ioport volatile int *)0x1B10 = 0x36;
	*(ioport volatile int *)0x1B12 = 0x00;

	return;
}


/*-----------------------------------------------------------------------**
** 	Function Name : ECG_SAR_INIT()                                       **
** 	Description : 	-  The function initializes the SAR module to select **
** 					   GPAIN1(AIN3) Also set the saradc interrupt for    **
** 					   all key press                                     **
**                                                                       **
** 	Parameters	:	- ECG_Status - Output : return the status ECG_OK on  **
** 											successful Init              **
** 	Return	: Status                                                     **
**-----------------------------------------------------------------------*/

ECG_Status ECG_SAR_INIT()
{
	ECG_Status Status = ECG_OK;
	
  	*SARCTRL = 0x0000;		/* reset the CTRL Reg */
    *SARCTRL = 0x3400; 		/* select AIN3, which is GPAIN1 */
	*SARCLKCTRL = 0x0064;   /* 100/100 = 1MHz */ 
    *SARPINCTRL = 0x7104;   /* Set the AD reference & PIN control values */
    *SARGPOCTRL = 0x0100;   /* Set the PEN interrupt */
 
	return Status;
}
	
/*-----------------------------------------------------------------------**
** 	Function Name : ECG_DMA_INIT                                         **
** 	Description : 	- The function initializes the DMA module to be used **
** 					  for any                                            **
** 					  other peripheral                                   **
** 	Parameters	:	- ECG_Status - Output : return the status ECG_OK on  **
** 											successful Init              **
** 	Return : Status                                                      **
**                                                                       **
**-----------------------------------------------------------------------*/
ECG_Status ECG_DMA_INIT()
{
	ECG_Status Status = ECG_OK;
	
		
	/* DMA hardware initialization */
	DMA_HwInit();

	/* DMA driver initialization */
	DMA_DrvInit();


	return Status;
}

/*----------------------------------------------------------------------------**
** 	Function Name : ECG_SPI_INIT                                              **
** 	Description : 	- 	The function initializes the SPI peripheral for       **
** 						the the Tx and Rx as required for the                 **
** 						ADS1258.The function uses:-                           **
**                                                                            **
** 						- SPI CLK 	 - 7 MHz                                  **
** 						- WordLength - 8                                      **
** 						- csDelay 	 - 0 (other delay not supported in C5505) **
** 	Parameters	:	- None                                                    **
** 	Return	: ECG_Status - Output : return the status ECG_OK on               **
** 											successful Init                   **
**----------------------------------------------------------------------------*/

ECG_Status ECG_SPI_INIT(Uint32 spiClkRate, Uint16 wordLen, Uint16 frameLen )
{
	ECG_Status Status;
	DDA_spiConfig spiConfig;
	Uint16 slaveNo = 0;			/* SlaveNo 0 */
	Uint16 dataDly = 0;			/* delay 0 */
	Uint16 csPol = 0;			/* cspol 0 */
	Uint16 clkPh = 1;			/* clkphase 0 */
	Uint16 clkPol = 0;			/* clkpol 0 */

    Status = SPI_FAILURE;

    /* Driver configuration initialization */
  
    spiConfig.spiCtrlMode   = SPI_MODE;			/* SPI Mode */
    spiConfig.spiMaster     = SPI_MASTER;		/* SPI Master */
    spiConfig.opMode        = SPI_POLLED;		/* SPI polling mode */
    spiConfig.instances     = SPI_SINGLE_INST;	/* Single instance */
    spiConfig.syncMode      = SPI_SYNC;			/* Sync Mode */

	/* Initialize the SPI with */
	Status = SPI_INIT(spiConfig,spiClkRate,frameLen,wordLen,slaveNo,dataDly,csPol,clkPh,clkPol);
	

	return Status;
}

/*---------------------------------------------------------------------**
** 	FunctionName : ECG_GPIO_INIT                                       **
** 	Description : 	- 	This function initializes the GPIO module and  **
** 						set the direction of the GPIO pin as required. **
**                                                                     **
**                    GPIO4 - SPI_START - output pin                   **
**                    GPIO10 - GPIO1- input pin                        **
**                    GPIO0 - DOUT2 - input pin                        **
**                    GPIO11 - CLKSEL - output pin                     **
**                    GPIO1 - RESETB - output  pin                     ** 
**                                                                     **
** 	Parameter : None                                                   **
** 	Returns	:	- ECG_Status - Output : return the status ECG_OK on    **
** 											successful Init            **
**---------------------------------------------------------------------*/
	

void ECG_GPIO_INIT()
{
	/*GPIO4- START as o/p, GPIO11 - CLKSEL as o/p, GPIO1 - RESETB as o/p
	all other pins as inputs*/

	*GPIO_DIR0_ADDR = *GPIO_DIR0_ADDR | 0x0C12;

//	*GPIO_DOUT0_ADDR = *GPIO_DOUT0_ADDR & 0xF7FF;
}
/*---------------------------------------------------------------------**
** 	FunctionName : ECG_FE_INIT                                         **
** 	Description : 	- 	This function initializes all the modules/     **
** 						components in the  ECG Front End.This includes **
**                                                                     **
** 					  	- FE detection                                 **
** 					  	- Leadoff initialization                       **
** 					  	- ADS1258 Initialization                       **
** 	Parameter : None                                                   **
** 	Return	:	- ECG_Status - Output : return the status ECG_OK on    **
** 											successful Init            **
**---------------------------------------------------------------------*/

ECG_Status ECG_FE_INIT()
{
	ECG_Status status = PSP_SOK;

	/*  Call the ADS1258 initialization function*/
	status  = ECG_ADS1298_INIT();
	if(status != ECG_OK)
	while(1);
	
	return status;
}

/*-------------------------------------------------------------------**
**                                                                   **
** 	Function Name : ECG_LCD_INIT                                     **
** 	Description : 	- The function sets the data to LCD registers of **
** 					  C5505. Also initializes the  LCD controller    **
** 					  registers                                      **
** 	Return	:	- None                                               **
** 					                                                 **
**-------------------------------------------------------------------*/

void ECG_LCD_INIT(void)
{
	Uint16  data;
	Uint32 i;
	
	/*Enable module clock - Do this first since some modules might expect clock ON during reset*/
	data = *(ioport volatile unsigned *)0x1C03;  
	data &= 0xfffe;
	*(ioport volatile unsigned *)0x1C03=data;
	/* Give some delay after reset*/
  	for (i=0; i< 100; i++);						
  	/* clear LCD module's interrupt flag in CPU*/
    *(volatile unsigned *) 0x001=0xffff;		
	/*Enable LCD interrupt*/
	*(volatile unsigned *)0x0= *(volatile unsigned *)0x0 | 0x1000;	
	
   /*Enable the Frame Done interrupt*/
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCRL,LCDC_LCDLIDDCRL_DONE_INT_EN,0x1); 
	
	 
	asm("\tBIT (ST1, #ST1_INTM) = #0");	  		/*Enable GLobal Int.Mask*/   


	/* Set AsyncMPU80 mode*/
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCRL,LCDC_LCDLIDDCRL_LIDD_MODE_SEL,0x3); 


	/* Max out strobe, setup and hold timings*/
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGL,LCDC_LCDLIDDCS0CONFIGL_R_SU,0xF); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGL,LCDC_LCDLIDDCS0CONFIGL_R_STROBE,0x3F); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGL,LCDC_LCDLIDDCS0CONFIGL_R_HOLD,0xF); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGL,LCDC_LCDLIDDCS0CONFIGL_TA,0x1); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGM,LCDC_LCDLIDDCS0CONFIGM_W_SU,0x1F); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGM,LCDC_LCDLIDDCS0CONFIGM_W_STROBE,0x3F); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGM,LCDC_LCDLIDDCS0CONFIGM_W_HOLD,0xF); 
	CSL_FINS(CSL_LCDC_REGS->LCDLIDDCS0CONFIGM,LCDC_LCDLIDDCS0CONFIGM_R_SU,0x1); 

	for(i=1;i<5000;i++ )
		asm("	nop");

	/*  Call LCD controller initialization function*/
	Init_LCD();    
}

void ECG_TIMER_INIT()
{

	/*  Timer0 Initialization so as to capture data of nine
	channels at an interval of 2 ms */

	/* TIM0 EN | AutoReload En | Prescale = 0(100/2 = 50MHz) */
	*CPU_TIM0_CTRL = 0x8002; 
	/*  Timer0 period set to 0x186A0 = 2msms */
	/*  Timer0 period set to 0x15F90 = 1.8ms */
//	*CPU_TIM0_PLWR = 0x7530;//.6ms//822D;-0.665ms/*C350 = 1ms*/
// 0.55 ms was tried but not working 0.58 ms is working
// 0x30D4 = .25ms ( 500 SPS)  0x2710 = .2ms  ;  0x2328 = .18ms 0x3633 = .2775(450SPS)

	//*CPU_TIM0_PLWR = 0x86A0;
	*CPU_TIM0_PLWR = 0x86A0;
	*CPU_TIM0_PHWR = 0x0001; 

	*CPU_TIM0_CLWR = 0x0000;
	*CPU_TIM0_CHWR = 0x0000;
	
	/*  Clearing timer Aggregation register*/

	*CPU_TIMINT_AGGR = 0x0000;

	
}
