/**============================================================================
*@file em_protocol.c
*@desc This file contains source for example code to implement host/master
*      protocol to drive SPI/I2C boot using AIS command/data stream for 
*      TMS320C672x devices.
*
*==============================================================================
* Copyright (C) 2004-2005 Texas Instruments Incorporated. 
* Use of this software us controlled by the terms and conditions found in the
* license agreement under which this software has been supplied
*=============================================================================*/

/*===========================================================================*/
/* This software is provided under the following                             */
/* License Agreement:                                                        */
/*===========================================================================*/
/* THIS PROGRAM IS PROVIDED "AS IS". TI MAKES NO WARRANTIES OR               */
/* REPRESENTATIONS, EITHER EXPRESS, IMPLIED OR STATUTORY,                    */
/* INCLUDING ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS              */
/* FOR A PARTICULAR PURPOSE, LACK OF VIRUSES, ACCURACY OR                    */
/* COMPLETENESS OF RESPONSES, RESULTS AND LACK OF NEGLIGENCE.                */
/* TI DISCLAIMS ANY WARRANTY OF TITLE, QUIET ENJOYMENT, QUIET                */
/* POSSESSION, AND NON-INFRINGEMENT OF ANY THIRD PARTY                       */
/* INTELLECTUAL PROPERTY RIGHTS WITH REGARD TO THE PROGRAM OR                */
/* YOUR USE OF THE PROGRAM.                                                  */
/*                                                                           */
/* IN NO EVENT SHALL TI BE LIABLE FOR ANY SPECIAL, INCIDENTAL,               */
/* CONSEQUENTIAL OR INDIRECT DAMAGES, HOWEVER CAUSED, ON ANY                 */
/* THEORY OF LIABILITY AND WHETHER OR NOT TI HAS BEEN ADVISED                */
/* OF THE POSSIBILITY OF SUCH DAMAGES, ARISING IN ANY WAY OUT                */
/* OF THIS AGREEMENT, THE PROGRAM, OR YOUR USE OF THE PROGRAM.               */
/* EXCLUDED DAMAGES INCLUDE, BUT ARE NOT LIMITED TO, COST OF                 */
/* REMOVAL OR REINSTALLATION, COMPUTER TIME, LABOR COSTS, LOSS               */
/* OF GOODWILL, LOSS OF PROFITS, LOSS OF SAVINGS, OR LOSS OF                 */
/* USE OR INTERRUPTION OF BUSINESS. IN NO EVENT WILL TI'S                    */
/* AGGREGATE LIABILITY UNDER THIS AGREEMENT OR ARISING OUT OF                */
/* YOUR USE OF THE PROGRAM EXCEED FIVE HUNDRED DOLLARS                       */
/* (U.S.$500).                                                               */
/*                                                                           */
/* Unless otherwise stated, the Program written and copyrighted              */
/* by Texas Instruments is distributed as "freeware".  You may,              */
/* only under TI's copyright in the Program, use and modify the              */
/* Program without any charge or restriction.  You may                       */
/* distribute to third parties, provided that you transfer a                 */
/* copy of this license to the third party and the third party               */
/* agrees to these terms by its first use of the Program. You                */
/* must reproduce the copyright notice and any other legend of               */
/* ownership on each copy or partial copy, of the Program.                   */
/*                                                                           */
/* You acknowledge and agree that the Program contains                       */
/* copyrighted material, trade secrets and other TI proprietary              */
/* information and is protected by copyright laws,                           */
/* international copyright treaties, and trade secret laws, as               */
/* well as other intellectual property laws.  To protect TI's                */
/* rights in the Program, you agree not to decompile, reverse                */
/* engineer, disassemble or otherwise translate any object code              */
/* versions of the Program to a human-readable form.  You agree              */
/* that in no event will you alter, remove or destroy any                    */
/* copyright notice included in the Program.  TI reserves all                */
/* rights not specifically granted under this license. Except                */
/* as specifically provided herein, nothing in this agreement                */
/* shall be construed as conferring by implication, estoppel,                */
/* or otherwise, upon you, any license or other right under any              */
/* TI patents, copyrights or trade secrets.                                  */
/*                                                                           */
/* You may not use the Program in non-TI devices.                            */
/*===========================================================================*/



#include <stdio.h>
#include <bootProtocol.h>
#include <em_protocol.h>

#define NOPINGS  10

//#define BOOT_TIME_BENCHMARK
//#define NODELAYS

#ifdef BOOT_TIME_BENCHMARK
	extern void timer_start(void);
	#define Timer_Begin(x)   ((x) = *(volatile unsigned int *)0x1980008)
	#define Timer_Acc(x, s)  ((x) = (*(volatile unsigned int *)0x1980008 - (s)))
	
	#define printf REMOVE_printf
	inline void REMOVE_printf(char *string, ...){}	
#endif

#ifdef NODELAYS
	#define sw_delay REMOVE_sw_delay
	inline void REMOVE_sw_delay(int a){}
#endif

#ifdef DSK6713
	#define CPUSPEED    225
#else
	#ifdef DSK6711
		#define CPUSPEED    150
	#else 
		#ifdef BOARD_M2
			#define CPUSPEED    225
		#endif
	#endif
#endif		
	

int delayVal;
int noPings;
unsigned char *address;
  
/* Opens the External Master Device */   
void EM_open(EM_Config *fp,int pllConfigured)
{
	int clockDivider;
	
	if(pllConfigured)
	//	clockDivider = 0x1B;
	 clockDivider = 0xFF;
	else
	   //	clockDivider = 0xE1;
	   clockDivider= 0xFF;
			
	switch(fp->device)
	{
		case EM_DEVICE_SPI_MCBSP0:
			DSM_open(0,fp->data_bits,clockDivider);
			break;
		case EM_DEVICE_SPI_MCBSP1:
		    printf("Opening MCBSP1\n");
			DSM_open(1,fp->data_bits,clockDivider);
			break;
		case EM_DEVICE_I2C_I2C0:
		    printf ("Opening I2C_I2C0\n");
			DIM_open(0,fp->data_bits);
			break;
		case EM_DEVICE_I2C_I2C1:
			DIM_open(1,fp->data_bits);
			break;
		case EM_DEVICE_SPI_SW:
			ESM_open();
			break;
	}
}

/*This function reads the 8/16/32 bit element depending on the protocol */
/*For SPI, MOSI is kept high by transmitting 0xff... during the read operation */
unsigned int EM_readDataWord(EM_Config *fp)
{
	unsigned int ret_val;
	switch(fp->device)
	{
		case EM_DEVICE_SPI_MCBSP0:	
			ret_val = DSM_readWord(0,fp->data_bits);
			break;
		case EM_DEVICE_SPI_MCBSP1:
			ret_val = DSM_readWord(1,fp->data_bits);
			break;
		case EM_DEVICE_I2C_I2C0:
			ret_val = DIM_readWord(0,fp->data_bits);
			break;
		case EM_DEVICE_I2C_I2C1:
			ret_val = DIM_readWord(1,fp->data_bits);
			break;
		case EM_DEVICE_SPI_SW:
			ret_val = ESM_readWord(fp->data_bits);
			break;
	}
	return ret_val;
}

/*This function writes the 8/16/32 bit element depending on the protocol */
/*For SPI the read value returned is ignored */
unsigned int EM_writeDataWord(unsigned int write_value, EM_Config *fp)
{
	unsigned int read_value=0;
	switch(fp->device)
	{
		case EM_DEVICE_SPI_MCBSP0:	
			read_value=DSM_writeWord(write_value,0,fp->data_bits);
			break;
		case EM_DEVICE_SPI_MCBSP1:
			read_value=DSM_writeWord(write_value,1,fp->data_bits);
			break;
		case EM_DEVICE_I2C_I2C0:
			DIM_writeWord(write_value,0,fp->data_bits);
			break;
		case EM_DEVICE_I2C_I2C1:
			DIM_writeWord(write_value,1,fp->data_bits);
			break;
		case EM_DEVICE_SPI_SW:
			ESM_writeWord(write_value,fp->data_bits);
			break;
	}
	return read_value;
}

/*Not used (ignore) */
unsigned int EM_readStartWord(EM_Config *fp)
{
	unsigned int ret_val;
	sw_delay(delayVal);
	ret_val = EM_readDataWord(fp);
	return ret_val;
}

/*This function is responsible for exchanging the start word */
/*It sends the Transmit start word and returns the read value back */
unsigned int EM_xferStartWord(EM_Config *fp)
{
	unsigned int write_value,read_value;
	
	write_value = EM_littleEndianWord(address);
	address+= 4;
	
	switch(fp->device)
	{
		case EM_DEVICE_SPI_MCBSP0:
			sw_delay(delayVal);
			read_value=EM_writeDataWord(write_value,fp);
			break;
		case EM_DEVICE_SPI_MCBSP1:
			sw_delay(delayVal);
			read_value=EM_writeDataWord(write_value,fp);
			break;
		case EM_DEVICE_I2C_I2C0:
			EM_writeDataWord(write_value,fp);
			read_value = EM_readDataWord(fp);
			break;
		case EM_DEVICE_I2C_I2C1:
			EM_writeDataWord(write_value,fp);
			read_value = EM_readDataWord(fp);
			break;
		case EM_DEVICE_SPI_SW:
			sw_delay(delayVal);
			EM_writeDataWord(write_value,fp);
			read_value = EM_readDataWord(fp);
			break;
	}	
		
	return read_value;
}

/* This function does a 32 bit read calling EM_readDataWord the appropriate number of times */ 
unsigned int EM_readWord(EM_Config *fp)
{
	unsigned int ret_val,ret_val1, ret_val2, ret_val3, ret_val4;
	switch(fp->data_bits)
	{
		case 8:
			sw_delay(delayVal);
			ret_val1 = EM_readDataWord(fp);
			sw_delay(delayVal);
			ret_val2 = EM_readDataWord(fp) << 8;
			sw_delay(delayVal);
			ret_val3 = EM_readDataWord(fp) << 16;
			sw_delay(delayVal);
			ret_val4 = EM_readDataWord(fp) << 24;
			ret_val = ret_val1 + ret_val2 + ret_val3 + ret_val4;
			break;
		case 16:
			sw_delay(delayVal);
			ret_val1 = EM_readDataWord(fp);
			sw_delay(delayVal);
			ret_val2 = EM_readDataWord(fp)<<16;
			ret_val = ret_val1 + ret_val2;
			break;
		case 32:
			sw_delay(delayVal);
			ret_val = EM_readDataWord(fp);
			break;
	}
	return ret_val;
}

/* This function does a 32 bit write calling EM_writeDataWord the appropriate number of times */ 
void EM_writeWord(unsigned int write_value, EM_Config *fp)
{
	switch(fp->data_bits)
	{
		case 8:
			sw_delay(delayVal);
			EM_writeDataWord( ( write_value & 0xFF ), fp);
			sw_delay(delayVal);
			EM_writeDataWord( ( ( write_value & 0xFF00 ) >> 8 ), fp);
			sw_delay(delayVal);
			EM_writeDataWord( ( ( write_value & 0xFF0000 ) >> 16 ), fp);
			sw_delay(delayVal);
			EM_writeDataWord( ( ( write_value & 0xFF000000 ) >> 24 ), fp);
			break;
		case 16:
			sw_delay(delayVal);
			EM_writeDataWord( ( write_value & 0xFFFF ), fp);
			sw_delay(delayVal);
			EM_writeDataWord( ( ( write_value & 0xFFFF0000 ) >> 16 ), fp);
			break;
		case 32:
			sw_delay(delayVal);
			EM_writeDataWord( write_value, fp);
			break;
	}		
}

/* This function returns the data from an AIS script in little endian format */
unsigned int EM_littleEndianWord(unsigned char *address)
{
	unsigned int a,b,c,d;
	a=address[0];
	b=address[1];
	c=address[2];
	d=address[3];
	return (a | b<<8 | c<<16 | d<<24);
}

/* This function returns 1 if it recieves a valid "Recieve Start Word" */
/* or a 0 if supplied with some other value */
unsigned int EM_isStartWord(unsigned int read_word, EM_Config *fp)
{
	switch(fp->data_bits)
	{
		case 8:
			if ( read_word == BOOT_PROTOCOL_RECV_START8)
				return 1;
			else
				return 0;
		case 16:
			if ( read_word == BOOT_PROTOCOL_RECV_START16)
				return 1;
			else
				return 0;
		case 32:
			if ( read_word == BOOT_PROTOCOL_RECV_START32)
				return 1;
			else
				return 0;
	}
	return 0;
}

/* Close the External Master Device */   
void EM_close(EM_Config *fp)
{
	switch(fp->device)
	{
		case EM_DEVICE_SPI_MCBSP0:	
			DSM_close(0);
			break;
		case EM_DEVICE_SPI_MCBSP1:
			DSM_close(1);
			break;
		case EM_DEVICE_I2C_I2C0:
			DIM_close(0);
			break;
		case EM_DEVICE_I2C_I2C1:
			DIM_close(1);
			break;
	}
}

/* AIS protocol implementing API */
void EM_protocolImplement(int state, EM_Config *fp)
{
	unsigned int read_value,i;
	unsigned int write_value,sectionSize,sectionAddr;
	unsigned int sectionPatternType,sectionFillPattern;
	int loopExit = 0;
	int pllConfigured = 0,bootTableWriteStart;
#ifdef BOOT_TIME_BENCHMARK	
	unsigned int startTimerValue,timerValueIncrease,timerIncreaseAfterBootTable=0;
#endif	
	
	address = (unsigned char *)fp->baseAddress;

#ifdef BOOT_TIME_BENCHMARK
	timer_start();
	Timer_Begin(startTimerValue);
#endif
		
	while(1)
	{
		/*Inter word Delay requirements vary before/after PLL configuration*/
		/* Before PLL setup for SPI: */
		/* A data rate of 500 Kbps with 0.5 msec interword delay is recommended */
		/* After PLL setup for SPI: */
		/* A data rate of 4 Mbps with no interword delay is recommended */
		/* Before/After PLL setup for I2C: */
		/* A data rate of 300 Kbps with no interword delay is recommended */		
		if(pllConfigured)
		{
			delayVal = 0;
			EM_close(fp);
			EM_open(fp,pllConfigured);
		}	
		else
			delayVal = 5919; //0.5 msec for 225 MHz (code in IRAM)
			
		switch(state)
		{
			/*This state implements Start Word Synchronization and Ping Opcode*/
			/*This is the first mandatory step required for DSP serial slave bootmodes*/	
			case EM_STATE_XCHANGE_STARTWORD:
				/*Keep sending Transmit Start Word till Recieve Start Word obtained*/
				read_value=EM_xferStartWord(fp);
				if(EM_isStartWord(read_value,fp))
				{
					state = EM_STATE_XCHANGE_SYNCWORD;
					
					/*Write Transmit Ping Device Opcode */
					write_value = BOOT_PROTOCOL_XMT_SYNC | BOOT_PROTOCOL_PING_DEVICE; 
					EM_writeWord(write_value, fp);
					read_value = EM_readWord(fp);
					
					/*Go back to Start Word synchronization if the next word read is not Recieve Ping Opcode*/			
					if(read_value!= (BOOT_PROTOCOL_RECV_SYNC | BOOT_PROTOCOL_PING_DEVICE) )
					{
						state = EM_STATE_XCHANGE_STARTWORD;
						break;				
					}
					else
					{
						/*Write the Ping Count*/
						EM_writeWord(NOPINGS, fp);
						read_value = EM_readWord(fp);
						
						/*Read the Ping Count back*/
						if(read_value != NOPINGS)
						{
							/*If error go back to Start Word Synchronization*/
							state = EM_STATE_XCHANGE_STARTWORD;
							break;
						}
												
						for(i=1;i<=NOPINGS;i++)
						{
							/*Send the ping count*/
							EM_writeWord(i, fp);
							
							/*Receive the ping count back*/
							read_value = EM_readWord(fp);
							if(read_value!=i)
							{
								/*If error go back to Start Word Synchronization*/
								state = EM_STATE_XCHANGE_STARTWORD;
								break;
							}
						}
					}
				}				
				else
				{
					/*If receive Start Word not obtained go back to Start Word Synchronization*/
					state = EM_STATE_XCHANGE_STARTWORD;
	                address = (unsigned char *)fp->baseAddress;
				}	
				break;

			/* This state writes the Opcode with the Transmit Sync */
			/* 1 msec delay is recommended during data trasmission for Opcode synchronization*/
			/* This implementation puts delay in the form of calls to sw_delay() */				
			/* The user should substitue sw_delay with appropriate delay routines */
			case EM_STATE_XCHANGE_SYNCWORD:
				/*Recommended way for Opcode Synchronization*/
			
				write_value = EM_littleEndianWord(address);
				if((write_value & 0xFFFFFF00) != BOOT_PROTOCOL_XMT_SYNC)
				{
					printf("Wrong Format Stored in the Master Device\n");
					return;
				}
				sw_delay(11838); // 1 msec delay
				
				/*Send the Transmit X Opcode and ignore received value (if SPI) */
				EM_writeWord(write_value, fp);
				
				sw_delay(11838); // 1 msec delay
				
				state = EM_STATE_PROCESS_SYNCWORD;
				break;

			/* This state completes the process of opcode synchronization */
			/* An attempt is made to receive the Opcode with the Recieve sync */
			/* In case of an error in read the control is returned back to EM_STATE_XCHANGE_SYNCWORD */			
			case EM_STATE_PROCESS_SYNCWORD:
			
				/*If SPI keep MOSI high (by sending 0xff...) and read the recieved value*/
				/*If recieved value not Recieve X Opocde go back to the previous state*/
				/*For I2C simply attempt reading the Recieve Opcode X */ 	
				read_value = EM_readWord(fp);
				
				sw_delay(11838); // 1 msec delay
								
				if((read_value & 0xFFFFFF00) != BOOT_PROTOCOL_RECV_SYNC)
				{
					state = EM_STATE_XCHANGE_SYNCWORD;
					break;
				}
				if((read_value & 0xFF) != (write_value & 0xFF))
				{
					state = EM_STATE_XCHANGE_SYNCWORD;
					break;
				}
				
				if(bootTableWriteStart && ((read_value & 0xFF)!= BOOT_PROTOCOL_BOOT_TABLE) )
				{
					#ifdef BOOT_TIME_BENCHMARK
						if(!timerIncreaseAfterBootTable)
							Timer_Acc(timerIncreaseAfterBootTable, startTimerValue);
					#endif
					pllConfigured = 1;
				}
				
				/* Jump to the appropriate state depending on the opcode in AIS */
				switch(read_value & 0xFF)
				{
					case BOOT_PROTOCOL_SECTION_LOAD:
						state = EM_STATE_PROCESS_OPCODE_SECTION_LOAD;
						break;
						
					case BOOT_PROTOCOL_JUMP:
						state = EM_STATE_PROCESS_OPCODE_JUMP;
						break;
						
					case BOOT_PROTOCOL_JUMP_CLOSE:
						state = EM_STATE_PROCESS_OPCODE_JUMP_CLOSE;
						break;
						
					case BOOT_PROTOCOL_DISABLE_CRC:
						address+=4;
						state = EM_STATE_XCHANGE_SYNCWORD;
						break;
						
					case BOOT_PROTOCOL_REQUEST_CRC:
						state = EM_STATE_PROCESS_OPCODE_REQUEST_CRC;
						break;
						
					case BOOT_PROTOCOL_ENABLE_CRC:
						address+=4;
						state = EM_STATE_XCHANGE_SYNCWORD;
						break;
						
					case BOOT_PROTOCOL_BOOT_TABLE:
						state = EM_STATE_PROCESS_OPCODE_BOOT_TABLE;
						bootTableWriteStart = 1;
						break;
						
					case BOOT_PROTOCOL_COMPSECTION_LOAD:
						state = EM_STATE_PROCESS_OPCODE_COMPSECTION_LOAD;
						break;
						
					case BOOT_PROTOCOL_SECTION_FILL:
						state = EM_STATE_PROCESS_OPCODE_SECTION_FILL;
						break;	
							
                    case BOOT_PROTOCOL_PING_DEVICE:
					    state = EM_STATE_PROCESS_OPCODE_PING_DEVICE;
						break;
					default:
						state = EM_STATE_XCHANGE_SYNCWORD;
						break;
				}
				break;

			/* This state implements Section Load Opcode */
			/* Section Load is a fundamental way to load code/data sections into the DSP */
			case EM_STATE_PROCESS_OPCODE_SECTION_LOAD:
				address+=4;
				/* Write the Section Address */
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value, fp);
				sectionAddr = write_value;
				address+=4;
	
				/* Write the Section Size */			
				sectionSize = EM_littleEndianWord(address);
				address+=4;
				EM_writeWord(sectionSize, fp);
				
				/* Write the Section Data */		
				for(i=0;i<sectionSize/4;i++)
				{
					write_value = EM_littleEndianWord(address);
					address+=4;
					EM_writeWord(write_value,fp);
				}
				if(sectionSize % 4)
				{
					write_value = EM_littleEndianWord(address);
					address += (sectionSize % 4);
					EM_writeWord(write_value,fp);
				}
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;

			/* This state implements CRC Request Opcode implementation */
			/* If the DSP supplied CRC value does not match with the one */
			/* present in AIS, Startover Opcode is sent, to signal the DSP to reset */
			/* its computed CRC value */
			case EM_STATE_PROCESS_OPCODE_REQUEST_CRC:
				read_value = EM_readWord(fp);
				address+=4;
				if(read_value != EM_littleEndianWord(address))
				{
					address+=4;
					address += EM_littleEndianWord(address);
					address+=4;
					state = EM_STATE_SEND_OPCODE_STARTOVER;	/* Start Over */
					printf("CRC Mismatch for Section Address: 0x%x : 0x%x\n",sectionAddr, read_value);
				}
				else
				{
					address+=8;
					state = EM_STATE_XCHANGE_SYNCWORD;
					printf("CRC Match for Section Address: 0x%x :)\n",sectionAddr);
				}
				break;
			
			/*This state sends the Transmit Start Over Opcode */
			case EM_STATE_SEND_OPCODE_STARTOVER:
				write_value = BOOT_PROTOCOL_XMT_SYNC | BOOT_PROTOCOL_STARTOVER;
				
				sw_delay(11838); // 1 msec delay
								
				/*Send Startover Opcode if CRC error detected*/
				EM_writeWord(write_value,fp);
				
				sw_delay(11838); // 1 msec delay
				
				state = EM_STATE_XCHANGE_OPCODE_STARTOVER;
				break;

			/* This state completes the process of Start Over opcode synchronization */
			/* An attempt is made to receive the Start Over Opcode with the Recieve sync */
			/* In case of an error in read the control is returned back to EM_STATE_SEND_OPCODE_STARTOVER */								
			case EM_STATE_XCHANGE_OPCODE_STARTOVER:
				read_value = EM_readWord(fp);
				
				sw_delay(11838); // 1 msec delay				
				
				if(read_value != (BOOT_PROTOCOL_RECV_SYNC | BOOT_PROTOCOL_STARTOVER))
					state = EM_STATE_SEND_OPCODE_STARTOVER;
				else
					state = EM_STATE_XCHANGE_SYNCWORD;
				break;
				
			/* This state implements Jump Opcode. */
			/* Jump Opcode is typically used to execute a portion of code (secondary bootloader) */
			/* downloaded through Section Load Opcodes. This enables running the seondary bootloader */
			/* and upon completion returning the control back to primary bootloader  */ 
			case EM_STATE_PROCESS_OPCODE_JUMP:
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
				pllConfigured = 1;
				address+=4;
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;
			
			/* This state implements Jump Close Opcode */
			/* Jump Close opcode is used at the end of application download to begin execution of the */
			/* downloaded application */
			case EM_STATE_PROCESS_OPCODE_JUMP_CLOSE:
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
				loopExit = 1;
				break;
			
			/* This state implements Boot Table Opcode */
			/* Boot table entries are typically used to do initialize the PLL and EMIF */
			/* at the start of the bootloading process */
			case EM_STATE_PROCESS_OPCODE_BOOT_TABLE:
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
				
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
				
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
				
				address+=4;
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value,fp);
	
				address+=4;
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;
			
			/* This state implements Compressed Section Load Opcode */
			/* Compressed Section Load opcode is used as a replacement to section load, where there */
			/* are restrictions to download size */
			/* The Compressed Section Load opcode is not supported by default. It has to be implemented */
			/* through the download of secondary bootloader. For more details contact TI support */
			case EM_STATE_PROCESS_OPCODE_COMPSECTION_LOAD:
				/*Write New Section*/
				address+=4;				
				write_value = EM_littleEndianWord(address);			
				EM_writeWord(write_value, fp);
				address+=4;
				
				if( (*(address - 4)) & 1 )
				{
					/* Write the Section Address */
					write_value = EM_littleEndianWord(address);
					sectionAddr = write_value;
					//printf("Write : 0x%x\n", write_value);
					EM_writeWord(write_value, fp);
					address+=4;
				}
				
				/* Write the Section Size */			
				sectionSize = EM_littleEndianWord(address);
				address+=4;
				EM_writeWord(sectionSize, fp);
				
				/* Write the compressed section data*/
				for(i=0;i<sectionSize/4;i++)
				{
					write_value = EM_littleEndianWord(address);
					address+=4;
					EM_writeWord(write_value,fp);
				}
				
				if(sectionSize % 4)
				{
					write_value = EM_littleEndianWord(address);
					address += (sectionSize % 4);
					EM_writeWord(write_value,fp);
				}
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;
			
			/* This state implements Section Fill Opcode */
			/* Section Fill opcode is used as a replacement to Section Load when a 8/16/32 bit pattern */
			/* is to be filled into a particular section */
			case EM_STATE_PROCESS_OPCODE_SECTION_FILL:
				address+=4;
				/* Write the Section Address */
				write_value = EM_littleEndianWord(address);
				EM_writeWord(write_value, fp);
				sectionAddr = write_value;
				
				address+=4;
				/* Write the Section Size */			
				sectionSize = EM_littleEndianWord(address);
				EM_writeWord(sectionSize, fp);
	
				address+=4;
				/* Write the Section Pattern Type */			
				sectionPatternType = EM_littleEndianWord(address);
				EM_writeWord(sectionPatternType, fp);
	
				address+=4;
				/* Write the Section Fill Pattern */			
				sectionFillPattern = EM_littleEndianWord(address);
				EM_writeWord(sectionFillPattern, fp);
	
				address+=4;
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;

			case EM_STATE_PROCESS_OPCODE_PING_DEVICE:
			    address+=4;
				noPings = EM_littleEndianWord(address);
				EM_writeWord(noPings, fp);
				address += 4;

				read_value = EM_readWord(fp);
						
						/*Read the Ping Count back*/
				if(read_value != noPings)
				{
				/*If error go back to Start Word Synchronization*/
					state = EM_STATE_XCHANGE_STARTWORD;
					address = (unsigned char*)fp->baseAddress;
					break;
				}
												
				for(i=1;i<=noPings;i++)
				{
					/*Send the ping count*/
				    write_value = EM_littleEndianWord(address);
					EM_writeWord(i, fp);
						
					/*Receive the ping count back*/
					read_value = EM_readWord(fp);
					if(read_value!=i)
					{
					/*If error go back to Start Word Synchronization*/
						state = EM_STATE_XCHANGE_STARTWORD;
					    address = (unsigned char*)fp->baseAddress;
						break;
					}

					address += 4;
				}
				state = EM_STATE_XCHANGE_SYNCWORD;
				break;
		}
		
		/*Check for bootloading over status */
		if(loopExit == 1)
		{
			printf("External Master Success!!\n");
			address = (unsigned char *)fp->baseAddress;
			loopExit=0;
			state = EM_STATE_XCHANGE_STARTWORD;
			break;
		}
	}

#ifdef BOOT_TIME_BENCHMARK
	#undef printf
	Timer_Acc(timerValueIncrease, startTimerValue);
	timerValueIncrease = (timerValueIncrease*4)/(CPUSPEED*1000);
	timerIncreaseAfterBootTable = (timerIncreaseAfterBootTable*4)/(CPUSPEED*1000);
	printf("Bootup time till Boottable entries: %d msec\n",(short)timerIncreaseAfterBootTable);
	printf("Total Bootup time: %d msec\n",(short)timerValueIncrease);
#endif	
	
}

